import { useEffect } from 'react';
import { useQuery } from '@tanstack/react-query';
import { useLocation } from 'wouter';
import { HeaderTag } from '@shared/schema';

// Extend window interface for analytics globals
declare global {
  interface Window {
    Shown?: {
      pageView: () => void;
    };
    gtag?: (...args: any[]) => void;
    dataLayer?: any[];
  }
}

interface SEOHeadProps {
  title?: string;
  description?: string;
  keywords?: string;
  ogTitle?: string;
  ogDescription?: string;
  ogImage?: string;
  canonicalUrl?: string;
  structuredData?: object;
  noindex?: boolean;
}

export function SEOHead({
  title = "LoveMedia Foundation - South Africa's Largest Community Platform",
  description = "Join 12,000+ members in South Africa's largest WhatsApp community platform. Access exclusive discounts, local business directory, community news and alerts across all 9 provinces.",
  keywords = "South Africa community, WhatsApp groups, business directory, discounts, savings, non-profit, community platform, local news, community alerts",
  ogTitle,
  ogDescription,
  ogImage = "/og-image.jpg",
  canonicalUrl,
  structuredData,
  noindex = false
}: SEOHeadProps) {
  
  // Track current location for route change analytics
  const [location] = useLocation();
  
  // Fetch header tags for injection
  const { data: headerTags } = useQuery<HeaderTag[]>({
    queryKey: ['/api/header-tags/active']
  });
  
  useEffect(() => {
    // Update document title
    document.title = title;
    
    // Update or create meta tags
    const updateMeta = (name: string, content: string, property?: boolean) => {
      const attribute = property ? 'property' : 'name';
      let meta = document.querySelector(`meta[${attribute}="${name}"]`);
      
      if (!meta) {
        meta = document.createElement('meta');
        meta.setAttribute(attribute, name);
        document.head.appendChild(meta);
      }
      
      meta.setAttribute('content', content);
    };
    
    // Basic meta tags
    updateMeta('description', description);
    updateMeta('keywords', keywords);
    updateMeta('robots', noindex ? 'noindex, nofollow' : 'index, follow');
    
    // Open Graph tags
    updateMeta('og:title', ogTitle || title, true);
    updateMeta('og:description', ogDescription || description, true);
    updateMeta('og:image', ogImage, true);
    updateMeta('og:type', 'website', true);
    updateMeta('og:locale', 'en_ZA', true);
    updateMeta('og:site_name', 'LoveMedia Foundation', true);
    
    // Twitter tags
    updateMeta('twitter:card', 'summary_large_image', true);
    updateMeta('twitter:title', ogTitle || title, true);
    updateMeta('twitter:description', ogDescription || description, true);
    updateMeta('twitter:image', ogImage, true);
    
    // Canonical URL
    if (canonicalUrl) {
      let canonical = document.querySelector('link[rel="canonical"]');
      if (!canonical) {
        canonical = document.createElement('link');
        canonical.setAttribute('rel', 'canonical');
        document.head.appendChild(canonical);
      }
      canonical.setAttribute('href', canonicalUrl);
    }
    
    // Structured Data
    if (structuredData) {
      let script = document.querySelector('script[type="application/ld+json"]#dynamic-structured-data');
      if (!script) {
        script = document.createElement('script');
        script.setAttribute('type', 'application/ld+json');
        script.setAttribute('id', 'dynamic-structured-data');
        document.head.appendChild(script);
      }
      script.textContent = JSON.stringify(structuredData);
    }
    
  }, [title, description, keywords, ogTitle, ogDescription, ogImage, canonicalUrl, structuredData, noindex]);

  // Inject header tags into document head with proper timing for analytics
  useEffect(() => {
    if (!headerTags) return;
    
    // Remove existing injected header tags
    const existingTags = document.querySelectorAll('[data-header-tag-injection="true"]');
    existingTags.forEach(tag => tag.remove());
    
    // Inject active header tags sorted by priority
    const activeTags = headerTags
      .filter(tag => tag.isActive)
      .sort((a, b) => a.priority - b.priority);
      
    activeTags.forEach(tag => {
      try {
        // Create a temporary div to parse the HTML
        const tempDiv = document.createElement('div');
        tempDiv.innerHTML = tag.code;
        
        // Process each element in the parsed HTML
        const elements = tempDiv.children;
        for (let i = 0; i < elements.length; i++) {
          const originalElement = elements[i];
          
          if (originalElement.tagName.toLowerCase() === 'script') {
            // Handle script tags specially for proper analytics loading
            const script = document.createElement('script');
            
            // Copy all attributes from the original script
            Array.from(originalElement.attributes).forEach(attr => {
              // Keep defer attribute for Shown pixel (it requires it), but remove for Google Analytics
              if (attr.name === 'defer' && tag.name.toLowerCase().includes('analytics')) {
                // Skip defer only for Google Analytics, keep it for Shown
                return;
              }
              script.setAttribute(attr.name, attr.value);
            });
            
            // For Google Analytics, ensure immediate loading, but preserve Shown's defer behavior
            if (originalElement.getAttribute('src') && tag.name.toLowerCase().includes('analytics')) {
              // Force immediate loading only for Google Analytics
              script.async = true;
            }
            
            // Copy script content if it's an inline script
            if (originalElement.textContent && !originalElement.getAttribute('src')) {
              script.textContent = originalElement.textContent;
            }
            
            // Add tracking attributes
            script.setAttribute('data-header-tag-injection', 'true');
            script.setAttribute('data-tag-id', tag.id.toString());
            script.setAttribute('data-tag-name', tag.name);
            
            // Add onload handler for analytics scripts
            if (tag.name.toLowerCase().includes('shown') || tag.name.toLowerCase().includes('analytics')) {
              script.onload = () => {
                console.log(`🚀 Analytics script loaded and ready: ${tag.name}`);
                
                // Only trigger initial page view for Google Analytics - Shown handles this automatically
                if (tag.name.toLowerCase().includes('analytics') && window.gtag) {
                  setTimeout(() => {
                    window.gtag?.('config', 'AW-17369528004', {
                      page_title: document.title,
                      page_location: window.location.href
                    });
                  }, 100);
                }
                
                // Shown pixel fires automatically when loaded with defer - no manual trigger needed
                if (tag.name.toLowerCase().includes('shown')) {
                  console.log('🔥 Shown pixel loaded - tracking will happen automatically');
                }
              };
            }
            
            document.head.appendChild(script);
          } else {
            // Handle non-script elements as before
            const element = originalElement.cloneNode(true) as Element;
            element.setAttribute('data-header-tag-injection', 'true');
            element.setAttribute('data-tag-id', tag.id.toString());
            element.setAttribute('data-tag-name', tag.name);
            document.head.appendChild(element);
          }
        }
        
        // If no child elements, treat as script content
        if (elements.length === 0 && tag.code.trim()) {
          const script = document.createElement('script');
          script.textContent = tag.code;
          script.setAttribute('data-header-tag-injection', 'true');
          script.setAttribute('data-tag-id', tag.id.toString());
          script.setAttribute('data-tag-name', tag.name);
          document.head.appendChild(script);
        }
        
        console.log(`✅ Successfully injected header tag: ${tag.name}`);
      } catch (error) {
        console.warn(`❌ Failed to inject header tag "${tag.name}":`, error);
      }
    });
  }, [headerTags]);

  // Track route changes for Google Analytics (SPA page views)
  // Shown handles its own route tracking automatically
  useEffect(() => {
    // Delay to ensure analytics scripts are loaded
    const timeoutId = setTimeout(() => {
      // Only trigger manual page views for Google Analytics
      // Shown pixel handles route changes automatically with its defer script
      if (window.gtag && typeof window.gtag === 'function') {
        try {
          window.gtag('config', 'AW-17369528004', {
            page_title: document.title,
            page_location: window.location.href,
            page_path: location
          });
          console.log(`📊 Google Analytics page view tracked for: ${location}`);
        } catch (error) {
          console.warn('Google Analytics gtag error:', error);
        }

        // Also trigger general gtag event
        try {
          window.gtag('event', 'page_view', {
            page_title: document.title,
            page_location: window.location.href,
            page_path: location
          });
        } catch (error) {
          console.warn('Google Analytics event error:', error);
        }
      }
    }, 500); // Wait 500ms to ensure scripts are loaded

    return () => clearTimeout(timeoutId);
  }, [location]); // Fire on every route change
  
  return null;
}

// SEO data for each page
export const seoData = {
  home: {
    title: "LoveMedia Foundation - South Africa's Largest Community Platform | 4G/5G Data Deals & Web Hosting",
    description: "Join 12,000+ members in South Africa's largest WhatsApp community. Get uncapped 4G/5G data from R299/month and premium web hosting from R129/month. Non-profit negotiated rates with R200-R100 donated to charity per sale.",
    keywords: "South Africa community, WhatsApp groups, 4G 5G uncapped data deals, web hosting south africa, cpanel hosting, non-profit discounts, community platform, charity donations, business directory, uncapped data sim, website hosting, local news, community alerts, Eastern Cape, Free State, Gauteng, KwaZulu-Natal, Limpopo, Mpumalanga, Northern Cape, North West, Western Cape",
    canonicalUrl: "https://lovemedia.foundation/",
    structuredData: {
      "@context": "https://schema.org",
      "@type": "Organization",
      "name": "LoveMedia Foundation",
      "description": "South Africa's largest community platform connecting over 12,000 members through WhatsApp groups, business directory, and exclusive discounts",
      "url": "https://lovemedia.foundation",
      "logo": "https://lovemedia.foundation/favicon.svg",
      "aggregateRating": {
        "@type": "AggregateRating",
        "ratingValue": "4.8",
        "ratingCount": "1247"
      }
    }
  },
  
  community: {
    title: "Community WhatsApp Groups - LoveMedia Foundation | Join 12,000+ Members",
    description: "Connect with 12,000+ South Africans across 6 WhatsApp groups. Get community alerts, local discussions, accommodation info, and exclusive discounts. Join South Africa's most active community platform.",
    keywords: "WhatsApp groups South Africa, community groups, local discussions, community alerts, accommodation groups, Christian community, South Africa discounts",
    canonicalUrl: "https://lovemedia.foundation/community",
    structuredData: {
      "@context": "https://schema.org",
      "@type": "SocialMediaPosting",
      "headline": "Join South Africa's Largest WhatsApp Community Groups",
      "description": "Connect with 12,000+ members across 6 specialized WhatsApp groups for discussions, alerts, and exclusive discounts",
      "author": {
        "@type": "Organization",
        "name": "LoveMedia Foundation"
      }
    }
  },
  
  businessDirectory: {
    title: "Business Directory - LoveMedia Foundation | South African Local Businesses",
    description: "Discover verified local businesses across South Africa. Browse by category and location. Connect with trusted service providers in your community. Business listings with reviews and contact details.",
    keywords: "South Africa business directory, local businesses, business listings, service providers, company directory, verified businesses",
    canonicalUrl: "https://lovemedia.foundation/business-directory",
    structuredData: {
      "@context": "https://schema.org",
      "@type": "ItemList",
      "name": "South Africa Business Directory",
      "description": "Comprehensive directory of verified local businesses across South Africa",
      "numberOfItems": "500+"
    }
  },
  
  auth: {
    title: "Join LoveMedia Foundation - Register for South Africa's Largest Community Platform",
    description: "Create your account to join 12,000+ members in South Africa's premier community platform. Access exclusive WhatsApp groups, business directory, and member discounts.",
    keywords: "register LoveMedia Foundation, join community, South Africa membership, WhatsApp groups registration, community platform signup",
    canonicalUrl: "https://lovemedia.foundation/auth",
    noindex: true
  },
  
  terms: {
    title: "Terms & Conditions - LoveMedia Foundation | Community Platform Guidelines",
    description: "Read the terms and conditions for LoveMedia Foundation's community platform. Understand your rights and responsibilities as a member of South Africa's largest WhatsApp community.",
    keywords: "LoveMedia Foundation terms, community guidelines, terms of service, platform rules, member agreement",
    canonicalUrl: "https://lovemedia.foundation/terms"
  },
  
  dataDeals: {
    title: "Uncapped 4G/5G Data Deals | Axxess Uncapped | Mweb Uncapped",
    description: "Get the best uncapped 4G/5G data SIM deals in South Africa. Compare Axxess uncapped, Mweb uncapped, and premium data packages. No contracts, unlimited internet, priority streaming & gaming protocols. R200 donated to charity per order.",
    keywords: "uncapped 4G 5G data deals, Axxess uncapped, Mweb uncapped, uncapped data SIM, unlimited 4G data South Africa, uncapped internet deals, no contract data deals, unlimited mobile data, uncapped LTE packages, 4G 5G SIM cards, uncapped data bundles, unlimited internet South Africa, mobile broadband uncapped, wireless internet deals, data only SIM deals, uncapped mobile internet, 4G uncapped packages, 5G uncapped deals, OpenWeb uncapped data, Telkom uncapped, MTN uncapped, Vodacom uncapped, Cell C uncapped, Rain uncapped, fiber alternative uncapped, unlimited streaming data, gaming data packages",
    canonicalUrl: "https://lovemedia.foundation/data-deals",
    structuredData: {
      "@context": "https://schema.org",
      "@type": "Product",
      "name": "Uncapped 4G/5G Data SIM Deals",
      "description": "Premium uncapped mobile data packages including Axxess uncapped and Mweb uncapped options with no contracts and unlimited internet access",
      "brand": [
        {
          "@type": "Brand",
          "name": "Axxess"
        },
        {
          "@type": "Brand", 
          "name": "Mweb"
        },
        {
          "@type": "Brand",
          "name": "OpenWeb"
        }
      ],
      "category": "Mobile Data Plans",
      "offers": {
        "@type": "AggregateOffer",
        "lowPrice": "299",
        "highPrice": "1199",
        "priceCurrency": "ZAR",
        "availability": "https://schema.org/InStock",
        "seller": {
          "@type": "Organization",
          "name": "LoveMedia Foundation"
        }
      },
      "additionalProperty": [
        {
          "@type": "PropertyValue",
          "name": "Data Type",
          "value": "Uncapped 4G/5G"
        },
        {
          "@type": "PropertyValue", 
          "name": "Contract",
          "value": "No Contract Required"
        },
        {
          "@type": "PropertyValue",
          "name": "Country",
          "value": "South Africa"
        }
      ]
    }
  },
  
  hosting: {
    title: "Premium Web Hosting South Africa - Fast, Reliable CPanel Hosting | LoveMedia Foundation",
    description: "Professional web hosting in South Africa with CPanel, free .co.za domains, unlimited traffic, and instant activation. Premium hosting plans from R129/month with free SSL certificates and 24/7 support.",
    keywords: "web hosting south africa, cpanel hosting, cheap hosting south africa, domain hosting, website hosting, email hosting, ssl certificate, wordpress hosting, php hosting, mysql hosting, shared hosting, business hosting, professional hosting, instant activation, free domain, unlimited traffic",
    canonicalUrl: "https://lovemedia.foundation/hosting",
    structuredData: {
      "@context": "https://schema.org",
      "@type": "Service",
      "name": "Premium Web Hosting South Africa",
      "description": "Professional web hosting services in South Africa with CPanel control panel, free .co.za domains, and instant activation",
      "provider": {
        "@type": "Organization",
        "name": "LoveMedia Foundation"
      },
      "areaServed": "South Africa",
      "hasOfferCatalog": {
        "@type": "OfferCatalog",
        "name": "Web Hosting Plans",
        "itemListElement": [
          {
            "@type": "Offer",
            "itemOffered": {
              "@type": "Service",
              "name": "One Lite Hosting"
            },
            "price": "129",
            "priceCurrency": "ZAR",
            "priceSpecification": {
              "@type": "UnitPriceSpecification",
              "price": "129",
              "priceCurrency": "ZAR",
              "billingPeriod": "P1M"
            }
          },
          {
            "@type": "Offer", 
            "itemOffered": {
              "@type": "Service",
              "name": "One Hosting"
            },
            "price": "149",
            "priceCurrency": "ZAR",
            "priceSpecification": {
              "@type": "UnitPriceSpecification",
              "price": "149", 
              "priceCurrency": "ZAR",
              "billingPeriod": "P1M"
            }
          },
          {
            "@type": "Offer",
            "itemOffered": {
              "@type": "Service", 
              "name": "One Pro Hosting"
            },
            "price": "299",
            "priceCurrency": "ZAR",
            "priceSpecification": {
              "@type": "UnitPriceSpecification",
              "price": "299",
              "priceCurrency": "ZAR", 
              "billingPeriod": "P1M"
            }
          },
          {
            "@type": "Offer",
            "itemOffered": {
              "@type": "Service",
              "name": "One Prime Hosting"
            },
            "price": "399",
            "priceCurrency": "ZAR",
            "priceSpecification": {
              "@type": "UnitPriceSpecification",
              "price": "399",
              "priceCurrency": "ZAR",
              "billingPeriod": "P1M"
            }
          },
          {
            "@type": "Offer",
            "itemOffered": {
              "@type": "Service",
              "name": "One VIP Hosting" 
            },
            "price": "499",
            "priceCurrency": "ZAR",
            "priceSpecification": {
              "@type": "UnitPriceSpecification",
              "price": "499",
              "priceCurrency": "ZAR",
              "billingPeriod": "P1M"
            }
          }
        ]
      },
      "additionalProperty": [
        {
          "@type": "PropertyValue",
          "name": "Control Panel",
          "value": "CPanel"
        },
        {
          "@type": "PropertyValue",
          "name": "Free Domain",
          "value": ".co.za domain included"
        },
        {
          "@type": "PropertyValue",
          "name": "SSL Certificate",
          "value": "Free SSL included"
        },
        {
          "@type": "PropertyValue",
          "name": "Setup Fee",
          "value": "No setup fee"
        },
        {
          "@type": "PropertyValue",
          "name": "Activation",
          "value": "Instant activation"
        }
      ]
    }
  }
};