import { useState } from "react";
import { useAuth } from "@/hooks/use-auth";
import { Button } from "@/components/ui/button";
import { ScrollArea } from "@/components/ui/scroll-area";
import { useToast } from "@/hooks/use-toast";
import { Link, useLocation } from "wouter";
import { 
  Users, FileText, MessageSquare, Settings, BarChart3, 
  Mail, Heart, LogOut, Crown, Building, Edit, Search, 
  Image, Star, CheckSquare, Globe, Code, Megaphone, ChevronDown, ChevronRight, Database 
} from "lucide-react";

interface MenuCategory {
  id: string;
  title: string;
  icon: React.ReactNode;
  items: {
    id: string;
    title: string;
    icon: React.ReactNode;
    path?: string;
    isInternal?: boolean;
  }[];
}

const menuCategories: MenuCategory[] = [
  {
    id: "overview",
    title: "Dashboard",
    icon: <BarChart3 className="w-4 h-4" />,
    items: [
      { id: "stats", title: "Overview", icon: <BarChart3 className="w-4 h-4" />, path: "/admin" }
    ]
  },
  {
    id: "business",
    title: "Business Management",
    icon: <Building className="w-4 h-4" />,
    items: [
      { id: "import", title: "Business Import", icon: <Database className="w-4 h-4" />, path: "/admin/business-import" },
      { id: "approvals", title: "Business Approvals", icon: <CheckSquare className="w-4 h-4" />, path: "/admin/business-approvals" },
      { id: "featured", title: "Featured Businesses", icon: <Star className="w-4 h-4" />, path: "/admin/featured-businesses" },
      { id: "images", title: "Business Images", icon: <Image className="w-4 h-4" />, path: "/admin/business-images" }
    ]
  },
  {
    id: "content",
    title: "Content & Website",
    icon: <FileText className="w-4 h-4" />,
    items: [
      { id: "website-editor", title: "Website Editor", icon: <Edit className="w-4 h-4" />, path: "/admin/website-editor" },
      { id: "seo", title: "SEO Management", icon: <Search className="w-4 h-4" />, path: "/admin/seo-management" }
    ]
  },
  {
    id: "communication",
    title: "Communication",
    icon: <Mail className="w-4 h-4" />,
    items: [
      { id: "data-deals-email", title: "Data Deals Email", icon: <Mail className="w-4 h-4" />, path: "/admin/data-deals/email-settings" },
      { id: "customer-email", title: "Customer Email", icon: <Heart className="w-4 h-4" />, path: "/admin/data-deals/customer-email-settings" },
      { id: "header-tags", title: "Header Tags", icon: <Code className="w-4 h-4" />, path: "/admin/header-tags" }
    ]
  },
  {
    id: "community",
    title: "Users & Community",
    icon: <Users className="w-4 h-4" />,
    items: [
      { id: "users", title: "User Management", icon: <Users className="w-4 h-4" />, path: "/admin" },
      { id: "submissions", title: "Contact Submissions", icon: <MessageSquare className="w-4 h-4" />, path: "/admin" }
    ]
  },
  {
    id: "settings",
    title: "System Settings",
    icon: <Settings className="w-4 h-4" />,
    items: [
      { id: "general", title: "General Settings", icon: <Settings className="w-4 h-4" />, path: "/admin" }
    ]
  }
];

interface AdminLayoutProps {
  children: React.ReactNode;
  title?: string;
  description?: string;
}

export function AdminLayout({ children, title, description }: AdminLayoutProps) {
  const { user, logoutMutation } = useAuth();
  const [location] = useLocation();
  const [expandedCategories, setExpandedCategories] = useState<string[]>(["overview", "business", "content", "communication"]);

  const toggleCategory = (categoryId: string) => {
    setExpandedCategories(prev => 
      prev.includes(categoryId) 
        ? prev.filter(id => id !== categoryId)
        : [...prev, categoryId]
    );
  };

  const isCurrentPath = (path: string) => {
    return location === path;
  };

  return (
    <div className="min-h-screen bg-gray-50 dark:bg-gray-900 flex">
      {/* Sidebar */}
      <div className="w-64 bg-white dark:bg-gray-800 border-r border-gray-200 dark:border-gray-700 flex flex-col">
        {/* Header */}
        <div className="p-4 border-b border-gray-200 dark:border-gray-700">
          <Link href="/" className="flex items-center space-x-3">
            <div className="w-8 h-8 bg-gradient-to-br from-red-500 to-pink-500 rounded-full flex items-center justify-center">
              <Heart className="w-4 h-4 text-white" />
            </div>
            <div>
              <h1 className="text-sm font-bold text-gray-900 dark:text-white">LoveMedia Foundation</h1>
              <p className="text-xs text-gray-500 dark:text-gray-400">Admin Dashboard</p>
            </div>
          </Link>
        </div>

        {/* Navigation */}
        <ScrollArea className="flex-1 p-4">
          <nav className="space-y-2">
            {menuCategories.map((category) => (
              <div key={category.id}>
                <button
                  onClick={() => toggleCategory(category.id)}
                  className="w-full flex items-center justify-between p-2 text-sm font-medium text-gray-700 dark:text-gray-300 hover:bg-gray-100 dark:hover:bg-gray-700 rounded-md transition-colors"
                >
                  <div className="flex items-center space-x-2">
                    {category.icon}
                    <span>{category.title}</span>
                  </div>
                  {expandedCategories.includes(category.id) ? (
                    <ChevronDown className="w-4 h-4" />
                  ) : (
                    <ChevronRight className="w-4 h-4" />
                  )}
                </button>

                {expandedCategories.includes(category.id) && (
                  <div className="ml-4 mt-1 space-y-1">
                    {category.items.map((item) => (
                      <Link key={item.id} href={item.path || "/admin"}>
                        <div className={`flex items-center space-x-2 p-2 text-sm transition-colors rounded-md cursor-pointer ${
                          isCurrentPath(item.path || "/admin")
                            ? "bg-blue-100 dark:bg-blue-900 text-blue-700 dark:text-blue-300"
                            : "text-gray-600 dark:text-gray-400 hover:bg-gray-100 dark:hover:bg-gray-700"
                        }`}>
                          {item.icon}
                          <span>{item.title}</span>
                        </div>
                      </Link>
                    ))}
                  </div>
                )}
              </div>
            ))}
          </nav>
        </ScrollArea>

        {/* User Info & Logout */}
        <div className="p-4 border-t border-gray-200 dark:border-gray-700">
          <div className="flex items-center space-x-3 mb-3">
            <div className="w-8 h-8 bg-gradient-to-br from-blue-500 to-purple-500 rounded-full flex items-center justify-center">
              <Crown className="w-4 h-4 text-white" />
            </div>
            <div className="flex-1 min-w-0">
              <p className="text-sm font-medium text-gray-900 dark:text-white truncate">
                {user?.fullName || user?.username}
              </p>
              <p className="text-xs text-gray-500 dark:text-gray-400">Administrator</p>
            </div>
          </div>
          <Button
            variant="outline"
            size="sm"
            onClick={() => logoutMutation.mutate()}
            disabled={logoutMutation.isPending}
            className="w-full"
          >
            <LogOut className="w-4 h-4 mr-2" />
            Logout
          </Button>
        </div>
      </div>

      {/* Main Content */}
      <div className="flex-1 overflow-auto">
        <div className="p-8">
          {title && (
            <div className="mb-6">
              <h2 className="text-3xl font-bold tracking-tight text-gray-900 dark:text-white">
                {title}
              </h2>
              {description && (
                <p className="text-gray-600 dark:text-gray-400 mt-2">
                  {description}
                </p>
              )}
            </div>
          )}
          {children}
        </div>
      </div>
    </div>
  );
}