import { useState } from "react";
import { useMutation, useQuery, useQueryClient } from "@tanstack/react-query";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import { Textarea } from "@/components/ui/textarea";
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card";
import { Tabs, TabsContent, TabsList, TabsTrigger } from "@/components/ui/tabs";
import { useToast } from "@/hooks/use-toast";
import { apiRequest, getQueryFn } from "@/lib/queryClient";
import { 
  Mail, 
  Send, 
  Settings, 
  TestTube, 
  Users, 
  Activity,
  CheckCircle,
  XCircle,
  Clock
} from "lucide-react";

interface SmtpConfig {
  id: number;
  host: string;
  port: number;
  username: string;
  fromEmail: string;
  fromName: string;
  isActive: boolean;
}

interface EmailNewsletter {
  id: number;
  email: string;
  isActive: boolean;
  createdAt: string;
}

interface EmailCampaign {
  id: number;
  subject: string;
  content: string;
  recipientCount: number;
  sentCount: number;
  failedCount: number;
  status: string;
  createdAt: string;
  sentAt?: string;
}

export function AdminMassMail() {
  const [smtpConfig, setSmtpConfig] = useState({
    host: "",
    port: 587,
    username: "",
    password: "",
    fromEmail: "",
    fromName: "LoveMedia Foundation",
  });
  
  const [emailData, setEmailData] = useState({
    subject: "",
    content: "",
  });

  const [testEmail, setTestEmail] = useState("");

  const { toast } = useToast();
  const queryClient = useQueryClient();

  // Fetch SMTP configuration
  const { data: currentSmtpConfig } = useQuery<SmtpConfig>({
    queryKey: ["/api/admin/smtp-config"],
    queryFn: getQueryFn({ on401: "returnNull" }),
  });

  // Fetch newsletter subscribers
  const { data: subscribers = [] } = useQuery<EmailNewsletter[]>({
    queryKey: ["/api/admin/newsletter/subscribers"],
    queryFn: getQueryFn({ on401: "returnNull" }),
  });

  // Fetch email campaigns
  const { data: campaigns = [] } = useQuery<EmailCampaign[]>({
    queryKey: ["/api/admin/email-campaigns"],
    queryFn: getQueryFn({ on401: "returnNull" }),
  });

  // SMTP Configuration mutation
  const smtpConfigMutation = useMutation({
    mutationFn: async (config: typeof smtpConfig) => {
      const res = await apiRequest("POST", "/api/admin/smtp-config", config);
      return await res.json();
    },
    onSuccess: () => {
      toast({
        title: "SMTP Configuration Saved",
        description: "Your email server settings have been updated successfully.",
      });
      queryClient.invalidateQueries({ queryKey: ["/api/admin/smtp-config"] });
    },
    onError: (error: Error) => {
      toast({
        title: "Configuration Failed",
        description: error.message,
        variant: "destructive",
      });
    },
  });

  // SMTP Test mutation
  const smtpTestMutation = useMutation({
    mutationFn: async (email: string) => {
      const res = await apiRequest("POST", "/api/admin/smtp-test", { testEmail: email });
      return await res.json();
    },
    onSuccess: (data) => {
      toast({
        title: "SMTP Test Successful",
        description: data.message,
      });
    },
    onError: (error: Error) => {
      toast({
        title: "SMTP Test Failed",
        description: error.message,
        variant: "destructive",
      });
    },
  });

  // Mass email mutation
  const massEmailMutation = useMutation({
    mutationFn: async (data: typeof emailData) => {
      const res = await apiRequest("POST", "/api/admin/send-mass-email", data);
      return await res.json();
    },
    onSuccess: (data) => {
      toast({
        title: "Mass Email Sent Successfully",
        description: `Sent to ${data.sentCount} of ${data.totalRecipients} subscribers.`,
      });
      setEmailData({ subject: "", content: "" });
      queryClient.invalidateQueries({ queryKey: ["/api/admin/email-campaigns"] });
    },
    onError: (error: Error) => {
      toast({
        title: "Mass Email Failed",
        description: error.message,
        variant: "destructive",
      });
    },
  });

  const handleSmtpSubmit = (e: React.FormEvent) => {
    e.preventDefault();
    smtpConfigMutation.mutate(smtpConfig);
  };

  const handleEmailSubmit = (e: React.FormEvent) => {
    e.preventDefault();
    if (!emailData.subject || !emailData.content) {
      toast({
        title: "Validation Error",
        description: "Please provide both subject and content for the email.",
        variant: "destructive",
      });
      return;
    }
    massEmailMutation.mutate(emailData);
  };

  const handleTestSmtp = () => {
    if (!testEmail) {
      toast({
        title: "Validation Error",
        description: "Please provide an email address to send the test email to.",
        variant: "destructive",
      });
      return;
    }
    smtpTestMutation.mutate(testEmail);
  };

  const activeSubscribers = subscribers.filter(sub => sub.isActive);

  return (
    <div className="space-y-6">
      <div className="flex items-center gap-2">
        <Mail className="w-6 h-6 text-red-500" />
        <h2 className="text-2xl font-bold text-gray-900 dark:text-white">Mass Mail System</h2>
      </div>

      <Tabs defaultValue="compose" className="w-full">
        <TabsList className="grid w-full grid-cols-4">
          <TabsTrigger value="compose">Compose Email</TabsTrigger>
          <TabsTrigger value="subscribers">Subscribers ({activeSubscribers.length})</TabsTrigger>
          <TabsTrigger value="campaigns">Campaigns</TabsTrigger>
          <TabsTrigger value="settings">SMTP Settings</TabsTrigger>
        </TabsList>

        <TabsContent value="compose" className="space-y-6">
          <Card>
            <CardHeader>
              <CardTitle className="flex items-center gap-2">
                <Send className="w-5 h-5" />
                Compose Newsletter
              </CardTitle>
              <CardDescription>
                Send newsletters to all {activeSubscribers.length} active subscribers
              </CardDescription>
            </CardHeader>
            <CardContent>
              <form onSubmit={handleEmailSubmit} className="space-y-4">
                <div>
                  <Label htmlFor="subject">Subject</Label>
                  <Input
                    id="subject"
                    placeholder="Enter email subject..."
                    value={emailData.subject}
                    onChange={(e) => setEmailData(prev => ({ ...prev, subject: e.target.value }))}
                    required
                  />
                </div>

                <div>
                  <Label htmlFor="content">Email Content (HTML)</Label>
                  <Textarea
                    id="content"
                    placeholder="Enter email content in HTML format..."
                    rows={12}
                    value={emailData.content}
                    onChange={(e) => setEmailData(prev => ({ ...prev, content: e.target.value }))}
                    required
                  />
                  <p className="text-sm text-gray-500 mt-1">
                    You can use HTML tags for formatting. Unsubscribe link will be added automatically.
                  </p>
                </div>

                <div className="flex items-center justify-between">
                  <div className="text-sm text-gray-600">
                    This email will be sent to {activeSubscribers.length} active subscribers
                  </div>
                  <Button 
                    type="submit" 
                    disabled={massEmailMutation.isPending || activeSubscribers.length === 0}
                    className="bg-red-500 hover:bg-red-600"
                  >
                    {massEmailMutation.isPending ? "Sending..." : "Send Mass Email"}
                  </Button>
                </div>
              </form>
            </CardContent>
          </Card>
        </TabsContent>

        <TabsContent value="subscribers" className="space-y-6">
          <Card>
            <CardHeader>
              <CardTitle className="flex items-center gap-2">
                <Users className="w-5 h-5" />
                Newsletter Subscribers
              </CardTitle>
              <CardDescription>
                Manage your newsletter subscriber list
              </CardDescription>
            </CardHeader>
            <CardContent>
              <div className="space-y-4">
                <div className="grid grid-cols-1 md:grid-cols-3 gap-4">
                  <div className="text-center p-4 bg-green-50 dark:bg-green-900/20 rounded-lg">
                    <div className="text-2xl font-bold text-green-600">{activeSubscribers.length}</div>
                    <div className="text-sm text-green-600">Active Subscribers</div>
                  </div>
                  <div className="text-center p-4 bg-gray-50 dark:bg-gray-800 rounded-lg">
                    <div className="text-2xl font-bold text-gray-600">{subscribers.length - activeSubscribers.length}</div>
                    <div className="text-sm text-gray-600">Unsubscribed</div>
                  </div>
                  <div className="text-center p-4 bg-blue-50 dark:bg-blue-900/20 rounded-lg">
                    <div className="text-2xl font-bold text-blue-600">{subscribers.length}</div>
                    <div className="text-sm text-blue-600">Total Subscribers</div>
                  </div>
                </div>

                <div className="max-h-96 overflow-y-auto">
                  <div className="space-y-2">
                    {subscribers.map((subscriber) => (
                      <div key={subscriber.id} className="flex items-center justify-between p-3 border rounded-lg">
                        <div className="flex items-center gap-3">
                          <div className={`w-3 h-3 rounded-full ${subscriber.isActive ? 'bg-green-500' : 'bg-gray-400'}`} />
                          <span className="font-medium">{subscriber.email}</span>
                        </div>
                        <div className="text-sm text-gray-500">
                          Joined {new Date(subscriber.createdAt).toLocaleDateString()}
                        </div>
                      </div>
                    ))}
                  </div>
                </div>
              </div>
            </CardContent>
          </Card>
        </TabsContent>

        <TabsContent value="campaigns" className="space-y-6">
          <Card>
            <CardHeader>
              <CardTitle className="flex items-center gap-2">
                <Activity className="w-5 h-5" />
                Email Campaigns
              </CardTitle>
              <CardDescription>
                View your sent email campaigns and their performance
              </CardDescription>
            </CardHeader>
            <CardContent>
              <div className="space-y-4">
                {campaigns.length === 0 ? (
                  <div className="text-center py-8 text-gray-500">
                    No email campaigns sent yet. Send your first newsletter!
                  </div>
                ) : (
                  campaigns.map((campaign) => (
                    <div key={campaign.id} className="border rounded-lg p-4">
                      <div className="flex items-start justify-between">
                        <div className="flex-1">
                          <h4 className="font-semibold text-lg">{campaign.subject}</h4>
                          <div className="flex items-center gap-4 mt-2 text-sm text-gray-600">
                            <span>Recipients: {campaign.recipientCount}</span>
                            <span>Sent: {campaign.sentCount}</span>
                            {campaign.failedCount > 0 && (
                              <span className="text-red-600">Failed: {campaign.failedCount}</span>
                            )}
                            <span>Date: {new Date(campaign.createdAt).toLocaleDateString()}</span>
                          </div>
                        </div>
                        <div className="flex items-center gap-2">
                          {campaign.status === 'sent' && <CheckCircle className="w-5 h-5 text-green-500" />}
                          {campaign.status === 'failed' && <XCircle className="w-5 h-5 text-red-500" />}
                          {campaign.status === 'sending' && <Clock className="w-5 h-5 text-yellow-500" />}
                          <span className="text-sm capitalize">{campaign.status}</span>
                        </div>
                      </div>
                    </div>
                  ))
                )}
              </div>
            </CardContent>
          </Card>
        </TabsContent>

        <TabsContent value="settings" className="space-y-6">
          <Card>
            <CardHeader>
              <CardTitle className="flex items-center gap-2">
                <Settings className="w-5 h-5" />
                SMTP Configuration
              </CardTitle>
              <CardDescription>
                Configure your email server settings for sending newsletters
              </CardDescription>
            </CardHeader>
            <CardContent>
              <form onSubmit={handleSmtpSubmit} className="space-y-4">
                <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                  <div>
                    <Label htmlFor="host">SMTP Host</Label>
                    <Input
                      id="host"
                      placeholder="smtp.gmail.com"
                      value={smtpConfig.host}
                      onChange={(e) => setSmtpConfig(prev => ({ ...prev, host: e.target.value }))}
                      required
                    />
                  </div>
                  <div>
                    <Label htmlFor="port">SMTP Port</Label>
                    <Input
                      id="port"
                      type="number"
                      placeholder="587"
                      value={smtpConfig.port}
                      onChange={(e) => setSmtpConfig(prev => ({ ...prev, port: parseInt(e.target.value) }))}
                      required
                    />
                  </div>
                </div>

                <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                  <div>
                    <Label htmlFor="username">Username</Label>
                    <Input
                      id="username"
                      type="email"
                      placeholder="your-email@domain.com"
                      value={smtpConfig.username}
                      onChange={(e) => setSmtpConfig(prev => ({ ...prev, username: e.target.value }))}
                      required
                    />
                  </div>
                  <div>
                    <Label htmlFor="password">Password</Label>
                    <Input
                      id="password"
                      type="password"
                      placeholder="Your email password or app password"
                      value={smtpConfig.password}
                      onChange={(e) => setSmtpConfig(prev => ({ ...prev, password: e.target.value }))}
                      required
                    />
                  </div>
                </div>

                <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                  <div>
                    <Label htmlFor="fromEmail">From Email</Label>
                    <Input
                      id="fromEmail"
                      type="email"
                      placeholder="noreply@lovemedia.org"
                      value={smtpConfig.fromEmail}
                      onChange={(e) => setSmtpConfig(prev => ({ ...prev, fromEmail: e.target.value }))}
                      required
                    />
                  </div>
                  <div>
                    <Label htmlFor="fromName">From Name</Label>
                    <Input
                      id="fromName"
                      placeholder="LoveMedia Foundation"
                      value={smtpConfig.fromName}
                      onChange={(e) => setSmtpConfig(prev => ({ ...prev, fromName: e.target.value }))}
                      required
                    />
                  </div>
                </div>

                {/* Test Email Section */}
                {currentSmtpConfig && (
                  <div className="border-t pt-6 mt-6">
                    <div className="space-y-4">
                      <div>
                        <Label htmlFor="testEmail" className="text-base font-medium">Send Test Email</Label>
                        <p className="text-sm text-gray-600 dark:text-gray-400 mb-2">
                          Send a test email to verify your SMTP configuration
                        </p>
                      </div>
                      <div className="flex gap-2">
                        <Input
                          id="testEmail"
                          type="email"
                          placeholder="test@example.com"
                          value={testEmail}
                          onChange={(e) => setTestEmail(e.target.value)}
                          className="flex-1"
                        />
                        <Button 
                          type="button"
                          variant="outline"
                          onClick={handleTestSmtp}
                          disabled={smtpTestMutation.isPending || !testEmail}
                          className="flex items-center gap-2"
                        >
                          <TestTube className="w-4 h-4" />
                          {smtpTestMutation.isPending ? "Sending..." : "Send Test Email"}
                        </Button>
                      </div>
                    </div>
                  </div>
                )}

                <div className="flex items-center gap-4">
                  <Button 
                    type="submit" 
                    disabled={smtpConfigMutation.isPending}
                    className="bg-blue-500 hover:bg-blue-600"
                  >
                    {smtpConfigMutation.isPending ? "Saving..." : "Save Configuration"}
                  </Button>
                </div>
              </form>

              {currentSmtpConfig && (
                <div className="mt-6 p-4 bg-green-50 dark:bg-green-900/20 rounded-lg">
                  <div className="flex items-center gap-2 text-green-700 dark:text-green-400">
                    <CheckCircle className="w-5 h-5" />
                    <span className="font-medium">SMTP Configuration Active</span>
                  </div>
                  <p className="text-sm text-green-600 dark:text-green-400 mt-1">
                    Your email server is configured and ready to send newsletters.
                  </p>
                </div>
              )}
            </CardContent>
          </Card>
        </TabsContent>
      </Tabs>
    </div>
  );
}