import { useState } from 'react';
import { useForm } from 'react-hook-form';
import { zodResolver } from '@hookform/resolvers/zod';
import { z } from 'zod';
import { useMutation } from '@tanstack/react-query';
import { apiRequest } from '@/lib/queryClient';
import { Button } from '@/components/ui/button';
import { Input } from '@/components/ui/input';
import { Label } from '@/components/ui/label';
import { Textarea } from '@/components/ui/textarea';
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from '@/components/ui/select';
import { Checkbox } from '@/components/ui/checkbox';
import { Card, CardContent, CardHeader, CardTitle } from '@/components/ui/card';
import { Badge } from '@/components/ui/badge';
import { AlertCircle, CheckCircle, Heart, Shield } from 'lucide-react';
import { useToast } from '@/hooks/use-toast';

// Data deals packages with pricing
const packages = [
  { id: '100gb-anytime', name: '100GB Anytime Data - R299', price: 299 },
  { id: 'uncapped-2mbps', name: 'Uncapped up to 2Mbps Data SIM - R337 pm', price: 337 },
  { id: 'wildcard-lite', name: 'Uncapped Wildcard Lite - R337 pm', price: 337 },
  { id: 'uncapped-10mbps-basic', name: 'Uncapped up to 10Mbps Basic Data SIM - R387 pm', price: 387 },
  { id: 'full-speed-lite', name: 'Uncapped Full Speed Lite - R397', price: 397 },
  { id: 'uncapped-wildcard', name: 'Uncapped Wildcard - R427 pm', price: 427 },
  { id: 'uncapped-5mbps', name: 'Uncapped up to 5Mbps Data SIM - R427 pm', price: 427 },
  { id: 'uncapped-10mbps', name: 'Uncapped up to 10Mbps Data SIM - R487 pm', price: 487 },
  { id: 'uncapped-20mbps', name: 'Uncapped up to 20Mbps Data SIM - R587 pm', price: 587 },
  { id: 'uncapped-50mbps', name: 'Uncapped up to 50Mbps Data SIM - R787 pm', price: 787 },
  { id: 'full-speed-pro', name: 'Uncapped Full Speed Pro - R987 pm', price: 987 }
];

const banks = [
  'ABSA', 'African Bank', 'Bidvest Bank', 'Capitec Bank', 'Discovery Bank', 
  'First National Bank (FNB)', 'Investec', 'Nedbank', 'Standard Bank', 'TymeBank'
];

const signupFormSchema = z.object({
  firstName: z.string().min(1, 'First name is required'),
  surname: z.string().min(1, 'Surname is required'),
  email: z.string().email('Valid email address is required'),
  idNumber: z.string().min(13, 'Valid ID/Passport number is required'),
  mobile: z.string().min(10, 'Valid mobile number is required'),
  installationAddress: z.string().min(10, 'Full installation address is required'),
  packageId: z.string().min(1, 'Please select a package'),
  upgradeToPremium: z.boolean().default(false),
  deliveryAddress: z.string().min(10, 'Full delivery address is required'),
  paymentMethod: z.enum(['debit_order', 'credit_card']),
  bank: z.string().min(1, 'Bank is required'),
  accountNumber: z.string().min(8, 'Bank account number is required'),
  accountType: z.enum(['cheque', 'savings']),
  branchCode: z.string().min(6, 'Branch code is required'),
  mandateSignature: z.string().min(1, 'Electronic signature is required'),
  agreeToTerms: z.boolean().refine(val => val === true, 'You must agree to the terms of service')
});

type SignupFormData = z.infer<typeof signupFormSchema>;

interface DataDealsSignupFormProps {
  selectedPackage?: string;
  onClose?: () => void;
}

export function DataDealsSignupForm({ selectedPackage, onClose }: DataDealsSignupFormProps) {
  const [isSubmitting, setIsSubmitting] = useState(false);
  const { toast } = useToast();

  const form = useForm<SignupFormData>({
    resolver: zodResolver(signupFormSchema),
    defaultValues: {
      packageId: selectedPackage || '',
      upgradeToPremium: false,
      paymentMethod: 'debit_order',
      accountType: 'cheque',
      agreeToTerms: false
    }
  });

  const submitOrderMutation = useMutation({
    mutationFn: (data: SignupFormData) => {
      // Get selected package details
      const selectedPackageData = packages.find(pkg => pkg.id === data.packageId);
      console.log('[FRONTEND] Selected package data:', selectedPackageData);
      console.log('[FRONTEND] Form data received:', data);
      
      if (!selectedPackageData) {
        throw new Error('Selected package not found');
      }

      // Calculate pricing in cents (backend expects cents)
      const basePrice = selectedPackageData.price * 100; // Convert to cents
      const premiumPrice = data.upgradeToPremium ? 18900 : 0; // R189 in cents
      const totalPrice = basePrice + premiumPrice;

      // Prepare complete order data with all required fields
      const orderData = {
        ...data,
        packageName: selectedPackageData.name,
        basePrice,
        premiumPrice,
        totalPrice
      };

      console.log('[FRONTEND] Complete order data being sent:', orderData);
      console.log('[FRONTEND] Making API request to /api/data-deals/orders');
      
      return apiRequest('POST', '/api/data-deals/orders', orderData);
    },
    onSuccess: () => {
      console.log('[FRONTEND] Order submission successful');
      toast({
        title: "Order Submitted Successfully!",
        description: "Redirecting to confirmation page...",
      });
      form.reset();
      if (onClose) onClose();
      // Redirect to thank you page
      window.location.href = '/thankyou';
    },
    onError: (error) => {
      console.error('[FRONTEND] Order submission error:', error);
      console.error('[FRONTEND] Error details:', {
        message: error.message,
        stack: error.stack,
        name: error.name
      });
      toast({
        title: "Order Submission Failed",
        description: "There was an error submitting your order. Please try again.",
        variant: "destructive",
      });
    }
  });

  const onSubmit = async (data: SignupFormData) => {
    console.log('[FRONTEND] Form submission started');
    console.log('[FRONTEND] Form data:', data);
    console.log('[FRONTEND] Form validation state:', form.formState);
    
    setIsSubmitting(true);
    try {
      console.log('[FRONTEND] Calling submitOrderMutation.mutateAsync');
      await submitOrderMutation.mutateAsync(data);
      console.log('[FRONTEND] Mutation completed successfully');
    } catch (error) {
      console.error('[FRONTEND] Mutation failed:', error);
    } finally {
      setIsSubmitting(false);
      console.log('[FRONTEND] Form submission completed, isSubmitting set to false');
    }
  };

  const selectedPackageData = packages.find(pkg => pkg.id === form.watch('packageId'));
  const basePrice = selectedPackageData?.price || 0;
  const premiumPrice = form.watch('upgradeToPremium') ? 189 : 0;
  const totalPrice = basePrice + premiumPrice;

  return (
    <div className="max-w-4xl mx-auto p-6">
      <Card>
        <CardHeader>
          <CardTitle className="text-2xl text-center">
            OpenWeb Signup - Uncapped 4G/5G Data SIM
          </CardTitle>
          <div className="text-center space-y-2">
            <div className="flex items-center justify-center space-x-2 text-sm text-red-600 dark:text-red-400">
              <Heart className="w-4 h-4" />
              <span>R200 donated to LoveMedia Foundation with each order</span>
            </div>
            <div className="flex items-center justify-center space-x-2 text-sm text-blue-600 dark:text-blue-400">
              <Shield className="w-4 h-4" />
              <span>Priority streaming & gaming protocols included</span>
            </div>
          </div>
        </CardHeader>
        <CardContent>
          <form onSubmit={form.handleSubmit(onSubmit)} className="space-y-6">
            {/* Personal Information */}
            <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
              <div>
                <Label htmlFor="firstName">First Name *</Label>
                <Input
                  id="firstName"
                  {...form.register('firstName')}
                  className={form.formState.errors.firstName ? 'border-red-500' : ''}
                />
                {form.formState.errors.firstName && (
                  <p className="text-red-500 text-sm mt-1">
                    {form.formState.errors.firstName.message}
                  </p>
                )}
              </div>

              <div>
                <Label htmlFor="surname">Surname *</Label>
                <Input
                  id="surname"
                  {...form.register('surname')}
                  className={form.formState.errors.surname ? 'border-red-500' : ''}
                />
                {form.formState.errors.surname && (
                  <p className="text-red-500 text-sm mt-1">
                    {form.formState.errors.surname.message}
                  </p>
                )}
              </div>
            </div>

            <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
              <div>
                <Label htmlFor="email">Email Address *</Label>
                <Input
                  id="email"
                  type="email"
                  {...form.register('email')}
                  className={form.formState.errors.email ? 'border-red-500' : ''}
                />
                {form.formState.errors.email && (
                  <p className="text-red-500 text-sm mt-1">
                    {form.formState.errors.email.message}
                  </p>
                )}
              </div>

              <div>
                <Label htmlFor="idNumber">ID Number / Passport Number *</Label>
                <Input
                  id="idNumber"
                  {...form.register('idNumber')}
                  className={form.formState.errors.idNumber ? 'border-red-500' : ''}
                />
                {form.formState.errors.idNumber && (
                  <p className="text-red-500 text-sm mt-1">
                    {form.formState.errors.idNumber.message}
                  </p>
                )}
              </div>
            </div>

            <div>
              <Label htmlFor="mobile">Mobile Number *</Label>
              <Input
                id="mobile"
                {...form.register('mobile')}
                placeholder="Enter your mobile number"
                className={form.formState.errors.mobile ? 'border-red-500' : ''}
              />
              {form.formState.errors.mobile && (
                <p className="text-red-500 text-sm mt-1">
                  {form.formState.errors.mobile.message}
                </p>
              )}
            </div>

            <div>
              <Label htmlFor="installationAddress">Full Installation Address *</Label>
              <Textarea
                id="installationAddress"
                {...form.register('installationAddress')}
                placeholder="Kindly enter the Full Address of where you will be using your 4G / 5G Mobile Data Wifi service."
                className={form.formState.errors.installationAddress ? 'border-red-500' : ''}
              />
              {form.formState.errors.installationAddress && (
                <p className="text-red-500 text-sm mt-1">
                  {form.formState.errors.installationAddress.message}
                </p>
              )}
            </div>

            {/* Package Selection */}
            <div>
              <Label htmlFor="packageId">Which package would you like to buy? *</Label>
              <Select 
                value={form.watch('packageId')} 
                onValueChange={(value) => form.setValue('packageId', value)}
              >
                <SelectTrigger className={form.formState.errors.packageId ? 'border-red-500' : ''}>
                  <SelectValue placeholder="Select a package" />
                </SelectTrigger>
                <SelectContent>
                  {packages.map((pkg) => (
                    <SelectItem key={pkg.id} value={pkg.id}>
                      {pkg.name}
                    </SelectItem>
                  ))}
                </SelectContent>
              </Select>
              {form.formState.errors.packageId && (
                <p className="text-red-500 text-sm mt-1">
                  {form.formState.errors.packageId.message}
                </p>
              )}
            </div>

            {/* Premium Upgrade */}
            <div className="bg-blue-50 dark:bg-blue-900/20 p-4 rounded-lg border">
              <div className="flex items-start space-x-3">
                <Checkbox
                  id="upgradeToPremium"
                  checked={form.watch('upgradeToPremium')}
                  onCheckedChange={(checked) => form.setValue('upgradeToPremium', !!checked)}
                />
                <div className="flex-1">
                  <Label htmlFor="upgradeToPremium" className="font-semibold">
                    Upgrade to Premium for R189 per month extra
                  </Label>
                  <p className="text-sm text-gray-600 dark:text-gray-400 mt-1">
                    Premium adds Priority to your account. You will take priority over other users on the same mobile tower, 
                    which translates into a faster experience even when the network is busy. Premium users are placed into 
                    the Premium Shared Pool for speeds from 1Mbps to 20Mbps when FUP is reached.
                  </p>
                </div>
              </div>
            </div>

            <div>
              <Label htmlFor="deliveryAddress">Preferred Delivery Address *</Label>
              <Textarea
                id="deliveryAddress"
                {...form.register('deliveryAddress')}
                placeholder="Please enter the full address where you would like your SIM to be delivered to."
                className={form.formState.errors.deliveryAddress ? 'border-red-500' : ''}
              />
              {form.formState.errors.deliveryAddress && (
                <p className="text-red-500 text-sm mt-1">
                  {form.formState.errors.deliveryAddress.message}
                </p>
              )}
            </div>

            {/* Payment Information */}
            <div>
              <Label>Payment Method *</Label>
              <Select 
                value={form.watch('paymentMethod')} 
                onValueChange={(value) => form.setValue('paymentMethod', value as 'debit_order' | 'credit_card')}
              >
                <SelectTrigger>
                  <SelectValue placeholder="Select payment method" />
                </SelectTrigger>
                <SelectContent>
                  <SelectItem value="debit_order">Debit Order</SelectItem>
                  <SelectItem value="credit_card">Credit Card</SelectItem>
                </SelectContent>
              </Select>
            </div>

            <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
              <div>
                <Label htmlFor="bank">Bank *</Label>
                <Select 
                  value={form.watch('bank')} 
                  onValueChange={(value) => form.setValue('bank', value)}
                >
                  <SelectTrigger className={form.formState.errors.bank ? 'border-red-500' : ''}>
                    <SelectValue placeholder="e.g. Absa, First National Bank etc" />
                  </SelectTrigger>
                  <SelectContent>
                    {banks.map((bank) => (
                      <SelectItem key={bank} value={bank}>
                        {bank}
                      </SelectItem>
                    ))}
                  </SelectContent>
                </Select>
                {form.formState.errors.bank && (
                  <p className="text-red-500 text-sm mt-1">
                    {form.formState.errors.bank.message}
                  </p>
                )}
              </div>

              <div>
                <Label htmlFor="accountNumber">Bank Account Number *</Label>
                <Input
                  id="accountNumber"
                  {...form.register('accountNumber')}
                  placeholder="Kindly enter your bank account number"
                  className={form.formState.errors.accountNumber ? 'border-red-500' : ''}
                />
                {form.formState.errors.accountNumber && (
                  <p className="text-red-500 text-sm mt-1">
                    {form.formState.errors.accountNumber.message}
                  </p>
                )}
              </div>
            </div>

            <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
              <div>
                <Label>Account Type *</Label>
                <Select 
                  value={form.watch('accountType')} 
                  onValueChange={(value) => form.setValue('accountType', value as 'cheque' | 'savings')}
                >
                  <SelectTrigger>
                    <SelectValue placeholder="Select account type" />
                  </SelectTrigger>
                  <SelectContent>
                    <SelectItem value="cheque">Cheque</SelectItem>
                    <SelectItem value="savings">Savings</SelectItem>
                  </SelectContent>
                </Select>
              </div>

              <div>
                <Label htmlFor="branchCode">Branch Code *</Label>
                <Input
                  id="branchCode"
                  {...form.register('branchCode')}
                  className={form.formState.errors.branchCode ? 'border-red-500' : ''}
                />
                {form.formState.errors.branchCode && (
                  <p className="text-red-500 text-sm mt-1">
                    {form.formState.errors.branchCode.message}
                  </p>
                )}
              </div>
            </div>

            {/* Electronic Mandate */}
            <div>
              <Label htmlFor="mandateSignature">Electronic Debit Order / Credit Order Mandate *</Label>
              <p className="text-sm text-gray-600 dark:text-gray-400 mb-2">
                By typing in your Name in the below box, you Electronically Sign and agree to the Electronic Debit Order / Credit Order Mandate:
              </p>
              <Input
                id="mandateSignature"
                {...form.register('mandateSignature')}
                placeholder="Type your full name to sign electronically"
                className={form.formState.errors.mandateSignature ? 'border-red-500' : ''}
              />
              {form.formState.errors.mandateSignature && (
                <p className="text-red-500 text-sm mt-1">
                  {form.formState.errors.mandateSignature.message}
                </p>
              )}
            </div>

            {/* Terms Agreement */}
            <div className="flex items-start space-x-3">
              <Checkbox
                id="agreeToTerms"
                checked={form.watch('agreeToTerms')}
                onCheckedChange={(checked) => form.setValue('agreeToTerms', !!checked)}
                className={form.formState.errors.agreeToTerms ? 'border-red-500' : ''}
              />
              <div className="flex-1">
                <Label htmlFor="agreeToTerms" className="text-sm">
                  I agree to the OpenWeb{' '}
                  <a 
                    href="http://openweb.co.za/terms-conditions/" 
                    target="_blank" 
                    rel="noopener noreferrer"
                    className="text-blue-600 dark:text-blue-400 underline hover:no-underline"
                  >
                    Terms of Service
                  </a>
                </Label>
                {form.formState.errors.agreeToTerms && (
                  <p className="text-red-500 text-sm mt-1">
                    {form.formState.errors.agreeToTerms.message}
                  </p>
                )}
              </div>
            </div>

            {/* Order Summary */}
            {selectedPackageData && (
              <div className="bg-gray-50 dark:bg-gray-800 p-4 rounded-lg border">
                <h3 className="font-semibold mb-3">Order Summary</h3>
                <div className="space-y-2 text-sm">
                  <div className="flex justify-between">
                    <span>Selected Package:</span>
                    <span className="font-medium">{selectedPackageData.name}</span>
                  </div>
                  <div className="flex justify-between">
                    <span>Base Price:</span>
                    <span>R{basePrice}</span>
                  </div>
                  {form.watch('upgradeToPremium') && (
                    <div className="flex justify-between">
                      <span>Premium Upgrade:</span>
                      <span>R{premiumPrice}</span>
                    </div>
                  )}
                  <div className="flex justify-between font-semibold text-base pt-2 border-t">
                    <span>Total Monthly:</span>
                    <span>R{totalPrice}</span>
                  </div>
                  <div className="bg-green-50 dark:bg-green-900/20 p-3 rounded-md mt-3">
                    <div className="flex justify-between items-center text-green-700 dark:text-green-400">
                      <span className="flex items-center text-sm">
                        <Heart className="w-4 h-4 mr-1" />
                        LoveMedia Foundation Donation
                      </span>
                      <span className="font-semibold">R200</span>
                    </div>
                    <p className="text-xs text-green-600 dark:text-green-400 mt-1">
                      ✓ Included by OpenWeb at no extra cost to you
                    </p>
                  </div>
                </div>
              </div>
            )}

            <Button 
              type="submit" 
              className="w-full bg-gradient-to-r from-blue-500 to-purple-500 hover:from-blue-600 hover:to-purple-600 text-white"
              disabled={isSubmitting || submitOrderMutation.isPending}
            >
              {isSubmitting || submitOrderMutation.isPending ? (
                <>
                  <div className="animate-spin rounded-full h-4 w-4 border-b-2 border-white mr-2"></div>
                  Submitting Order...
                </>
              ) : (
                'Signup Now'
              )}
            </Button>

            <div className="text-center text-sm text-gray-600 dark:text-gray-400">
              <p>
                By clicking the 'Signup Now' button, you confirm that you have read and agree to the OpenWeb{' '}
                <a 
                  href="http://openweb.co.za/terms-conditions/" 
                  target="_blank" 
                  rel="noopener noreferrer"
                  className="text-blue-600 dark:text-blue-400 underline hover:no-underline"
                >
                  Terms of Service
                </a>.
              </p>
            </div>
          </form>
        </CardContent>
      </Card>
    </div>
  );
}