import { useState } from "react";
import { useMutation, useQueryClient } from "@tanstack/react-query";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import { Checkbox } from "@/components/ui/checkbox";
import {
  Dialog,
  DialogContent,
  DialogDescription,
  DialogFooter,
  DialogHeader,
  DialogTitle,
} from "@/components/ui/dialog";
import { useToast } from "@/hooks/use-toast";
import { apiRequest } from "@/lib/queryClient";
import { Mail, Gift, Zap, DollarSign, Newspaper } from "lucide-react";

interface NewsletterSignupModalProps {
  isOpen: boolean;
  onClose: () => void;
}

export function NewsletterSignupModal({ isOpen, onClose }: NewsletterSignupModalProps) {
  const [email, setEmail] = useState("");
  const [agreeToTerms, setAgreeToTerms] = useState(false);
  const { toast } = useToast();
  const queryClient = useQueryClient();

  const newsletterMutation = useMutation({
    mutationFn: async (data: { email: string }) => {
      const res = await apiRequest("POST", "/api/newsletter/subscribe", data);
      return await res.json();
    },
    onSuccess: () => {
      toast({
        title: "Welcome to our community!",
        description: "You've successfully subscribed to our newsletter. Get ready for amazing deals and updates!",
      });
      setEmail("");
      setAgreeToTerms(false);
      onClose();
    },
    onError: (error: Error) => {
      toast({
        title: "Subscription failed",
        description: error.message,
        variant: "destructive",
      });
    },
  });

  const handleSubmit = (e: React.FormEvent) => {
    e.preventDefault();
    
    if (!email) {
      toast({
        title: "Email required",
        description: "Please enter your email address",
        variant: "destructive",
      });
      return;
    }

    if (!agreeToTerms) {
      toast({
        title: "Terms agreement required",
        description: "Please agree to our Terms of Service to continue",
        variant: "destructive",
      });
      return;
    }

    newsletterMutation.mutate({ email });
  };

  const benefits = [
    {
      icon: Gift,
      title: "Product Discounts",
      description: "Exclusive deals saving you thousands of rands monthly"
    },
    {
      icon: Zap,
      title: "Loadshedding Alerts",
      description: "Stay prepared with real-time power outage notifications"
    },
    {
      icon: Newspaper,
      title: "South Africa News",
      description: "Important local news and community updates"
    },
    {
      icon: DollarSign,
      title: "Money-Saving Deals",
      description: "Negotiated discounts exclusively for our community"
    }
  ];

  return (
    <Dialog open={isOpen} onOpenChange={onClose}>
      <DialogContent className="sm:max-w-[500px]">
        <DialogHeader>
          <DialogTitle className="flex items-center gap-2 text-xl">
            <Mail className="w-6 h-6 text-red-500" />
            Join LoveMedia Foundation Newsletter
          </DialogTitle>
          <DialogDescription className="text-base">
            Get exclusive access to discounts, alerts, and community updates from South Africa's most trusted non-profit organization.
          </DialogDescription>
        </DialogHeader>

        <div className="grid gap-4 py-4">
          {/* Benefits */}
          <div className="space-y-3">
            <h4 className="font-semibold text-gray-900 dark:text-white">What you'll receive:</h4>
            <div className="grid gap-2">
              {benefits.map((benefit, index) => {
                const Icon = benefit.icon;
                return (
                  <div key={index} className="flex items-start gap-3 p-2 rounded-lg bg-gray-50 dark:bg-gray-800">
                    <Icon className="w-5 h-5 text-red-500 mt-0.5 flex-shrink-0" />
                    <div>
                      <p className="font-medium text-sm text-gray-900 dark:text-white">{benefit.title}</p>
                      <p className="text-xs text-gray-600 dark:text-gray-400">{benefit.description}</p>
                    </div>
                  </div>
                );
              })}
            </div>
          </div>

          {/* Email Form */}
          <form id="newsletter" onSubmit={handleSubmit} className="space-y-4">
            <div className="space-y-2">
              <Label htmlFor="email">Email Address</Label>
              <Input
                id="email"
                type="email"
                placeholder="your.email@example.com"
                value={email}
                onChange={(e) => setEmail(e.target.value)}
                required
              />
            </div>

            <div className="flex items-center space-x-2">
              <Checkbox
                id="agree-terms"
                checked={agreeToTerms}
                onCheckedChange={(checked) => setAgreeToTerms(checked as boolean)}
              />
              <Label htmlFor="agree-terms" className="text-sm text-gray-600 dark:text-gray-400">
                I agree to the{" "}
                <a href="/terms" target="_blank" className="text-red-500 hover:underline">
                  Terms of Service
                </a>{" "}
                and consent to receive newsletters and promotional emails
              </Label>
            </div>
          </form>
        </div>

        <DialogFooter className="gap-2">
          <Button type="button" variant="outline" onClick={onClose}>
            Maybe Later
          </Button>
          <Button 
            onClick={handleSubmit}
            disabled={newsletterMutation.isPending}
            className="bg-red-500 hover:bg-red-600"
          >
            {newsletterMutation.isPending ? "Subscribing..." : "Subscribe Now"}
          </Button>
        </DialogFooter>
      </DialogContent>
    </Dialog>
  );
}