import { PageSection } from "@shared/schema";
import { Button } from "@/components/ui/button";
import { Card, CardContent } from "@/components/ui/card";
import { MessageSquare, ExternalLink, Users } from "lucide-react";

interface SectionPreviewProps {
  section: PageSection;
}

export function SectionPreview({ section }: SectionPreviewProps) {
  const sectionStyle = {
    background: section.styles.background,
    color: section.styles.textColor,
    padding: section.styles.padding,
    margin: section.styles.margin,
  };

  const renderHeroSection = () => (
    <section style={sectionStyle}>
      <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
        <div className="text-center">
          <h1 className="text-5xl lg:text-6xl font-bold mb-6">
            {section.content.heading}
          </h1>
          <p className="text-xl lg:text-2xl font-light mb-8">
            {section.content.subheading}
          </p>
          <p className="text-lg mb-12 max-w-3xl mx-auto">
            {section.content.description}
          </p>
          
          {section.content.stats && (
            <div className="grid grid-cols-1 md:grid-cols-3 gap-8 mb-12">
              {section.content.stats.map((stat: any, index: number) => (
                <div key={index} className="text-center">
                  <div className="text-4xl font-bold mb-2">{stat.number}</div>
                  <div className="text-lg opacity-80">{stat.label}</div>
                </div>
              ))}
            </div>
          )}
          
          {section.content.buttonText && (
            <Button 
              size="lg"
              className="bg-white text-gray-900 hover:bg-gray-100"
            >
              {section.content.buttonText}
            </Button>
          )}
        </div>
      </div>
    </section>
  );

  const renderCommunitySection = () => (
    <section style={sectionStyle}>
      <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
        <div className="text-center mb-16">
          <h2 className="text-4xl lg:text-5xl font-bold mb-6">
            {section.content.heading}
          </h2>
          <p className="text-xl max-w-3xl mx-auto">
            {section.content.description}
          </p>
        </div>

        {section.content.groups && (
          <div className="grid md:grid-cols-2 lg:grid-cols-3 gap-8">
            {section.content.groups.map((group: any, index: number) => (
              <Card key={index} className="hover:shadow-xl transition-all duration-300">
                <CardContent className="p-8">
                  <div className="flex items-center gap-4 mb-6">
                    <div className="w-16 h-16 bg-green-500 rounded-full flex items-center justify-center">
                      <MessageSquare className="w-8 h-8 text-white" />
                    </div>
                    <div>
                      <h3 className="text-xl font-bold">{group.name}</h3>
                      <p className="text-sm text-gray-600">{group.members} members</p>
                    </div>
                  </div>
                  <p className="text-gray-600 mb-6">{group.description}</p>
                  <Button className="w-full bg-green-500 hover:bg-green-600 text-white">
                    <MessageSquare className="w-4 h-4 mr-2" />
                    Join Group
                    <ExternalLink className="w-4 h-4 ml-2" />
                  </Button>
                </CardContent>
              </Card>
            ))}
          </div>
        )}
      </div>
    </section>
  );

  const renderAboutSection = () => (
    <section style={sectionStyle}>
      <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
        <div className="grid lg:grid-cols-2 gap-12 items-center">
          <div>
            <h2 className="text-4xl lg:text-5xl font-bold mb-6">
              {section.content.heading}
            </h2>
            <p className="text-xl mb-8">
              {section.content.description}
            </p>
            
            {section.content.features && (
              <div className="space-y-4">
                {section.content.features.map((feature: string, index: number) => (
                  <div key={index} className="flex items-center space-x-3">
                    <div className="w-2 h-2 bg-blue-500 rounded-full"></div>
                    <span className="text-lg">{feature}</span>
                  </div>
                ))}
              </div>
            )}
          </div>
          
          {section.content.image && (
            <div className="text-center">
              <img
                src={section.content.image}
                alt="About section image"
                className="rounded-2xl shadow-2xl w-full h-64 object-cover"
              />
            </div>
          )}
        </div>
      </div>
    </section>
  );

  const renderCustomSection = () => (
    <section style={sectionStyle}>
      <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
        <div 
          dangerouslySetInnerHTML={{ __html: section.content.html || "<p>Custom content</p>" }}
        />
      </div>
    </section>
  );

  const renderDefaultSection = () => (
    <section style={sectionStyle}>
      <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
        <div className="text-center">
          <h2 className="text-4xl font-bold mb-6">
            {section.content.heading || section.title}
          </h2>
          <p className="text-xl">
            {section.content.description || "Section content"}
          </p>
        </div>
      </div>
    </section>
  );

  const renderSection = () => {
    switch (section.type) {
      case "hero":
        return renderHeroSection();
      case "community":
        return renderCommunitySection();
      case "about":
        return renderAboutSection();
      case "custom":
        return renderCustomSection();
      default:
        return renderDefaultSection();
    }
  };

  return (
    <div className="relative">
      {renderSection()}
      {section.styles.customCSS && (
        <style dangerouslySetInnerHTML={{ __html: section.styles.customCSS }} />
      )}
    </div>
  );
}