import { useState } from "react";
import { useQuery, useMutation } from "@tanstack/react-query";
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Badge } from "@/components/ui/badge";
import { Progress } from "@/components/ui/progress";
import { Alert, AlertDescription } from "@/components/ui/alert";
import { Separator } from "@/components/ui/separator";
import { 
  Upload,
  Image,
  AlertCircle,
  CheckCircle,
  RefreshCw,
  Database,
  FileImage,
  Zap
} from "lucide-react";
import { AdminLayout } from "@/components/admin-layout";
import { useToast } from "@/hooks/use-toast";
import { apiRequest } from "@/lib/queryClient";

interface ImportLog {
  id: string;
  message: string;
  type: 'info' | 'success' | 'warning' | 'error';
  timestamp: string;
}

interface ImportStatus {
  isRunning: boolean;
  progress: number;
  currentStep: string;
  totalImages: number;
  processedImages: number;
  matchedImages: number;
  errorCount: number;
  logs: ImportLog[];
}

interface ImageStats {
  totalBusinesses: number;
  businessesWithImages: number;
  availableImages: number;
}

export default function AdminBusinessImages() {
  const { toast } = useToast();
  const [importStatus, setImportStatus] = useState<ImportStatus | null>(null);

  // Fetch image import status
  const { data: status, isLoading, refetch } = useQuery<ImportStatus>({
    queryKey: ["/api/admin/business-images/status"],
    refetchInterval: importStatus?.isRunning ? 2000 : false, // Poll every 2 seconds if running
  });

  // Fetch business images statistics
  const { data: stats } = useQuery<ImageStats>({
    queryKey: ["/api/admin/business-images/stats"],
  });

  // Start image import mutation
  const startImportMutation = useMutation({
    mutationFn: async () => {
      const response = await apiRequest("POST", "/api/admin/business-images/start-import");
      return response.json();
    },
    onSuccess: (data) => {
      setImportStatus(data);
      toast({
        title: "Image import started",
        description: "The business image import process has begun. This may take several minutes.",
      });
      refetch();
    },
    onError: (error: any) => {
      toast({
        title: "Failed to start import",
        description: error.message || "An error occurred while starting the image import.",
        variant: "destructive",
      });
    },
  });

  // Stop image import mutation
  const stopImportMutation = useMutation({
    mutationFn: async () => {
      const response = await apiRequest("POST", "/api/admin/business-images/stop-import");
      return response.json();
    },
    onSuccess: () => {
      setImportStatus(null);
      toast({
        title: "Import stopped",
        description: "The image import process has been stopped.",
      });
      refetch();
    },
    onError: (error: any) => {
      toast({
        title: "Failed to stop import",
        description: error.message || "An error occurred while stopping the import.",
        variant: "destructive",
      });
    },
  });

  const handleStartImport = () => {
    startImportMutation.mutate();
  };

  const handleStopImport = () => {
    stopImportMutation.mutate();
  };

  return (
    <AdminLayout title="Business Images Import" description="Manage business directory images and import from Excel files">
      <div>
        <div className="space-y-6">
          {/* Statistics Cards */}
          <div className="grid grid-cols-1 md:grid-cols-4 gap-6">
            <Card>
              <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
                <CardTitle className="text-sm font-medium">Total Businesses</CardTitle>
                <Database className="h-4 w-4 text-muted-foreground" />
              </CardHeader>
              <CardContent>
                <div className="text-2xl font-bold">{stats?.totalBusinesses || 0}</div>
                <p className="text-xs text-muted-foreground">In directory</p>
              </CardContent>
            </Card>

            <Card>
              <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
                <CardTitle className="text-sm font-medium">With Images</CardTitle>
                <FileImage className="h-4 w-4 text-muted-foreground" />
              </CardHeader>
              <CardContent>
                <div className="text-2xl font-bold">{stats?.businessesWithImages || 0}</div>
                <p className="text-xs text-muted-foreground">
                  {stats && stats.totalBusinesses > 0 
                    ? `${Math.round((stats.businessesWithImages || 0) / stats.totalBusinesses * 100)}% coverage`
                    : '0% coverage'
                  }
                </p>
              </CardContent>
            </Card>

            <Card>
              <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
                <CardTitle className="text-sm font-medium">Available Images</CardTitle>
                <Image className="h-4 w-4 text-muted-foreground" />
              </CardHeader>
              <CardContent>
                <div className="text-2xl font-bold">{stats?.availableImages || 0}</div>
                <p className="text-xs text-muted-foreground">Ready to import</p>
              </CardContent>
            </Card>

            <Card>
              <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
                <CardTitle className="text-sm font-medium">Import Status</CardTitle>
                <Zap className="h-4 w-4 text-muted-foreground" />
              </CardHeader>
              <CardContent>
                <div className="text-2xl font-bold">
                  {status?.isRunning ? 'Running' : 'Idle'}
                </div>
                <p className="text-xs text-muted-foreground">
                  {status?.isRunning ? 'Import in progress' : 'Ready to import'}
                </p>
              </CardContent>
            </Card>
          </div>

          {/* Import Controls */}
          <Card>
            <CardHeader>
              <CardTitle className="flex items-center gap-2">
                <Upload className="w-5 h-5" />
                Image Import Control
              </CardTitle>
              <CardDescription>
                Import and match images to business directory listings automatically
              </CardDescription>
            </CardHeader>
            <CardContent className="space-y-4">
              <Alert>
                <AlertCircle className="h-4 w-4" />
                <AlertDescription>
                  This process will scan the images-to-import directory and automatically match images to businesses based on name similarity. 
                  The process may take several minutes depending on the number of images.
                </AlertDescription>
              </Alert>

              <div className="flex items-center gap-4">
                {!status?.isRunning ? (
                  <Button 
                    onClick={handleStartImport}
                    disabled={startImportMutation.isPending}
                    className="flex items-center gap-2"
                  >
                    <Upload className="w-4 h-4" />
                    Start Image Import
                  </Button>
                ) : (
                  <Button 
                    onClick={handleStopImport}
                    disabled={stopImportMutation.isPending}
                    variant="destructive"
                    className="flex items-center gap-2"
                  >
                    <RefreshCw className="w-4 h-4" />
                    Stop Import
                  </Button>
                )}
                
                <Button variant="outline" onClick={() => refetch()}>
                  <RefreshCw className="w-4 h-4 mr-2" />
                  Refresh Status
                </Button>
              </div>
            </CardContent>
          </Card>

          {/* Import Progress */}
          {status?.isRunning && (
            <Card>
              <CardHeader>
                <CardTitle className="flex items-center gap-2">
                  <RefreshCw className="w-5 h-5 animate-spin" />
                  Import Progress
                </CardTitle>
              </CardHeader>
              <CardContent className="space-y-4">
                <div className="space-y-2">
                  <div className="flex justify-between text-sm">
                    <span>Overall Progress</span>
                    <span>{status.processedImages}/{status.totalImages} images ({status.progress}%)</span>
                  </div>
                  <Progress value={status.progress} className="h-3" />
                  <div className="text-xs text-muted-foreground text-center">
                    Processing in batches of 30 to prevent server overload
                  </div>
                </div>

                <div className="grid grid-cols-1 md:grid-cols-3 gap-4 text-sm">
                  <div>
                    <span className="font-medium">Current Step:</span>
                    <p className="text-muted-foreground">{status.currentStep}</p>
                  </div>
                  <div>
                    <span className="font-medium">Matched:</span>
                    <p className="text-green-600">{status.matchedImages} images</p>
                  </div>
                  <div>
                    <span className="font-medium">Errors:</span>
                    <p className="text-red-600">{status.errorCount} issues</p>
                  </div>
                </div>
              </CardContent>
            </Card>
          )}

          {/* Recent Logs */}
          {status?.logs && status.logs.length > 0 && (
            <Card>
              <CardHeader>
                <CardTitle>Import Logs</CardTitle>
                <CardDescription>Recent activity from the image import process</CardDescription>
              </CardHeader>
              <CardContent>
                <div className="space-y-2 max-h-96 overflow-y-auto">
                  {status.logs.slice(-20).reverse().map((log) => (
                    <div key={log.id} className="flex items-start gap-3 p-3 rounded-lg bg-gray-50 dark:bg-gray-800">
                      {log.type === 'success' && <CheckCircle className="w-4 h-4 text-green-500 mt-0.5" />}
                      {log.type === 'error' && <AlertCircle className="w-4 h-4 text-red-500 mt-0.5" />}
                      {log.type === 'warning' && <AlertCircle className="w-4 h-4 text-yellow-500 mt-0.5" />}
                      {log.type === 'info' && <RefreshCw className="w-4 h-4 text-blue-500 mt-0.5" />}
                      
                      <div className="flex-1 min-w-0">
                        <p className="text-sm">{log.message}</p>
                        <p className="text-xs text-muted-foreground">{new Date(log.timestamp).toLocaleString()}</p>
                      </div>
                    </div>
                  ))}
                </div>
              </CardContent>
            </Card>
          )}

          {/* Instructions */}
          <Card>
            <CardHeader>
              <CardTitle>How to Use</CardTitle>
            </CardHeader>
            <CardContent className="space-y-4">
              <div className="space-y-2">
                <h4 className="font-medium">1. Automatic Excel Processing</h4>
                <p className="text-sm text-muted-foreground">
                  The system automatically reads image URLs from the 16 Excel files in the attached_assets directory. 
                  No manual preparation needed - all image URLs are extracted automatically.
                </p>
              </div>

              <Separator />

              <div className="space-y-2">
                <h4 className="font-medium">2. Batch Processing</h4>
                <p className="text-sm text-muted-foreground">
                  Click "Start Image Import" to begin processing. The system will:
                </p>
                <ul className="text-sm text-muted-foreground ml-4 space-y-1">
                  <li>• Process images in batches of 30 to prevent server overload</li>
                  <li>• Download images directly from URLs in Excel files</li>
                  <li>• Match images to businesses using intelligent name matching</li>
                  <li>• Update business records with downloaded image paths</li>
                  <li>• Wait 3 seconds between batches for optimal performance</li>
                </ul>
              </div>

              <Separator />

              <div className="space-y-2">
                <h4 className="font-medium">3. Monitor Progress</h4>
                <p className="text-sm text-muted-foreground">
                  Watch the real-time progress bar and logs. The process automatically continues until all image URLs 
                  have been processed. Each batch of 30 images is processed sequentially with status updates.
                </p>
              </div>
            </CardContent>
          </Card>
        </div>
      </div>
    </AdminLayout>
  );
}