import { useState, useEffect } from "react";
import { useQuery, useMutation, useQueryClient } from "@tanstack/react-query";
import { AdminLayout } from "@/components/admin-layout";
import { Button } from "@/components/ui/button";
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card";
import { Progress } from "@/components/ui/progress";
import { Badge } from "@/components/ui/badge";
import { Alert, AlertDescription } from "@/components/ui/alert";
import { Separator } from "@/components/ui/separator";
import { Switch } from "@/components/ui/switch";
import { Label } from "@/components/ui/label";
import { useToast } from "@/hooks/use-toast";
import { apiRequest } from "@/lib/queryClient";
import { PlayCircle, StopCircle, Database, FileSpreadsheet, CheckCircle, Clock, AlertTriangle, Loader2 } from "lucide-react";

interface ImportStatus {
  isRunning: boolean;
  currentFile?: string;
  currentCategory?: string;
  totalFiles: number;
  processedFiles: number;
  totalBusinesses: number;
  importedBusinesses: number;
  failedBusinesses: number;
  startTime?: string;
  endTime?: string;
  error?: string;
  categoryStats: Record<string, number>;
  filesCompleted: string[];
}

export default function AdminBusinessImport() {
  const [clearExisting, setClearExisting] = useState(false);
  const { toast } = useToast();
  const queryClient = useQueryClient();

  // Poll for import status every 2 seconds
  const { data: importStatus, isLoading } = useQuery<ImportStatus>({
    queryKey: ["/api/admin/import/status"],
    refetchInterval: 2000, // Poll every 2 seconds
    refetchIntervalInBackground: true,
  });

  const startImportMutation = useMutation({
    mutationFn: async () => {
      return await apiRequest("POST", "/api/admin/import/start", { clearExisting });
    },
    onSuccess: () => {
      toast({
        title: "Import Started",
        description: "Business import has been initiated successfully.",
      });
      queryClient.invalidateQueries({ queryKey: ["/api/admin/import/status"] });
    },
    onError: (error) => {
      toast({
        title: "Import Failed",
        description: error.message,
        variant: "destructive",
      });
    },
  });

  const stopImportMutation = useMutation({
    mutationFn: async () => {
      return await apiRequest("POST", "/api/admin/import/stop");
    },
    onSuccess: () => {
      toast({
        title: "Import Stopped",
        description: "Business import has been stopped.",
        variant: "destructive",
      });
      queryClient.invalidateQueries({ queryKey: ["/api/admin/import/status"] });
    },
    onError: (error) => {
      toast({
        title: "Stop Failed",
        description: error.message,
        variant: "destructive",
      });
    },
  });

  const formatDuration = (startTime?: string, endTime?: string) => {
    if (!startTime) return "N/A";
    const start = new Date(startTime);
    const end = endTime ? new Date(endTime) : new Date();
    const diffMs = end.getTime() - start.getTime();
    const diffMinutes = Math.floor(diffMs / 60000);
    const diffSeconds = Math.floor((diffMs % 60000) / 1000);
    return `${diffMinutes}m ${diffSeconds}s`;
  };

  const getProgressPercentage = () => {
    if (!importStatus || importStatus.totalFiles === 0) return 0;
    return Math.round((importStatus.processedFiles / importStatus.totalFiles) * 100);
  };

  const categoryEntries = Object.entries(importStatus?.categoryStats || {})
    .sort(([,a], [,b]) => b - a);

  if (isLoading) {
    return (
      <AdminLayout>
        <div className="flex items-center justify-center p-8">
          <Loader2 className="h-8 w-8 animate-spin" />
        </div>
      </AdminLayout>
    );
  }

  return (
    <AdminLayout>
      <div className="container mx-auto p-6 space-y-6">
        <div className="flex items-center justify-between">
          <div>
            <h1 className="text-3xl font-bold text-gray-900 dark:text-white">
              Business Import System
            </h1>
            <p className="text-gray-600 dark:text-gray-300 mt-2">
              Import and manage your comprehensive business directory from Excel files
            </p>
          </div>
          
          <div className="flex items-center space-x-4">
            {importStatus?.isRunning ? (
              <Button
                onClick={() => stopImportMutation.mutate()}
                disabled={stopImportMutation.isPending}
                variant="destructive"
                data-testid="button-stop-import"
              >
                <StopCircle className="w-4 h-4 mr-2" />
                {stopImportMutation.isPending ? "Stopping..." : "Stop Import"}
              </Button>
            ) : (
              <div className="flex items-center space-x-4">
                <div className="flex items-center space-x-2">
                  <Switch
                    id="clear-existing"
                    checked={clearExisting}
                    onCheckedChange={setClearExisting}
                    data-testid="switch-clear-existing"
                  />
                  <Label htmlFor="clear-existing" className="text-sm">
                    Clear existing data
                  </Label>
                </div>
                
                <Button
                  onClick={() => startImportMutation.mutate()}
                  disabled={startImportMutation.isPending}
                  data-testid="button-start-import"
                >
                  <PlayCircle className="w-4 h-4 mr-2" />
                  {startImportMutation.isPending ? "Starting..." : "Start Import"}
                </Button>
              </div>
            )}
          </div>
        </div>

        {/* Status Overview */}
        <div className="grid grid-cols-1 md:grid-cols-4 gap-4">
          <Card>
            <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
              <CardTitle className="text-sm font-medium">Status</CardTitle>
              <Database className="h-4 w-4 text-muted-foreground" />
            </CardHeader>
            <CardContent>
              <div className="flex items-center space-x-2">
                {importStatus?.isRunning ? (
                  <>
                    <Loader2 className="h-4 w-4 animate-spin text-blue-500" />
                    <Badge variant="default" data-testid="status-running">Running</Badge>
                  </>
                ) : importStatus?.error ? (
                  <>
                    <AlertTriangle className="h-4 w-4 text-red-500" />
                    <Badge variant="destructive" data-testid="status-error">Error</Badge>
                  </>
                ) : (
                  <>
                    <CheckCircle className="h-4 w-4 text-green-500" />
                    <Badge variant="secondary" data-testid="status-idle">Idle</Badge>
                  </>
                )}
              </div>
            </CardContent>
          </Card>

          <Card>
            <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
              <CardTitle className="text-sm font-medium">Total Businesses</CardTitle>
              <Database className="h-4 w-4 text-muted-foreground" />
            </CardHeader>
            <CardContent>
              <div className="text-2xl font-bold" data-testid="text-total-businesses">
                {importStatus?.totalBusinesses.toLocaleString() || 0}
              </div>
              <p className="text-xs text-muted-foreground">
                In database
              </p>
            </CardContent>
          </Card>

          <Card>
            <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
              <CardTitle className="text-sm font-medium">Import Progress</CardTitle>
              <FileSpreadsheet className="h-4 w-4 text-muted-foreground" />
            </CardHeader>
            <CardContent>
              <div className="text-2xl font-bold" data-testid="text-progress">
                {importStatus?.processedFiles || 0} / {importStatus?.totalFiles || 0}
              </div>
              <p className="text-xs text-muted-foreground">
                Files processed
              </p>
            </CardContent>
          </Card>

          <Card>
            <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
              <CardTitle className="text-sm font-medium">Duration</CardTitle>
              <Clock className="h-4 w-4 text-muted-foreground" />
            </CardHeader>
            <CardContent>
              <div className="text-2xl font-bold" data-testid="text-duration">
                {formatDuration(importStatus?.startTime, importStatus?.endTime)}
              </div>
              <p className="text-xs text-muted-foreground">
                {importStatus?.isRunning ? "Running time" : "Last duration"}
              </p>
            </CardContent>
          </Card>
        </div>

        {/* Progress Bar */}
        {importStatus?.isRunning && (
          <Card>
            <CardHeader>
              <CardTitle>Import Progress</CardTitle>
              <CardDescription>
                Currently processing: {importStatus.currentFile} ({importStatus.currentCategory})
              </CardDescription>
            </CardHeader>
            <CardContent className="space-y-4">
              <div className="space-y-2">
                <div className="flex justify-between text-sm">
                  <span>File Progress</span>
                  <span>{getProgressPercentage()}%</span>
                </div>
                <Progress value={getProgressPercentage()} className="w-full" data-testid="progress-import" />
              </div>
              
              <div className="text-sm text-muted-foreground">
                <div>Imported: {importStatus.importedBusinesses.toLocaleString()} businesses</div>
                <div>Failed: {importStatus.failedBusinesses.toLocaleString()} businesses</div>
              </div>
            </CardContent>
          </Card>
        )}

        {/* Error Display */}
        {importStatus?.error && (
          <Alert variant="destructive">
            <AlertTriangle className="h-4 w-4" />
            <AlertDescription data-testid="text-error">
              {importStatus.error}
            </AlertDescription>
          </Alert>
        )}

        {/* Category Statistics */}
        <Card>
          <CardHeader>
            <CardTitle>Category Statistics</CardTitle>
            <CardDescription>
              Businesses imported by category
            </CardDescription>
          </CardHeader>
          <CardContent>
            {categoryEntries.length > 0 ? (
              <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-4">
                {categoryEntries.map(([category, count]) => (
                  <div
                    key={category}
                    className="flex items-center justify-between p-3 border rounded-lg"
                    data-testid={`category-${category.toLowerCase().replace(/\s+/g, '-')}`}
                  >
                    <div className="font-medium">{category}</div>
                    <Badge variant="outline">{count.toLocaleString()}</Badge>
                  </div>
                ))}
              </div>
            ) : (
              <div className="text-center py-8 text-muted-foreground">
                No businesses imported yet
              </div>
            )}
          </CardContent>
        </Card>

        {/* Files Status */}
        {importStatus && importStatus.totalFiles > 0 && (
          <Card>
            <CardHeader>
              <CardTitle>File Processing Status</CardTitle>
              <CardDescription>
                Excel files and their processing status
              </CardDescription>
            </CardHeader>
            <CardContent>
              <div className="space-y-2">
                <div className="flex justify-between text-sm font-medium">
                  <span>Files Completed</span>
                  <span>{importStatus.filesCompleted.length} / {importStatus.totalFiles}</span>
                </div>
                
                {importStatus.filesCompleted.length > 0 && (
                  <div className="text-sm text-muted-foreground">
                    <div className="max-h-32 overflow-y-auto space-y-1">
                      {importStatus.filesCompleted.map((filename, index) => (
                        <div key={index} className="flex items-center space-x-2">
                          <CheckCircle className="h-3 w-3 text-green-500" />
                          <span>{filename}</span>
                        </div>
                      ))}
                    </div>
                  </div>
                )}
              </div>
            </CardContent>
          </Card>
        )}

        {/* Import Instructions */}
        <Card>
          <CardHeader>
            <CardTitle>Import Instructions</CardTitle>
          </CardHeader>
          <CardContent className="space-y-4">
            <div className="text-sm space-y-2">
              <p><strong>Intelligent Resume:</strong> The system will automatically skip categories that have already been imported completely.</p>
              <p><strong>Clear Existing Data:</strong> Enable this option to remove all existing businesses before importing (recommended for fresh imports).</p>
              <p><strong>Process Time:</strong> Importing 15,000+ businesses typically takes 20-30 minutes.</p>
              <p><strong>Background Processing:</strong> The import runs in the background, so you can navigate away from this page.</p>
            </div>
            
            <Separator />
            
            <div className="text-sm">
              <h4 className="font-medium mb-2">Expected Files (16 categories):</h4>
              <div className="grid grid-cols-2 gap-2 text-muted-foreground">
                {Object.values(CATEGORY_MAPPING).map(category => (
                  <div key={category}>• {category}</div>
                ))}
              </div>
            </div>
          </CardContent>
        </Card>
      </div>
    </AdminLayout>
  );
}

const CATEGORY_MAPPING = {
  'Agriculture': 'Agriculture',
  'Car_&_Automotive': 'Car & Automotive', 
  'Education_Training_Lessons': 'Education & Training',
  'Engineering': 'Engineering',
  'Events_&_entertainment': 'Events & Entertainment',
  'Financial_&_Insurance_Services': 'Financial & Insurance Services',
  'Home_Building_&_Trade': 'Home Building & Trade',
  'Hospitality': 'Hospitality',
  'IT_Computer_&_Technology': 'IT & Technology',
  'Marketing_Business,_and_Legal': 'Marketing & Legal',
  'Medical_Wellness_&_Beauty': 'Medical & Beauty',
  'Public_Service_&_Government': 'Public Service & Government',
  'Security_Systems_&_Protection': 'Security & Protection',
  'Telecommunication': 'Telecommunication',
  'Trade_&_Industry': 'Trade & Industry',
  'Transport': 'Transport'
};