import { useState, useEffect } from "react";
import { useQuery, useMutation, useQueryClient } from "@tanstack/react-query";
import { useForm } from "react-hook-form";
import { zodResolver } from "@hookform/resolvers/zod";
import { Button } from "@/components/ui/button";
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card";
import { Form, FormControl, FormDescription, FormField, FormItem, FormLabel, FormMessage } from "@/components/ui/form";
import { Input } from "@/components/ui/input";
import { Textarea } from "@/components/ui/textarea";
import { Switch } from "@/components/ui/switch";
import { Separator } from "@/components/ui/separator";
import { Badge } from "@/components/ui/badge";
import { Alert, AlertDescription } from "@/components/ui/alert";
import { useToast } from "@/hooks/use-toast";
import { insertDataDealsEmailSettingsSchema, type DataDealsEmailSettings, type InsertDataDealsEmailSettings } from "@shared/schema";
import { apiRequest } from "@/lib/queryClient";
import { Save, Mail, Info, CheckCircle, AlertTriangle } from "lucide-react";
import { AdminLayout } from "@/components/admin-layout";

// Available variables for email templates
const AVAILABLE_VARIABLES = [
  "orderNumber", "orderDate", "customerName", "customerEmail", "customerMobile", 
  "customerIdNumber", "installationAddress", "deliveryAddress", "packageName", 
  "basePrice", "premiumPrice", "totalPrice", "upgradeToPremium", "paymentMethodText", 
  "bank", "accountNumber", "accountType", "branchCode", "mandateSignature", "donationAmount"
];

export default function AdminDataDealsEmailPage() {
  const { toast } = useToast();
  const queryClient = useQueryClient();
  const [showVariables, setShowVariables] = useState(false);

  // Query to fetch current email settings
  const { data: emailSettings, isLoading, error } = useQuery({
    queryKey: ["/api/admin/data-deals/email-settings"],
    retry: (failureCount, error) => {
      console.error('[FRONTEND] Query failed:', {
        attempt: failureCount + 1,
        error: error?.message,
        timestamp: new Date().toISOString()
      });
      return failureCount < 2; // Retry up to 2 times
    },
    retryDelay: (attemptIndex) => Math.min(1000 * 2 ** attemptIndex, 30000),
  });

  // Form setup
  const form = useForm<InsertDataDealsEmailSettings>({
    resolver: zodResolver(insertDataDealsEmailSettingsSchema),
    defaultValues: {
      isActive: true,
      toEmail: "wow@openweb.email",
      ccEmail: "info@lovemedia.foundation",
      fromEmail: "orders@lovemedia.foundation",
      fromName: "LoveMedia Foundation",
      replyToEmail: "{{customerEmail}}",
      subject: "New Data Deal Order - {{orderNumber}} - LoveMedia Foundation Partnership",
      emailTemplate: `<h2>New LoveMedia Foundation Data Deal Order</h2>
<p><strong>Order Number:</strong> {{orderNumber}}</p>
<p><strong>Order Date:</strong> {{orderDate}}</p>

<h3>Customer Information</h3>
<ul>
  <li><strong>Name:</strong> {{customerName}}</li>
  <li><strong>Email:</strong> {{customerEmail}}</li>
  <li><strong>Mobile:</strong> {{customerMobile}}</li>
  <li><strong>ID/Passport:</strong> {{customerIdNumber}}</li>
</ul>

<h3>Installation & Delivery</h3>
<ul>
  <li><strong>Installation Address:</strong><br>{{installationAddress}}</li>
  <li><strong>Delivery Address:</strong><br>{{deliveryAddress}}</li>
</ul>

<h3>Package Details</h3>
<ul>
  <li><strong>Package:</strong> {{packageName}}</li>
  <li><strong>Base Price:</strong> R{{basePrice}}</li>
  {{#if upgradeToPremium}}<li><strong>Premium Upgrade:</strong> R{{premiumPrice}}</li>{{/if}}
  <li><strong>Total Monthly Price:</strong> R{{totalPrice}}</li>
</ul>

<h3>Payment Information</h3>
<ul>
  <li><strong>Payment Method:</strong> {{paymentMethodText}}</li>
  <li><strong>Bank:</strong> {{bank}}</li>
  <li><strong>Account Number:</strong> {{accountNumber}}</li>
  <li><strong>Account Type:</strong> {{accountType}}</li>
  <li><strong>Branch Code:</strong> {{branchCode}}</li>
  <li><strong>Electronic Mandate Signature:</strong> {{mandateSignature}}</li>
</ul>

<h3>LoveMedia Foundation Partnership</h3>
<p><strong>Donation to LoveMedia Foundation:</strong> R{{donationAmount}}</p>
<p>This order includes priority streaming & gaming protocols and exclusive discounts for LoveMedia Foundation members.</p>

<hr>
<p><em>This order was submitted through the LoveMedia Foundation platform and includes our partnership agreement for R{{donationAmount}} donation per order.</em></p>
<p><strong>For questions contact:</strong> info@lovemedia.foundation</p>`
    }
  });

  // Update form when data is loaded
  useEffect(() => {
    if (emailSettings) {
      form.reset({
        isActive: emailSettings.isActive ?? true,
        toEmail: emailSettings.toEmail,
        ccEmail: emailSettings.ccEmail || "",
        fromEmail: emailSettings.fromEmail,
        fromName: emailSettings.fromName,
        replyToEmail: emailSettings.replyToEmail,
        subject: emailSettings.subject,
        emailTemplate: emailSettings.emailTemplate
      });
    }
  }, [emailSettings, form]);

  // Mutation to update email settings
  const updateSettingsMutation = useMutation({
    mutationFn: async (data: InsertDataDealsEmailSettings) => {
      console.log('[FRONTEND] Attempting to update email settings:', {
        toEmail: data.toEmail,
        fromEmail: data.fromEmail,
        timestamp: new Date().toISOString()
      });
      
      try {
        const response = await apiRequest("PUT", "/api/admin/data-deals/email-settings", data);
        console.log('[FRONTEND] Email settings update successful');
        return response;
      } catch (error) {
        console.error('[FRONTEND] Email settings update failed:', {
          error: error.message,
          stack: error.stack,
          timestamp: new Date().toISOString()
        });
        throw error;
      }
    },
    onSuccess: (data) => {
      console.log('[FRONTEND] Mutation success, invalidating cache...');
      queryClient.invalidateQueries({ queryKey: ["/api/admin/data-deals/email-settings"] });
      toast({
        title: "Email settings updated",
        description: "Data deals email configuration has been saved successfully.",
      });
    },
    onError: (error: any) => {
      console.error('[FRONTEND] Mutation error:', {
        message: error.message,
        response: error.response,
        timestamp: new Date().toISOString()
      });
      
      let errorMessage = "Failed to update email settings. Please try again.";
      
      // Handle different error types
      if (error.response?.data?.message) {
        errorMessage = error.response.data.message;
      } else if (error.response?.data?.details) {
        errorMessage = `Validation errors: ${error.response.data.details.join(', ')}`;
      } else if (error.message) {
        errorMessage = error.message;
      }
      
      toast({
        title: "Update failed",
        description: errorMessage,
        variant: "destructive",
      });
    },
  });

  const onSubmit = (data: InsertDataDealsEmailSettings) => {
    updateSettingsMutation.mutate(data);
  };

  if (isLoading) {
    return (
      <div className="min-h-screen bg-background p-8">
        <div className="max-w-4xl mx-auto">
          <div className="animate-pulse space-y-4">
            <div className="h-8 bg-muted rounded w-1/3"></div>
            <div className="h-64 bg-muted rounded"></div>
          </div>
        </div>
      </div>
    );
  }

  if (error) {
    return (
      <div className="min-h-screen bg-background p-8">
        <div className="max-w-4xl mx-auto">
          <Alert variant="destructive">
            <AlertTriangle className="h-4 w-4" />
            <AlertDescription>
              Failed to load email settings: {error?.message || 'Unknown error'}. Please refresh the page or check console logs for details.
            </AlertDescription>
          </Alert>
        </div>
      </div>
    );
  }

  return (
    <AdminLayout title="Data Deals Email Settings" description="Configure email templates and settings for data deals order notifications">
      <div className="grid grid-cols-1 lg:grid-cols-3 gap-8">
          {/* Main Configuration */}
          <div className="lg:col-span-2">
            <Card>
              <CardHeader>
                <CardTitle className="flex items-center gap-2">
                  <Mail className="w-5 h-5" />
                  Email Configuration
                </CardTitle>
                <CardDescription>
                  Customize how data deals order emails are sent to OpenWeb and other recipients
                </CardDescription>
              </CardHeader>
              <CardContent>
                <Form {...form}>
                  <form onSubmit={form.handleSubmit(onSubmit)} className="space-y-6">
                    {/* Active Status */}
                    <FormField
                      control={form.control}
                      name="isActive"
                      render={({ field }) => (
                        <FormItem className="flex flex-row items-center justify-between rounded-lg border p-4">
                          <div className="space-y-0.5">
                            <FormLabel className="text-base">Active Configuration</FormLabel>
                            <FormDescription>
                              Enable or disable this email configuration
                            </FormDescription>
                          </div>
                          <FormControl>
                            <Switch
                              checked={field.value}
                              onCheckedChange={field.onChange}
                            />
                          </FormControl>
                        </FormItem>
                      )}
                    />

                    <Separator />

                    {/* Email Addresses */}
                    <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                      <FormField
                        control={form.control}
                        name="toEmail"
                        render={({ field }) => (
                          <FormItem>
                            <FormLabel>To Email</FormLabel>
                            <FormControl>
                              <Input placeholder="wow@openweb.email" {...field} />
                            </FormControl>
                            <FormDescription>Primary recipient (OpenWeb)</FormDescription>
                            <FormMessage />
                          </FormItem>
                        )}
                      />

                      <FormField
                        control={form.control}
                        name="ccEmail"
                        render={({ field }) => (
                          <FormItem>
                            <FormLabel>CC Email</FormLabel>
                            <FormControl>
                              <Input placeholder="info@lovemedia.foundation" {...field} />
                            </FormControl>
                            <FormDescription>Copy recipient (optional)</FormDescription>
                            <FormMessage />
                          </FormItem>
                        )}
                      />

                      <FormField
                        control={form.control}
                        name="fromEmail"
                        render={({ field }) => (
                          <FormItem>
                            <FormLabel>From Email</FormLabel>
                            <FormControl>
                              <Input placeholder="orders@lovemedia.foundation" {...field} />
                            </FormControl>
                            <FormDescription>Sender email address</FormDescription>
                            <FormMessage />
                          </FormItem>
                        )}
                      />

                      <FormField
                        control={form.control}
                        name="fromName"
                        render={({ field }) => (
                          <FormItem>
                            <FormLabel>From Name</FormLabel>
                            <FormControl>
                              <Input placeholder="LoveMedia Foundation" {...field} />
                            </FormControl>
                            <FormDescription>Sender display name</FormDescription>
                            <FormMessage />
                          </FormItem>
                        )}
                      />
                    </div>

                    <FormField
                      control={form.control}
                      name="replyToEmail"
                      render={({ field }) => (
                        <FormItem>
                          <FormLabel>Reply-To Email</FormLabel>
                          <FormControl>
                            <Input placeholder="{{customerEmail}}" {...field} />
                          </FormControl>
                          <FormDescription>
                            Email for replies. Use {`{{customerEmail}}`} to make customer's email the reply-to address
                          </FormDescription>
                          <FormMessage />
                        </FormItem>
                      )}
                    />

                    {/* Subject Line */}
                    <FormField
                      control={form.control}
                      name="subject"
                      render={({ field }) => (
                        <FormItem>
                          <FormLabel>Email Subject</FormLabel>
                          <FormControl>
                            <Input placeholder="New Data Deal Order - {{orderNumber}}" {...field} />
                          </FormControl>
                          <FormDescription>
                            Subject line supports variable substitution (e.g., {`{{orderNumber}}`})
                          </FormDescription>
                          <FormMessage />
                        </FormItem>
                      )}
                    />

                    {/* Email Template */}
                    <FormField
                      control={form.control}
                      name="emailTemplate"
                      render={({ field }) => (
                        <FormItem>
                          <FormLabel>Email Template</FormLabel>
                          <FormControl>
                            <Textarea
                              placeholder="Enter HTML email template..."
                              className="min-h-[400px] font-mono text-sm"
                              {...field}
                            />
                          </FormControl>
                          <FormDescription>
                            HTML email template with variable substitution support. Use variables like {`{{customerName}}`}, {`{{orderNumber}}`}, etc.
                          </FormDescription>
                          <FormMessage />
                        </FormItem>
                      )}
                    />

                    {/* Submit Button */}
                    <div className="flex gap-3">
                      <Button 
                        type="submit" 
                        disabled={updateSettingsMutation.isPending}
                        className="flex-1"
                      >
                        <Save className="w-4 h-4 mr-2" />
                        {updateSettingsMutation.isPending ? "Saving..." : "Save Email Settings"}
                      </Button>
                    </div>
                  </form>
                </Form>
              </CardContent>
            </Card>
          </div>

          {/* Variables Reference */}
          <div className="space-y-6">
            <Card>
              <CardHeader>
                <CardTitle className="flex items-center gap-2">
                  <Info className="w-5 h-5" />
                  Available Variables
                </CardTitle>
                <CardDescription>
                  Use these variables in your email template and subject line
                </CardDescription>
              </CardHeader>
              <CardContent>
                <div className="space-y-2">
                  {AVAILABLE_VARIABLES.map((variable) => (
                    <div key={variable} className="flex items-center gap-2">
                      <Badge variant="secondary" className="font-mono text-xs">
                        {`{{${variable}}}`}
                      </Badge>
                    </div>
                  ))}
                </div>
                
                <Separator className="my-4" />
                
                <div className="space-y-2">
                  <h4 className="font-semibold text-sm">Conditional Blocks</h4>
                  <div className="text-sm text-muted-foreground space-y-1">
                    <p>For optional content based on conditions:</p>
                    <Badge variant="outline" className="font-mono text-xs">
                      {`{{#if upgradeToPremium}}Premium content{{/if}}`}
                    </Badge>
                  </div>
                </div>
              </CardContent>
            </Card>

            <Card>
              <CardHeader>
                <CardTitle className="flex items-center gap-2">
                  <CheckCircle className="w-5 h-5" />
                  Current Status
                </CardTitle>
              </CardHeader>
              <CardContent>
                <div className="space-y-3">
                  <div className="flex items-center justify-between">
                    <span className="text-sm text-muted-foreground">Configuration</span>
                    <Badge variant={emailSettings?.isActive ? "default" : "secondary"}>
                      {emailSettings?.isActive ? "Active" : "Inactive"}
                    </Badge>
                  </div>
                  
                  <div className="flex items-center justify-between">
                    <span className="text-sm text-muted-foreground">Primary Recipient</span>
                    <span className="text-sm font-mono">{emailSettings?.toEmail}</span>
                  </div>
                  
                  <div className="flex items-center justify-between">
                    <span className="text-sm text-muted-foreground">From Address</span>
                    <span className="text-sm font-mono">{emailSettings?.fromEmail}</span>
                  </div>
                  
                  {emailSettings?.ccEmail && (
                    <div className="flex items-center justify-between">
                      <span className="text-sm text-muted-foreground">CC Address</span>
                      <span className="text-sm font-mono">{emailSettings.ccEmail}</span>
                    </div>
                  )}
                </div>
              </CardContent>
            </Card>

            <Alert>
              <Info className="h-4 w-4" />
              <AlertDescription className="text-sm">
                <strong>Important:</strong> Changes to email settings will apply to all new data deals orders. 
                Make sure to test your email template thoroughly before activating.
              </AlertDescription>
            </Alert>
          </div>
        </div>
    </AdminLayout>
  );
}