import { useState } from "react";
import { useQuery, useMutation, useQueryClient } from "@tanstack/react-query";
import { apiRequest } from "@/lib/queryClient";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Badge } from "@/components/ui/badge";
import { Switch } from "@/components/ui/switch";
import { Input } from "@/components/ui/input";
import { ProBadge } from "@/components/ui/pro-badge";
import { useToast } from "@/hooks/use-toast";
import { useAuth } from "@/hooks/use-auth";
import { Link } from "wouter";
import { 
  MapPin, 
  Phone, 
  Mail, 
  Globe, 
  Star, 
  Eye, 
  Search 
} from "lucide-react";
import { AdminLayout } from "@/components/admin-layout";

interface Business {
  id: number;
  name: string;
  description: string;
  category: string;
  location: string;
  contactEmail?: string;
  contactPhone?: string;
  website?: string;
  rating: number;
  reviewCount: number;
  imageUrl?: string;
  isVerified: boolean;
  featuredOnHomepage: boolean;
  homepageViewCount: number;
}

export default function AdminFeaturedBusinesses() {
  const { toast } = useToast();
  const { user, logoutMutation } = useAuth();
  const queryClient = useQueryClient();
  const [searchTerm, setSearchTerm] = useState("");

  // Get all businesses for selection
  const { data: allBusinessesData, isLoading: businessesLoading } = useQuery({
    queryKey: ["/api/businesses", "", "", "", 1, 50],
  });

  // Get featured businesses
  const { data: featuredBusinesses, isLoading: featuredLoading } = useQuery<Business[]>({
    queryKey: ["/api/admin/businesses/featured"],
  });

  // Update business featured status
  const updateFeaturedMutation = useMutation({
    mutationFn: async ({ businessId, featured }: { businessId: number; featured: boolean }) => {
      const response = await fetch(`/api/admin/businesses/${businessId}/featured`, {
        method: "PATCH",
        headers: { "Content-Type": "application/json" },
        credentials: "include",
        body: JSON.stringify({ featured }),
      });
      if (!response.ok) throw new Error("Failed to update featured status");
      return response.json();
    },
    onSuccess: () => {
      toast({
        title: "Success",
        description: "Business featured status updated successfully",
      });
      queryClient.invalidateQueries({ queryKey: ["/api/admin/businesses/featured"] });
      queryClient.invalidateQueries({ queryKey: ["/api/businesses/featured-homepage"] });
    },
    onError: (error: any) => {
      toast({
        title: "Error",
        description: error.message || "Failed to update featured status",
        variant: "destructive",
      });
    },
  });

  const handleToggleFeatured = (businessId: number, featured: boolean) => {
    updateFeaturedMutation.mutate({ businessId, featured });
  };

  const filteredBusinesses = allBusinessesData?.businesses?.filter((business: Business) =>
    business.name.toLowerCase().includes(searchTerm.toLowerCase()) ||
    business.category.toLowerCase().includes(searchTerm.toLowerCase()) ||
    business.location.toLowerCase().includes(searchTerm.toLowerCase())
  ) || [];

  const currentFeaturedCount = featuredBusinesses?.length || 0;

  return (
    <AdminLayout title="Featured Businesses" description="Manage businesses that appear in the 'Recommended Businesses' section on the homepage. Featured businesses rotate with equal visibility.">
      <div className="space-y-6">
        <div className="flex justify-between items-center">
          <Badge variant="secondary" className="text-lg px-4 py-2">
            {currentFeaturedCount} Featured
          </Badge>
        </div>
        {/* Current Featured Businesses */}

          {/* Current Featured Businesses */}
          <Card>
        <CardHeader>
          <CardTitle className="flex items-center gap-2">
            <Eye className="h-5 w-5" />
            Currently Featured Businesses
          </CardTitle>
        </CardHeader>
        <CardContent>
          {featuredLoading ? (
            <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-4">
              {[...Array(3)].map((_, i) => (
                <Card key={i} className="animate-pulse">
                  <CardHeader>
                    <div className="h-6 bg-gray-200 rounded w-3/4"></div>
                    <div className="h-4 bg-gray-200 rounded w-1/2"></div>
                  </CardHeader>
                  <CardContent>
                    <div className="h-4 bg-gray-200 rounded"></div>
                  </CardContent>
                </Card>
              ))}
            </div>
          ) : featuredBusinesses?.length === 0 ? (
            <p className="text-gray-500 text-center py-8">
              No businesses are currently featured on the homepage.
            </p>
          ) : (
            <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-4">
              {featuredBusinesses?.map((business) => (
                <Card key={business.id} className="border-blue-200 bg-blue-50">
                  <CardHeader>
                    <div className="flex justify-between items-start">
                      <div className="flex-1">
                        <CardTitle className="flex items-center gap-2 text-lg">
                          {business.name}
                          <ProBadge />
                        </CardTitle>
                        <div className="flex items-center gap-1 text-gray-600 text-sm">
                          <MapPin className="h-3 w-3" />
                          {business.location}
                        </div>
                      </div>
                      <Switch
                        checked={true}
                        onCheckedChange={(checked) => handleToggleFeatured(business.id, checked)}
                        disabled={updateFeaturedMutation.isPending}
                      />
                    </div>
                  </CardHeader>
                  <CardContent>
                    <p className="text-sm text-gray-600 mb-3 line-clamp-2">
                      {business.description}
                    </p>
                    <div className="flex justify-between items-center text-xs text-gray-500">
                      <span>Category: {business.category}</span>
                      <div className="flex items-center gap-1">
                        <Eye className="h-3 w-3" />
                        {business.homepageViewCount} views
                      </div>
                    </div>
                  </CardContent>
                </Card>
              ))}
            </div>
          )}
          </CardContent>
          </Card>

          {/* All Businesses for Selection */}
          <Card>
        <CardHeader>
          <CardTitle>Add Businesses to Featured</CardTitle>
          <div className="flex items-center gap-4">
            <div className="relative flex-1">
              <Search className="absolute left-3 top-3 h-4 w-4 text-gray-400" />
              <Input
                placeholder="Search businesses by name, category, or location..."
                value={searchTerm}
                onChange={(e) => setSearchTerm(e.target.value)}
                className="pl-10"
                data-testid="input-search-businesses"
              />
            </div>
          </div>
        </CardHeader>
        <CardContent>
          {businessesLoading ? (
            <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-4">
              {[...Array(6)].map((_, i) => (
                <Card key={i} className="animate-pulse">
                  <CardHeader>
                    <div className="h-6 bg-gray-200 rounded w-3/4"></div>
                    <div className="h-4 bg-gray-200 rounded w-1/2"></div>
                  </CardHeader>
                  <CardContent>
                    <div className="h-4 bg-gray-200 rounded"></div>
                  </CardContent>
                </Card>
              ))}
            </div>
          ) : (
            <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-4 max-h-96 overflow-y-auto">
              {filteredBusinesses.map((business: Business) => (
                <Card 
                  key={business.id} 
                  className={business.featuredOnHomepage ? "border-blue-200 bg-blue-50" : ""}
                >
                  <CardHeader>
                    <div className="flex justify-between items-start">
                      <div className="flex-1">
                        <CardTitle className="flex items-center gap-2 text-lg">
                          {business.name}
                          {business.featuredOnHomepage && <ProBadge />}
                        </CardTitle>
                        <div className="flex items-center gap-1 text-gray-600 text-sm">
                          <MapPin className="h-3 w-3" />
                          {business.location}
                        </div>
                      </div>
                      <Switch
                        checked={business.featuredOnHomepage}
                        onCheckedChange={(checked) => handleToggleFeatured(business.id, checked)}
                        disabled={updateFeaturedMutation.isPending}
                        data-testid={`switch-featured-${business.id}`}
                      />
                    </div>
                  </CardHeader>
                  <CardContent>
                    <p className="text-sm text-gray-600 mb-3 line-clamp-2">
                      {business.description}
                    </p>
                    <div className="flex justify-between items-center text-xs text-gray-500">
                      <span>Category: {business.category}</span>
                      {business.featuredOnHomepage && (
                        <div className="flex items-center gap-1">
                          <Eye className="h-3 w-3" />
                          {business.homepageViewCount || 0} views
                        </div>
                      )}
                    </div>
                  </CardContent>
                </Card>
              ))}
            </div>
          )}
          </CardContent>
          </Card>

          {/* Info Card */}
          <Card className="bg-blue-50 border-blue-200">
        <CardContent className="pt-6">
          <h3 className="font-semibold text-blue-800 mb-2">How Featured Businesses Work:</h3>
          <ul className="text-sm text-blue-700 space-y-1">
            <li>• Featured businesses appear in the "Recommended Businesses" section on the homepage</li>
            <li>• The system shows 3 businesses at a time, rotating fairly based on view counts</li>
            <li>• Businesses with fewer homepage views are prioritized for display</li>
            <li>• Each time a business is shown, its view count increases by 1</li>
            <li>• You can feature unlimited businesses - they'll all get equal exposure over time</li>
          </ul>
          </CardContent>
          </Card>
      </div>
    </AdminLayout>
  );
}