import { useState } from "react";
import { useQuery, useMutation, useQueryClient } from "@tanstack/react-query";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import { Textarea } from "@/components/ui/textarea";
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card";
import { Switch } from "@/components/ui/switch";
import { Badge } from "@/components/ui/badge";
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select";
import { Dialog, DialogContent, DialogDescription, DialogHeader, DialogTitle, DialogTrigger } from "@/components/ui/dialog";
import { useToast } from "@/hooks/use-toast";
import { Plus, Edit2, Trash2, Save, X, Search, Globe, Tag } from "lucide-react";
import { SeoSettings, InsertSeoSettings } from "@shared/schema";
import { AdminLayout } from "@/components/admin-layout";

export default function AdminSeoManagement() {
  const [editingSettings, setEditingSettings] = useState<SeoSettings | null>(null);
  const [isCreating, setIsCreating] = useState(false);
  const [newSettings, setNewSettings] = useState<Partial<InsertSeoSettings>>({
    pageName: "",
    title: "",
    description: "",
    keywords: "",
    ogTitle: "",
    ogDescription: "",
    ogImage: "",
    canonicalUrl: "",
    noindex: false,
  });
  const { toast } = useToast();
  const queryClient = useQueryClient();

  const { data: seoSettings, isLoading } = useQuery<SeoSettings[]>({
    queryKey: ["/api/admin/seo-settings"],
    queryFn: async () => {
      const response = await fetch("/api/admin/seo-settings");
      if (!response.ok) throw new Error("Failed to fetch SEO settings");
      return response.json();
    },
  });

  const createMutation = useMutation({
    mutationFn: async (settings: InsertSeoSettings) => {
      const response = await fetch("/api/admin/seo-settings", {
        method: "POST",
        headers: { "Content-Type": "application/json" },
        body: JSON.stringify(settings),
      });
      if (!response.ok) throw new Error("Failed to create SEO settings");
      return response.json();
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/admin/seo-settings"] });
      setIsCreating(false);
      setNewSettings({
        pageName: "",
        title: "",
        description: "",
        keywords: "",
        ogTitle: "",
        ogDescription: "",
        ogImage: "",
        canonicalUrl: "",
        noindex: false,
      });
      toast({
        title: "Success",
        description: "SEO settings created successfully",
      });
    },
    onError: (error) => {
      toast({
        title: "Error",
        description: error.message,
        variant: "destructive",
      });
    },
  });

  const updateMutation = useMutation({
    mutationFn: async (settings: SeoSettings) => {
      const response = await fetch(`/api/admin/seo-settings/${settings.id}`, {
        method: "PUT",
        headers: { "Content-Type": "application/json" },
        body: JSON.stringify(settings),
      });
      if (!response.ok) throw new Error("Failed to update SEO settings");
      return response.json();
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/admin/seo-settings"] });
      setEditingSettings(null);
      toast({
        title: "Success",
        description: "SEO settings updated successfully",
      });
    },
    onError: (error) => {
      toast({
        title: "Error",
        description: error.message,
        variant: "destructive",
      });
    },
  });

  const deleteMutation = useMutation({
    mutationFn: async (id: number) => {
      const response = await fetch(`/api/admin/seo-settings/${id}`, {
        method: "DELETE",
      });
      if (!response.ok) throw new Error("Failed to delete SEO settings");
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/admin/seo-settings"] });
      toast({
        title: "Success",
        description: "SEO settings deleted successfully",
      });
    },
    onError: (error) => {
      toast({
        title: "Error",
        description: error.message,
        variant: "destructive",
      });
    },
  });

  const handleCreateSettings = () => {
    if (!newSettings.pageName || !newSettings.title || !newSettings.description) {
      toast({
        title: "Error",
        description: "Page name, title, and description are required",
        variant: "destructive",
      });
      return;
    }
    createMutation.mutate(newSettings as InsertSeoSettings);
  };

  const handleUpdateSettings = () => {
    if (!editingSettings) return;
    updateMutation.mutate(editingSettings);
  };

  const handleDeleteSettings = (id: number) => {
    if (confirm("Are you sure you want to delete these SEO settings?")) {
      deleteMutation.mutate(id);
    }
  };

  const predefinedPages = [
    { value: "home", label: "Homepage" },
    { value: "about", label: "About Us" },
    { value: "contact", label: "Contact" },
    { value: "business-directory", label: "Business Directory" },
    { value: "community", label: "Community" },
    { value: "auth", label: "Authentication" },
    { value: "terms", label: "Terms of Service" },
    { value: "privacy", label: "Privacy Policy" },
  ];

  if (isLoading) {
    return (
      <div className="flex items-center justify-center min-h-96">
        <div className="animate-spin rounded-full h-8 w-8 border-b-2 border-blue-600"></div>
      </div>
    );
  }

  return (
    <AdminLayout title="SEO Management" description="Manage SEO settings for different pages of your website">
      <div className="space-y-6">
        <div className="flex justify-between items-center">
        <Dialog open={isCreating} onOpenChange={setIsCreating}>
          <DialogTrigger asChild>
            <Button className="bg-blue-600 hover:bg-blue-700">
              <Plus className="w-4 h-4 mr-2" />
              Add SEO Settings
            </Button>
          </DialogTrigger>
          <DialogContent className="max-w-2xl">
            <DialogHeader>
              <DialogTitle>Create SEO Settings</DialogTitle>
              <DialogDescription>
                Create SEO settings for a new page
              </DialogDescription>
            </DialogHeader>
            <div className="space-y-4">
              <div>
                <Label htmlFor="new-pageName">Page Name</Label>
                <Select
                  value={newSettings.pageName}
                  onValueChange={(value) => setNewSettings({ ...newSettings, pageName: value })}
                >
                  <SelectTrigger>
                    <SelectValue placeholder="Select a page" />
                  </SelectTrigger>
                  <SelectContent>
                    {predefinedPages.map(page => (
                      <SelectItem key={page.value} value={page.value}>
                        {page.label}
                      </SelectItem>
                    ))}
                  </SelectContent>
                </Select>
              </div>
              <div>
                <Label htmlFor="new-title">Page Title</Label>
                <Input
                  id="new-title"
                  value={newSettings.title}
                  onChange={(e) => setNewSettings({ ...newSettings, title: e.target.value })}
                  placeholder="Enter page title"
                />
              </div>
              <div>
                <Label htmlFor="new-description">Meta Description</Label>
                <Textarea
                  id="new-description"
                  value={newSettings.description}
                  onChange={(e) => setNewSettings({ ...newSettings, description: e.target.value })}
                  placeholder="Enter meta description"
                  rows={3}
                />
              </div>
              <div>
                <Label htmlFor="new-keywords">Keywords</Label>
                <Input
                  id="new-keywords"
                  value={newSettings.keywords}
                  onChange={(e) => setNewSettings({ ...newSettings, keywords: e.target.value })}
                  placeholder="Enter keywords separated by commas"
                />
              </div>
              <div>
                <Label htmlFor="new-ogTitle">Open Graph Title</Label>
                <Input
                  id="new-ogTitle"
                  value={newSettings.ogTitle || ""}
                  onChange={(e) => setNewSettings({ ...newSettings, ogTitle: e.target.value })}
                  placeholder="Enter Open Graph title (optional)"
                />
              </div>
              <div>
                <Label htmlFor="new-ogDescription">Open Graph Description</Label>
                <Textarea
                  id="new-ogDescription"
                  value={newSettings.ogDescription || ""}
                  onChange={(e) => setNewSettings({ ...newSettings, ogDescription: e.target.value })}
                  placeholder="Enter Open Graph description (optional)"
                  rows={2}
                />
              </div>
              <div>
                <Label htmlFor="new-ogImage">Open Graph Image URL</Label>
                <Input
                  id="new-ogImage"
                  value={newSettings.ogImage || ""}
                  onChange={(e) => setNewSettings({ ...newSettings, ogImage: e.target.value })}
                  placeholder="Enter Open Graph image URL (optional)"
                />
              </div>
              <div>
                <Label htmlFor="new-canonicalUrl">Canonical URL</Label>
                <Input
                  id="new-canonicalUrl"
                  value={newSettings.canonicalUrl || ""}
                  onChange={(e) => setNewSettings({ ...newSettings, canonicalUrl: e.target.value })}
                  placeholder="Enter canonical URL (optional)"
                />
              </div>
              <div className="flex items-center space-x-2">
                <Switch
                  id="new-noindex"
                  checked={newSettings.noindex}
                  onCheckedChange={(checked) => setNewSettings({ ...newSettings, noindex: checked })}
                />
                <Label htmlFor="new-noindex">No Index (prevent search engines from indexing)</Label>
              </div>
              <div className="flex gap-2">
                <Button 
                  onClick={handleCreateSettings}
                  className="bg-green-600 hover:bg-green-700"
                  disabled={createMutation.isPending}
                >
                  <Save className="w-4 h-4 mr-2" />
                  Create
                </Button>
                <Button
                  onClick={() => setIsCreating(false)}
                  variant="outline"
                >
                  <X className="w-4 h-4 mr-2" />
                  Cancel
                </Button>
              </div>
            </div>
          </DialogContent>
        </Dialog>
      </div>

      <div className="grid gap-4">
        {seoSettings?.map((settings) => (
          <Card key={settings.id} className="border-l-4 border-l-blue-500">
            <CardHeader>
              <div className="flex justify-between items-start">
                <div>
                  <CardTitle className="flex items-center gap-2">
                    <Globe className="w-5 h-5" />
                    {predefinedPages.find(p => p.value === settings.pageName)?.label || settings.pageName}
                    <Badge variant="secondary">{settings.pageName}</Badge>
                  </CardTitle>
                  <CardDescription>{settings.title}</CardDescription>
                </div>
                <div className="flex gap-2">
                  <Button
                    onClick={() => setEditingSettings(settings)}
                    variant="outline"
                    size="sm"
                  >
                    <Edit2 className="w-4 h-4" />
                  </Button>
                  <Button
                    onClick={() => handleDeleteSettings(settings.id)}
                    variant="outline"
                    size="sm"
                    className="text-red-600 hover:text-red-700"
                  >
                    <Trash2 className="w-4 h-4" />
                  </Button>
                </div>
              </div>
            </CardHeader>
            <CardContent className="space-y-4">
              <div>
                <strong>Description:</strong>
                <p className="text-gray-600 dark:text-gray-400 mt-1">{settings.description}</p>
              </div>
              {settings.keywords && (
                <div>
                  <strong>Keywords:</strong>
                  <p className="text-gray-600 dark:text-gray-400 mt-1">{settings.keywords}</p>
                </div>
              )}
              {settings.ogTitle && (
                <div>
                  <strong>Open Graph Title:</strong>
                  <p className="text-gray-600 dark:text-gray-400 mt-1">{settings.ogTitle}</p>
                </div>
              )}
              {settings.canonicalUrl && (
                <div>
                  <strong>Canonical URL:</strong>
                  <p className="text-gray-600 dark:text-gray-400 mt-1">{settings.canonicalUrl}</p>
                </div>
              )}
              <div className="flex items-center gap-4">
                <Badge variant={settings.noindex ? "destructive" : "secondary"}>
                  {settings.noindex ? "No Index" : "Indexed"}
                </Badge>
                <span className="text-sm text-gray-500">
                  Created: {new Date(settings.createdAt).toLocaleDateString()}
                </span>
              </div>
            </CardContent>
          </Card>
        ))}
      </div>

      {seoSettings?.length === 0 && (
        <Card>
          <CardContent className="text-center py-8">
            <Search className="w-12 h-12 text-gray-400 mx-auto mb-4" />
            <h3 className="text-lg font-semibold text-gray-900 dark:text-white mb-2">
              No SEO Settings Found
            </h3>
            <p className="text-gray-600 dark:text-gray-400 mb-4">
              Get started by creating SEO settings for your pages
            </p>
            <Button onClick={() => setIsCreating(true)} className="bg-blue-600 hover:bg-blue-700">
              <Plus className="w-4 h-4 mr-2" />
              Create First SEO Settings
            </Button>
          </CardContent>
        </Card>
      )}

      {/* Edit Dialog */}
      <Dialog open={!!editingSettings} onOpenChange={(open) => !open && setEditingSettings(null)}>
        <DialogContent className="max-w-2xl">
          <DialogHeader>
            <DialogTitle>Edit SEO Settings</DialogTitle>
            <DialogDescription>
              Update SEO settings for {editingSettings?.pageName}
            </DialogDescription>
          </DialogHeader>
          {editingSettings && (
            <div className="space-y-4">
              <div>
                <Label htmlFor="edit-title">Page Title</Label>
                <Input
                  id="edit-title"
                  value={editingSettings.title}
                  onChange={(e) => setEditingSettings({ ...editingSettings, title: e.target.value })}
                />
              </div>
              <div>
                <Label htmlFor="edit-description">Meta Description</Label>
                <Textarea
                  id="edit-description"
                  value={editingSettings.description}
                  onChange={(e) => setEditingSettings({ ...editingSettings, description: e.target.value })}
                  rows={3}
                />
              </div>
              <div>
                <Label htmlFor="edit-keywords">Keywords</Label>
                <Input
                  id="edit-keywords"
                  value={editingSettings.keywords}
                  onChange={(e) => setEditingSettings({ ...editingSettings, keywords: e.target.value })}
                />
              </div>
              <div>
                <Label htmlFor="edit-ogTitle">Open Graph Title</Label>
                <Input
                  id="edit-ogTitle"
                  value={editingSettings.ogTitle || ""}
                  onChange={(e) => setEditingSettings({ ...editingSettings, ogTitle: e.target.value })}
                />
              </div>
              <div>
                <Label htmlFor="edit-ogDescription">Open Graph Description</Label>
                <Textarea
                  id="edit-ogDescription"
                  value={editingSettings.ogDescription || ""}
                  onChange={(e) => setEditingSettings({ ...editingSettings, ogDescription: e.target.value })}
                  rows={2}
                />
              </div>
              <div>
                <Label htmlFor="edit-ogImage">Open Graph Image URL</Label>
                <Input
                  id="edit-ogImage"
                  value={editingSettings.ogImage || ""}
                  onChange={(e) => setEditingSettings({ ...editingSettings, ogImage: e.target.value })}
                />
              </div>
              <div>
                <Label htmlFor="edit-canonicalUrl">Canonical URL</Label>
                <Input
                  id="edit-canonicalUrl"
                  value={editingSettings.canonicalUrl || ""}
                  onChange={(e) => setEditingSettings({ ...editingSettings, canonicalUrl: e.target.value })}
                />
              </div>
              <div className="flex items-center space-x-2">
                <Switch
                  id="edit-noindex"
                  checked={editingSettings.noindex}
                  onCheckedChange={(checked) => setEditingSettings({ ...editingSettings, noindex: checked })}
                />
                <Label htmlFor="edit-noindex">No Index</Label>
              </div>
              <div className="flex gap-2">
                <Button 
                  onClick={handleUpdateSettings}
                  className="bg-green-600 hover:bg-green-700"
                  disabled={updateMutation.isPending}
                >
                  <Save className="w-4 h-4 mr-2" />
                  Save
                </Button>
                <Button
                  onClick={() => setEditingSettings(null)}
                  variant="outline"
                >
                  <X className="w-4 h-4 mr-2" />
                  Cancel
                </Button>
              </div>
            </div>
          )}
        </DialogContent>
      </Dialog>
      </div>
    </AdminLayout>
  );
}