import { useState, useEffect } from "react";
import { useQuery } from "@tanstack/react-query";
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card";
import { Badge } from "@/components/ui/badge";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select";
import { Dialog, DialogContent, DialogDescription, DialogHeader, DialogTitle, DialogTrigger } from "@/components/ui/dialog";
import { Form, FormControl, FormField, FormItem, FormLabel, FormMessage } from "@/components/ui/form";
import { Textarea } from "@/components/ui/textarea";
import { useForm } from "react-hook-form";
import { zodResolver } from "@hookform/resolvers/zod";
import { z } from "zod";
import { Search, MapPin, Phone, Mail, Globe, Star, Plus, ChevronLeft, ChevronRight } from "lucide-react";
import { SEOHead } from "@/components/SEOHead";
import Navigation from "@/components/navigation";
import Footer from "@/components/footer";
import { useToast } from "@/hooks/use-toast";
import { apiRequest } from "@/lib/queryClient";

const businessListingSchema = z.object({
  name: z.string().min(2, "Business name must be at least 2 characters"),
  description: z.string().min(10, "Description must be at least 10 characters"),
  category: z.string().min(1, "Please select a category"),
  location: z.string().min(5, "Please provide a detailed location"),
  contactEmail: z.string().email("Please enter a valid email address").optional().or(z.literal("")),
  contactPhone: z.string().min(10, "Please enter a valid phone number").optional().or(z.literal("")),
  website: z.string().url("Please enter a valid website URL").optional().or(z.literal("")),
  whatsappNumber: z.string().optional(),
});

type BusinessListingForm = z.infer<typeof businessListingSchema>;

// Business categories for better organization
const businessCategories = [
  "Agriculture", "Car & Automotive", "Education Training Lessons", "Engineering", 
  "Events & entertainment", "Financial & Insurance Services", "Home Building & Trade", 
  "Hospitality", "IT Computer & Technology", "Marketing Business, and Legal", 
  "Medical Wellness & Beauty", "Public Service & Government", "Security Systems & Protection", 
  "Telecommunication", "Trade & Industry", "Transport"
];

// South African provinces for location selection
const southAfricanProvinces = [
  "Eastern Cape", "Free State", "Gauteng", "KwaZulu-Natal", "Limpopo", 
  "Mpumalanga", "Northern Cape", "North West", "Western Cape"
];

export default function BusinessDirectory() {
  const [searchTerm, setSearchTerm] = useState("");
  const [selectedCategory, setSelectedCategory] = useState<string>("all");
  const [selectedLocation, setSelectedLocation] = useState<string>("all");
  const [currentPage, setCurrentPage] = useState(1);
  const [showListingDialog, setShowListingDialog] = useState(false);
  const [isSubmitting, setIsSubmitting] = useState(false);
  const { toast } = useToast();

  const itemsPerPage = 10;

  // Get businesses data with pagination and filters
  const { data: businessesData, isLoading: businessesLoading } = useQuery({
    queryKey: ['/api/businesses', searchTerm, selectedCategory, selectedLocation, currentPage],
    queryFn: () => {
      const params = new URLSearchParams({
        search: searchTerm,
        category: selectedCategory,
        location: selectedLocation,
        page: currentPage.toString(),
        limit: itemsPerPage.toString()
      });
      return fetch(`/api/businesses?${params}`, { credentials: 'include' })
        .then(res => res.json());
    },
  });

  const businesses = businessesData?.businesses || [];
  const totalCount = businessesData?.total || 0;
  const totalPages = Math.ceil(totalCount / itemsPerPage);

  // Reset to page 1 when filters change
  const resetPagination = () => {
    setCurrentPage(1);
  };

  // Update filter handlers to reset pagination
  const handleCategoryChange = (value: string) => {
    setSelectedCategory(value);
    resetPagination();
  };

  const handleLocationChange = (value: string) => {
    setSelectedLocation(value);
    resetPagination();
  };

  const handleSearchChange = (value: string) => {
    setSearchTerm(value);
    resetPagination();
  };

  const form = useForm<BusinessListingForm>({
    resolver: zodResolver(businessListingSchema),
    defaultValues: {
      name: "",
      description: "",
      category: "",
      location: "",
      contactEmail: "",
      contactPhone: "",
      website: "",
      whatsappNumber: "",
    },
  });

  const handleSubmitBusiness = async (data: BusinessListingForm) => {
    try {
      setIsSubmitting(true);
      
      await apiRequest("POST", "/api/businesses/submit", data);

      toast({
        title: "Business submitted successfully!",
        description: "Your business listing has been submitted for admin review. You'll be notified once it's approved.",
      });

      form.reset();
      setShowListingDialog(false);
    } catch (error) {
      toast({
        title: "Submission failed",
        description: "There was an error submitting your business listing. Please try again.",
        variant: "destructive",
      });
    } finally {
      setIsSubmitting(false);
    }
  };

  return (
    <>
      <SEOHead 
        title="Business Directory - LoveMedia Foundation"
        description="Discover thousands of businesses across South Africa. Find local services, compare options, and support your community through our comprehensive business directory."
        keywords="business directory, South Africa businesses, local services, companies, professional services"
      />
      
      <Navigation />
      
      <div className="min-h-screen bg-gradient-to-br from-blue-50 via-white to-purple-50 pt-16">
        {/* Hero Section */}
        <div className="bg-gradient-to-r from-blue-600 to-purple-600 text-white py-16">
          <div className="container mx-auto px-4">
            <div className="text-center max-w-4xl mx-auto">
              <h1 className="text-4xl md:text-6xl font-bold mb-6">
                South Africa's Premier Business Directory
              </h1>
              <p className="text-xl md:text-2xl mb-8 opacity-90">
                Discover over 15,000 verified businesses across 16 categories. 
                Find the services you need and support local enterprises.
              </p>
              
              <Dialog open={showListingDialog} onOpenChange={setShowListingDialog}>
                <DialogTrigger asChild>
                  <Button size="lg" className="bg-white text-blue-600 hover:bg-gray-100 text-lg px-8 py-3">
                    <Plus className="mr-2 h-5 w-5" />
                    List Your Business - FREE
                  </Button>
                </DialogTrigger>
                <DialogContent className="max-w-4xl max-h-[90vh] overflow-y-auto">
                  <DialogHeader>
                    <DialogTitle>List Your Business - FREE</DialogTitle>
                    <DialogDescription>
                      Submit your business for free! All listings are reviewed by our admin team to ensure quality. Your business will appear in our directory once approved.
                    </DialogDescription>
                  </DialogHeader>
                  
                  <div className="bg-green-50 border border-green-200 rounded-lg p-4 mb-6">
                    <h4 className="font-semibold text-green-800 mb-2">🎉 Free Business Listings!</h4>
                    <p className="text-green-700 text-sm">
                      Submit your business information below at no cost. Our admin team will review your submission 
                      and publish it to our directory once approved. This helps maintain quality and accuracy for all our users.
                    </p>
                  </div>

                  <Form {...form}>
                    <form onSubmit={form.handleSubmit(handleSubmitBusiness)} className="space-y-4">
                      <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                        <FormField
                          control={form.control}
                          name="name"
                          render={({ field }) => (
                            <FormItem>
                              <FormLabel>Business Name *</FormLabel>
                              <FormControl>
                                <Input placeholder="Your Business Name" {...field} data-testid="input-business-name" />
                              </FormControl>
                              <FormMessage />
                            </FormItem>
                          )}
                        />
                        
                        <FormField
                          control={form.control}
                          name="category"
                          render={({ field }) => (
                            <FormItem>
                              <FormLabel>Category *</FormLabel>
                              <Select onValueChange={field.onChange} defaultValue={field.value}>
                                <FormControl>
                                  <SelectTrigger data-testid="select-category">
                                    <SelectValue placeholder="Select a category" />
                                  </SelectTrigger>
                                </FormControl>
                                <SelectContent>
                                  {businessCategories.map((category) => (
                                    <SelectItem key={category} value={category}>
                                      {category}
                                    </SelectItem>
                                  ))}
                                </SelectContent>
                              </Select>
                              <FormMessage />
                            </FormItem>
                          )}
                        />
                      </div>

                      <FormField
                        control={form.control}
                        name="description"
                        render={({ field }) => (
                          <FormItem>
                            <FormLabel>Business Description *</FormLabel>
                            <FormControl>
                              <Textarea 
                                placeholder="Describe your business, services, and what makes you unique..."
                                className="min-h-[100px]"
                                {...field}
                                data-testid="textarea-description"
                              />
                            </FormControl>
                            <FormMessage />
                          </FormItem>
                        )}
                      />

                      <FormField
                        control={form.control}
                        name="location"
                        render={({ field }) => (
                          <FormItem>
                            <FormLabel>Full Address *</FormLabel>
                            <FormControl>
                              <Input 
                                placeholder="Street address, City, Province, Postal Code"
                                {...field}
                                data-testid="input-location"
                              />
                            </FormControl>
                            <FormMessage />
                          </FormItem>
                        )}
                      />

                      <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                        <FormField
                          control={form.control}
                          name="contactEmail"
                          render={({ field }) => (
                            <FormItem>
                              <FormLabel>Email Address</FormLabel>
                              <FormControl>
                                <Input 
                                  type="email"
                                  placeholder="business@example.com"
                                  {...field}
                                  data-testid="input-email"
                                />
                              </FormControl>
                              <FormMessage />
                            </FormItem>
                          )}
                        />

                        <FormField
                          control={form.control}
                          name="contactPhone"
                          render={({ field }) => (
                            <FormItem>
                              <FormLabel>Phone Number</FormLabel>
                              <FormControl>
                                <Input 
                                  placeholder="+27 XX XXX XXXX"
                                  {...field}
                                  data-testid="input-phone"
                                />
                              </FormControl>
                              <FormMessage />
                            </FormItem>
                          )}
                        />
                      </div>

                      <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                        <FormField
                          control={form.control}
                          name="website"
                          render={({ field }) => (
                            <FormItem>
                              <FormLabel>Website URL</FormLabel>
                              <FormControl>
                                <Input 
                                  placeholder="https://www.yourwebsite.com"
                                  {...field}
                                  data-testid="input-website"
                                />
                              </FormControl>
                              <FormMessage />
                            </FormItem>
                          )}
                        />

                        <FormField
                          control={form.control}
                          name="whatsappNumber"
                          render={({ field }) => (
                            <FormItem>
                              <FormLabel>WhatsApp Number</FormLabel>
                              <FormControl>
                                <Input 
                                  placeholder="+27 XX XXX XXXX"
                                  {...field}
                                  data-testid="input-whatsapp"
                                />
                              </FormControl>
                              <FormMessage />
                            </FormItem>
                          )}
                        />
                      </div>

                      <div className="bg-blue-50 border border-blue-200 rounded-lg p-4">
                        <h4 className="font-semibold text-blue-800 mb-2">📋 Review Process</h4>
                        <p className="text-blue-700 text-sm">
                          Your submission will be reviewed by our admin team within 24-48 hours. 
                          You'll receive an email notification once your business is approved and published.
                        </p>
                      </div>

                      <div className="flex gap-4 pt-4">
                        <Button 
                          type="button" 
                          variant="outline" 
                          onClick={() => setShowListingDialog(false)}
                          data-testid="button-cancel"
                        >
                          Cancel
                        </Button>
                        <Button 
                          type="submit"
                          disabled={isSubmitting}
                          data-testid="button-submit"
                        >
                          {isSubmitting ? "Submitting..." : "Submit for Review"}
                        </Button>
                      </div>
                    </form>
                  </Form>
                </DialogContent>
              </Dialog>
            </div>
          </div>
        </div>

        {/* Search and Filter Section */}
        <div className="container mx-auto px-4 py-8">
          <div className="bg-white rounded-xl shadow-sm p-6 mb-8">
            <div className="grid grid-cols-1 md:grid-cols-4 gap-4">
              <div className="md:col-span-2">
                <div className="relative">
                  <Search className="absolute left-3 top-1/2 transform -translate-y-1/2 text-gray-400 h-4 w-4" />
                  <Input
                    placeholder="Search businesses, services, or keywords..."
                    value={searchTerm}
                    onChange={(e) => handleSearchChange(e.target.value)}
                    className="pl-10"
                    data-testid="input-search"
                  />
                </div>
              </div>
              <Select value={selectedCategory} onValueChange={handleCategoryChange}>
                <SelectTrigger data-testid="select-filter-category">
                  <SelectValue placeholder="All Categories" />
                </SelectTrigger>
                <SelectContent>
                  <SelectItem value="all">All Categories</SelectItem>
                  {businessCategories.map((category) => (
                    <SelectItem key={category} value={category}>
                      {category}
                    </SelectItem>
                  ))}
                </SelectContent>
              </Select>
              <Select value={selectedLocation} onValueChange={handleLocationChange}>
                <SelectTrigger data-testid="select-filter-location">
                  <SelectValue placeholder="All Provinces" />
                </SelectTrigger>
                <SelectContent>
                  <SelectItem value="all">All Provinces</SelectItem>
                  {southAfricanProvinces.map((province) => (
                    <SelectItem key={province} value={province}>
                      {province}
                    </SelectItem>
                  ))}
                </SelectContent>
              </Select>
            </div>
          </div>

          {/* Results Info */}
          <div className="flex justify-between items-center mb-6">
            <p className="text-gray-600">
              {businessesLoading ? 'Loading...' : `Showing ${businesses.length} of ${totalCount} businesses`}
              {selectedCategory !== 'all' && ` in ${selectedCategory}`}
              {selectedLocation !== 'all' && ` in ${selectedLocation}`}
            </p>
            {totalPages > 1 && (
              <p className="text-gray-600">Page {currentPage} of {totalPages}</p>
            )}
          </div>

          {/* Business Cards Grid */}
          {businessesLoading ? (
            <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6">
              {[...Array(6)].map((_, i) => (
                <Card key={i} className="animate-pulse">
                  <CardHeader>
                    <div className="h-6 bg-gray-200 rounded w-3/4"></div>
                    <div className="h-4 bg-gray-200 rounded w-1/2"></div>
                  </CardHeader>
                  <CardContent>
                    <div className="h-4 bg-gray-200 rounded mb-2"></div>
                    <div className="h-4 bg-gray-200 rounded w-2/3"></div>
                  </CardContent>
                </Card>
              ))}
            </div>
          ) : businesses.length === 0 ? (
            <div className="text-center py-12">
              <p className="text-gray-500 text-lg">No businesses found matching your criteria.</p>
              <p className="text-gray-400 text-sm mt-2">Try adjusting your search or filters.</p>
            </div>
          ) : (
            <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6">
              {businesses.map((business: any) => (
                <Card 
                  key={business.id} 
                  className="hover:shadow-lg transition-shadow cursor-pointer group"
                  data-testid={`card-business-${business.id}`}
                >
                  <CardHeader>
                    <div className="flex justify-between items-start">
                      <div className="flex-1">
                        <CardTitle className="group-hover:text-blue-600 transition-colors">
                          {business.name}
                        </CardTitle>
                        <CardDescription className="flex items-center gap-1 mt-1">
                          <MapPin className="h-3 w-3" />
                          {business.location}
                        </CardDescription>
                      </div>
                      {(business.isFeatured || business.topPlacement) && (
                        <Badge variant="secondary" className="ml-2">
                          {business.topPlacement ? <Star className="h-3 w-3 mr-1" /> : null}
                          {business.topPlacement ? 'Premium' : 'Featured'}
                        </Badge>
                      )}
                    </div>
                  </CardHeader>
                  <CardContent>
                    <Badge variant="outline" className="mb-3">
                      {business.category}
                    </Badge>
                    <p className="text-sm text-gray-600 mb-4 line-clamp-3">
                      {business.description}
                    </p>
                    <div className="flex flex-wrap gap-2">
                      {business.contactPhone && (
                        <Button variant="outline" size="sm" className="flex items-center gap-1">
                          <Phone className="h-3 w-3" />
                          Call
                        </Button>
                      )}
                      {business.contactEmail && (
                        <Button variant="outline" size="sm" className="flex items-center gap-1">
                          <Mail className="h-3 w-3" />
                          Email
                        </Button>
                      )}
                      {business.website && (
                        <Button variant="outline" size="sm" className="flex items-center gap-1">
                          <Globe className="h-3 w-3" />
                          Website
                        </Button>
                      )}
                    </div>
                  </CardContent>
                </Card>
              ))}
            </div>
          )}

          {/* Pagination */}
          {totalPages > 1 && (
            <div className="flex justify-center items-center gap-2 mt-8">
              <Button
                variant="outline"
                size="sm"
                onClick={() => setCurrentPage(Math.max(1, currentPage - 1))}
                disabled={currentPage === 1}
                data-testid="button-prev-page"
              >
                <ChevronLeft className="h-4 w-4 mr-1" />
                Previous
              </Button>
              
              {/* Page numbers */}
              {Array.from({ length: Math.min(5, totalPages) }, (_, i) => {
                const pageNumber = Math.max(1, Math.min(totalPages - 4, currentPage - 2)) + i;
                if (pageNumber <= totalPages) {
                  return (
                    <Button
                      key={pageNumber}
                      variant={currentPage === pageNumber ? "default" : "outline"}
                      size="sm"
                      onClick={() => setCurrentPage(pageNumber)}
                      data-testid={`button-page-${pageNumber}`}
                    >
                      {pageNumber}
                    </Button>
                  );
                }
                return null;
              })}
              
              <Button
                variant="outline"
                size="sm"
                onClick={() => setCurrentPage(Math.min(totalPages, currentPage + 1))}
                disabled={currentPage === totalPages}
                data-testid="button-next-page"
              >
                Next
                <ChevronRight className="h-4 w-4 ml-1" />
              </Button>
            </div>
          )}
        </div>
      </div>
      
      <Footer />
    </>
  );
}