import { useQuery } from "@tanstack/react-query";
import { useAuth } from "@/hooks/use-auth";
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Badge } from "@/components/ui/badge";
import { 
  Heart, 
  MessageSquare, 
  Bell, 
  Building, 
  LogOut, 
  User,
  ExternalLink,
  Calendar,
  MapPin
} from "lucide-react";
import { Link } from "wouter";

export default function ClientDashboard() {
  const { user, logoutMutation } = useAuth();

  // Fetch latest content
  const { data: content = [], isLoading: contentLoading } = useQuery({
    queryKey: ["/api/content", { status: "published" }],
  });

  // Fetch WhatsApp groups
  const { data: whatsappGroups = [], isLoading: groupsLoading } = useQuery({
    queryKey: ["/api/whatsapp-groups"],
  });

  // Fetch businesses
  const { data: businesses = [], isLoading: businessesLoading } = useQuery({
    queryKey: ["/api/businesses"],
  });

  return (
    <div className="min-h-screen bg-gray-50 dark:bg-gray-900">
      {/* Header */}
      <header className="bg-white dark:bg-gray-800 border-b border-gray-200 dark:border-gray-700">
        <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
          <div className="flex justify-between items-center h-16">
            <div className="flex items-center space-x-4">
              <Link href="/" className="flex items-center space-x-3">
                <div className="w-10 h-10 bg-gradient-to-br from-red-500 to-pink-500 rounded-full flex items-center justify-center">
                  <Heart className="w-5 h-5 text-white" />
                </div>
                <div>
                  <h1 className="text-lg font-bold text-gray-900 dark:text-white">LoveMedia Foundation</h1>
                  <p className="text-xs text-gray-500 dark:text-gray-400">Member Dashboard</p>
                </div>
              </Link>
            </div>

            <div className="flex items-center space-x-4">
              <div className="flex items-center space-x-2">
                <User className="w-4 h-4 text-blue-500" />
                <span className="text-sm font-medium text-gray-700 dark:text-gray-300">
                  {user?.fullName}
                </span>
              </div>
              <Button
                variant="outline"
                size="sm"
                onClick={() => logoutMutation.mutate()}
                disabled={logoutMutation.isPending}
              >
                <LogOut className="w-4 h-4 mr-2" />
                Logout
              </Button>
            </div>
          </div>
        </div>
      </header>

      <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-8">
        {/* Welcome Section */}
        <div className="mb-8">
          <h2 className="text-3xl font-bold text-gray-900 dark:text-white mb-2">
            Welcome back, {user?.fullName?.split(' ')[0]}!
          </h2>
          <p className="text-gray-600 dark:text-gray-400">
            Stay connected with your South African community. Here's what's happening in your area.
          </p>
        </div>

        <div className="grid lg:grid-cols-3 gap-6">
          {/* Main Content */}
          <div className="lg:col-span-2 space-y-6">
            {/* Latest Community Updates */}
            <Card>
              <CardHeader>
                <CardTitle className="flex items-center gap-2">
                  <Bell className="w-5 h-5 text-orange-500" />
                  Latest Community Updates
                </CardTitle>
                <CardDescription>
                  Stay informed with the latest news and alerts from your community
                </CardDescription>
              </CardHeader>
              <CardContent>
                {contentLoading ? (
                  <div className="space-y-4">
                    {[...Array(3)].map((_, i) => (
                      <div key={i} className="animate-pulse">
                        <div className="h-4 bg-gray-200 dark:bg-gray-700 rounded mb-2"></div>
                        <div className="h-3 bg-gray-200 dark:bg-gray-700 rounded w-3/4 mb-2"></div>
                        <div className="h-3 bg-gray-200 dark:bg-gray-700 rounded w-1/2"></div>
                      </div>
                    ))}
                  </div>
                ) : (
                  <div className="space-y-4">
                    {content?.slice(0, 5).map((item: any) => (
                      <div key={item.id} className="p-4 bg-gray-50 dark:bg-gray-800 rounded-lg">
                        <div className="flex justify-between items-start mb-2">
                          <h4 className="font-medium text-gray-900 dark:text-white">{item.title}</h4>
                          <div className="flex gap-2">
                            <Badge variant={item.type === 'alert' ? 'destructive' : 'default'}>
                              {item.type}
                            </Badge>
                            {item.priority === 'urgent' && (
                              <Badge variant="destructive">Urgent</Badge>
                            )}
                          </div>
                        </div>
                        <p className="text-sm text-gray-600 dark:text-gray-400 mb-2">
                          {item.body.substring(0, 150)}...
                        </p>
                        <div className="flex items-center gap-2 text-xs text-gray-500">
                          <Calendar className="w-3 h-3" />
                          {new Date(item.publishedAt || item.createdAt).toLocaleDateString()}
                        </div>
                      </div>
                    ))}
                    {(!content || content.length === 0) && (
                      <p className="text-gray-500 dark:text-gray-400 text-center py-8">
                        No updates available at the moment.
                      </p>
                    )}
                  </div>
                )}
              </CardContent>
            </Card>

            {/* Business Directory Preview - Temporarily hidden until Excel import */}
            {/* <Card>
              <CardHeader>
                <div className="flex justify-between items-center">
                  <div>
                    <CardTitle className="flex items-center gap-2">
                      <Building className="w-5 h-5 text-purple-500" />
                      Local Business Directory
                    </CardTitle>
                    <CardDescription>
                      Discover local businesses in your area
                    </CardDescription>
                  </div>
                  <Button variant="outline" size="sm" asChild>
                    <Link href="/business-directory">
                      View All
                      <ExternalLink className="w-4 h-4 ml-2" />
                    </Link>
                  </Button>
                </div>
              </CardHeader>
              <CardContent>
                {businessesLoading ? (
                  <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                    {[...Array(4)].map((_, i) => (
                      <div key={i} className="animate-pulse p-4 border border-gray-200 dark:border-gray-700 rounded-lg">
                        <div className="h-4 bg-gray-200 dark:bg-gray-700 rounded mb-2"></div>
                        <div className="h-3 bg-gray-200 dark:bg-gray-700 rounded w-3/4"></div>
                      </div>
                    ))}
                  </div>
                ) : (
                  <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                    {businesses?.slice(0, 4).map((business: any) => (
                      <div key={business.id} className="p-4 border border-gray-200 dark:border-gray-700 rounded-lg">
                        <h4 className="font-medium text-gray-900 dark:text-white mb-1">{business.name}</h4>
                        <p className="text-sm text-gray-600 dark:text-gray-400 mb-2">
                          {business.description?.substring(0, 80)}...
                        </p>
                        <div className="flex items-center gap-2 text-xs text-gray-500">
                          <MapPin className="w-3 h-3" />
                          {business.location}
                        </div>
                      </div>
                    ))}
                    {(!businesses || businesses.length === 0) && (
                      <p className="text-gray-500 dark:text-gray-400 text-center py-8 col-span-2">
                        No businesses listed yet.
                      </p>
                    )}
                  </div>
                )}
              </CardContent>
            </Card> */}
          </div>

          {/* Sidebar */}
          <div className="space-y-6">
            {/* User Profile */}
            <Card>
              <CardHeader>
                <CardTitle className="text-lg">Your Profile</CardTitle>
              </CardHeader>
              <CardContent>
                <div className="space-y-3">
                  <div>
                    <p className="text-sm font-medium text-gray-700 dark:text-gray-300">Name</p>
                    <p className="text-gray-900 dark:text-white">{user?.fullName}</p>
                  </div>
                  <div>
                    <p className="text-sm font-medium text-gray-700 dark:text-gray-300">Email</p>
                    <p className="text-gray-900 dark:text-white">{user?.email}</p>
                  </div>
                  <div>
                    <p className="text-sm font-medium text-gray-700 dark:text-gray-300">Location</p>
                    <p className="text-gray-900 dark:text-white capitalize">
                      {user?.location?.replace('-', ' ')}
                    </p>
                  </div>
                  <div>
                    <p className="text-sm font-medium text-gray-700 dark:text-gray-300">Member Since</p>
                    <p className="text-gray-900 dark:text-white">
                      {user?.createdAt ? new Date(user.createdAt).toLocaleDateString() : 'Recently'}
                    </p>
                  </div>
                </div>
              </CardContent>
            </Card>

            {/* WhatsApp Groups */}
            <Card>
              <CardHeader>
                <CardTitle className="flex items-center gap-2 text-lg">
                  <MessageSquare className="w-5 h-5 text-green-500" />
                  WhatsApp Groups
                </CardTitle>
                <CardDescription>
                  Join our active community groups
                </CardDescription>
              </CardHeader>
              <CardContent>
                {groupsLoading ? (
                  <div className="space-y-3">
                    {[...Array(4)].map((_, i) => (
                      <div key={i} className="animate-pulse">
                        <div className="h-4 bg-gray-200 dark:bg-gray-700 rounded mb-1"></div>
                        <div className="h-3 bg-gray-200 dark:bg-gray-700 rounded w-3/4"></div>
                      </div>
                    ))}
                  </div>
                ) : (
                  <div className="space-y-3">
                    {whatsappGroups?.map((group: any) => (
                      <div key={group.id} className="p-3 bg-gray-50 dark:bg-gray-800 rounded-lg">
                        <div className="flex justify-between items-start mb-2">
                          <h4 className="font-medium text-gray-900 dark:text-white text-sm">
                            {group.name}
                          </h4>
                          <Badge variant="outline" className="text-xs">
                            {group.memberCount}+ members
                          </Badge>
                        </div>
                        <p className="text-xs text-gray-600 dark:text-gray-400 mb-2">
                          {group.description}
                        </p>
                        <Button 
                          size="sm" 
                          className="w-full bg-green-600 hover:bg-green-700" 
                          asChild
                        >
                          <a href={group.groupUrl} target="_blank" rel="noopener noreferrer">
                            <MessageSquare className="w-3 h-3 mr-1" />
                            Join Group
                          </a>
                        </Button>
                      </div>
                    ))}
                  </div>
                )}
              </CardContent>
            </Card>

            {/* Quick Actions */}
            <Card>
              <CardHeader>
                <CardTitle className="text-lg">Quick Actions</CardTitle>
              </CardHeader>
              <CardContent>
                <div className="space-y-2">
                  <Button variant="outline" className="w-full justify-start" asChild>
                    <Link href="/community">
                      <MessageSquare className="w-4 h-4 mr-2" />
                      Community Page
                    </Link>
                  </Button>
                  <Button variant="outline" className="w-full justify-start" asChild>
                    <Link href="/directory">
                      <Building className="w-4 h-4 mr-2" />
                      Business Directory
                    </Link>
                  </Button>
                  <Button variant="outline" className="w-full justify-start" asChild>
                    <Link href="/#contact">
                      <Heart className="w-4 h-4 mr-2" />
                      Contact Us
                    </Link>
                  </Button>
                </div>
              </CardContent>
            </Card>
          </div>
        </div>
      </div>
    </div>
  );
}
