var __defProp = Object.defineProperty;
var __export = (target, all) => {
  for (var name in all)
    __defProp(target, name, { get: all[name], enumerable: true });
};

// server/index.ts
import express2 from "express";

// server/routes.ts
import { createServer } from "http";

// server/auth.ts
import passport from "passport";
import { Strategy as LocalStrategy } from "passport-local";
import session2 from "express-session";
import { scrypt, randomBytes, timingSafeEqual } from "crypto";
import { promisify } from "util";

// shared/schema.ts
var schema_exports = {};
__export(schema_exports, {
  businesses: () => businesses,
  businessesRelations: () => businessesRelations,
  contactSubmissions: () => contactSubmissions,
  content: () => content,
  contentRelations: () => contentRelations,
  emailCampaigns: () => emailCampaigns,
  emailNewsletters: () => emailNewsletters,
  headerTags: () => headerTags,
  insertBusinessSchema: () => insertBusinessSchema,
  insertContactSchema: () => insertContactSchema,
  insertContentSchema: () => insertContentSchema,
  insertEmailCampaignSchema: () => insertEmailCampaignSchema,
  insertEmailNewsletterSchema: () => insertEmailNewsletterSchema,
  insertHeaderTagSchema: () => insertHeaderTagSchema,
  insertPageLayoutSchema: () => insertPageLayoutSchema,
  insertProductSchema: () => insertProductSchema,
  insertSeoSettingsSchema: () => insertSeoSettingsSchema,
  insertSmtpConfigSchema: () => insertSmtpConfigSchema,
  insertUserSchema: () => insertUserSchema,
  pageLayouts: () => pageLayouts,
  products: () => products,
  registrationSchema: () => registrationSchema,
  seoSettings: () => seoSettings,
  smtpConfig: () => smtpConfig,
  southAfricanProvinces: () => southAfricanProvinces,
  users: () => users,
  usersRelations: () => usersRelations,
  whatsappGroups: () => whatsappGroups
});
import { pgTable, text, serial, integer, boolean, timestamp, json } from "drizzle-orm/pg-core";
import { relations } from "drizzle-orm";
import { createInsertSchema } from "drizzle-zod";
import { z } from "zod";
var users = pgTable("users", {
  id: serial("id").primaryKey(),
  username: text("username").notNull().unique(),
  email: text("email").notNull().unique(),
  password: text("password").notNull(),
  // Legacy fields (required for backward compatibility)
  fullName: text("full_name").notNull(),
  location: text("location"),
  // New fields
  firstName: text("first_name"),
  lastName: text("last_name"),
  mobile: text("mobile"),
  province: text("province"),
  // South African province
  role: text("role").notNull().default("client"),
  // 'admin' or 'client'
  isActive: boolean("is_active").notNull().default(true),
  createdAt: timestamp("created_at").defaultNow().notNull(),
  lastLogin: timestamp("last_login")
});
var businesses = pgTable("businesses", {
  id: serial("id").primaryKey(),
  name: text("name").notNull(),
  description: text("description"),
  category: text("category").notNull(),
  location: text("location").notNull(),
  contactEmail: text("contact_email"),
  contactPhone: text("contact_phone"),
  website: text("website"),
  whatsappNumber: text("whatsapp_number"),
  isVerified: boolean("is_verified").notNull().default(false),
  rating: integer("rating").default(0),
  reviewCount: integer("review_count").default(0),
  createdBy: integer("created_by").references(() => users.id),
  createdAt: timestamp("created_at").defaultNow().notNull()
});
var content = pgTable("content", {
  id: serial("id").primaryKey(),
  title: text("title").notNull(),
  body: text("body").notNull(),
  type: text("type").notNull(),
  // 'news', 'alert', 'update', 'event'
  status: text("status").notNull().default("draft"),
  // 'draft', 'published', 'archived'
  priority: text("priority").notNull().default("normal"),
  // 'low', 'normal', 'high', 'urgent'
  tags: json("tags").$type().default([]),
  authorId: integer("author_id").references(() => users.id),
  createdAt: timestamp("created_at").defaultNow().notNull(),
  updatedAt: timestamp("updated_at").defaultNow().notNull(),
  publishedAt: timestamp("published_at")
});
var whatsappGroups = pgTable("whatsapp_groups", {
  id: serial("id").primaryKey(),
  name: text("name").notNull(),
  description: text("description"),
  groupUrl: text("group_url").notNull(),
  memberCount: integer("member_count").default(0),
  category: text("category").notNull(),
  // 'discussions', 'alerts', 'accommodation', 'christian', 'other'
  isActive: boolean("is_active").notNull().default(true),
  createdAt: timestamp("created_at").defaultNow().notNull()
});
var contactSubmissions = pgTable("contact_submissions", {
  id: serial("id").primaryKey(),
  name: text("name").notNull(),
  email: text("email").notNull(),
  subject: text("subject").notNull(),
  message: text("message").notNull(),
  status: text("status").notNull().default("pending"),
  // 'pending', 'responded', 'closed'
  createdAt: timestamp("created_at").defaultNow().notNull()
});
var emailNewsletters = pgTable("email_newsletters", {
  id: serial("id").primaryKey(),
  email: text("email").notNull().unique(),
  isActive: boolean("is_active").notNull().default(true),
  createdAt: timestamp("created_at").defaultNow().notNull(),
  unsubscribedAt: timestamp("unsubscribed_at")
});
var smtpConfig = pgTable("smtp_config", {
  id: serial("id").primaryKey(),
  host: text("host").notNull(),
  port: integer("port").notNull(),
  username: text("username").notNull(),
  password: text("password").notNull(),
  // encrypted
  fromEmail: text("from_email").notNull(),
  fromName: text("from_name").notNull(),
  isActive: boolean("is_active").notNull().default(true),
  createdAt: timestamp("created_at").defaultNow().notNull(),
  updatedAt: timestamp("updated_at").defaultNow().notNull()
});
var emailCampaigns = pgTable("email_campaigns", {
  id: serial("id").primaryKey(),
  subject: text("subject").notNull(),
  content: text("content").notNull(),
  recipientCount: integer("recipient_count").notNull(),
  sentCount: integer("sent_count").default(0),
  failedCount: integer("failed_count").default(0),
  status: text("status").notNull().default("draft"),
  // 'draft', 'sending', 'sent', 'failed'
  sentBy: integer("sent_by").references(() => users.id),
  createdAt: timestamp("created_at").defaultNow().notNull(),
  sentAt: timestamp("sent_at")
});
var products = pgTable("products", {
  id: serial("id").primaryKey(),
  name: text("name").notNull(),
  description: text("description"),
  category: text("category").notNull(),
  // '4g', '5g', 'partner_deal'
  wholesalePrice: integer("wholesale_price"),
  // in cents
  adminFeePercentage: integer("admin_fee_percentage").default(5),
  finalPrice: integer("final_price"),
  // calculated: wholesale + admin fee
  isActive: boolean("is_active").notNull().default(true),
  createdAt: timestamp("created_at").defaultNow().notNull(),
  updatedAt: timestamp("updated_at").defaultNow().notNull()
});
var pageLayouts = pgTable("page_layouts", {
  id: serial("id").primaryKey(),
  pageName: text("page_name").notNull().unique(),
  // 'home', 'about', 'community', etc.
  sections: json("sections").$type().notNull(),
  isActive: boolean("is_active").notNull().default(true),
  createdBy: integer("created_by").references(() => users.id),
  createdAt: timestamp("created_at").defaultNow().notNull(),
  updatedAt: timestamp("updated_at").defaultNow().notNull()
});
var headerTags = pgTable("header_tags", {
  id: serial("id").primaryKey(),
  name: text("name").notNull(),
  // Display name for admin
  type: text("type").notNull(),
  // 'gtm', 'facebook_pixel', 'custom'
  code: text("code").notNull(),
  // The actual tag code
  isActive: boolean("is_active").notNull().default(true),
  priority: integer("priority").notNull().default(1),
  // Order of execution (lower numbers first)
  createdAt: timestamp("created_at").defaultNow().notNull(),
  updatedAt: timestamp("updated_at").defaultNow().notNull(),
  createdBy: integer("created_by").references(() => users.id)
});
var seoSettings = pgTable("seo_settings", {
  id: serial("id").primaryKey(),
  pageName: text("page_name").notNull().unique(),
  // 'home', 'about', 'contact', 'business-directory', etc.
  title: text("title").notNull(),
  description: text("description").notNull(),
  keywords: text("keywords").notNull(),
  ogTitle: text("og_title"),
  ogDescription: text("og_description"),
  ogImage: text("og_image"),
  canonicalUrl: text("canonical_url"),
  structuredData: json("structured_data"),
  // JSON-LD structured data
  noindex: boolean("noindex").notNull().default(false),
  createdAt: timestamp("created_at").defaultNow().notNull(),
  updatedAt: timestamp("updated_at").defaultNow().notNull(),
  createdBy: integer("created_by").references(() => users.id)
});
var usersRelations = relations(users, ({ many }) => ({
  businesses: many(businesses),
  content: many(content)
}));
var businessesRelations = relations(businesses, ({ one }) => ({
  createdBy: one(users, {
    fields: [businesses.createdBy],
    references: [users.id]
  })
}));
var contentRelations = relations(content, ({ one }) => ({
  author: one(users, {
    fields: [content.authorId],
    references: [users.id]
  })
}));
var insertUserSchema = createInsertSchema(users, {
  email: z.string().email("Please enter a valid email address"),
  mobile: z.string().min(10, "Please enter a valid mobile number"),
  firstName: z.string().min(2, "First name must be at least 2 characters"),
  lastName: z.string().min(2, "Last name must be at least 2 characters"),
  province: z.string().min(1, "Please select a province")
}).omit({ id: true, createdAt: true, lastLogin: true, isActive: true, role: true });
var insertBusinessSchema = createInsertSchema(businesses).omit({
  id: true,
  createdAt: true,
  createdBy: true,
  isVerified: true,
  rating: true,
  reviewCount: true
});
var insertContentSchema = createInsertSchema(content).omit({
  id: true,
  createdAt: true,
  updatedAt: true,
  authorId: true
});
var insertContactSchema = createInsertSchema(contactSubmissions).omit({
  id: true,
  createdAt: true,
  status: true
});
var insertProductSchema = createInsertSchema(products).omit({
  id: true,
  createdAt: true,
  updatedAt: true,
  finalPrice: true
});
var insertEmailNewsletterSchema = createInsertSchema(emailNewsletters, {
  email: z.string().email("Please enter a valid email address")
}).omit({
  id: true,
  createdAt: true,
  unsubscribedAt: true,
  isActive: true
});
var insertSmtpConfigSchema = createInsertSchema(smtpConfig).omit({
  id: true,
  createdAt: true,
  updatedAt: true
});
var insertEmailCampaignSchema = createInsertSchema(emailCampaigns).omit({
  id: true,
  createdAt: true,
  sentAt: true,
  sentBy: true,
  sentCount: true,
  failedCount: true
});
var insertPageLayoutSchema = createInsertSchema(pageLayouts).omit({
  id: true,
  createdAt: true,
  updatedAt: true,
  createdBy: true
});
var insertHeaderTagSchema = createInsertSchema(headerTags).omit({
  id: true,
  createdAt: true,
  updatedAt: true,
  createdBy: true
});
var insertSeoSettingsSchema = createInsertSchema(seoSettings).omit({
  id: true,
  createdAt: true,
  updatedAt: true,
  createdBy: true
});
var registrationSchema = insertUserSchema.extend({
  confirmPassword: z.string(),
  agreeToTerms: z.boolean().refine((val) => val === true, {
    message: "You must agree to the Terms of Service"
  })
}).refine((data) => data.password === data.confirmPassword, {
  message: "Passwords do not match",
  path: ["confirmPassword"]
});
var southAfricanProvinces = [
  "Eastern Cape",
  "Free State",
  "Gauteng",
  "KwaZulu-Natal",
  "Limpopo",
  "Mpumalanga",
  "Northern Cape",
  "North West",
  "Western Cape"
];

// server/db.ts
import { Pool, neonConfig } from "@neondatabase/serverless";
import { drizzle } from "drizzle-orm/neon-serverless";
import ws from "ws";
neonConfig.webSocketConstructor = ws;
if (!process.env.DATABASE_URL) {
  throw new Error(
    "DATABASE_URL must be set. Did you forget to provision a database?"
  );
}
var pool = new Pool({ connectionString: process.env.DATABASE_URL });
var db = drizzle({ client: pool, schema: schema_exports });

// server/storage.ts
import { eq, desc, and, like } from "drizzle-orm";
import session from "express-session";
import connectPg from "connect-pg-simple";
var PostgresSessionStore = connectPg(session);
var DatabaseStorage = class {
  sessionStore;
  constructor() {
    this.sessionStore = new PostgresSessionStore({
      pool,
      createTableIfMissing: true
    });
  }
  // User management
  async getUser(id) {
    const [user] = await db.select().from(users).where(eq(users.id, id));
    return user || void 0;
  }
  async getUserByUsername(username) {
    const [user] = await db.select().from(users).where(eq(users.username, username));
    return user || void 0;
  }
  async getUserByEmail(email) {
    const [user] = await db.select().from(users).where(eq(users.email, email));
    return user || void 0;
  }
  async createUser(insertUser) {
    const [user] = await db.insert(users).values(insertUser).returning();
    return user;
  }
  async updateUser(id, updates) {
    const [user] = await db.update(users).set(updates).where(eq(users.id, id)).returning();
    return user || void 0;
  }
  async getAllUsers() {
    return await db.select().from(users).orderBy(desc(users.createdAt));
  }
  // Business directory
  async getBusinesses(category, location) {
    let query = db.select().from(businesses);
    const conditions = [];
    if (category) conditions.push(eq(businesses.category, category));
    if (location) conditions.push(like(businesses.location, `%${location}%`));
    if (conditions.length > 0) {
      query = query.where(and(...conditions));
    }
    return await query.orderBy(desc(businesses.createdAt));
  }
  async getBusiness(id) {
    const [business] = await db.select().from(businesses).where(eq(businesses.id, id));
    return business || void 0;
  }
  async createBusiness(business) {
    const [newBusiness] = await db.insert(businesses).values(business).returning();
    return newBusiness;
  }
  async updateBusiness(id, updates) {
    const [business] = await db.update(businesses).set(updates).where(eq(businesses.id, id)).returning();
    return business || void 0;
  }
  async deleteBusiness(id) {
    const result = await db.delete(businesses).where(eq(businesses.id, id));
    return result.rowCount ? result.rowCount > 0 : false;
  }
  // Content management
  async getContent(type, status) {
    let query = db.select().from(content);
    const conditions = [];
    if (type) conditions.push(eq(content.type, type));
    if (status) conditions.push(eq(content.status, status));
    if (conditions.length > 0) {
      query = query.where(and(...conditions));
    }
    return await query.orderBy(desc(content.createdAt));
  }
  async getContentById(id) {
    const [contentItem] = await db.select().from(content).where(eq(content.id, id));
    return contentItem || void 0;
  }
  async createContent(contentData) {
    const [newContent] = await db.insert(content).values({
      ...contentData,
      updatedAt: /* @__PURE__ */ new Date()
    }).returning();
    return newContent;
  }
  async updateContent(id, updates) {
    const [contentItem] = await db.update(content).set({
      ...updates,
      updatedAt: /* @__PURE__ */ new Date()
    }).where(eq(content.id, id)).returning();
    return contentItem || void 0;
  }
  async deleteContent(id) {
    const result = await db.delete(content).where(eq(content.id, id));
    return result.rowCount ? result.rowCount > 0 : false;
  }
  async getPublishedContent(type) {
    let query = db.select().from(content).where(eq(content.status, "published"));
    if (type) {
      query = query.where(and(eq(content.status, "published"), eq(content.type, type)));
    }
    return await query.orderBy(desc(content.publishedAt));
  }
  // WhatsApp groups
  async getWhatsAppGroups() {
    return await db.select().from(whatsappGroups).orderBy(desc(whatsappGroups.memberCount));
  }
  async getActiveWhatsAppGroups() {
    return await db.select().from(whatsappGroups).where(eq(whatsappGroups.isActive, true)).orderBy(desc(whatsappGroups.memberCount));
  }
  async updateWhatsAppGroup(id, updates) {
    const [group] = await db.update(whatsappGroups).set(updates).where(eq(whatsappGroups.id, id)).returning();
    return group || void 0;
  }
  // Contact submissions
  async createContactSubmission(submission) {
    const [newSubmission] = await db.insert(contactSubmissions).values(submission).returning();
    return newSubmission;
  }
  async getContactSubmissions(status) {
    let query = db.select().from(contactSubmissions);
    if (status) {
      query = query.where(eq(contactSubmissions.status, status));
    }
    return await query.orderBy(desc(contactSubmissions.createdAt));
  }
  async updateContactSubmission(id, updates) {
    const [submission] = await db.update(contactSubmissions).set(updates).where(eq(contactSubmissions.id, id)).returning();
    return submission || void 0;
  }
  // Products and pricing
  async getProducts(category) {
    let query = db.select().from(products).where(eq(products.isActive, true));
    if (category) {
      query = query.where(and(eq(products.isActive, true), eq(products.category, category)));
    }
    return await query.orderBy(desc(products.createdAt));
  }
  async getProduct(id) {
    const [product] = await db.select().from(products).where(eq(products.id, id));
    return product || void 0;
  }
  async createProduct(product) {
    const finalPrice = product.wholesalePrice ? product.wholesalePrice + Math.round(product.wholesalePrice * (product.adminFeePercentage || 5) / 100) : 0;
    const [newProduct] = await db.insert(products).values({
      ...product,
      finalPrice,
      updatedAt: /* @__PURE__ */ new Date()
    }).returning();
    return newProduct;
  }
  async updateProduct(id, updates) {
    if (updates.wholesalePrice || updates.adminFeePercentage) {
      const currentProduct = await this.getProduct(id);
      if (currentProduct) {
        const wholesalePrice = updates.wholesalePrice ?? currentProduct.wholesalePrice;
        const adminFeePercentage = updates.adminFeePercentage ?? currentProduct.adminFeePercentage;
        if (wholesalePrice) {
          updates.finalPrice = wholesalePrice + Math.round(wholesalePrice * adminFeePercentage / 100);
        }
      }
    }
    const [product] = await db.update(products).set({
      ...updates,
      updatedAt: /* @__PURE__ */ new Date()
    }).where(eq(products.id, id)).returning();
    return product || void 0;
  }
  async deleteProduct(id) {
    const result = await db.delete(products).where(eq(products.id, id));
    return result.rowCount ? result.rowCount > 0 : false;
  }
  // Newsletter management
  async subscribeToNewsletter(email) {
    const existing = await this.getNewsletterSubscription(email);
    if (existing) {
      if (!existing.isActive) {
        const [reactivated] = await db.update(emailNewsletters).set({ isActive: true, unsubscribedAt: null }).where(eq(emailNewsletters.email, email)).returning();
        return reactivated;
      }
      return existing;
    }
    const [subscription] = await db.insert(emailNewsletters).values({ email }).returning();
    return subscription;
  }
  async unsubscribeFromNewsletter(email) {
    const result = await db.update(emailNewsletters).set({ isActive: false, unsubscribedAt: /* @__PURE__ */ new Date() }).where(eq(emailNewsletters.email, email));
    return result.rowCount ? result.rowCount > 0 : false;
  }
  async getNewsletterSubscribers(activeOnly = true) {
    let query = db.select().from(emailNewsletters);
    if (activeOnly) {
      query = query.where(eq(emailNewsletters.isActive, true));
    }
    return await query.orderBy(desc(emailNewsletters.createdAt));
  }
  async getNewsletterSubscription(email) {
    const [subscription] = await db.select().from(emailNewsletters).where(eq(emailNewsletters.email, email));
    return subscription || void 0;
  }
  // SMTP configuration
  async getSmtpConfig() {
    const [config] = await db.select().from(smtpConfig).where(eq(smtpConfig.isActive, true)).orderBy(desc(smtpConfig.createdAt));
    return config || void 0;
  }
  async updateSmtpConfig(config) {
    await db.update(smtpConfig).set({ isActive: false });
    const [newConfig] = await db.insert(smtpConfig).values({ ...config, updatedAt: /* @__PURE__ */ new Date() }).returning();
    return newConfig;
  }
  async testSmtpConnection(config) {
    return true;
  }
  // Email campaigns
  async createEmailCampaign(campaign) {
    const [newCampaign] = await db.insert(emailCampaigns).values(campaign).returning();
    return newCampaign;
  }
  async updateEmailCampaign(id, updates) {
    const [campaign] = await db.update(emailCampaigns).set(updates).where(eq(emailCampaigns.id, id)).returning();
    return campaign || void 0;
  }
  async getEmailCampaigns() {
    return await db.select().from(emailCampaigns).orderBy(desc(emailCampaigns.createdAt));
  }
  async getEmailCampaign(id) {
    const [campaign] = await db.select().from(emailCampaigns).where(eq(emailCampaigns.id, id));
    return campaign || void 0;
  }
  // Page layouts for website editor
  async getPageLayout(pageName) {
    const [layout] = await db.select().from(pageLayouts).where(eq(pageLayouts.pageName, pageName));
    return layout || void 0;
  }
  async createPageLayout(layout) {
    const [newLayout] = await db.insert(pageLayouts).values(layout).returning();
    return newLayout;
  }
  async updatePageLayout(pageName, updates) {
    const [updatedLayout] = await db.update(pageLayouts).set({ ...updates, updatedAt: /* @__PURE__ */ new Date() }).where(eq(pageLayouts.pageName, pageName)).returning();
    return updatedLayout || void 0;
  }
  async deletePageLayout(pageName) {
    const result = await db.delete(pageLayouts).where(eq(pageLayouts.pageName, pageName));
    return result.rowCount > 0;
  }
  async getAllPageLayouts() {
    return await db.select().from(pageLayouts).orderBy(pageLayouts.createdAt);
  }
  // Header tags for tracking codes
  async getHeaderTags() {
    return await db.select().from(headerTags).orderBy(headerTags.priority);
  }
  async getActiveHeaderTags() {
    return await db.select().from(headerTags).where(eq(headerTags.isActive, true)).orderBy(headerTags.priority);
  }
  async getHeaderTag(id) {
    const [tag] = await db.select().from(headerTags).where(eq(headerTags.id, id));
    return tag || void 0;
  }
  async createHeaderTag(tag) {
    const [newTag] = await db.insert(headerTags).values(tag).returning();
    return newTag;
  }
  async updateHeaderTag(id, updates) {
    const [updatedTag] = await db.update(headerTags).set({ ...updates, updatedAt: /* @__PURE__ */ new Date() }).where(eq(headerTags.id, id)).returning();
    return updatedTag || void 0;
  }
  async deleteHeaderTag(id) {
    const result = await db.delete(headerTags).where(eq(headerTags.id, id));
    return result.rowCount > 0;
  }
  // SEO settings for pages
  async getSeoSettings() {
    return await db.select().from(seoSettings).orderBy(seoSettings.pageName);
  }
  async getSeoSettingsByPage(pageName) {
    const [settings] = await db.select().from(seoSettings).where(eq(seoSettings.pageName, pageName));
    return settings || void 0;
  }
  async getSeoSettingsById(id) {
    const [settings] = await db.select().from(seoSettings).where(eq(seoSettings.id, id));
    return settings || void 0;
  }
  async createSeoSettings(settings) {
    const [newSettings] = await db.insert(seoSettings).values(settings).returning();
    return newSettings;
  }
  async updateSeoSettings(id, updates) {
    const [updatedSettings] = await db.update(seoSettings).set({ ...updates, updatedAt: /* @__PURE__ */ new Date() }).where(eq(seoSettings.id, id)).returning();
    return updatedSettings || void 0;
  }
  async deleteSeoSettings(id) {
    const result = await db.delete(seoSettings).where(eq(seoSettings.id, id));
    return result.rowCount > 0;
  }
};
var storage = new DatabaseStorage();

// server/auth.ts
var scryptAsync = promisify(scrypt);
async function hashPassword(password) {
  const salt = randomBytes(16).toString("hex");
  const buf = await scryptAsync(password, salt, 64);
  return `${buf.toString("hex")}.${salt}`;
}
async function comparePasswords(supplied, stored) {
  const [hashed, salt] = stored.split(".");
  const hashedBuf = Buffer.from(hashed, "hex");
  const suppliedBuf = await scryptAsync(supplied, salt, 64);
  return timingSafeEqual(hashedBuf, suppliedBuf);
}
function setupAuth(app2) {
  const sessionSettings = {
    secret: process.env.SESSION_SECRET,
    resave: false,
    saveUninitialized: false,
    store: storage.sessionStore
  };
  app2.set("trust proxy", 1);
  app2.use(session2(sessionSettings));
  app2.use(passport.initialize());
  app2.use(passport.session());
  passport.use(
    new LocalStrategy(async (username, password, done) => {
      const user = await storage.getUserByUsername(username);
      if (!user || !await comparePasswords(password, user.password)) {
        return done(null, false);
      } else {
        return done(null, user);
      }
    })
  );
  passport.serializeUser((user, done) => done(null, user.id));
  passport.deserializeUser(async (id, done) => {
    const user = await storage.getUser(id);
    done(null, user);
  });
  app2.post("/api/register", async (req, res, next) => {
    try {
      const existingUser = await storage.getUserByUsername(req.body.username);
      if (existingUser) {
        return res.status(400).send("Username already exists");
      }
      const fullName = req.body.fullName || "";
      const nameParts = fullName.trim().split(" ");
      const firstName = nameParts[0] || "";
      const lastName = nameParts.slice(1).join(" ") || "";
      const user = await storage.createUser({
        username: req.body.username,
        email: req.body.email,
        password: await hashPassword(req.body.password),
        // Support both old and new schema fields
        fullName: req.body.fullName || "",
        // Old schema field
        firstName,
        lastName,
        mobile: req.body.mobile || "",
        location: req.body.location || "",
        // Old schema field
        province: req.body.location || ""
        // New schema field
      });
      req.login(user, (err) => {
        if (err) return next(err);
        res.status(201).json(user);
      });
    } catch (error) {
      next(error);
    }
  });
  app2.post("/api/login", passport.authenticate("local"), (req, res) => {
    res.status(200).json(req.user);
  });
  app2.post("/api/logout", (req, res, next) => {
    req.logout((err) => {
      if (err) return next(err);
      res.sendStatus(200);
    });
  });
  app2.get("/api/user", (req, res) => {
    if (!req.isAuthenticated()) return res.sendStatus(401);
    res.json(req.user);
  });
}

// server/routes.ts
import { z as z2 } from "zod";
import nodemailer from "nodemailer";
import path from "path";
function registerRoutes(app2) {
  setupAuth(app2);
  app2.post("/api/contact", async (req, res) => {
    try {
      const validatedData = insertContactSchema.parse(req.body);
      const submission = await storage.createContactSubmission(validatedData);
      res.status(201).json(submission);
    } catch (error) {
      console.error("Contact submission error:", error);
      res.status(400).json({ message: "Invalid submission data" });
    }
  });
  app2.get("/api/businesses", async (req, res) => {
    try {
      const { category, location } = req.query;
      const businesses2 = await storage.getBusinesses(
        category,
        location
      );
      res.json(businesses2);
    } catch (error) {
      console.error("Get businesses error:", error);
      res.status(500).json({ message: "Failed to fetch businesses" });
    }
  });
  app2.post("/api/businesses", async (req, res) => {
    try {
      if (!req.user) {
        return res.status(401).json({ message: "Authentication required" });
      }
      const validatedData = insertBusinessSchema.parse(req.body);
      const business = await storage.createBusiness({
        ...validatedData,
        createdBy: req.user.id
      });
      res.status(201).json(business);
    } catch (error) {
      console.error("Create business error:", error);
      res.status(400).json({ message: "Invalid business data" });
    }
  });
  app2.get("/api/content", async (req, res) => {
    try {
      const { type, status } = req.query;
      if (status === "published" || !req.user) {
        const content3 = await storage.getPublishedContent(type);
        return res.json(content3);
      }
      if (req.user?.role === "admin") {
        const content3 = await storage.getContent(type, status);
        return res.json(content3);
      }
      const content2 = await storage.getPublishedContent(type);
      res.json(content2);
    } catch (error) {
      console.error("Get content error:", error);
      res.status(500).json({ message: "Failed to fetch content" });
    }
  });
  app2.post("/api/content", async (req, res) => {
    try {
      if (!req.user || req.user.role !== "admin") {
        return res.status(403).json({ message: "Admin access required" });
      }
      const validatedData = insertContentSchema.parse(req.body);
      const content2 = await storage.createContent({
        ...validatedData,
        authorId: req.user.id
      });
      res.status(201).json(content2);
    } catch (error) {
      console.error("Create content error:", error);
      res.status(400).json({ message: "Invalid content data" });
    }
  });
  app2.put("/api/content/:id", async (req, res) => {
    try {
      if (!req.user || req.user.role !== "admin") {
        return res.status(403).json({ message: "Admin access required" });
      }
      const id = parseInt(req.params.id);
      const updates = req.body;
      if (updates.status === "published" && !updates.publishedAt) {
        updates.publishedAt = /* @__PURE__ */ new Date();
      }
      const content2 = await storage.updateContent(id, updates);
      if (!content2) {
        return res.status(404).json({ message: "Content not found" });
      }
      res.json(content2);
    } catch (error) {
      console.error("Update content error:", error);
      res.status(400).json({ message: "Failed to update content" });
    }
  });
  app2.delete("/api/content/:id", async (req, res) => {
    try {
      if (!req.user || req.user.role !== "admin") {
        return res.status(403).json({ message: "Admin access required" });
      }
      const id = parseInt(req.params.id);
      const deleted = await storage.deleteContent(id);
      if (!deleted) {
        return res.status(404).json({ message: "Content not found" });
      }
      res.status(204).send();
    } catch (error) {
      console.error("Delete content error:", error);
      res.status(500).json({ message: "Failed to delete content" });
    }
  });
  app2.get("/api/whatsapp-groups", async (req, res) => {
    try {
      const groups = await storage.getActiveWhatsAppGroups();
      res.json(groups);
    } catch (error) {
      console.error("Get WhatsApp groups error:", error);
      res.status(500).json({ message: "Failed to fetch WhatsApp groups" });
    }
  });
  app2.get("/api/products", async (req, res) => {
    try {
      const { category } = req.query;
      const products2 = await storage.getProducts(category);
      res.json(products2);
    } catch (error) {
      console.error("Get products error:", error);
      res.status(500).json({ message: "Failed to fetch products" });
    }
  });
  app2.post("/api/products", async (req, res) => {
    try {
      if (!req.user || req.user.role !== "admin") {
        return res.status(403).json({ message: "Admin access required" });
      }
      const validatedData = insertProductSchema.parse(req.body);
      const product = await storage.createProduct(validatedData);
      res.status(201).json(product);
    } catch (error) {
      console.error("Create product error:", error);
      res.status(400).json({ message: "Invalid product data" });
    }
  });
  app2.get("/api/admin/users", async (req, res) => {
    try {
      if (!req.user || req.user.role !== "admin") {
        return res.status(403).json({ message: "Admin access required" });
      }
      const users2 = await storage.getAllUsers();
      const safeUsers = users2.map(({ password, ...user }) => user);
      res.json(safeUsers);
    } catch (error) {
      console.error("Get users error:", error);
      res.status(500).json({ message: "Failed to fetch users" });
    }
  });
  app2.get("/api/admin/contact-submissions", async (req, res) => {
    try {
      if (!req.user || req.user.role !== "admin") {
        return res.status(403).json({ message: "Admin access required" });
      }
      const { status } = req.query;
      const submissions = await storage.getContactSubmissions(status);
      res.json(submissions);
    } catch (error) {
      console.error("Get contact submissions error:", error);
      res.status(500).json({ message: "Failed to fetch contact submissions" });
    }
  });
  app2.put("/api/admin/users/:id", async (req, res) => {
    try {
      if (!req.user || req.user.role !== "admin") {
        return res.status(403).json({ message: "Admin access required" });
      }
      const id = parseInt(req.params.id);
      const updates = req.body;
      delete updates.password;
      const user = await storage.updateUser(id, updates);
      if (!user) {
        return res.status(404).json({ message: "User not found" });
      }
      const { password, ...safeUser } = user;
      res.json(safeUser);
    } catch (error) {
      console.error("Update user error:", error);
      res.status(400).json({ message: "Failed to update user" });
    }
  });
  app2.get("/api/admin/stats", async (req, res) => {
    try {
      if (!req.user || req.user.role !== "admin") {
        return res.status(403).json({ message: "Admin access required" });
      }
      const [users2, businesses2, contentItems, groups, submissions] = await Promise.all([
        storage.getAllUsers(),
        storage.getBusinesses(),
        storage.getContent(),
        storage.getActiveWhatsAppGroups(),
        storage.getContactSubmissions("pending")
      ]);
      const stats = {
        totalUsers: users2.length,
        activeUsers: users2.filter((u) => u.isActive).length,
        totalBusinesses: businesses2.length,
        totalContent: contentItems.length,
        publishedContent: contentItems.filter((c) => c.status === "published").length,
        whatsappGroups: groups.length,
        totalMembers: groups.reduce((sum, group) => sum + (group.memberCount || 0), 0),
        pendingSubmissions: submissions.length
      };
      res.json(stats);
    } catch (error) {
      console.error("Get admin stats error:", error);
      res.status(500).json({ message: "Failed to fetch admin stats" });
    }
  });
  app2.post("/api/newsletter/subscribe", async (req, res) => {
    try {
      const validatedData = insertEmailNewsletterSchema.parse(req.body);
      const subscription = await storage.subscribeToNewsletter(validatedData.email);
      res.status(201).json(subscription);
    } catch (error) {
      console.error("Newsletter subscription error:", error);
      if (error instanceof z2.ZodError) {
        return res.status(400).json({ message: error.errors[0]?.message || "Invalid email address" });
      }
      res.status(400).json({ message: "Subscription failed" });
    }
  });
  app2.post("/api/newsletter/unsubscribe", async (req, res) => {
    try {
      const { email } = req.body;
      if (!email) {
        return res.status(400).json({ message: "Email is required" });
      }
      const success = await storage.unsubscribeFromNewsletter(email);
      if (success) {
        res.json({ message: "Successfully unsubscribed" });
      } else {
        res.status(404).json({ message: "Email not found" });
      }
    } catch (error) {
      console.error("Newsletter unsubscribe error:", error);
      res.status(500).json({ message: "Unsubscribe failed" });
    }
  });
  app2.get("/api/admin/newsletter/subscribers", async (req, res) => {
    try {
      if (!req.user || req.user.role !== "admin") {
        return res.status(403).json({ message: "Admin access required" });
      }
      const activeOnly = req.query.active !== "false";
      const subscribers = await storage.getNewsletterSubscribers(activeOnly);
      res.json(subscribers);
    } catch (error) {
      console.error("Get newsletter subscribers error:", error);
      res.status(500).json({ message: "Failed to fetch subscribers" });
    }
  });
  app2.get("/api/admin/smtp-config", async (req, res) => {
    try {
      if (!req.user || req.user.role !== "admin") {
        return res.status(403).json({ message: "Admin access required" });
      }
      const config = await storage.getSmtpConfig();
      if (config) {
        const { password, ...safeConfig } = config;
        res.json(safeConfig);
      } else {
        res.status(404).json({ message: "No SMTP configuration found" });
      }
    } catch (error) {
      console.error("Get SMTP config error:", error);
      res.status(500).json({ message: "Failed to fetch SMTP configuration" });
    }
  });
  app2.post("/api/admin/smtp-config", async (req, res) => {
    try {
      if (!req.user || req.user.role !== "admin") {
        return res.status(403).json({ message: "Admin access required" });
      }
      const validatedData = insertSmtpConfigSchema.parse(req.body);
      const config = await storage.updateSmtpConfig(validatedData);
      const { password, ...safeConfig } = config;
      res.status(201).json(safeConfig);
    } catch (error) {
      console.error("Update SMTP config error:", error);
      if (error instanceof z2.ZodError) {
        return res.status(400).json({ message: error.errors[0]?.message || "Invalid SMTP configuration" });
      }
      res.status(500).json({ message: "Failed to update SMTP configuration" });
    }
  });
  app2.post("/api/admin/smtp-test", async (req, res) => {
    try {
      if (!req.user || req.user.role !== "admin") {
        return res.status(403).json({ message: "Admin access required" });
      }
      const config = await storage.getSmtpConfig();
      if (!config) {
        return res.status(404).json({ message: "No SMTP configuration found" });
      }
      const transporter = nodemailer.createTransport({
        host: config.host,
        port: config.port,
        auth: {
          user: config.username,
          pass: config.password
        }
      });
      await transporter.verify();
      await transporter.sendMail({
        from: `"${config.fromName}" <${config.fromEmail}>`,
        to: req.user.email,
        subject: "LoveMedia Foundation - SMTP Test",
        text: "This is a test email to verify your SMTP configuration is working correctly.",
        html: `
          <h2>SMTP Test Successful</h2>
          <p>This is a test email to verify your SMTP configuration is working correctly.</p>
          <p>Your email system is now ready to send newsletters to your community!</p>
          <br>
          <p>Best regards,<br>LoveMedia Foundation</p>
        `
      });
      res.json({ message: "SMTP test successful! Check your email for confirmation." });
    } catch (error) {
      console.error("SMTP test error:", error);
      res.status(500).json({ message: `SMTP test failed: ${error.message}` });
    }
  });
  app2.post("/api/admin/send-mass-email", async (req, res) => {
    try {
      if (!req.user || req.user.role !== "admin") {
        return res.status(403).json({ message: "Admin access required" });
      }
      const { subject, content: content2 } = req.body;
      if (!subject || !content2) {
        return res.status(400).json({ message: "Subject and content are required" });
      }
      const config = await storage.getSmtpConfig();
      if (!config) {
        return res.status(404).json({ message: "No SMTP configuration found. Please configure SMTP first." });
      }
      const subscribers = await storage.getNewsletterSubscribers(true);
      if (subscribers.length === 0) {
        return res.status(400).json({ message: "No active subscribers found" });
      }
      const campaign = await storage.createEmailCampaign({
        subject,
        content: content2,
        recipientCount: subscribers.length,
        status: "sending",
        sentBy: req.user.id
      });
      const transporter = nodemailer.createTransport({
        host: config.host,
        port: config.port,
        auth: {
          user: config.username,
          pass: config.password
        }
      });
      let sentCount = 0;
      let failedCount = 0;
      const batchSize = 10;
      for (let i = 0; i < subscribers.length; i += batchSize) {
        const batch = subscribers.slice(i, i + batchSize);
        await Promise.allSettled(
          batch.map(async (subscriber) => {
            try {
              await transporter.sendMail({
                from: `"${config.fromName}" <${config.fromEmail}>`,
                to: subscriber.email,
                subject,
                html: `
                  ${content2}
                  <br><br>
                  <hr>
                  <small>
                    <p>You received this email because you subscribed to LoveMedia Foundation newsletter.</p>
                    <p><a href="${process.env.REPLIT_DOMAIN || "http://localhost:5000"}/api/newsletter/unsubscribe?email=${encodeURIComponent(subscriber.email)}">Unsubscribe</a></p>
                  </small>
                `
              });
              sentCount++;
            } catch (error) {
              console.error(`Failed to send email to ${subscriber.email}:`, error);
              failedCount++;
            }
          })
        );
        if (i + batchSize < subscribers.length) {
          await new Promise((resolve) => setTimeout(resolve, 1e3));
        }
      }
      await storage.updateEmailCampaign(campaign.id, {
        status: failedCount === 0 ? "sent" : "failed",
        sentCount,
        failedCount,
        sentAt: /* @__PURE__ */ new Date()
      });
      res.json({
        message: `Mass email sent successfully!`,
        campaignId: campaign.id,
        totalRecipients: subscribers.length,
        sentCount,
        failedCount
      });
    } catch (error) {
      console.error("Mass email error:", error);
      res.status(500).json({ message: "Failed to send mass email" });
    }
  });
  app2.get("/api/admin/email-campaigns", async (req, res) => {
    try {
      if (!req.user || req.user.role !== "admin") {
        return res.status(403).json({ message: "Admin access required" });
      }
      const campaigns = await storage.getEmailCampaigns();
      res.json(campaigns);
    } catch (error) {
      console.error("Get email campaigns error:", error);
      res.status(500).json({ message: "Failed to fetch email campaigns" });
    }
  });
  app2.get("/api/admin/page-layout/:pageName", async (req, res) => {
    try {
      if (!req.user || req.user.role !== "admin") {
        return res.status(403).json({ message: "Admin access required" });
      }
      const { pageName } = req.params;
      const layout = await storage.getPageLayout(pageName);
      if (!layout) {
        return res.status(404).json({ message: "Page layout not found" });
      }
      res.json(layout);
    } catch (error) {
      console.error("Get page layout error:", error);
      res.status(500).json({ message: "Failed to fetch page layout" });
    }
  });
  app2.post("/api/admin/page-layout", async (req, res) => {
    try {
      if (!req.user || req.user.role !== "admin") {
        return res.status(403).json({ message: "Admin access required" });
      }
      const validatedData = insertPageLayoutSchema.parse(req.body);
      const existingLayout = await storage.getPageLayout(validatedData.pageName);
      if (existingLayout) {
        const updatedLayout = await storage.updatePageLayout(
          validatedData.pageName,
          validatedData
        );
        return res.json(updatedLayout);
      }
      const layout = await storage.createPageLayout({
        ...validatedData,
        createdBy: req.user.id
      });
      res.status(201).json(layout);
    } catch (error) {
      console.error("Create/update page layout error:", error);
      res.status(400).json({ message: "Invalid page layout data" });
    }
  });
  app2.put("/api/admin/page-layout/:pageName", async (req, res) => {
    try {
      if (!req.user || req.user.role !== "admin") {
        return res.status(403).json({ message: "Admin access required" });
      }
      const { pageName } = req.params;
      const updates = req.body;
      const layout = await storage.updatePageLayout(pageName, updates);
      if (!layout) {
        return res.status(404).json({ message: "Page layout not found" });
      }
      res.json(layout);
    } catch (error) {
      console.error("Update page layout error:", error);
      res.status(400).json({ message: "Failed to update page layout" });
    }
  });
  app2.delete("/api/admin/page-layout/:pageName", async (req, res) => {
    try {
      if (!req.user || req.user.role !== "admin") {
        return res.status(403).json({ message: "Admin access required" });
      }
      const { pageName } = req.params;
      const deleted = await storage.deletePageLayout(pageName);
      if (!deleted) {
        return res.status(404).json({ message: "Page layout not found" });
      }
      res.status(204).send();
    } catch (error) {
      console.error("Delete page layout error:", error);
      res.status(500).json({ message: "Failed to delete page layout" });
    }
  });
  app2.get("/api/admin/page-layouts", async (req, res) => {
    try {
      if (!req.user || req.user.role !== "admin") {
        return res.status(403).json({ message: "Admin access required" });
      }
      const layouts = await storage.getAllPageLayouts();
      res.json(layouts);
    } catch (error) {
      console.error("Get page layouts error:", error);
      res.status(500).json({ message: "Failed to fetch page layouts" });
    }
  });
  app2.get("/api/admin/header-tags", async (req, res) => {
    try {
      if (!req.user || req.user.role !== "admin") {
        return res.status(403).json({ message: "Admin access required" });
      }
      const tags = await storage.getHeaderTags();
      res.json(tags);
    } catch (error) {
      console.error("Get header tags error:", error);
      res.status(500).json({ message: "Failed to fetch header tags" });
    }
  });
  app2.get("/api/header-tags/active", async (req, res) => {
    try {
      const tags = await storage.getActiveHeaderTags();
      res.json(tags);
    } catch (error) {
      console.error("Get active header tags error:", error);
      res.status(500).json({ message: "Failed to fetch active header tags" });
    }
  });
  app2.post("/api/admin/header-tags", async (req, res) => {
    try {
      if (!req.user || req.user.role !== "admin") {
        return res.status(403).json({ message: "Admin access required" });
      }
      const validatedData = insertHeaderTagSchema.parse(req.body);
      const tag = await storage.createHeaderTag({
        ...validatedData,
        createdBy: req.user.id
      });
      res.status(201).json(tag);
    } catch (error) {
      console.error("Create header tag error:", error);
      res.status(400).json({ message: "Invalid header tag data" });
    }
  });
  app2.put("/api/admin/header-tags/:id", async (req, res) => {
    try {
      if (!req.user || req.user.role !== "admin") {
        return res.status(403).json({ message: "Admin access required" });
      }
      const { id } = req.params;
      const updates = req.body;
      const tag = await storage.updateHeaderTag(parseInt(id), updates);
      if (!tag) {
        return res.status(404).json({ message: "Header tag not found" });
      }
      res.json(tag);
    } catch (error) {
      console.error("Update header tag error:", error);
      res.status(400).json({ message: "Failed to update header tag" });
    }
  });
  app2.delete("/api/admin/header-tags/:id", async (req, res) => {
    try {
      if (!req.user || req.user.role !== "admin") {
        return res.status(403).json({ message: "Admin access required" });
      }
      const { id } = req.params;
      const deleted = await storage.deleteHeaderTag(parseInt(id));
      if (!deleted) {
        return res.status(404).json({ message: "Header tag not found" });
      }
      res.status(204).send();
    } catch (error) {
      console.error("Delete header tag error:", error);
      res.status(500).json({ message: "Failed to delete header tag" });
    }
  });
  app2.get("/api/admin/seo-settings", async (req, res) => {
    try {
      if (!req.user || req.user.role !== "admin") {
        return res.status(403).json({ message: "Admin access required" });
      }
      const settings = await storage.getSeoSettings();
      res.json(settings);
    } catch (error) {
      console.error("Error fetching SEO settings:", error);
      res.status(500).json({ error: "Failed to fetch SEO settings" });
    }
  });
  app2.get("/api/admin/seo-settings/:id", async (req, res) => {
    try {
      if (!req.user || req.user.role !== "admin") {
        return res.status(403).json({ message: "Admin access required" });
      }
      const id = parseInt(req.params.id);
      const settings = await storage.getSeoSettingsById(id);
      if (!settings) {
        return res.status(404).json({ error: "SEO settings not found" });
      }
      res.json(settings);
    } catch (error) {
      console.error("Error fetching SEO settings:", error);
      res.status(500).json({ error: "Failed to fetch SEO settings" });
    }
  });
  app2.get("/api/seo-settings/:pageName", async (req, res) => {
    try {
      const pageName = req.params.pageName;
      const settings = await storage.getSeoSettingsByPage(pageName);
      if (!settings) {
        return res.status(404).json({ error: "SEO settings not found" });
      }
      res.json(settings);
    } catch (error) {
      console.error("Error fetching SEO settings:", error);
      res.status(500).json({ error: "Failed to fetch SEO settings" });
    }
  });
  app2.post("/api/admin/seo-settings", async (req, res) => {
    try {
      if (!req.user || req.user.role !== "admin") {
        return res.status(403).json({ message: "Admin access required" });
      }
      const settings = await storage.createSeoSettings({ ...req.body, createdBy: req.user.id });
      res.status(201).json(settings);
    } catch (error) {
      console.error("Error creating SEO settings:", error);
      res.status(500).json({ error: "Failed to create SEO settings" });
    }
  });
  app2.put("/api/admin/seo-settings/:id", async (req, res) => {
    try {
      if (!req.user || req.user.role !== "admin") {
        return res.status(403).json({ message: "Admin access required" });
      }
      const id = parseInt(req.params.id);
      const settings = await storage.updateSeoSettings(id, req.body);
      if (!settings) {
        return res.status(404).json({ error: "SEO settings not found" });
      }
      res.json(settings);
    } catch (error) {
      console.error("Error updating SEO settings:", error);
      res.status(500).json({ error: "Failed to update SEO settings" });
    }
  });
  app2.delete("/api/admin/seo-settings/:id", async (req, res) => {
    try {
      if (!req.user || req.user.role !== "admin") {
        return res.status(403).json({ message: "Admin access required" });
      }
      const id = parseInt(req.params.id);
      const deleted = await storage.deleteSeoSettings(id);
      if (!deleted) {
        return res.status(404).json({ error: "SEO settings not found" });
      }
      res.status(204).send();
    } catch (error) {
      console.error("Error deleting SEO settings:", error);
      res.status(500).json({ error: "Failed to delete SEO settings" });
    }
  });
  app2.get("/sitemap.xml", (req, res) => {
    res.setHeader("Content-Type", "application/xml");
    res.setHeader("Cache-Control", "public, max-age=86400");
    res.sendFile(path.join(process.cwd(), "client/public/sitemap.xml"));
  });
  app2.get("/robots.txt", (req, res) => {
    res.setHeader("Content-Type", "text/plain");
    res.setHeader("Cache-Control", "public, max-age=86400");
    res.sendFile(path.join(process.cwd(), "client/public/robots.txt"));
  });
  const httpServer = createServer(app2);
  return httpServer;
}

// server/vite.ts
import express from "express";
import fs from "fs";
import path3 from "path";
import { createServer as createViteServer, createLogger } from "vite";

// vite.config.ts
import { defineConfig } from "vite";
import react from "@vitejs/plugin-react";
import path2 from "path";
import runtimeErrorOverlay from "@replit/vite-plugin-runtime-error-modal";
var vite_config_default = defineConfig({
  plugins: [
    react(),
    runtimeErrorOverlay(),
    ...process.env.NODE_ENV !== "production" && process.env.REPL_ID !== void 0 ? [
      await import("@replit/vite-plugin-cartographer").then(
        (m) => m.cartographer()
      )
    ] : []
  ],
  resolve: {
    alias: {
      "@": path2.resolve(import.meta.dirname, "client", "src"),
      "@shared": path2.resolve(import.meta.dirname, "shared"),
      "@assets": path2.resolve(import.meta.dirname, "attached_assets")
    }
  },
  root: path2.resolve(import.meta.dirname, "client"),
  build: {
    outDir: path2.resolve(import.meta.dirname, "dist/public"),
    emptyOutDir: true
  },
  server: {
    fs: {
      strict: true,
      deny: ["**/.*"]
    }
  }
});

// server/vite.ts
import { nanoid } from "nanoid";
var viteLogger = createLogger();
function log(message, source = "express") {
  const formattedTime = (/* @__PURE__ */ new Date()).toLocaleTimeString("en-US", {
    hour: "numeric",
    minute: "2-digit",
    second: "2-digit",
    hour12: true
  });
  console.log(`${formattedTime} [${source}] ${message}`);
}
async function setupVite(app2, server) {
  const serverOptions = {
    middlewareMode: true,
    hmr: { server },
    allowedHosts: true
  };
  const vite = await createViteServer({
    ...vite_config_default,
    configFile: false,
    customLogger: {
      ...viteLogger,
      error: (msg, options) => {
        viteLogger.error(msg, options);
        process.exit(1);
      }
    },
    server: serverOptions,
    appType: "custom"
  });
  app2.use(vite.middlewares);
  app2.use("*", async (req, res, next) => {
    const url = req.originalUrl;
    try {
      const clientTemplate = path3.resolve(
        import.meta.dirname,
        "..",
        "client",
        "index.html"
      );
      let template = await fs.promises.readFile(clientTemplate, "utf-8");
      template = template.replace(
        `src="/src/main.tsx"`,
        `src="/src/main.tsx?v=${nanoid()}"`
      );
      const page = await vite.transformIndexHtml(url, template);
      res.status(200).set({ "Content-Type": "text/html" }).end(page);
    } catch (e) {
      vite.ssrFixStacktrace(e);
      next(e);
    }
  });
}
function serveStatic(app2) {
  const distPath = path3.resolve(import.meta.dirname, "public");
  if (!fs.existsSync(distPath)) {
    throw new Error(
      `Could not find the build directory: ${distPath}, make sure to build the client first`
    );
  }
  app2.use(express.static(distPath));
  app2.use("*", (_req, res) => {
    res.sendFile(path3.resolve(distPath, "index.html"));
  });
}

// server/index.ts
var app = express2();
app.use(express2.json());
app.use(express2.urlencoded({ extended: false }));
app.use((req, res, next) => {
  const start = Date.now();
  const path4 = req.path;
  let capturedJsonResponse = void 0;
  const originalResJson = res.json;
  res.json = function(bodyJson, ...args) {
    capturedJsonResponse = bodyJson;
    return originalResJson.apply(res, [bodyJson, ...args]);
  };
  res.on("finish", () => {
    const duration = Date.now() - start;
    if (path4.startsWith("/api")) {
      let logLine = `${req.method} ${path4} ${res.statusCode} in ${duration}ms`;
      if (capturedJsonResponse) {
        logLine += ` :: ${JSON.stringify(capturedJsonResponse)}`;
      }
      if (logLine.length > 80) {
        logLine = logLine.slice(0, 79) + "\u2026";
      }
      log(logLine);
    }
  });
  next();
});
(async () => {
  const server = await registerRoutes(app);
  app.use((err, _req, res, _next) => {
    const status = err.status || err.statusCode || 500;
    const message = err.message || "Internal Server Error";
    res.status(status).json({ message });
    throw err;
  });
  if (app.get("env") === "development") {
    await setupVite(app, server);
  } else {
    serveStatic(app);
  }
  const port = parseInt(process.env.PORT || "5000", 10);
  server.listen({
    port,
    host: "0.0.0.0",
    reusePort: true
  }, () => {
    log(`serving on port ${port}`);
  });
})();
