import fs from 'fs';
import path from 'path';
import XLSX from 'xlsx';
import { db } from '../server/db.ts';
import { businesses, businessCategories } from '../shared/schema.ts';
import { eq } from 'drizzle-orm';

// Category mapping for the Excel files
const CATEGORY_MAPPING = {
  'Agriculture': 'Agriculture',
  'Car_&_Automotive': 'Car & Automotive', 
  'Education_Training_Lessons': 'Education & Training',
  'Engineering': 'Engineering',
  'Events_&_entertainment': 'Events & Entertainment',
  'Financial_&_Insurance_Services': 'Financial & Insurance Services',
  'Home_Building_&_Trade': 'Home Building & Trade',
  'Hospitality': 'Hospitality',
  'IT_Computer_&_Technology': 'IT & Technology',
  'Marketing_Business,_and_Legal': 'Marketing & Legal',
  'Medical_Wellness_&_Beauty': 'Medical & Beauty',
  'Public_Service_&_Government': 'Public Service & Government',
  'Security_Systems_&_Protection': 'Security & Protection',
  'Telecommunication': 'Telecommunication',
  'Trade_&_Industry': 'Trade & Industry',
  'Transport': 'Transport'
};

const CATEGORY_ICONS = {
  'Agriculture': 'Sprout',
  'Car & Automotive': 'Car',
  'Education & Training': 'GraduationCap',
  'Engineering': 'Cog',
  'Events & Entertainment': 'Calendar',
  'Financial & Insurance Services': 'DollarSign',
  'Home Building & Trade': 'Home',
  'Hospitality': 'Coffee',
  'IT & Technology': 'Monitor',
  'Marketing & Legal': 'Briefcase',
  'Medical & Beauty': 'Heart',
  'Public Service & Government': 'Building',
  'Security & Protection': 'Shield',
  'Telecommunication': 'Phone',
  'Trade & Industry': 'Factory',
  'Transport': 'Truck'
};

function extractCategoryFromFilename(filename) {
  // Extract category from filename like "yep-co-za_Agriculture_1753630080604.xlsx"
  const parts = filename.split('_');
  if (parts.length >= 2) {
    let categoryPart = parts.slice(1, -1).join('_'); // Remove first part (yep-co-za) and last part (timestamp)
    return CATEGORY_MAPPING[categoryPart] || categoryPart.replace(/_/g, ' ');
  }
  return 'Other';
}

function normalizeBusinessData(row, category) {
  // Handle various possible column names from Excel files
  const name = row.Name || row.name || row.business_name || row['Business Name'] || '';
  const description = row.Description || row.description || row.about || row.About || '';
  const location = row.Location || row.location || row.address || row.Address || row.Province || row.province || 'South Africa';
  const phone = row.Phone || row.phone || row.contact_phone || row['Contact Phone'] || row.telephone || row.Telephone || '';
  const email = row.Email || row.email || row.contact_email || row['Contact Email'] || '';
  const website = row.Website || row.website || row.url || row.URL || '';
  const contact_person = row['Contact Person'] || row.contact_person || row.owner || row.Owner || '';
  const slogan = row.Slogan || row.slogan || row.tagline || row.Tagline || '';
  
  // Handle coordinates
  let latitude = null;
  let longitude = null;
  if (row.Latitude || row.latitude) {
    latitude = parseFloat(row.Latitude || row.latitude);
  }
  if (row.Longitude || row.longitude) {
    longitude = parseFloat(row.Longitude || row.longitude);
  }

  // Clean and validate data
  const cleanName = String(name).trim();
  if (!cleanName || cleanName.length < 2) {
    return null; // Skip invalid entries
  }

  return {
    name: cleanName,
    description: String(description || `${cleanName} - ${category} services`).trim(),
    category: category,
    location: String(location).trim() || 'South Africa',
    contactEmail: String(email || '').trim() || null,
    contactPhone: String(phone || '').trim() || null,
    website: String(website || '').trim() || null,
    whatsappNumber: null,
    contactPerson: String(contact_person || '').trim() || null,
    slogan: String(slogan || '').trim() || null,
    latitude: latitude ? String(latitude) : null,
    longitude: longitude ? String(longitude) : null,
    imageUrl: null,
    originalImageUrl: null,
    isVerified: false,
    rating: 0,
    reviewCount: 0,
    isFeatured: false,
    featuredUntil: null,
    highlightColor: null,
    topPlacement: false,
    topPlacementUntil: null,
    createdBy: null,
    storeId: row.store_id || row.Store_ID || row.ID || null
  };
}

async function processExcelFile(filePath) {
  console.log(`\n📄 Processing: ${path.basename(filePath)}`);
  
  const category = extractCategoryFromFilename(path.basename(filePath));
  console.log(`   Category: ${category}`);
  
  try {
    const workbook = XLSX.readFile(filePath);
    const sheetName = workbook.SheetNames[0];
    const worksheet = workbook.Sheets[sheetName];
    const data = XLSX.utils.sheet_to_json(worksheet);
    
    console.log(`   Raw rows: ${data.length}`);
    
    const validBusinesses = [];
    let skippedCount = 0;
    
    for (const row of data) {
      const business = normalizeBusinessData(row, category);
      if (business) {
        validBusinesses.push(business);
      } else {
        skippedCount++;
      }
    }
    
    console.log(`   Valid businesses: ${validBusinesses.length}`);
    console.log(`   Skipped invalid: ${skippedCount}`);
    
    return validBusinesses;
    
  } catch (error) {
    console.error(`   ❌ Error processing ${filePath}:`, error.message);
    return [];
  }
}

async function importBusinessesToDatabase(businesses) {
  const batchSize = 50; // Smaller batches to avoid timeouts
  let totalImported = 0;
  let totalSkipped = 0;
  
  console.log(`\n📊 Starting database import of ${businesses.length} businesses in batches of ${batchSize}`);
  
  for (let i = 0; i < businesses.length; i += batchSize) {
    const batch = businesses.slice(i, i + batchSize);
    const batchNumber = Math.floor(i / batchSize) + 1;
    const totalBatches = Math.ceil(businesses.length / batchSize);
    
    console.log(`\n⏳ Processing batch ${batchNumber}/${totalBatches} (${batch.length} businesses)`);
    
    try {
      // Use onConflictDoUpdate to overwrite existing businesses
      for (const business of batch) {
        try {
          await db.insert(businesses).values(business)
            .onConflictDoUpdate({
              target: businesses.name, // Assuming name is the conflict target
              set: business
            });
          totalImported++;
        } catch (err) {
          // Try individual insert without conflict handling for new records
          try {
            await db.insert(businesses).values(business);
            totalImported++;
          } catch (finalErr) {
            console.error(`     ❌ Failed to import: ${business.name} - ${finalErr.message}`);
            totalSkipped++;
          }
        }
      }
      
      console.log(`     ✅ Batch ${batchNumber} completed`);
      
      // Add small delay between batches to prevent overwhelming the database
      await new Promise(resolve => setTimeout(resolve, 100));
      
    } catch (error) {
      console.error(`     ❌ Batch ${batchNumber} failed:`, error.message);
      totalSkipped += batch.length;
    }
    
    // Log progress every 10 batches
    if (batchNumber % 10 === 0 || batchNumber === totalBatches) {
      console.log(`📈 Progress: ${totalImported} imported, ${totalSkipped} skipped (${((i + batchSize) / businesses.length * 100).toFixed(1)}%)`);
    }
  }
  
  return { imported: totalImported, skipped: totalSkipped };
}

async function setupCategories() {
  console.log('\n📂 Setting up business categories...');
  
  const categories = Object.values(CATEGORY_MAPPING);
  
  for (let i = 0; i < categories.length; i++) {
    const category = categories[i];
    try {
      await db.insert(businessCategories).values({
        name: category,
        description: `${category} businesses and services`,
        iconName: CATEGORY_ICONS[category] || 'Building2',
        displayOrder: i + 1,
        isActive: true
      }).onConflictDoNothing();
      console.log(`   ✅ ${category}`);
    } catch (error) {
      console.log(`   ⚠️  ${category} (${error.message})`);
    }
  }
}

async function clearExistingBusinesses() {
  console.log('\n🗑️  Clearing existing businesses (as requested to overwrite)...');
  try {
    const result = await db.delete(businesses);
    console.log(`   ✅ Cleared existing business data`);
  } catch (error) {
    console.log(`   ⚠️  Could not clear existing data: ${error.message}`);
  }
}

async function importAllBusinesses() {
  console.log('🚀 Starting comprehensive business import from Excel files');
  console.log('⏰ This process may take 15-30 minutes for 15,000+ businesses\n');
  
  const startTime = Date.now();
  
  try {
    // Find all latest Excel files (with 1753630080xxx timestamp)
    const attachedAssetsDir = path.join(process.cwd(), 'attached_assets');
    const excelFiles = fs.readdirSync(attachedAssetsDir)
      .filter(file => file.endsWith('.xlsx') && file.includes('1753630080'))
      .map(file => path.join(attachedAssetsDir, file));
    
    if (excelFiles.length === 0) {
      console.error('❌ No Excel files found with timestamp 1753630080xxx');
      return;
    }
    
    console.log(`📁 Found ${excelFiles.length} Excel files to process:`);
    excelFiles.forEach(file => console.log(`   • ${path.basename(file)}`));
    
    // Setup categories first
    await setupCategories();
    
    // Clear existing businesses (overwrite as requested)
    await clearExistingBusinesses();
    
    // Process all Excel files
    let allBusinesses = [];
    
    for (const filePath of excelFiles) {
      const businesses = await processExcelFile(filePath);
      allBusinesses = allBusinesses.concat(businesses);
      
      // Add delay between files to prevent overwhelming
      await new Promise(resolve => setTimeout(resolve, 500));
    }
    
    console.log(`\n📊 Total businesses extracted from all files: ${allBusinesses.length}`);
    
    if (allBusinesses.length === 0) {
      console.error('❌ No valid businesses found in any Excel files');
      return;
    }
    
    // Import to database
    const result = await importBusinessesToDatabase(allBusinesses);
    
    const endTime = Date.now();
    const duration = Math.round((endTime - startTime) / 1000);
    
    console.log('\n🎉 Import completed!');
    console.log(`✅ Successfully imported: ${result.imported} businesses`);
    console.log(`❌ Failed/skipped: ${result.skipped} businesses`);
    console.log(`⏱️  Total time: ${duration} seconds`);
    console.log(`📁 Categories: ${Object.keys(CATEGORY_MAPPING).length}`);
    
    // Show breakdown by category
    console.log('\n📈 Business distribution by category:');
    const categoryCount = {};
    allBusinesses.forEach(b => {
      categoryCount[b.category] = (categoryCount[b.category] || 0) + 1;
    });
    
    Object.entries(categoryCount)
      .sort(([,a], [,b]) => b - a)
      .forEach(([category, count]) => {
        console.log(`   ${category}: ${count} businesses`);
      });
    
  } catch (error) {
    console.error('💥 Import failed:', error);
    throw error;
  }
}

// Run the import if this file is executed directly
if (import.meta.url === `file://${process.argv[1]}`) {
  importAllBusinesses()
    .then(() => {
      console.log('\n✅ All done! Your business directory should now have 15,000+ businesses.');
      process.exit(0);
    })
    .catch(error => {
      console.error('\n💥 Import failed:', error);
      process.exit(1);
    });
}

export { importAllBusinesses };