import fs from 'fs';
import path from 'path';
import https from 'https';
import http from 'http';
import { fileURLToPath } from 'url';
import xlsx from 'xlsx';
import { drizzle } from 'drizzle-orm/neon-http';
import { neon } from '@neondatabase/serverless';
import { businesses } from '../shared/schema.js';
import { eq, ilike, or } from 'drizzle-orm';

const __filename = fileURLToPath(import.meta.url);
const __dirname = path.dirname(__filename);

// Database connection
const sql = neon(process.env.DATABASE_URL);
const db = drizzle(sql);

// Create images directory if it doesn't exist
const imagesDir = path.join(__dirname, '..', 'public', 'business-images');
if (!fs.existsSync(imagesDir)) {
  fs.mkdirSync(imagesDir, { recursive: true });
}

// Progress tracking
let progressCallback = null;
let shouldStop = false;

// Set progress callback for real-time updates
export function setProgressCallback(callback) {
  progressCallback = callback;
}

// Stop the import process
export function stopImport() {
  shouldStop = true;
}

// Utility function to log progress
function logProgress(message, type = 'info') {
  const log = {
    id: Date.now().toString(),
    message,
    type,
    timestamp: new Date().toISOString()
  };
  
  console.log(`[${type.toUpperCase()}] ${message}`);
  
  if (progressCallback) {
    progressCallback(log);
  }
}

// Function to sanitize filename
function sanitizeFilename(filename) {
  return filename
    .replace(/[^a-z0-9]/gi, '-')
    .replace(/-+/g, '-')
    .replace(/^-|-$/g, '')
    .toLowerCase();
}

// Function to download image from URL
function downloadImage(url, filepath) {
  return new Promise((resolve, reject) => {
    const protocol = url.startsWith('https:') ? https : http;
    
    const request = protocol.get(url, (response) => {
      // Check if response is successful
      if (response.statusCode >= 200 && response.statusCode < 300) {
        const fileStream = fs.createWriteStream(filepath);
        response.pipe(fileStream);
        
        fileStream.on('finish', () => {
          fileStream.close();
          resolve(filepath);
        });
        
        fileStream.on('error', (err) => {
          fs.unlink(filepath, () => {}); // Delete partial file
          reject(err);
        });
      } else if (response.statusCode >= 300 && response.statusCode < 400 && response.headers.location) {
        // Handle redirects
        downloadImage(response.headers.location, filepath)
          .then(resolve)
          .catch(reject);
      } else {
        reject(new Error(`HTTP ${response.statusCode}: ${response.statusMessage}`));
      }
    });
    
    request.on('error', reject);
    request.setTimeout(30000, () => {
      request.destroy();
      reject(new Error('Request timeout'));
    });
  });
}

// Function to find similar business names
function calculateSimilarity(str1, str2) {
  const clean1 = str1.toLowerCase().replace(/[^a-z0-9]/g, '');
  const clean2 = str2.toLowerCase().replace(/[^a-z0-9]/g, '');
  
  if (clean1 === clean2) return 1.0;
  if (clean1.includes(clean2) || clean2.includes(clean1)) return 0.8;
  
  // Simple Levenshtein distance calculation
  const matrix = [];
  for (let i = 0; i <= clean2.length; i++) {
    matrix[i] = [i];
  }
  for (let j = 0; j <= clean1.length; j++) {
    matrix[0][j] = j;
  }
  
  for (let i = 1; i <= clean2.length; i++) {
    for (let j = 1; j <= clean1.length; j++) {
      if (clean2.charAt(i - 1) === clean1.charAt(j - 1)) {
        matrix[i][j] = matrix[i - 1][j - 1];
      } else {
        matrix[i][j] = Math.min(
          matrix[i - 1][j - 1] + 1,
          matrix[i][j - 1] + 1,
          matrix[i - 1][j] + 1
        );
      }
    }
  }
  
  const maxLength = Math.max(clean1.length, clean2.length);
  return 1.0 - (matrix[clean2.length][clean1.length] / maxLength);
}

// Function to find matching business in database
async function findMatchingBusiness(businessName, location = null) {
  try {
    // Try exact match first
    let matchingBusinesses = await db
      .select()
      .from(businesses)
      .where(eq(businesses.name, businessName));
    
    if (matchingBusinesses.length > 0) {
      return matchingBusinesses[0];
    }
    
    // Try case-insensitive match
    matchingBusinesses = await db
      .select()
      .from(businesses)
      .where(ilike(businesses.name, businessName));
    
    if (matchingBusinesses.length > 0) {
      return matchingBusinesses[0];
    }
    
    // Try partial matches
    const allBusinesses = await db.select().from(businesses);
    let bestMatch = null;
    let bestScore = 0;
    
    for (const business of allBusinesses) {
      const similarity = calculateSimilarity(businessName, business.name);
      if (similarity > bestScore && similarity > 0.7) {
        bestScore = similarity;
        bestMatch = business;
      }
    }
    
    return bestMatch;
  } catch (error) {
    logProgress(`Error finding business match: ${error.message}`, 'error');
    return null;
  }
}

// Function to process Excel file and extract image URLs
async function processExcelFile(filePath) {
  try {
    const filename = path.basename(filePath);
    logProgress(`Processing Excel file: ${filename}`, 'info');
    
    const workbook = xlsx.readFile(filePath);
    const sheetName = workbook.SheetNames[0];
    const worksheet = workbook.Sheets[sheetName];
    const data = xlsx.utils.sheet_to_json(worksheet);
    
    logProgress(`Found ${data.length} rows in ${filename}`, 'info');
    
    if (data.length === 0) {
      logProgress(`No data found in ${filename}`, 'warning');
      return [];
    }
    
    // Get all column names for verbose logging
    const allColumns = Object.keys(data[0] || {});
    logProgress(`Columns in ${filename}: ${allColumns.join(', ')}`, 'info');
    
    // Look for image URL columns with extensive pattern matching
    const imageUrlPatterns = [
      /image.*url/i,
      /imageurl/i,
      /img.*url/i,
      /photo.*url/i,
      /picture.*url/i,
      /logo.*url/i,
      /^image$/i,
      /^photo$/i,
      /^picture$/i,
      /^logo$/i,
      /url.*image/i,
      /url.*photo/i,
      /website/i,
      /link/i
    ];
    
    const imageUrlKeys = allColumns.filter(key => 
      imageUrlPatterns.some(pattern => pattern.test(key))
    );
    
    logProgress(`Found potential image URL columns: ${imageUrlKeys.join(', ') || 'NONE'}`, 'info');
    
    // Look for business name columns with extensive pattern matching
    const namePatterns = [
      /^name$/i,
      /business.*name/i,
      /company.*name/i,
      /^business$/i,
      /^company$/i,
      /entity.*name/i,
      /organisation.*name/i,
      /organization.*name/i,
      /title/i,
      /firm/i
    ];
    
    const nameKeys = allColumns.filter(key => 
      namePatterns.some(pattern => pattern.test(key))
    );
    
    logProgress(`Found potential name columns: ${nameKeys.join(', ') || 'NONE'}`, 'info');
    
    // Check all columns for URL-like content in sample data
    const urlPattern = /https?:\/\/[^\s]+/;
    const columnsWithUrls = [];
    
    // Sample first few rows to find URL-like content
    const sampleSize = Math.min(10, data.length);
    for (let i = 0; i < sampleSize; i++) {
      const row = data[i];
      Object.entries(row).forEach(([key, value]) => {
        if (urlPattern.test(String(value)) && !columnsWithUrls.includes(key)) {
          columnsWithUrls.push(key);
        }
      });
    }
    
    logProgress(`Columns containing URL-like content: ${columnsWithUrls.join(', ') || 'NONE'}`, 'info');
    
    // Use the best available image URL column
    let imageUrlKey = null;
    if (imageUrlKeys.length > 0) {
      imageUrlKey = imageUrlKeys[0];
    } else if (columnsWithUrls.length > 0) {
      imageUrlKey = columnsWithUrls[0];
      logProgress(`No explicit image URL column found, using URL column: ${imageUrlKey}`, 'warning');
    }
    
    // Use the best available name column
    let nameKey = null;
    if (nameKeys.length > 0) {
      nameKey = nameKeys[0];
    } else if (allColumns.includes('name')) {
      nameKey = 'name';
    } else if (allColumns.length > 0) {
      nameKey = allColumns[0]; // Fallback to first column
      logProgress(`No explicit name column found, using first column: ${nameKey}`, 'warning');
    }
    
    if (!imageUrlKey) {
      logProgress(`No image URL column found in ${filename} - skipping`, 'error');
      logProgress(`Available columns: ${allColumns.join(', ')}`, 'error');
      return [];
    }
    
    if (!nameKey) {
      logProgress(`No business name column found in ${filename} - skipping`, 'error');
      return [];
    }
    
    logProgress(`Using image URL column: "${imageUrlKey}", name column: "${nameKey}"`, 'info');
    
    const imageData = [];
    let extractedCount = 0;
    let validUrlCount = 0;
    let totalChecked = 0;
    
    for (const row of data) {
      if (shouldStop) {
        logProgress('Import stopped by user', 'warning');
        return imageData;
      }
      
      totalChecked++;
      const imageUrl = row[imageUrlKey];
      const businessName = row[nameKey];
      const location = row.location || row.address || row.province || row.Location || row.Address || row.Province || '';
      
      // Log first few samples for debugging
      if (totalChecked <= 5) {
        logProgress(`Sample row ${totalChecked}: imageUrl="${imageUrl}", businessName="${businessName}"`, 'info');
      }
      
      if (imageUrl && String(imageUrl).trim()) {
        const urlStr = String(imageUrl).trim();
        if (urlStr.startsWith('http://') || urlStr.startsWith('https://')) {
          validUrlCount++;
          if (businessName && String(businessName).trim()) {
            imageData.push({
              businessName: String(businessName).trim(),
              imageUrl: urlStr,
              location: String(location).trim() || null
            });
            extractedCount++;
          }
        }
      }
    }
    
    logProgress(`Checked ${totalChecked} rows, found ${validUrlCount} valid URLs, extracted ${extractedCount} complete records from ${filename}`, 'info');
    return imageData;
  } catch (error) {
    logProgress(`Error processing Excel file ${filePath}: ${error.message}`, 'error');
    logProgress(`Error stack: ${error.stack}`, 'error');
    return [];
  }
}

// Main import function
export async function importBusinessImages() {
  try {
    shouldStop = false;
    logProgress('Starting business image import process');
    
    // Find all Excel files in attached_assets directory
    const attachedAssetsDir = path.join(__dirname, '..', 'attached_assets');
    const excelFiles = fs.readdirSync(attachedAssetsDir)
      .filter(file => file.endsWith('.xlsx') && file.includes('1753526977'))
      .map(file => path.join(attachedAssetsDir, file));
    
    if (excelFiles.length === 0) {
      logProgress('No Excel files found in attached_assets directory', 'warning');
      return {
        success: false,
        message: 'No Excel files found',
        totalImages: 0,
        processedImages: 0,
        matchedImages: 0,
        errorCount: 1
      };
    }
    
    logProgress(`Found ${excelFiles.length} Excel files to process`);
    
    // Process all Excel files to extract image data
    let allImageData = [];
    for (const filePath of excelFiles) {
      if (shouldStop) break;
      
      const imageData = await processExcelFile(filePath);
      allImageData = allImageData.concat(imageData);
    }
    
    logProgress(`Total image URLs extracted across all files: ${allImageData.length}`, 'info');
    
    if (allImageData.length === 0) {
      logProgress('No image URLs found in Excel files', 'warning');
      return {
        success: false,
        message: 'No image URLs found in any Excel files',
        totalImages: 0,
        processedImages: 0,
        matchedImages: 0,
        errorCount: 1
      };
    }
    
    logProgress(`Found ${allImageData.length} total image URLs to process`);
    
    let processedImages = 0;
    let matchedImages = 0;
    let errorCount = 0;
    const batchSize = 30;
    
    // Process images in batches of 30
    for (let batchStart = 0; batchStart < allImageData.length; batchStart += batchSize) {
      if (shouldStop) {
        logProgress('Import stopped by user', 'warning');
        break;
      }
      
      const batchEnd = Math.min(batchStart + batchSize, allImageData.length);
      const currentBatch = allImageData.slice(batchStart, batchEnd);
      const batchNumber = Math.floor(batchStart / batchSize) + 1;
      const totalBatches = Math.ceil(allImageData.length / batchSize);
      
      logProgress(`Starting batch ${batchNumber}/${totalBatches} (${currentBatch.length} images)`, 'info');
      
      // Update progress callback with batch info
      if (progressCallback) {
        progressCallback({
          id: Date.now().toString(),
          message: `Processing batch ${batchNumber}/${totalBatches}: ${batchStart + 1}-${batchEnd} of ${allImageData.length}`,
          type: 'info',
          timestamp: new Date().toISOString()
        });
      }
      
      // Process each image in the current batch
      for (let i = 0; i < currentBatch.length; i++) {
        if (shouldStop) {
          logProgress('Import stopped by user', 'warning');
          break;
        }
        
        const imageItem = currentBatch[i];
        const globalIndex = batchStart + i + 1;
        
        try {
          logProgress(`Processing image ${globalIndex}/${allImageData.length}: ${imageItem.businessName}`);
          
          // Find matching business in database
          const matchingBusiness = await findMatchingBusiness(imageItem.businessName, imageItem.location);
          
          if (!matchingBusiness) {
            logProgress(`No matching business found for: ${imageItem.businessName}`, 'warning');
            errorCount++;
            processedImages++;
            continue;
          }
          
          // Create filename for the image
          const businessId = matchingBusiness.id;
          const sanitizedName = sanitizeFilename(matchingBusiness.name);
          const url = new URL(imageItem.imageUrl);
          const urlPath = url.pathname;
          const extension = path.extname(urlPath) || '.jpg';
          const filename = `${businessId}-${sanitizedName}${extension}`;
          const filepath = path.join(imagesDir, filename);
          
          // Check if image already exists
          if (fs.existsSync(filepath)) {
            logProgress(`Image already exists for: ${matchingBusiness.name}`, 'info');
            processedImages++;
            matchedImages++;
            continue;
          }
          
          // Download the image
          try {
            await downloadImage(imageItem.imageUrl, filepath);
            
            // Update business record with image path
            const imagePath = `/business-images/${filename}`;
            await db
              .update(businesses)
              .set({ imageUrl: imagePath })
              .where(eq(businesses.id, businessId));
            
            logProgress(`Successfully downloaded and assigned image for: ${matchingBusiness.name}`, 'success');
            matchedImages++;
            
          } catch (downloadError) {
            logProgress(`Failed to download image for ${matchingBusiness.name}: ${downloadError.message}`, 'error');
            errorCount++;
          }
          
          processedImages++;
          
          // Small delay to avoid overwhelming the server
          await new Promise(resolve => setTimeout(resolve, 200));
          
        } catch (error) {
          logProgress(`Error processing image for ${imageItem.businessName}: ${error.message}`, 'error');
          errorCount++;
          processedImages++;
        }
      }
      
      // Log batch completion
      const batchProgress = Math.round((processedImages / allImageData.length) * 100);
      logProgress(`Batch ${batchNumber}/${totalBatches} completed. Progress: ${processedImages}/${allImageData.length} (${batchProgress}%)`, 'success');
      
      // Longer delay between batches to prevent server overload
      if (batchEnd < allImageData.length && !shouldStop) {
        logProgress(`Waiting 3 seconds before starting next batch...`, 'info');
        await new Promise(resolve => setTimeout(resolve, 3000));
      }
    }
    
    const result = {
      success: true,
      message: 'Image import completed',
      totalImages: allImageData.length,
      processedImages,
      matchedImages,
      errorCount
    };
    
    logProgress(`Import completed: ${matchedImages}/${allImageData.length} images successfully matched and downloaded`);
    
    return result;
    
  } catch (error) {
    logProgress(`Fatal error during import: ${error.message}`, 'error');
    return {
      success: false,
      message: error.message,
      totalImages: 0,
      processedImages: 0,
      matchedImages: 0,
      errorCount: 1
    };
  }
}

// If running directly from command line
if (import.meta.url === `file://${process.argv[1]}`) {
  importBusinessImages()
    .then(result => {
      console.log('Import result:', result);
      process.exit(0);
    })
    .catch(error => {
      console.error('Import failed:', error);
      process.exit(1);
    });
}