import fs from 'fs';
import path from 'path';
import XLSX from 'xlsx';
import { db } from '../server/db.ts';
import { businesses, businessCategories } from '../shared/schema.ts';

const CATEGORY_MAPPING = {
  'Agriculture': 'Agriculture',
  'Car_&_Automotive': 'Car & Automotive', 
  'Education_Training_Lessons': 'Education & Training',
  'Engineering': 'Engineering',
  'Events_&_entertainment': 'Events & Entertainment',
  'Financial_&_Insurance_Services': 'Financial & Insurance Services',
  'Home_Building_&_Trade': 'Home Building & Trade',
  'Hospitality': 'Hospitality',
  'IT_Computer_&_Technology': 'IT & Technology',
  'Marketing_Business,_and_Legal': 'Marketing & Legal',
  'Medical_Wellness_&_Beauty': 'Medical & Beauty',
  'Public_Service_&_Government': 'Public Service & Government',
  'Security_Systems_&_Protection': 'Security & Protection',
  'Telecommunication': 'Telecommunication',
  'Trade_&_Industry': 'Trade & Industry',
  'Transport': 'Transport'
};

function extractCategoryFromFilename(filename) {
  const parts = filename.split('_');
  if (parts.length >= 2) {
    let categoryPart = parts.slice(1, -1).join('_');
    return CATEGORY_MAPPING[categoryPart] || categoryPart.replace(/_/g, ' ');
  }
  return 'Other';
}

async function setupCategories() {
  console.log('📂 Setting up business categories...');
  const categories = Object.values(CATEGORY_MAPPING);
  
  for (const category of categories) {
    try {
      await db.insert(businessCategories).values({
        name: category,
        description: `${category} businesses and services`,
        iconName: 'Building2',
        displayOrder: 1,
        isActive: true
      }).onConflictDoNothing();
    } catch (error) {
      // Category already exists
    }
  }
}

async function importAllFiles() {
  console.log('🚀 Starting working business import...');
  
  // Clear existing data first
  await db.delete(businesses);
  console.log('✅ Cleared existing businesses');
  
  // Setup categories
  await setupCategories();
  
  // Find Excel files
  const attachedAssetsDir = './attached_assets';
  const excelFiles = fs.readdirSync(attachedAssetsDir)
    .filter(file => file.endsWith('.xlsx') && file.includes('1753630080'))
    .map(file => path.join(attachedAssetsDir, file));
  
  console.log(`📁 Found ${excelFiles.length} Excel files to process`);
  
  let totalImported = 0;
  let totalFailed = 0;
  
  for (const filePath of excelFiles) {
    const filename = path.basename(filePath);
    const category = extractCategoryFromFilename(filename);
    
    console.log(`\n📄 Processing: ${filename} (${category})`);
    
    try {
      const workbook = XLSX.readFile(filePath);
      const sheetName = workbook.SheetNames[0];
      const worksheet = workbook.Sheets[sheetName];
      const data = XLSX.utils.sheet_to_json(worksheet);
      
      console.log(`   Found ${data.length} rows`);
      
      let fileImported = 0;
      let fileFailed = 0;
      
      // Process in smaller batches
      for (let i = 0; i < data.length; i += 50) {
        const batch = data.slice(i, i + 50);
        
        for (const row of batch) {
          const businessName = String(row.name || '').trim();
          if (!businessName || businessName.length < 2) {
            fileFailed++;
            continue;
          }
          
          const business = {
            storeId: row.store_id?.toString() || null,
            name: businessName,
            description: String(row.description || `${businessName} - ${category} services`).trim(),
            category: category,
            location: String(row.address || 'South Africa').trim(),
            contactEmail: String(row.email || '').trim() || null,
            contactPhone: String(row.phone || '').trim() || null,
            website: String(row.website || '').trim() || null,
            whatsappNumber: null,
            contactPerson: String(row.contact_person || '').trim() || null,
            slogan: String(row.slogan || '').trim() || null,
            latitude: null,
            longitude: null,
            imageUrl: null,
            originalImageUrl: null,
            isVerified: Boolean(row.is_verified) || false,
            rating: 0,
            reviewCount: 0,
            isFeatured: false,
            featuredUntil: null,
            highlightColor: null,
            topPlacement: false,
            topPlacementUntil: null,
            featuredOnHomepage: false,
            homepageViewCount: 0,
            status: 'approved',
            approvedAt: new Date(),
            approvedBy: null,
            rejectionReason: null,
            createdBy: null
          };
          
          try {
            await db.insert(businesses).values(business);
            fileImported++;
            totalImported++;
            
            if (totalImported % 1000 === 0) {
              console.log(`     📊 Progress: ${totalImported} total imported`);
            }
          } catch (error) {
            fileFailed++;
            totalFailed++;
            if (fileFailed % 100 === 0) {
              console.log(`     ⚠️  Failed: ${fileFailed} in this file`);
            }
          }
        }
        
        // Small delay between batches
        if (i % 500 === 0 && i > 0) {
          await new Promise(resolve => setTimeout(resolve, 100));
        }
      }
      
      console.log(`   ✅ File complete: ${fileImported} imported, ${fileFailed} failed`);
      
    } catch (error) {
      console.error(`   ❌ Error processing ${filename}:`, error.message);
    }
  }
  
  console.log('\n🎉 IMPORT COMPLETE!');
  console.log(`✅ Total imported: ${totalImported} businesses`);
  console.log(`❌ Total failed: ${totalFailed} businesses`);
  
  // Final count check
  const finalCount = await db.select().from(businesses);
  console.log(`📊 Database contains: ${finalCount.length} businesses`);
  
  return { imported: totalImported, failed: totalFailed, final: finalCount.length };
}

importAllFiles()
  .then(result => {
    console.log('\n✅ Your business directory is now fully populated!');
    console.log('📱 Visit /directory to see your comprehensive business listings');
    process.exit(0);
  })
  .catch(error => {
    console.error('\n💥 Import failed:', error);
    process.exit(1);
  });