import fs from 'fs';
import path from 'path';
import XLSX from 'xlsx';
import { db } from './db.ts';
import { businesses, businessCategories } from '../shared/schema.ts';

interface ImportStatus {
  isRunning: boolean;
  currentFile?: string;
  currentCategory?: string;
  totalFiles: number;
  processedFiles: number;
  totalBusinesses: number;
  importedBusinesses: number;
  failedBusinesses: number;
  startTime?: Date;
  endTime?: Date;
  error?: string;
  categoryStats: Record<string, number>;
  filesCompleted: string[];
}

interface ImportProgress {
  status: ImportStatus;
  onProgress?: (status: ImportStatus) => void;
}

const CATEGORY_MAPPING = {
  'Agriculture': 'Agriculture',
  'Car_&_Automotive': 'Car & Automotive', 
  'Education_Training_Lessons': 'Education & Training',
  'Engineering': 'Engineering',
  'Events_&_entertainment': 'Events & Entertainment',
  'Financial_&_Insurance_Services': 'Financial & Insurance Services',
  'Home_Building_&_Trade': 'Home Building & Trade',
  'Hospitality': 'Hospitality',
  'IT_Computer_&_Technology': 'IT & Technology',
  'Marketing_Business,_and_Legal': 'Marketing & Legal',
  'Medical_Wellness_&_Beauty': 'Medical & Beauty',
  'Public_Service_&_Government': 'Public Service & Government',
  'Security_Systems_&_Protection': 'Security & Protection',
  'Telecommunication': 'Telecommunication',
  'Trade_&_Industry': 'Trade & Industry',
  'Transport': 'Transport'
};

class ImportManager {
  private static instance: ImportManager;
  private importStatus: ImportStatus;
  private progressCallbacks: ((status: ImportStatus) => void)[] = [];

  private constructor() {
    this.importStatus = {
      isRunning: false,
      totalFiles: 0,
      processedFiles: 0,
      totalBusinesses: 0,
      importedBusinesses: 0,
      failedBusinesses: 0,
      categoryStats: {},
      filesCompleted: []
    };
    // Initialize with current database state
    this.initializeStatus();
  }

  private async initializeStatus() {
    try {
      // Get current category stats
      const categoryStats = await this.getCurrentCategoryStats();
      const totalBusinesses = Object.values(categoryStats).reduce((sum, count) => sum + count, 0);
      
      // Find available Excel files
      const attachedAssetsDir = path.join(process.cwd(), 'attached_assets');
      const allFiles = fs.readdirSync(attachedAssetsDir);
      const excelFiles = allFiles
        .filter(file => file.endsWith('.xlsx') && file.startsWith('yep-co-za_') && file.includes('1753630080'))
        .length;
      
      this.importStatus = {
        ...this.importStatus,
        totalFiles: excelFiles,
        totalBusinesses,
        categoryStats
      };
      
      console.log(`Initialized ImportManager: ${totalBusinesses} businesses, ${excelFiles} Excel files available`);
    } catch (error) {
      console.error('Failed to initialize ImportManager status:', error);
    }
  }

  static getInstance(): ImportManager {
    if (!ImportManager.instance) {
      ImportManager.instance = new ImportManager();
    }
    return ImportManager.instance;
  }

  async getStatus(): Promise<ImportStatus> {
    // Refresh current stats if not running an import
    if (!this.importStatus.isRunning) {
      try {
        const categoryStats = await this.getCurrentCategoryStats();
        const totalBusinesses = Object.values(categoryStats).reduce((sum, count) => sum + count, 0);
        
        // Update total files count if not set
        if (this.importStatus.totalFiles === 0) {
          const attachedAssetsDir = path.join(process.cwd(), 'attached_assets');
          const allFiles = fs.readdirSync(attachedAssetsDir);
          const excelFiles = allFiles
            .filter(file => file.endsWith('.xlsx') && file.startsWith('yep-co-za_') && file.includes('1753630080'))
            .length;
          
          this.importStatus.totalFiles = excelFiles;
        }
        
        this.importStatus.totalBusinesses = totalBusinesses;
        this.importStatus.categoryStats = categoryStats;
      } catch (error) {
        console.error('Failed to refresh status:', error);
      }
    }
    
    return { ...this.importStatus };
  }

  onProgress(callback: (status: ImportStatus) => void) {
    this.progressCallbacks.push(callback);
  }

  private updateStatus(updates: Partial<ImportStatus>) {
    this.importStatus = { ...this.importStatus, ...updates };
    this.progressCallbacks.forEach(callback => callback({ ...this.importStatus }));
  }

  private extractCategoryFromFilename(filename: string): string {
    const parts = filename.split('_');
    if (parts.length >= 2) {
      let categoryPart = parts.slice(1, -1).join('_');
      return CATEGORY_MAPPING[categoryPart as keyof typeof CATEGORY_MAPPING] || categoryPart.replace(/_/g, ' ');
    }
    return 'Other';
  }

  private async getCurrentCategoryStats(): Promise<Record<string, number>> {
    const businessList = await db.select().from(businesses);
    const stats: Record<string, number> = {};
    businessList.forEach(biz => {
      stats[biz.category] = (stats[biz.category] || 0) + 1;
    });
    return stats;
  }

  private async setupCategories() {
    const categories = Object.values(CATEGORY_MAPPING);
    for (const category of categories) {
      try {
        await db.insert(businessCategories).values({
          name: category,
          description: `${category} businesses and services`,
          iconName: 'Building2',
          displayOrder: 1,
          isActive: true
        }).onConflictDoNothing();
      } catch (error) {
        // Category already exists
      }
    }
  }

  private async processExcelFile(filePath: string, skipCount: number = 0): Promise<{ imported: number; failed: number }> {
    const filename = path.basename(filePath);
    const category = this.extractCategoryFromFilename(filename);
    
    this.updateStatus({
      currentFile: filename,
      currentCategory: category
    });

    const workbook = XLSX.readFile(filePath);
    const sheetName = workbook.SheetNames[0];
    const worksheet = workbook.Sheets[sheetName];
    const data = XLSX.utils.sheet_to_json(worksheet);

    let imported = 0;
    let failed = 0;

    // Process in batches for better performance
    const batchSize = 50;
    for (let i = skipCount; i < data.length; i += batchSize) {
      const batch = data.slice(i, Math.min(i + batchSize, data.length));
      
      for (const row of batch) {
        const rowData = row as any;
        const businessName = String(rowData.name || '').trim();
        if (!businessName || businessName.length < 2) {
          failed++;
          continue;
        }

        const business = {
          storeId: rowData.store_id?.toString() || null,
          name: businessName,
          description: String(rowData.description || `${businessName} - ${category} services`).trim(),
          category: category,
          location: String(rowData.address || 'South Africa').trim(),
          contactEmail: String(rowData.email || '').trim() || null,
          contactPhone: String(rowData.phone || '').trim() || null,
          website: String(rowData.website || '').trim() || null,
          whatsappNumber: null,
          contactPerson: String(rowData.contact_person || '').trim() || null,
          slogan: String(rowData.slogan || '').trim() || null,
          latitude: null,
          longitude: null,
          imageUrl: null,
          originalImageUrl: null,
          isVerified: Boolean(rowData.is_verified) || false,
          rating: 0,
          reviewCount: 0,
          isFeatured: false,
          featuredUntil: null,
          highlightColor: null,
          topPlacement: false,
          topPlacementUntil: null,
          featuredOnHomepage: false,
          homepageViewCount: 0,
          status: 'approved',
          approvedAt: new Date(),
          approvedBy: null,
          rejectionReason: null,
          createdBy: null
        };

        try {
          await db.insert(businesses).values(business);
          imported++;
          
          // Update progress every 100 imports
          if (imported % 100 === 0) {
            this.updateStatus({
              importedBusinesses: this.importStatus.importedBusinesses + imported,
              failedBusinesses: this.importStatus.failedBusinesses + failed
            });
          }
        } catch (error) {
          failed++;
        }
      }

      // Small delay to prevent overwhelming the database
      if (i % 500 === 0) {
        await new Promise(resolve => setTimeout(resolve, 50));
      }
    }

    return { imported, failed };
  }

  async startImport(clearExisting: boolean = false): Promise<void> {
    if (this.importStatus.isRunning) {
      throw new Error('Import is already running');
    }

    this.updateStatus({
      isRunning: true,
      startTime: new Date(),
      endTime: undefined,
      error: undefined,
      importedBusinesses: 0,
      failedBusinesses: 0,
      processedFiles: 0,
      filesCompleted: []
    });

    try {
      // Setup categories
      await this.setupCategories();

      // Clear existing data if requested
      if (clearExisting) {
        await db.delete(businesses);
        this.updateStatus({ categoryStats: {} });
      }

      // Get current state
      const currentStats = await this.getCurrentCategoryStats();
      this.updateStatus({ categoryStats: currentStats });

      // Find Excel files - use the most recent versions (1753630080* timestamp)
      const attachedAssetsDir = path.join(process.cwd(), 'attached_assets');
      const allFiles = fs.readdirSync(attachedAssetsDir);
      console.log('All Excel files found:', allFiles.filter(f => f.endsWith('.xlsx')));
      
      // Get the latest files with 1753630080* timestamp
      const latestFiles = allFiles
        .filter(file => file.endsWith('.xlsx') && file.startsWith('yep-co-za_') && file.includes('1753630080'))
        .map(file => path.join(attachedAssetsDir, file));
        
      console.log('Selected latest Excel files:', latestFiles.map(f => path.basename(f)));
      
      const excelFiles = latestFiles;

      this.updateStatus({ totalFiles: excelFiles.length });

      let totalImported = 0;
      let totalFailed = 0;

      for (const filePath of excelFiles) {
        const filename = path.basename(filePath);
        const category = this.extractCategoryFromFilename(filename);
        
        // Check if this category already has data (intelligent resume)
        const currentCategoryCount = currentStats[category] || 0;
        let skipCount = 0;
        
        if (currentCategoryCount > 0 && !clearExisting) {
          // For partially imported files, we need to determine how many to skip
          if (category === 'Car & Automotive' && currentCategoryCount < 10000) {
            skipCount = currentCategoryCount;
          } else if (currentCategoryCount > 0) {
            // Category appears complete, skip entire file
            this.updateStatus({ 
              filesCompleted: [...this.importStatus.filesCompleted, filename],
              processedFiles: this.importStatus.processedFiles + 1
            });
            continue;
          }
        }

        const result = await this.processExcelFile(filePath, skipCount);
        totalImported += result.imported;
        totalFailed += result.failed;

        // Update progress after each file
        const updatedStats = await this.getCurrentCategoryStats();
        this.updateStatus({
          processedFiles: this.importStatus.processedFiles + 1,
          importedBusinesses: totalImported,
          failedBusinesses: totalFailed,
          categoryStats: updatedStats,
          filesCompleted: [...this.importStatus.filesCompleted, filename]
        });
      }

      // Final update
      const finalStats = await this.getCurrentCategoryStats();
      const totalBusinesses = Object.values(finalStats).reduce((sum, count) => sum + count, 0);
      
      this.updateStatus({
        isRunning: false,
        endTime: new Date(),
        totalBusinesses: totalBusinesses,
        categoryStats: finalStats
      });

    } catch (error) {
      this.updateStatus({
        isRunning: false,
        endTime: new Date(),
        error: error instanceof Error ? error.message : 'Unknown error'
      });
      throw error;
    }
  }

  async stopImport(): Promise<void> {
    if (!this.importStatus.isRunning) {
      throw new Error('No import is currently running');
    }

    this.updateStatus({
      isRunning: false,
      endTime: new Date(),
      error: 'Import stopped by user'
    });
  }
}

export default ImportManager;