import type { Express } from "express";
import { createServer, type Server } from "http";
import express from "express";
import { setupAuth } from "./auth";
import { storage } from "./storage";
import { insertContactSchema, insertBusinessSchema, insertContentSchema, insertProductSchema, insertEmailNewsletterSchema, insertSmtpConfigSchema, insertEmailCampaignSchema, insertPageLayoutSchema, insertHeaderTagSchema, insertSystemSettingsSchema, insertDataDealsOrderSchema, insertDataDealsEmailSettingsSchema, insertDataDealsCustomerEmailSettingsSchema } from "@shared/schema";
import { z } from "zod";
import nodemailer from "nodemailer";
import path from "path";
import ImportManager from "./importManager";

export function registerRoutes(app: Express): Server {
  // Serve business images statically
  app.use('/business-images', express.static(path.join(process.cwd(), 'public', 'business-images')));
  
  // Setup authentication routes
  setupAuth(app);

  // Contact form submission
  app.post("/api/contact", async (req, res) => {
    try {
      const validatedData = insertContactSchema.parse(req.body);
      const submission = await storage.createContactSubmission(validatedData);
      res.status(201).json(submission);
    } catch (error) {
      console.error("Contact submission error:", error);
      res.status(400).json({ message: "Invalid submission data" });
    }
  });

  // Business directory routes
  app.get("/api/businesses", async (req, res) => {
    try {
      const { category, location, search, page, limit } = req.query;
      const result = await storage.getBusinesses(
        category as string,
        location as string,
        search as string,
        page ? parseInt(page as string) : undefined,
        limit ? parseInt(limit as string) : undefined
      );
      res.json(result);
    } catch (error) {
      console.error("Get businesses error:", error);
      res.status(500).json({ message: "Failed to fetch businesses" });
    }
  });

  app.get("/api/business-categories", async (req, res) => {
    try {
      const categories = await storage.getBusinessCategories();
      res.json(categories);
    } catch (error) {
      console.error("Get business categories error:", error);
      res.status(500).json({ message: "Failed to fetch categories" });
    }
  });

  app.get("/api/business-locations", async (req, res) => {
    try {
      const locations = await storage.getBusinessLocations();
      res.json(locations);
    } catch (error) {
      console.error("Get business locations error:", error);
      res.status(500).json({ message: "Failed to fetch locations" });
    }
  });

  app.post("/api/businesses", async (req, res) => {
    try {
      if (!req.user) {
        return res.status(401).json({ message: "Authentication required" });
      }

      const validatedData = insertBusinessSchema.parse(req.body);
      const business = await storage.createBusiness({
        ...validatedData,
        createdBy: req.user.id,
      });
      res.status(201).json(business);
    } catch (error) {
      console.error("Create business error:", error);
      res.status(400).json({ message: "Invalid business data" });
    }
  });

  app.post("/api/business-listings", async (req, res) => {
    try {
      const { business, paymentOption } = req.body;
      
      // For now, create business without payment (will add Stripe integration next)
      const newBusiness = await storage.createBusiness({
        ...business,
        createdBy: req.user?.id || null,
        isVerified: false,
        isFeatured: paymentOption.id === 'featured' || paymentOption.id === 'top_placement',
        topPlacement: paymentOption.id === 'top_placement',
        highlightColor: paymentOption.id === 'highlight' ? 'blue' : null,
        featuredUntil: (paymentOption.id === 'featured' || paymentOption.id === 'top_placement') 
          ? new Date(Date.now() + 30 * 24 * 60 * 60 * 1000) 
          : null,
        topPlacementUntil: paymentOption.id === 'top_placement' 
          ? new Date(Date.now() + 30 * 24 * 60 * 60 * 1000) 
          : null,
      });

      res.status(201).json({ 
        business: newBusiness,
        message: "Business listing submitted successfully. Payment integration coming soon!"
      });
    } catch (error) {
      console.error("Create business listing error:", error);
      res.status(400).json({ message: "Invalid business listing data" });
    }
  });

  // Business submission for approval workflow
  app.post("/api/businesses/submit", async (req, res) => {
    try {
      const businessData = req.body;
      const business = await storage.submitBusiness(businessData);
      res.status(201).json(business);
    } catch (error) {
      console.error("Business submission error:", error);
      res.status(400).json({ message: "Failed to submit business for review" });
    }
  });

  // Admin: Get pending business submissions
  app.get("/api/admin/businesses/pending", async (req, res) => {
    try {
      if (!req.user || req.user.role !== 'admin') {
        return res.status(403).json({ message: "Admin access required" });
      }
      
      const pendingBusinesses = await storage.getPendingBusinesses();
      res.json(pendingBusinesses);
    } catch (error) {
      console.error("Error fetching pending businesses:", error);
      res.status(500).json({ message: "Failed to fetch pending businesses" });
    }
  });

  // Admin: Approve business
  app.post("/api/admin/businesses/:id/approve", async (req, res) => {
    try {
      if (!req.user || req.user.role !== 'admin') {
        return res.status(403).json({ message: "Admin access required" });
      }
      
      const { id } = req.params;
      const business = await storage.approveBusiness(parseInt(id), req.user.id);
      res.json(business);
    } catch (error) {
      console.error("Error approving business:", error);
      res.status(500).json({ message: "Failed to approve business" });
    }
  });

  // Admin: Reject business
  app.post("/api/admin/businesses/:id/reject", async (req, res) => {
    try {
      if (!req.user || req.user.role !== 'admin') {
        return res.status(403).json({ message: "Admin access required" });
      }
      
      const { id } = req.params;
      const { reason } = req.body;
      const business = await storage.rejectBusiness(parseInt(id), req.user.id, reason || "No reason provided");
      res.json(business);
    } catch (error) {
      console.error("Error rejecting business:", error);
      res.status(500).json({ message: "Failed to reject business" });
    }
  });

  // Featured businesses for homepage
  app.get("/api/businesses/featured-homepage", async (req, res) => {
    try {
      const featuredBusinesses = await storage.getFeaturedBusinessesForHomepage();
      res.json(featuredBusinesses);
    } catch (error) {
      console.error("Error fetching featured businesses:", error);
      res.status(500).json({ message: "Failed to fetch featured businesses" });
    }
  });

  // Admin: Update business featured status
  app.patch("/api/admin/businesses/:id/featured", async (req, res) => {
    try {
      if (!req.user || req.user.role !== 'admin') {
        return res.status(403).json({ message: "Admin access required" });
      }
      
      const { id } = req.params;
      const { featured } = req.body;
      
      const updatedBusiness = await storage.updateBusinessFeaturedStatus(
        parseInt(id), 
        featured
      );
      
      if (!updatedBusiness) {
        return res.status(404).json({ message: "Business not found" });
      }
      
      res.json(updatedBusiness);
    } catch (error) {
      console.error("Error updating business featured status:", error);
      res.status(500).json({ message: "Failed to update business featured status" });
    }
  });

  // Admin: Get all featured businesses
  app.get("/api/admin/businesses/featured", async (req, res) => {
    try {
      if (!req.user || req.user.role !== 'admin') {
        return res.status(403).json({ message: "Admin access required" });
      }
      
      const featuredBusinesses = await storage.getAllFeaturedBusinesses();
      res.json(featuredBusinesses);
    } catch (error) {
      console.error("Error fetching all featured businesses:", error);
      res.status(500).json({ message: "Failed to fetch featured businesses" });
    }
  });

  // Content routes (news, alerts, updates)
  app.get("/api/content", async (req, res) => {
    try {
      const { type, status } = req.query;
      
      // Public endpoint only shows published content
      if (status === 'published' || !req.user) {
        const content = await storage.getPublishedContent(type as string);
        return res.json(content);
      }

      // Admin users can see all content
      if (req.user?.role === 'admin') {
        const content = await storage.getContent(type as string, status as string);
        return res.json(content);
      }

      // Regular users only see published content
      const content = await storage.getPublishedContent(type as string);
      res.json(content);
    } catch (error) {
      console.error("Get content error:", error);
      res.status(500).json({ message: "Failed to fetch content" });
    }
  });

  app.post("/api/content", async (req, res) => {
    try {
      if (!req.user || req.user.role !== 'admin') {
        return res.status(403).json({ message: "Admin access required" });
      }

      const validatedData = insertContentSchema.parse(req.body);
      const content = await storage.createContent({
        ...validatedData,
        authorId: req.user.id,
      });
      res.status(201).json(content);
    } catch (error) {
      console.error("Create content error:", error);
      res.status(400).json({ message: "Invalid content data" });
    }
  });

  app.put("/api/content/:id", async (req, res) => {
    try {
      if (!req.user || req.user.role !== 'admin') {
        return res.status(403).json({ message: "Admin access required" });
      }

      const id = parseInt(req.params.id);
      const updates = req.body;
      
      // If publishing, set publishedAt timestamp
      if (updates.status === 'published' && !updates.publishedAt) {
        updates.publishedAt = new Date();
      }

      const content = await storage.updateContent(id, updates);
      if (!content) {
        return res.status(404).json({ message: "Content not found" });
      }

      res.json(content);
    } catch (error) {
      console.error("Update content error:", error);
      res.status(400).json({ message: "Failed to update content" });
    }
  });

  app.delete("/api/content/:id", async (req, res) => {
    try {
      if (!req.user || req.user.role !== 'admin') {
        return res.status(403).json({ message: "Admin access required" });
      }

      const id = parseInt(req.params.id);
      const deleted = await storage.deleteContent(id);
      
      if (!deleted) {
        return res.status(404).json({ message: "Content not found" });
      }

      res.status(204).send();
    } catch (error) {
      console.error("Delete content error:", error);
      res.status(500).json({ message: "Failed to delete content" });
    }
  });

  // WhatsApp groups
  app.get("/api/whatsapp-groups", async (req, res) => {
    try {
      const groups = await storage.getActiveWhatsAppGroups();
      res.json(groups);
    } catch (error) {
      console.error("Get WhatsApp groups error:", error);
      res.status(500).json({ message: "Failed to fetch WhatsApp groups" });
    }
  });

  // Products and pricing
  app.get("/api/products", async (req, res) => {
    try {
      const { category } = req.query;
      const products = await storage.getProducts(category as string);
      res.json(products);
    } catch (error) {
      console.error("Get products error:", error);
      res.status(500).json({ message: "Failed to fetch products" });
    }
  });

  app.post("/api/products", async (req, res) => {
    try {
      if (!req.user || req.user.role !== 'admin') {
        return res.status(403).json({ message: "Admin access required" });
      }

      const validatedData = insertProductSchema.parse(req.body);
      const product = await storage.createProduct(validatedData);
      res.status(201).json(product);
    } catch (error) {
      console.error("Create product error:", error);
      res.status(400).json({ message: "Invalid product data" });
    }
  });

  // Admin routes
  app.get("/api/admin/users", async (req, res) => {
    try {
      if (!req.user || req.user.role !== 'admin') {
        return res.status(403).json({ message: "Admin access required" });
      }

      const users = await storage.getAllUsers();
      // Remove password from response
      const safeUsers = users.map(({ password, ...user }) => user);
      res.json(safeUsers);
    } catch (error) {
      console.error("Get users error:", error);
      res.status(500).json({ message: "Failed to fetch users" });
    }
  });

  app.get("/api/admin/contact-submissions", async (req, res) => {
    try {
      if (!req.user || req.user.role !== 'admin') {
        return res.status(403).json({ message: "Admin access required" });
      }

      const { status } = req.query;
      const submissions = await storage.getContactSubmissions(status as string);
      res.json(submissions);
    } catch (error) {
      console.error("Get contact submissions error:", error);
      res.status(500).json({ message: "Failed to fetch contact submissions" });
    }
  });

  app.put("/api/admin/users/:id", async (req, res) => {
    try {
      if (!req.user || req.user.role !== 'admin') {
        return res.status(403).json({ message: "Admin access required" });
      }

      const id = parseInt(req.params.id);
      const updates = req.body;
      
      // Don't allow password updates through this endpoint
      delete updates.password;

      const user = await storage.updateUser(id, updates);
      if (!user) {
        return res.status(404).json({ message: "User not found" });
      }

      // Remove password from response
      const { password, ...safeUser } = user;
      res.json(safeUser);
    } catch (error) {
      console.error("Update user error:", error);
      res.status(400).json({ message: "Failed to update user" });
    }
  });

  // Dashboard stats for admins
  app.get("/api/admin/stats", async (req, res) => {
    try {
      if (!req.user || req.user.role !== 'admin') {
        return res.status(403).json({ message: "Admin access required" });
      }

      const [users, businesses, contentItems, groups, submissions] = await Promise.all([
        storage.getAllUsers(),
        storage.getBusinesses(),
        storage.getContent(),
        storage.getActiveWhatsAppGroups(),
        storage.getContactSubmissions('pending'),
      ]);

      const stats = {
        totalUsers: users.length,
        activeUsers: users.filter(u => u.isActive).length,
        totalBusinesses: businesses.length,
        totalContent: contentItems.length,
        publishedContent: contentItems.filter(c => c.status === 'published').length,
        whatsappGroups: groups.length,
        totalMembers: groups.reduce((sum, group) => sum + (group.memberCount || 0), 0),
        pendingSubmissions: submissions.length,
      };

      res.json(stats);
    } catch (error) {
      console.error("Get admin stats error:", error);
      res.status(500).json({ message: "Failed to fetch admin stats" });
    }
  });

  // Newsletter subscription routes
  app.post("/api/newsletter/subscribe", async (req, res) => {
    try {
      const validatedData = insertEmailNewsletterSchema.parse(req.body);
      const subscription = await storage.subscribeToNewsletter(validatedData.email);
      res.status(201).json(subscription);
    } catch (error) {
      console.error("Newsletter subscription error:", error);
      if (error instanceof z.ZodError) {
        return res.status(400).json({ message: error.errors[0]?.message || "Invalid email address" });
      }
      res.status(400).json({ message: "Subscription failed" });
    }
  });

  app.post("/api/newsletter/unsubscribe", async (req, res) => {
    try {
      const { email } = req.body;
      if (!email) {
        return res.status(400).json({ message: "Email is required" });
      }
      
      const success = await storage.unsubscribeFromNewsletter(email);
      if (success) {
        res.json({ message: "Successfully unsubscribed" });
      } else {
        res.status(404).json({ message: "Email not found" });
      }
    } catch (error) {
      console.error("Newsletter unsubscribe error:", error);
      res.status(500).json({ message: "Unsubscribe failed" });
    }
  });

  // Admin newsletter management routes
  app.get("/api/admin/newsletter/subscribers", async (req, res) => {
    try {
      if (!req.user || req.user.role !== "admin") {
        return res.status(403).json({ message: "Admin access required" });
      }

      const activeOnly = req.query.active !== "false";
      const subscribers = await storage.getNewsletterSubscribers(activeOnly);
      res.json(subscribers);
    } catch (error) {
      console.error("Get newsletter subscribers error:", error);
      res.status(500).json({ message: "Failed to fetch subscribers" });
    }
  });

  // SMTP configuration routes
  app.get("/api/admin/smtp-config", async (req, res) => {
    try {
      if (!req.user || req.user.role !== "admin") {
        return res.status(403).json({ message: "Admin access required" });
      }

      const config = await storage.getSmtpConfig();
      if (config) {
        // Don't expose password in response
        const { password, ...safeConfig } = config;
        res.json(safeConfig);
      } else {
        res.status(404).json({ message: "No SMTP configuration found" });
      }
    } catch (error) {
      console.error("Get SMTP config error:", error);
      res.status(500).json({ message: "Failed to fetch SMTP configuration" });
    }
  });

  app.post("/api/admin/smtp-config", async (req, res) => {
    try {
      if (!req.user || req.user.role !== "admin") {
        return res.status(403).json({ message: "Admin access required" });
      }

      const validatedData = insertSmtpConfigSchema.parse(req.body);
      const config = await storage.updateSmtpConfig(validatedData);
      
      // Don't expose password in response
      const { password, ...safeConfig } = config;
      res.status(201).json(safeConfig);
    } catch (error) {
      console.error("Update SMTP config error:", error);
      if (error instanceof z.ZodError) {
        return res.status(400).json({ message: error.errors[0]?.message || "Invalid SMTP configuration" });
      }
      res.status(500).json({ message: "Failed to update SMTP configuration" });
    }
  });

  app.post("/api/admin/smtp-test", async (req, res) => {
    try {
      if (!req.user || req.user.role !== "admin") {
        return res.status(403).json({ message: "Admin access required" });
      }

      const { testEmail } = req.body;
      const targetEmail = testEmail || req.user.email;

      const config = await storage.getSmtpConfig();
      if (!config) {
        return res.status(404).json({ message: "No SMTP configuration found" });
      }

      // Test SMTP connection
      const transporter = nodemailer.createTransport({
        host: config.host,
        port: config.port,
        auth: {
          user: config.username,
          pass: config.password,
        },
      });

      await transporter.verify();
      
      // Send test email
      await transporter.sendMail({
        from: `"${config.fromName}" <${config.fromEmail}>`,
        to: targetEmail,
        subject: "LoveMedia Foundation - SMTP Test",
        text: "This is a test email to verify your SMTP configuration is working correctly.",
        html: `
          <h2>SMTP Test Successful</h2>
          <p>This is a test email to verify your SMTP configuration is working correctly.</p>
          <p>Your email system is now ready to send newsletters to your community!</p>
          <br>
          <p>Best regards,<br>LoveMedia Foundation</p>
        `,
      });

      res.json({ message: `SMTP test successful! Check ${targetEmail} for confirmation.` });
    } catch (error: any) {
      console.error("SMTP test error:", error);
      res.status(500).json({ message: `SMTP test failed: ${error.message}` });
    }
  });

  // Mass email campaign routes
  app.post("/api/admin/send-mass-email", async (req, res) => {
    try {
      if (!req.user || req.user.role !== "admin") {
        return res.status(403).json({ message: "Admin access required" });
      }

      const { subject, content } = req.body;
      if (!subject || !content) {
        return res.status(400).json({ message: "Subject and content are required" });
      }

      const config = await storage.getSmtpConfig();
      if (!config) {
        return res.status(404).json({ message: "No SMTP configuration found. Please configure SMTP first." });
      }

      const subscribers = await storage.getNewsletterSubscribers(true);
      if (subscribers.length === 0) {
        return res.status(400).json({ message: "No active subscribers found" });
      }

      // Create email campaign record
      const campaign = await storage.createEmailCampaign({
        subject,
        content,
        recipientCount: subscribers.length,
        status: "sending",
        sentBy: req.user.id,
      });

      // Setup nodemailer transporter
      const transporter = nodemailer.createTransport({
        host: config.host,
        port: config.port,
        auth: {
          user: config.username,
          pass: config.password,
        },
      });

      let sentCount = 0;
      let failedCount = 0;

      // Send emails in batches to avoid overwhelming the server
      const batchSize = 10;
      for (let i = 0; i < subscribers.length; i += batchSize) {
        const batch = subscribers.slice(i, i + batchSize);
        
        await Promise.allSettled(
          batch.map(async (subscriber) => {
            try {
              await transporter.sendMail({
                from: `"${config.fromName}" <${config.fromEmail}>`,
                to: subscriber.email,
                subject,
                html: `
                  ${content}
                  <br><br>
                  <hr>
                  <small>
                    <p>You received this email because you subscribed to LoveMedia Foundation newsletter.</p>
                    <p><a href="${process.env.REPLIT_DOMAIN || 'http://localhost:5000'}/api/newsletter/unsubscribe?email=${encodeURIComponent(subscriber.email)}">Unsubscribe</a></p>
                  </small>
                `,
              });
              sentCount++;
            } catch (error) {
              console.error(`Failed to send email to ${subscriber.email}:`, error);
              failedCount++;
            }
          })
        );

        // Small delay between batches
        if (i + batchSize < subscribers.length) {
          await new Promise(resolve => setTimeout(resolve, 1000));
        }
      }

      // Update campaign status
      await storage.updateEmailCampaign(campaign.id, {
        status: failedCount === 0 ? "sent" : "failed",
        sentCount,
        failedCount,
        sentAt: new Date(),
      });

      res.json({
        message: `Mass email sent successfully!`,
        campaignId: campaign.id,
        totalRecipients: subscribers.length,
        sentCount,
        failedCount,
      });
    } catch (error) {
      console.error("Mass email error:", error);
      res.status(500).json({ message: "Failed to send mass email" });
    }
  });

  app.get("/api/admin/email-campaigns", async (req, res) => {
    try {
      if (!req.user || req.user.role !== "admin") {
        return res.status(403).json({ message: "Admin access required" });
      }

      const campaigns = await storage.getEmailCampaigns();
      res.json(campaigns);
    } catch (error) {
      console.error("Get email campaigns error:", error);
      res.status(500).json({ message: "Failed to fetch email campaigns" });
    }
  });

  // Page layout editor routes (Admin only)
  app.get("/api/admin/page-layout/:pageName", async (req, res) => {
    try {
      if (!req.user || req.user.role !== 'admin') {
        return res.status(403).json({ message: "Admin access required" });
      }

      const { pageName } = req.params;
      const layout = await storage.getPageLayout(pageName);
      
      if (!layout) {
        return res.status(404).json({ message: "Page layout not found" });
      }

      res.json(layout);
    } catch (error) {
      console.error("Get page layout error:", error);
      res.status(500).json({ message: "Failed to fetch page layout" });
    }
  });

  app.post("/api/admin/page-layout", async (req, res) => {
    try {
      if (!req.user || req.user.role !== 'admin') {
        return res.status(403).json({ message: "Admin access required" });
      }

      const validatedData = insertPageLayoutSchema.parse(req.body);
      
      // Check if page layout already exists
      const existingLayout = await storage.getPageLayout(validatedData.pageName);
      
      if (existingLayout) {
        // Update existing layout
        const updatedLayout = await storage.updatePageLayout(
          validatedData.pageName,
          validatedData
        );
        return res.json(updatedLayout);
      }

      // Create new layout
      const layout = await storage.createPageLayout({
        ...validatedData,
        createdBy: req.user.id,
      });
      
      res.status(201).json(layout);
    } catch (error) {
      console.error("Create/update page layout error:", error);
      res.status(400).json({ message: "Invalid page layout data" });
    }
  });

  app.put("/api/admin/page-layout/:pageName", async (req, res) => {
    try {
      if (!req.user || req.user.role !== 'admin') {
        return res.status(403).json({ message: "Admin access required" });
      }

      const { pageName } = req.params;
      const updates = req.body;

      const layout = await storage.updatePageLayout(pageName, updates);
      
      if (!layout) {
        return res.status(404).json({ message: "Page layout not found" });
      }

      res.json(layout);
    } catch (error) {
      console.error("Update page layout error:", error);
      res.status(400).json({ message: "Failed to update page layout" });
    }
  });

  app.delete("/api/admin/page-layout/:pageName", async (req, res) => {
    try {
      if (!req.user || req.user.role !== 'admin') {
        return res.status(403).json({ message: "Admin access required" });
      }

      const { pageName } = req.params;
      const deleted = await storage.deletePageLayout(pageName);
      
      if (!deleted) {
        return res.status(404).json({ message: "Page layout not found" });
      }

      res.status(204).send();
    } catch (error) {
      console.error("Delete page layout error:", error);
      res.status(500).json({ message: "Failed to delete page layout" });
    }
  });

  app.get("/api/admin/page-layouts", async (req, res) => {
    try {
      if (!req.user || req.user.role !== 'admin') {
        return res.status(403).json({ message: "Admin access required" });
      }

      const layouts = await storage.getAllPageLayouts();
      res.json(layouts);
    } catch (error) {
      console.error("Get page layouts error:", error);
      res.status(500).json({ message: "Failed to fetch page layouts" });
    }
  });

  // Header Tags API routes
  app.get("/api/admin/header-tags", async (req, res) => {
    try {
      if (!req.user || req.user.role !== 'admin') {
        return res.status(403).json({ message: "Admin access required" });
      }

      const tags = await storage.getHeaderTags();
      res.json(tags);
    } catch (error) {
      console.error("Get header tags error:", error);
      res.status(500).json({ message: "Failed to fetch header tags" });
    }
  });

  app.get("/api/header-tags/active", async (req, res) => {
    try {
      const tags = await storage.getActiveHeaderTags();
      res.json(tags);
    } catch (error) {
      console.error("Get active header tags error:", error);
      res.status(500).json({ message: "Failed to fetch active header tags" });
    }
  });

  app.post("/api/admin/header-tags", async (req, res) => {
    try {
      if (!req.user || req.user.role !== 'admin') {
        return res.status(403).json({ message: "Admin access required" });
      }

      const validatedData = insertHeaderTagSchema.parse(req.body);
      
      const tag = await storage.createHeaderTag({
        ...validatedData,
        createdBy: req.user.id,
      });
      
      res.status(201).json(tag);
    } catch (error) {
      console.error("Create header tag error:", error);
      res.status(400).json({ message: "Invalid header tag data" });
    }
  });

  app.put("/api/admin/header-tags/:id", async (req, res) => {
    try {
      if (!req.user || req.user.role !== 'admin') {
        return res.status(403).json({ message: "Admin access required" });
      }

      const { id } = req.params;
      const updates = req.body;

      const tag = await storage.updateHeaderTag(parseInt(id), updates);
      
      if (!tag) {
        return res.status(404).json({ message: "Header tag not found" });
      }

      res.json(tag);
    } catch (error) {
      console.error("Update header tag error:", error);
      res.status(400).json({ message: "Failed to update header tag" });
    }
  });

  app.delete("/api/admin/header-tags/:id", async (req, res) => {
    try {
      if (!req.user || req.user.role !== 'admin') {
        return res.status(403).json({ message: "Admin access required" });
      }

      const { id } = req.params;
      const deleted = await storage.deleteHeaderTag(parseInt(id));
      
      if (!deleted) {
        return res.status(404).json({ message: "Header tag not found" });
      }

      res.status(204).send();
    } catch (error) {
      console.error("Delete header tag error:", error);
      res.status(500).json({ message: "Failed to delete header tag" });
    }
  });

  // SEO Settings management (admin only)
  app.get("/api/admin/seo-settings", async (req, res) => {
    try {
      if (!req.user || req.user.role !== 'admin') {
        return res.status(403).json({ message: "Admin access required" });
      }
      
      const settings = await storage.getSeoSettings();
      res.json(settings);
    } catch (error) {
      console.error("Error fetching SEO settings:", error);
      res.status(500).json({ error: "Failed to fetch SEO settings" });
    }
  });

  app.get("/api/admin/seo-settings/:id", async (req, res) => {
    try {
      if (!req.user || req.user.role !== 'admin') {
        return res.status(403).json({ message: "Admin access required" });
      }
      
      const id = parseInt(req.params.id);
      const settings = await storage.getSeoSettingsById(id);
      if (!settings) {
        return res.status(404).json({ error: "SEO settings not found" });
      }
      res.json(settings);
    } catch (error) {
      console.error("Error fetching SEO settings:", error);
      res.status(500).json({ error: "Failed to fetch SEO settings" });
    }
  });

  app.get("/api/seo-settings/:pageName", async (req, res) => {
    try {
      const pageName = req.params.pageName;
      const settings = await storage.getSeoSettingsByPage(pageName);
      if (!settings) {
        return res.status(404).json({ error: "SEO settings not found" });
      }
      res.json(settings);
    } catch (error) {
      console.error("Error fetching SEO settings:", error);
      res.status(500).json({ error: "Failed to fetch SEO settings" });
    }
  });

  app.post("/api/admin/seo-settings", async (req, res) => {
    try {
      if (!req.user || req.user.role !== 'admin') {
        return res.status(403).json({ message: "Admin access required" });
      }
      
      const settings = await storage.createSeoSettings({ ...req.body, createdBy: req.user.id });
      res.status(201).json(settings);
    } catch (error) {
      console.error("Error creating SEO settings:", error);
      res.status(500).json({ error: "Failed to create SEO settings" });
    }
  });

  app.put("/api/admin/seo-settings/:id", async (req, res) => {
    try {
      if (!req.user || req.user.role !== 'admin') {
        return res.status(403).json({ message: "Admin access required" });
      }
      
      const id = parseInt(req.params.id);
      const settings = await storage.updateSeoSettings(id, req.body);
      if (!settings) {
        return res.status(404).json({ error: "SEO settings not found" });
      }
      res.json(settings);
    } catch (error) {
      console.error("Error updating SEO settings:", error);
      res.status(500).json({ error: "Failed to update SEO settings" });
    }
  });

  app.delete("/api/admin/seo-settings/:id", async (req, res) => {
    try {
      if (!req.user || req.user.role !== 'admin') {
        return res.status(403).json({ message: "Admin access required" });
      }
      
      const id = parseInt(req.params.id);
      const deleted = await storage.deleteSeoSettings(id);
      if (!deleted) {
        return res.status(404).json({ error: "SEO settings not found" });
      }
      res.status(204).send();
    } catch (error) {
      console.error("Error deleting SEO settings:", error);
      res.status(500).json({ error: "Failed to delete SEO settings" });
    }
  });

  // SEO Routes
  app.get('/sitemap.xml', (req, res) => {
    res.setHeader('Content-Type', 'application/xml');
    res.setHeader('Cache-Control', 'public, max-age=86400'); // Cache for 24 hours
    res.sendFile(path.join(process.cwd(), 'client/public/sitemap.xml'));
  });

  app.get('/robots.txt', (req, res) => {
    res.setHeader('Content-Type', 'text/plain');
    res.setHeader('Cache-Control', 'public, max-age=86400'); // Cache for 24 hours
    res.sendFile(path.join(process.cwd(), 'client/public/robots.txt'));
  });

  app.get('/llms.txt', (req, res) => {
    res.setHeader('Content-Type', 'text/plain');
    res.setHeader('Cache-Control', 'public, max-age=86400'); // Cache for 24 hours
    res.sendFile(path.join(process.cwd(), 'client/public/llms.txt'));
  });

  // System Settings management (admin only)
  app.get("/api/admin/system-settings", async (req, res) => {
    try {
      if (!req.user || req.user.role !== 'admin') {
        return res.status(403).json({ message: "Admin access required" });
      }
      
      const settings = await storage.getSystemSettings();
      res.json(settings);
    } catch (error) {
      console.error("Error fetching system settings:", error);
      res.status(500).json({ error: "Failed to fetch system settings" });
    }
  });

  app.put("/api/admin/system-settings", async (req, res) => {
    try {
      if (!req.user || req.user.role !== 'admin') {
        return res.status(403).json({ message: "Admin access required" });
      }

      const validatedData = insertSystemSettingsSchema.parse(req.body);
      const settings = await storage.updateSystemSettings({
        ...validatedData,
        updatedBy: req.user.id,
      });
      
      res.json(settings);
    } catch (error) {
      console.error("Update system settings error:", error);
      res.status(400).json({ message: "Failed to update system settings" });
    }
  });

  // Data Deals Orders API routes
  app.post("/api/data-deals/orders", async (req, res) => {
    try {
      console.log('[BACKEND] Data deals order submission started');
      console.log('[BACKEND] Request body received:', JSON.stringify(req.body, null, 2));
      console.log('[BACKEND] Request headers:', req.headers);
      console.log('[BACKEND] User authenticated:', !!req.user);
      
      // Validate the incoming data
      console.log('[BACKEND] Validating data against schema...');
      const validatedData = insertDataDealsOrderSchema.parse(req.body);
      console.log('[BACKEND] Data validation successful:', JSON.stringify(validatedData, null, 2));
      
      // Create the order in database
      console.log('[BACKEND] Creating order in database...');
      const order = await storage.createDataDealsOrder(validatedData);
      console.log('[BACKEND] Order created successfully:', JSON.stringify(order, null, 2));
      
      // Send email to OpenWeb
      console.log('[BACKEND] Sending email to OpenWeb...');
      await sendOrderToOpenWeb(order);
      console.log('[BACKEND] Email sent successfully');
      
      // Mark as sent
      console.log('[BACKEND] Marking order as sent to OpenWeb...');
      await storage.markOrderAsSentToOpenWeb(order.id, order.orderNumber!);
      console.log('[BACKEND] Order marked as sent successfully');

      // Send customer confirmation email
      console.log('[BACKEND] Sending customer confirmation email...');
      try {
        await sendCustomerConfirmationEmail(order);
        console.log('[BACKEND] Customer confirmation email sent successfully');
      } catch (emailError) {
        console.error('[BACKEND] Failed to send customer confirmation email:', emailError);
        // Don't fail the order if customer email fails
      }
      
      const response = { 
        success: true, 
        message: "Order submitted successfully!",
        orderNumber: order.orderNumber 
      };
      console.log('[BACKEND] Sending success response:', response);
      
      res.status(201).json(response);
    } catch (error: any) {
      console.error('[BACKEND] Data deals order submission error:', error);
      console.error('[BACKEND] Error type:', error.constructor?.name);
      console.error('[BACKEND] Error message:', error.message);
      if (error.stack) {
        console.error('[BACKEND] Error stack:', error.stack);
      }
      if (error.issues) {
        console.error('[BACKEND] Validation issues:', error.issues);
      }
      
      const errorResponse = { 
        success: false, 
        message: "Failed to submit order. Please try again.",
        details: error.message
      };
      console.log('[BACKEND] Sending error response:', errorResponse);
      
      res.status(400).json(errorResponse);
    }
  });

  // Admin route to view data deals orders
  app.get("/api/admin/data-deals/orders", async (req, res) => {
    try {
      if (!req.user || req.user.role !== 'admin') {
        return res.status(403).json({ message: "Admin access required" });
      }

      const { status } = req.query;
      const orders = await storage.getDataDealsOrders(status as string);
      res.json(orders);
    } catch (error) {
      console.error("Get data deals orders error:", error);
      res.status(500).json({ message: "Failed to fetch orders" });
    }
  });

  app.get("/api/admin/data-deals/orders/:id", async (req, res) => {
    try {
      if (!req.user || req.user.role !== 'admin') {
        return res.status(403).json({ message: "Admin access required" });
      }

      const { id } = req.params;
      const order = await storage.getDataDealsOrder(parseInt(id));
      
      if (!order) {
        return res.status(404).json({ message: "Order not found" });
      }

      res.json(order);
    } catch (error) {
      console.error("Get data deals order error:", error);
      res.status(500).json({ message: "Failed to fetch order" });
    }
  });

  // Data deals email settings management routes
  app.get("/api/admin/data-deals/email-settings", async (req, res) => {
    try {
      console.log('[API] GET /api/admin/data-deals/email-settings - User:', req.user?.email);
      
      if (!req.user || req.user.role !== 'admin') {
        console.warn('[API] Unauthorized access attempt to email settings');
        return res.status(403).json({ message: "Admin access required" });
      }

      console.log('[API] Fetching email settings from storage...');
      const settings = await storage.getDataDealsEmailSettings();
      
      console.log('[API] Email settings retrieved:', settings ? 'Settings found' : 'No settings, will create defaults');
      res.json(settings);
    } catch (error: any) {
      console.error("[API ERROR] Get data deals email settings error:", {
        message: error.message,
        stack: error.stack,
        user: req.user?.email,
        timestamp: new Date().toISOString()
      });
      
      res.status(500).json({ 
        message: "Failed to fetch email settings", 
        error: process.env.NODE_ENV === 'development' ? error.message : undefined 
      });
    }
  });

  app.put("/api/admin/data-deals/email-settings", async (req, res) => {
    try {
      console.log('[API] PUT /api/admin/data-deals/email-settings - User:', req.user?.email);
      console.log('[API] Request body:', {
        toEmail: req.body?.toEmail,
        fromEmail: req.body?.fromEmail,
        subject: req.body?.subject?.substring(0, 50) + '...'
      });
      
      if (!req.user || req.user.role !== 'admin') {
        console.warn('[API] Unauthorized access attempt to update email settings');
        return res.status(403).json({ message: "Admin access required" });
      }

      console.log('[API] Validating request data...');
      const validatedData = insertDataDealsEmailSettingsSchema.parse(req.body);
      console.log('[API] Data validation successful');
      
      // Get existing settings to update or create new ones
      console.log('[API] Checking for existing settings...');
      const existingSettings = await storage.getDataDealsEmailSettings();
      
      let updatedSettings;
      if (existingSettings) {
        console.log('[API] Updating existing settings with ID:', existingSettings.id);
        updatedSettings = await storage.updateDataDealsEmailSettings(existingSettings.id, validatedData);
      } else {
        console.log('[API] Creating new settings for user:', req.user.id);
        updatedSettings = await storage.createDataDealsEmailSettings({
          ...validatedData,
          createdBy: req.user.id,
        });
      }

      console.log('[API] Email settings operation successful');
      res.json(updatedSettings);
    } catch (error) {
      if (error instanceof z.ZodError) {
        console.error("[API VALIDATION ERROR] Data validation failed:", {
          errors: error.errors,
          user: req.user?.email,
          timestamp: new Date().toISOString()
        });
        return res.status(400).json({ 
          message: "Invalid data provided", 
          errors: error.errors,
          details: error.errors.map(e => `${e.path.join('.')}: ${e.message}`)
        });
      }
      
      console.error("[API ERROR] Update data deals email settings error:", {
        message: (error as any).message,
        stack: (error as any).stack,
        user: req.user?.email,
        timestamp: new Date().toISOString()
      });
      
      res.status(500).json({ 
        message: "Failed to update email settings", 
        error: process.env.NODE_ENV === 'development' ? (error as any).message : undefined 
      });
    }
  });

  // Customer confirmation email settings management routes
  app.get("/api/admin/data-deals/customer-email-settings", async (req, res) => {
    try {
      console.log('[API] GET /api/admin/data-deals/customer-email-settings - User:', req.user?.email);
      
      if (!req.user || req.user.role !== 'admin') {
        console.warn('[API] Unauthorized access attempt to customer email settings');
        return res.status(403).json({ message: "Admin access required" });
      }

      console.log('[API] Fetching customer email settings from storage...');
      const settings = await storage.getDataDealsCustomerEmailSettings();
      
      console.log('[API] Customer email settings retrieved:', settings ? 'Settings found' : 'No settings, will create defaults');
      res.json(settings);
    } catch (error: any) {
      console.error("[API ERROR] Get customer email settings error:", {
        message: error.message,
        stack: error.stack,
        user: req.user?.email,
        timestamp: new Date().toISOString()
      });
      
      res.status(500).json({ 
        message: "Failed to fetch customer email settings", 
        error: process.env.NODE_ENV === 'development' ? error.message : undefined 
      });
    }
  });

  app.put("/api/admin/data-deals/customer-email-settings", async (req, res) => {
    try {
      console.log('[API] PUT /api/admin/data-deals/customer-email-settings - User:', req.user?.email);
      console.log('[API] Request body:', {
        fromEmail: req.body?.fromEmail,
        subject: req.body?.subject?.substring(0, 50) + '...'
      });
      
      if (!req.user || req.user.role !== 'admin') {
        console.warn('[API] Unauthorized access attempt to update customer email settings');
        return res.status(403).json({ message: "Admin access required" });
      }

      console.log('[API] Validating customer email request data...');
      const validatedData = insertDataDealsCustomerEmailSettingsSchema.parse(req.body);
      console.log('[API] Customer email data validation successful');
      
      // Get existing settings to update or create new ones
      console.log('[API] Checking for existing customer email settings...');
      const existingSettings = await storage.getDataDealsCustomerEmailSettings();
      
      let updatedSettings;
      if (existingSettings) {
        console.log('[API] Updating existing customer email settings with ID:', existingSettings.id);
        updatedSettings = await storage.updateDataDealsCustomerEmailSettings(existingSettings.id, validatedData);
      } else {
        console.log('[API] Creating new customer email settings for user:', req.user.id);
        updatedSettings = await storage.createDataDealsCustomerEmailSettings({
          ...validatedData,
          createdBy: req.user.id,
        });
      }

      console.log('[API] Customer email settings operation successful');
      res.json(updatedSettings);
    } catch (error) {
      if (error instanceof z.ZodError) {
        console.error("[API VALIDATION ERROR] Customer email data validation failed:", {
          errors: error.errors,
          user: req.user?.email,
          timestamp: new Date().toISOString()
        });
        return res.status(400).json({ 
          message: "Invalid data provided", 
          errors: error.errors,
          details: error.errors.map(e => `${e.path.join('.')}: ${e.message}`)
        });
      }
      
      console.error("[API ERROR] Update customer email settings error:", {
        message: (error as any).message,
        stack: (error as any).stack,
        user: req.user?.email,
        timestamp: new Date().toISOString()
      });
      
      res.status(500).json({ 
        message: "Failed to update customer email settings", 
        error: process.env.NODE_ENV === 'development' ? (error as any).message : undefined 
      });
    }
  });

  // Function to substitute variables in email templates
  function substituteVariables(template: string, order: any): string {
    const variables = {
      orderNumber: order.orderNumber || 'N/A',
      orderDate: new Date(order.createdAt).toLocaleDateString(),
      customerName: `${order.firstName} ${order.surname}`,
      customerEmail: order.email,
      customerMobile: order.mobile,
      customerIdNumber: order.idNumber,
      installationAddress: order.installationAddress,
      deliveryAddress: order.deliveryAddress,
      packageName: order.packageName,
      basePrice: (order.basePrice / 100).toFixed(2),
      premiumPrice: (order.premiumPrice / 100).toFixed(2),
      totalPrice: (order.totalPrice / 100).toFixed(2),
      upgradeToPremium: order.upgradeToPremium,
      paymentMethodText: order.paymentMethod === 'debit_order' ? 'Debit Order' : 'Credit Card',
      bank: order.bank,
      accountNumber: order.accountNumber,
      accountType: order.accountType,
      branchCode: order.branchCode,
      mandateSignature: order.mandateSignature,
      donationAmount: (order.donationAmount / 100).toFixed(0) // R200
    };

    let result = template;
    
    // Replace all variables
    Object.entries(variables).forEach(([key, value]) => {
      const regex = new RegExp(`{{${key}}}`, 'g');
      result = result.replace(regex, String(value));
    });

    // Handle conditional blocks like {{#if upgradeToPremium}}content{{/if}}
    result = result.replace(/{{#if\s+upgradeToPremium}}(.*?){{\/if}}/g, (match, content) => {
      return order.upgradeToPremium ? content : '';
    });

    return result;
  }

  // Business Images Admin API endpoints
  interface ImportLog {
    id: string;
    message: string;
    type: 'info' | 'success' | 'warning' | 'error';
    timestamp: string;
  }
  
  let importProgress: {
    isRunning: boolean;
    progress: number;
    currentStep: string;
    totalImages: number;
    processedImages: number;
    matchedImages: number;  
    errorCount: number;
    logs: ImportLog[];
  } = {
    isRunning: false,
    progress: 0,
    currentStep: "Ready to import",
    totalImages: 0,
    processedImages: 0,
    matchedImages: 0,
    errorCount: 0,
    logs: []
  };

  app.get("/api/admin/business-images/status", async (req, res) => {
    try {
      if (!req.user || req.user.role !== 'admin') {
        return res.status(403).json({ message: "Admin access required" });
      }

      res.json(importProgress);
    } catch (error) {
      console.error("Get image import status error:", error);
      res.status(500).json({ message: "Failed to get import status" });
    }
  });

  app.get("/api/admin/business-images/stats", async (req, res) => {
    try {
      if (!req.user || req.user.role !== 'admin') {
        return res.status(403).json({ message: "Admin access required" });
      }

      // Get business statistics
      const allBusinesses = await storage.getBusinesses();
      const totalBusinesses = allBusinesses.total;
      
      // Count businesses with images (those that have imageUrl field)
      const businessesWithImages = allBusinesses.businesses.filter(b => b.imageUrl).length;
      
      // Count available Excel files with image data
      const fs = await import('fs');
      const path = await import('path');
      const attachedAssetsDir = path.join(process.cwd(), 'attached_assets');
      let availableImages = 0;
      
      try {
        const excelFiles = fs.readdirSync(attachedAssetsDir)
          .filter(file => file.endsWith('.xlsx') && file.includes('1753526977'));
        availableImages = excelFiles.length * 50; // Rough estimate
      } catch (err) {
        console.log('Could not count available images:', (err as Error).message);
      }

      const stats = {
        totalBusinesses,
        businessesWithImages,
        availableImages
      };
      
      res.json(stats);
    } catch (error) {
      console.error("Get image stats error:", error);
      res.status(500).json({ message: "Failed to get image statistics" });
    }
  });

  app.post("/api/admin/business-images/start-import", async (req, res) => {
    try {
      if (!req.user || req.user.role !== 'admin') {
        return res.status(403).json({ message: "Admin access required" });
      }

      if (importProgress.isRunning) {
        return res.status(400).json({ message: "Import already running" });
      }

      // Initialize import progress
      importProgress = {
        isRunning: true,
        progress: 0,
        currentStep: "Starting import process...",
        totalImages: 0,
        processedImages: 0,
        matchedImages: 0,
        errorCount: 0,
        logs: [{
          id: Date.now().toString(),
          message: "Image import process started",
          type: "info",
          timestamp: new Date().toISOString()
        }]
      };

      // Start the import process asynchronously
      (async () => {
        try {
          // Import the image import script
          const { importBusinessImages, setProgressCallback } = await import('../scripts/import-images.js') as any;
          
          // Set up progress callback
          setProgressCallback((log: ImportLog) => {
            importProgress.logs.push(log);
            if (importProgress.logs.length > 100) {
              importProgress.logs = importProgress.logs.slice(-100); // Keep last 100 logs
            }
            
            // Update progress based on processed images
            if (log.message.includes('Processing image')) {
              const match = log.message.match(/(\d+)\/(\d+)/);
              if (match) {
                const processed = parseInt(match[1]);
                const total = parseInt(match[2]);
                importProgress.processedImages = processed;
                importProgress.totalImages = total;
                importProgress.progress = Math.round((processed / total) * 100);
                importProgress.currentStep = `Processing image ${processed}/${total}`;
              }
            }
            
            // Update batch progress
            if (log.message.includes('Processing batch')) {
              importProgress.currentStep = log.message;
            }
            
            // Count matched images
            if (log.type === 'success' && log.message.includes('Successfully downloaded')) {
              importProgress.matchedImages++;
            }
            
            // Count errors
            if (log.type === 'error') {
              importProgress.errorCount++;
            }
          });

          // Run the import
          const result = await importBusinessImages();
          
          // Update final progress
          importProgress = {
            ...importProgress,
            isRunning: false,
            progress: 100,
            currentStep: result.success ? "Import completed successfully" : "Import failed",
            totalImages: result.totalImages,
            processedImages: result.processedImages,
            matchedImages: result.matchedImages,
            errorCount: result.errorCount,
            logs: [
              ...importProgress.logs,
              {
                id: Date.now().toString(),
                message: `Import completed: ${result.matchedImages}/${result.totalImages} images processed successfully`,
                type: result.success ? "success" : "error",
                timestamp: new Date().toISOString()
              }
            ]
          };

        } catch (error) {
          console.error("Import process error:", error);
          importProgress = {
            ...importProgress,
            isRunning: false,
            currentStep: "Import failed with error",
            errorCount: importProgress.errorCount + 1,
            logs: [
              ...importProgress.logs,
              {
                id: Date.now().toString(),
                message: `Import failed: ${(error as Error).message}`,
                type: "error",
                timestamp: new Date().toISOString()
              }
            ]
          };
        }
      })();
      
      res.json(importProgress);
    } catch (error) {
      console.error("Start image import error:", error);
      res.status(500).json({ message: "Failed to start image import" });
    }
  });

  app.post("/api/admin/business-images/stop-import", async (req, res) => {
    try {
      if (!req.user || req.user.role !== 'admin') {
        return res.status(403).json({ message: "Admin access required" });
      }

      if (!importProgress.isRunning) {
        return res.status(400).json({ message: "No import process running" });
      }

      try {
        // Import and call the stop function
        const { stopImport } = await import('../scripts/import-images.js') as any;
        stopImport();
        
        importProgress.logs.push({
          id: Date.now().toString(),
          message: "Import process stopped by admin",
          type: "warning",
          timestamp: new Date().toISOString()
        });

        res.json({ message: "Import process stopped" });
      } catch (error) {
        res.json({ message: "Import process stopped" });
      }
    } catch (error) {
      console.error("Stop image import error:", error);
      res.status(500).json({ message: "Failed to stop image import" });
    }
  });

  // Function to send order email to OpenWeb
  async function sendOrderToOpenWeb(order: any) {
    try {
      // Get SMTP configuration
      const smtpSettings = await storage.getSmtpConfig();
      
      if (!smtpSettings) {
        console.warn("No SMTP configuration found, order saved but email not sent");
        return;
      }

      // Get data deals email settings
      const emailSettings = await storage.getDataDealsEmailSettings();
      
      if (!emailSettings) {
        console.warn("No data deals email settings found, using defaults");
        return;
      }

      // Create transporter
      const transporter = nodemailer.createTransport({
        host: smtpSettings.host,
        port: smtpSettings.port,
        secure: smtpSettings.port === 465,
        auth: {
          user: smtpSettings.username,
          pass: smtpSettings.password,
        },
      });

      // Substitute variables in email template and subject
      const emailContent = substituteVariables(emailSettings.emailTemplate, order);
      const emailSubject = substituteVariables(emailSettings.subject, order);
      const replyToEmail = substituteVariables(emailSettings.replyToEmail || "", order);

      // Send email
      await transporter.sendMail({
        from: `${emailSettings.fromName} <${emailSettings.fromEmail}>`,
        to: emailSettings.toEmail,
        cc: emailSettings.ccEmail || undefined,
        replyTo: replyToEmail,
        subject: emailSubject,
        html: emailContent
      });

      console.log(`Order email sent successfully for order ${order.orderNumber}`);
    } catch (error) {
      console.error("Failed to send order email:", error);
      // Don't throw error - order should still be saved even if email fails
    }
  }

  // Function to send customer confirmation email
  async function sendCustomerConfirmationEmail(order: any) {
    try {
      console.log('[CUSTOMER EMAIL] Starting customer confirmation email process...');
      
      // Get SMTP configuration
      const smtpSettings = await storage.getSmtpConfig();
      
      if (!smtpSettings) {
        console.warn("[CUSTOMER EMAIL] No SMTP configuration found, customer email not sent");
        return;
      }

      // Get customer email settings
      const customerEmailSettings = await storage.getDataDealsCustomerEmailSettings();
      
      if (!customerEmailSettings) {
        console.warn("[CUSTOMER EMAIL] No customer email settings found, using defaults");
        return;
      }

      // Create transporter
      const transporter = nodemailer.createTransport({
        host: smtpSettings.host,
        port: smtpSettings.port,
        secure: smtpSettings.port === 465,
        auth: {
          user: smtpSettings.username,
          pass: smtpSettings.password,
        },
      });

      // Substitute variables in email template and subject
      const emailContent = substituteVariables(customerEmailSettings.emailTemplate, order);
      const emailSubject = substituteVariables(customerEmailSettings.subject, order);

      // Send email to customer
      await transporter.sendMail({
        from: `${customerEmailSettings.fromName} <${customerEmailSettings.fromEmail}>`,
        to: order.email, // Customer's email address
        replyTo: customerEmailSettings.replyToEmail || undefined,
        subject: emailSubject,
        html: emailContent
      });

      console.log(`[CUSTOMER EMAIL] Customer confirmation email sent successfully to ${order.email} for order ${order.orderNumber}`);
    } catch (error) {
      console.error("[CUSTOMER EMAIL] Failed to send customer confirmation email:", error);
      throw error; // Let the calling function handle this error
    }
  }

  // Import management routes (admin only)
  const importManager = ImportManager.getInstance();

  app.get("/api/admin/import/status", async (req, res) => {
    if (!req.user || req.user.role !== 'admin') {
      return res.status(403).json({ message: "Admin access required" });
    }
    
    try {
      const status = await importManager.getStatus();
      res.json(status);
    } catch (error) {
      console.error("Get import status error:", error);
      res.status(500).json({ message: "Failed to get import status" });
    }
  });

  app.post("/api/admin/import/start", async (req, res) => {
    if (!req.user || req.user.role !== 'admin') {
      return res.status(403).json({ message: "Admin access required" });
    }
    
    try {
      const { clearExisting } = req.body;
      await importManager.startImport(clearExisting);
      const status = await importManager.getStatus();
      res.json({ message: "Import started", status });
    } catch (error) {
      console.error("Start import error:", error);
      res.status(400).json({ message: error instanceof Error ? error.message : "Failed to start import" });
    }
  });

  app.post("/api/admin/import/stop", async (req, res) => {
    if (!req.user || req.user.role !== 'admin') {
      return res.status(403).json({ message: "Admin access required" });
    }
    
    try {
      await importManager.stopImport();
      const status = await importManager.getStatus();
      res.json({ message: "Import stopped", status });
    } catch (error) {
      console.error("Stop import error:", error);
      res.status(400).json({ message: error instanceof Error ? error.message : "Failed to stop import" });
    }
  });

  const httpServer = createServer(app);
  return httpServer;
}
