import { pgTable, text, serial, integer, boolean, timestamp, varchar, json } from "drizzle-orm/pg-core";
import { relations } from "drizzle-orm";
import { createInsertSchema } from "drizzle-zod";
import { z } from "zod";

// Users table with role-based access
export const users = pgTable("users", {
  id: serial("id").primaryKey(),
  username: text("username").notNull().unique(),
  email: text("email").notNull().unique(),
  password: text("password").notNull(),
  // Legacy fields (required for backward compatibility)
  fullName: text("full_name").notNull(),
  location: text("location"),
  // New fields
  firstName: text("first_name"),
  lastName: text("last_name"),
  mobile: text("mobile"),
  province: text("province"), // South African province
  role: text("role").notNull().default("client"), // 'admin' or 'client'
  isActive: boolean("is_active").notNull().default(true),
  createdAt: timestamp("created_at").defaultNow().notNull(),
  lastLogin: timestamp("last_login"),
});

// Business directory entries
export const businesses = pgTable("businesses", {
  id: serial("id").primaryKey(),
  storeId: text("store_id"), // Original store ID from Excel data
  name: text("name").notNull(),
  description: text("description"),
  category: text("category").notNull(),
  location: text("location").notNull(),
  contactEmail: text("contact_email"),
  contactPhone: text("contact_phone"),
  website: text("website"),
  whatsappNumber: text("whatsapp_number"),
  contactPerson: text("contact_person"),
  slogan: text("slogan"),
  latitude: text("latitude"), // Store as text to preserve precision
  longitude: text("longitude"), // Store as text to preserve precision
  imageUrl: text("image_url"), // Local image path
  originalImageUrl: text("original_image_url"), // Original remote image URL
  isVerified: boolean("is_verified").notNull().default(false),
  rating: integer("rating").default(0),
  reviewCount: integer("review_count").default(0),
  // Payment and premium features
  isFeatured: boolean("is_featured").notNull().default(false),
  featuredUntil: timestamp("featured_until"),
  highlightColor: text("highlight_color"), // Color code for highlighted listings
  topPlacement: boolean("top_placement").notNull().default(false),
  topPlacementUntil: timestamp("top_placement_until"),
  // Homepage featured businesses
  featuredOnHomepage: boolean("featured_on_homepage").notNull().default(false),
  homepageViewCount: integer("homepage_view_count").default(0), // Track views for rotation
  // Approval workflow for free listings
  status: text("status").notNull().default("pending"), // 'pending', 'approved', 'rejected'
  approvedAt: timestamp("approved_at"),
  approvedBy: integer("approved_by").references(() => users.id),
  rejectionReason: text("rejection_reason"),
  // User management
  createdBy: integer("created_by").references(() => users.id),
  createdAt: timestamp("created_at").defaultNow().notNull(),
  updatedAt: timestamp("updated_at").defaultNow().notNull(),
});

// Community content (news, alerts, updates)
export const content = pgTable("content", {
  id: serial("id").primaryKey(),
  title: text("title").notNull(),
  body: text("body").notNull(),
  type: text("type").notNull(), // 'news', 'alert', 'update', 'event'
  status: text("status").notNull().default("draft"), // 'draft', 'published', 'archived'
  priority: text("priority").notNull().default("normal"), // 'low', 'normal', 'high', 'urgent'
  tags: json("tags").$type<string[]>().default([]),
  authorId: integer("author_id").references(() => users.id),
  createdAt: timestamp("created_at").defaultNow().notNull(),
  updatedAt: timestamp("updated_at").defaultNow().notNull(),
  publishedAt: timestamp("published_at"),
});

// WhatsApp groups
export const whatsappGroups = pgTable("whatsapp_groups", {
  id: serial("id").primaryKey(),
  name: text("name").notNull(),
  description: text("description"),
  groupUrl: text("group_url").notNull(),
  memberCount: integer("member_count").default(0),
  category: text("category").notNull(), // 'discussions', 'alerts', 'accommodation', 'christian', 'other'
  isActive: boolean("is_active").notNull().default(true),
  createdAt: timestamp("created_at").defaultNow().notNull(),
});

// Contact form submissions
export const contactSubmissions = pgTable("contact_submissions", {
  id: serial("id").primaryKey(),
  name: text("name").notNull(),
  email: text("email").notNull(),
  subject: text("subject").notNull(),
  message: text("message").notNull(),
  status: text("status").notNull().default("pending"), // 'pending', 'responded', 'closed'
  createdAt: timestamp("created_at").defaultNow().notNull(),
});

// Email newsletter subscriptions (separate from registered users)
export const emailNewsletters = pgTable("email_newsletters", {
  id: serial("id").primaryKey(),
  email: text("email").notNull().unique(),
  isActive: boolean("is_active").notNull().default(true),
  createdAt: timestamp("created_at").defaultNow().notNull(),
  unsubscribedAt: timestamp("unsubscribed_at"),
});

// SMTP configuration for admin mass mail
export const smtpConfig = pgTable("smtp_config", {
  id: serial("id").primaryKey(),
  host: text("host").notNull(),
  port: integer("port").notNull(),
  username: text("username").notNull(),
  password: text("password").notNull(), // encrypted
  fromEmail: text("from_email").notNull(),
  fromName: text("from_name").notNull(),
  isActive: boolean("is_active").notNull().default(true),
  createdAt: timestamp("created_at").defaultNow().notNull(),
  updatedAt: timestamp("updated_at").defaultNow().notNull(),
});

// Mass email campaigns tracking
export const emailCampaigns = pgTable("email_campaigns", {
  id: serial("id").primaryKey(),
  subject: text("subject").notNull(),
  content: text("content").notNull(),
  recipientCount: integer("recipient_count").notNull(),
  sentCount: integer("sent_count").default(0),
  failedCount: integer("failed_count").default(0),
  status: text("status").notNull().default("draft"), // 'draft', 'sending', 'sent', 'failed'
  sentBy: integer("sent_by").references(() => users.id),
  createdAt: timestamp("created_at").defaultNow().notNull(),
  sentAt: timestamp("sent_at"),
});

// Product updates and pricing
export const products = pgTable("products", {
  id: serial("id").primaryKey(),
  name: text("name").notNull(),
  description: text("description"),
  category: text("category").notNull(), // '4g', '5g', 'partner_deal'
  wholesalePrice: integer("wholesale_price"), // in cents
  adminFeePercentage: integer("admin_fee_percentage").default(5),
  finalPrice: integer("final_price"), // calculated: wholesale + admin fee
  isActive: boolean("is_active").notNull().default(true),
  createdAt: timestamp("created_at").defaultNow().notNull(),
  updatedAt: timestamp("updated_at").defaultNow().notNull(),
});

// Page layout configurations for drag and drop editor
export const pageLayouts = pgTable("page_layouts", {
  id: serial("id").primaryKey(),
  pageName: text("page_name").notNull().unique(), // 'home', 'about', 'community', etc.
  sections: json("sections").$type<PageSection[]>().notNull(),
  isActive: boolean("is_active").notNull().default(true),
  createdBy: integer("created_by").references(() => users.id),
  createdAt: timestamp("created_at").defaultNow().notNull(),
  updatedAt: timestamp("updated_at").defaultNow().notNull(),
});

// Business listing payments (Stripe integration)
export const businessListingPayments = pgTable("business_listing_payments", {
  id: serial("id").primaryKey(),
  businessId: integer("business_id").references(() => businesses.id),
  userId: integer("user_id").references(() => users.id),
  stripePaymentIntentId: text("stripe_payment_intent_id").notNull(),
  amount: integer("amount").notNull(), // Amount in cents
  currency: text("currency").notNull().default("usd"),
  description: text("description"), // e.g., "Business listing fee"
  status: text("status").notNull().default("pending"), // 'pending', 'completed', 'failed', 'refunded'
  paymentType: text("payment_type").notNull(), // 'listing', 'featured', 'highlight', 'top_placement'
  featureDuration: integer("feature_duration"), // Duration in days for premium features
  createdAt: timestamp("created_at").defaultNow().notNull(),
  completedAt: timestamp("completed_at"),
});

// Business directory categories
export const businessCategories = pgTable("business_categories", {
  id: serial("id").primaryKey(),
  name: text("name").notNull().unique(),
  description: text("description"),
  iconName: text("icon_name"), // Lucide icon name
  displayOrder: integer("display_order").default(0),
  isActive: boolean("is_active").notNull().default(true),
  createdAt: timestamp("created_at").defaultNow().notNull(),
});

// Header tags for Google Tag Manager, Facebook Pixel, and custom tracking
export const headerTags = pgTable("header_tags", {
  id: serial("id").primaryKey(),
  name: text("name").notNull(), // Display name for admin
  type: text("type").notNull(), // 'gtm', 'facebook_pixel', 'custom'
  code: text("code").notNull(), // The actual tag code
  isActive: boolean("is_active").notNull().default(true),
  priority: integer("priority").notNull().default(1), // Order of execution (lower numbers first)
  createdAt: timestamp("created_at").defaultNow().notNull(),
  updatedAt: timestamp("updated_at").defaultNow().notNull(),
  createdBy: integer("created_by").references(() => users.id),
});

// SEO settings for different pages
export const seoSettings = pgTable("seo_settings", {
  id: serial("id").primaryKey(),
  pageName: text("page_name").notNull().unique(), // 'home', 'about', 'contact', 'business-directory', etc.
  title: text("title").notNull(),
  description: text("description").notNull(),
  keywords: text("keywords").notNull(),
  ogTitle: text("og_title"),
  ogDescription: text("og_description"),
  ogImage: text("og_image"),
  canonicalUrl: text("canonical_url"),
  structuredData: json("structured_data"), // JSON-LD structured data
  noindex: boolean("noindex").notNull().default(false),
  createdAt: timestamp("created_at").defaultNow().notNull(),
  updatedAt: timestamp("updated_at").defaultNow().notNull(),
  createdBy: integer("created_by").references(() => users.id),
});

// Data deals orders from OpenWeb partnership
export const dataDealsOrders = pgTable("data_deals_orders", {
  id: serial("id").primaryKey(),
  // Customer Information
  firstName: text("first_name").notNull(),
  surname: text("surname").notNull(),
  email: text("email").notNull(),
  idNumber: text("id_number").notNull(),
  mobile: text("mobile").notNull(),
  installationAddress: text("installation_address").notNull(),
  deliveryAddress: text("delivery_address").notNull(),
  // Package Information
  packageId: text("package_id").notNull(),
  packageName: text("package_name").notNull(),
  basePrice: integer("base_price").notNull(), // in cents
  upgradeToPremium: boolean("upgrade_to_premium").notNull().default(false),
  premiumPrice: integer("premium_price").default(0), // in cents
  totalPrice: integer("total_price").notNull(), // in cents
  // Payment Information
  paymentMethod: text("payment_method").notNull(), // 'debit_order' | 'credit_card'
  bank: text("bank").notNull(),
  accountNumber: text("account_number").notNull(),
  accountType: text("account_type").notNull(), // 'cheque' | 'savings'
  branchCode: text("branch_code").notNull(),
  mandateSignature: text("mandate_signature").notNull(),
  // Order Status
  status: text("status").notNull().default("pending"), // 'pending', 'sent', 'confirmed', 'cancelled'
  orderNumber: text("order_number"), // Generated order reference
  sentToOpenWeb: boolean("sent_to_openweb").notNull().default(false),
  donationAmount: integer("donation_amount").default(20000), // R200 in cents
  createdAt: timestamp("created_at").defaultNow().notNull(),
  processedAt: timestamp("processed_at"),
});

// Data deals email configuration for admins
export const dataDealsEmailSettings = pgTable("data_deals_email_settings", {
  id: serial("id").primaryKey(),
  isActive: boolean("is_active").default(true),
  
  // Email Configuration
  toEmail: text("to_email").notNull().default("wow@openweb.email"),
  ccEmail: text("cc_email").default("info@lovemedia.foundation"),
  fromEmail: text("from_email").notNull(),
  fromName: text("from_name").notNull().default("LoveMedia Foundation"),
  replyToEmail: text("reply_to_email").default("{{customerEmail}}"), // Variable substitution
  subject: text("subject").notNull().default("New Data Deal Order - {{orderNumber}} - LoveMedia Foundation Partnership"),
  
  // Email Template with variable support
  emailTemplate: text("email_template").notNull().default(`<h2>New LoveMedia Foundation Data Deal Order</h2>
<p><strong>Order Number:</strong> {{orderNumber}}</p>
<p><strong>Order Date:</strong> {{orderDate}}</p>

<h3>Customer Information</h3>
<ul>
  <li><strong>Name:</strong> {{customerName}}</li>
  <li><strong>Email:</strong> {{customerEmail}}</li>
  <li><strong>Mobile:</strong> {{customerMobile}}</li>
  <li><strong>ID/Passport:</strong> {{customerIdNumber}}</li>
</ul>

<h3>Installation & Delivery</h3>
<ul>
  <li><strong>Installation Address:</strong><br>{{installationAddress}}</li>
  <li><strong>Delivery Address:</strong><br>{{deliveryAddress}}</li>
</ul>

<h3>Package Details</h3>
<ul>
  <li><strong>Package:</strong> {{packageName}}</li>
  <li><strong>Base Price:</strong> R{{basePrice}}</li>
  {{#if upgradeToPremium}}<li><strong>Premium Upgrade:</strong> R{{premiumPrice}}</li>{{/if}}
  <li><strong>Total Monthly Price:</strong> R{{totalPrice}}</li>
</ul>

<h3>Payment Information</h3>
<ul>
  <li><strong>Payment Method:</strong> {{paymentMethodText}}</li>
  <li><strong>Bank:</strong> {{bank}}</li>
  <li><strong>Account Number:</strong> {{accountNumber}}</li>
  <li><strong>Account Type:</strong> {{accountType}}</li>
  <li><strong>Branch Code:</strong> {{branchCode}}</li>
  <li><strong>Electronic Mandate Signature:</strong> {{mandateSignature}}</li>
</ul>

<h3>LoveMedia Foundation Partnership</h3>
<p><strong>Donation to LoveMedia Foundation:</strong> R{{donationAmount}}</p>
<p>This order includes priority streaming & gaming protocols and exclusive discounts for LoveMedia Foundation members.</p>

<hr>
<p><em>This order was submitted through the LoveMedia Foundation platform and includes our partnership agreement for R{{donationAmount}} donation per order.</em></p>
<p><strong>For questions contact:</strong> info@lovemedia.foundation</p>`),
  
  // Available Variables Documentation
  availableVariables: text("available_variables").default("orderNumber,orderDate,customerName,customerEmail,customerMobile,customerIdNumber,installationAddress,deliveryAddress,packageName,basePrice,premiumPrice,totalPrice,upgradeToPremium,paymentMethodText,bank,accountNumber,accountType,branchCode,mandateSignature,donationAmount"),
  
  // Metadata
  createdBy: integer("created_by").references(() => users.id),
  createdAt: timestamp("created_at").defaultNow(),
  updatedAt: timestamp("updated_at").defaultNow(),
});

// Customer confirmation email settings for data deals
export const dataDealsCustomerEmailSettings = pgTable("data_deals_customer_email_settings", {
  id: serial("id").primaryKey(),
  isActive: boolean("is_active").default(true),
  
  // Email Configuration
  fromEmail: text("from_email").notNull(),
  fromName: text("from_name").notNull().default("LoveMedia Foundation"),
  replyToEmail: text("reply_to_email").default("info@lovemedia.foundation"),
  subject: text("subject").notNull().default("Thank You for Your Data Deal Order - {{orderNumber}}"),
  
  // Email Template with variable support
  emailTemplate: text("email_template").notNull().default(`<div style="font-family: Arial, sans-serif; max-width: 600px; margin: 0 auto; padding: 20px; color: #333;">
  <div style="text-align: center; margin-bottom: 30px;">
    <h1 style="color: #e11d48; margin-bottom: 10px;">Thank You for Your Order!</h1>
    <p style="font-size: 18px; color: #666;">Order Number: <strong>{{orderNumber}}</strong></p>
  </div>

  <div style="background: linear-gradient(135deg, #3b82f6, #8b5cf6); color: white; padding: 20px; border-radius: 10px; margin-bottom: 30px;">
    <h2 style="margin: 0 0 10px 0;">Hi {{customerName}}!</h2>
    <p style="margin: 0; font-size: 16px;">Thank you for choosing our exclusive 4G/5G data deals through the LoveMedia Foundation partnership.</p>
  </div>

  <div style="background: #f8fafc; padding: 20px; border-radius: 8px; margin-bottom: 20px;">
    <h3 style="color: #374151; margin-top: 0;">What Happens Next?</h3>
    <ul style="color: #6b7280; line-height: 1.6;">
      <li><strong>Order Processing:</strong> Your order is being processed by our partners at OpenWeb.co.za</li>
      <li><strong>Client Ambassador Contact:</strong> An OpenWeb Client Ambassador will reach out to you within 24-48 hours to complete your order setup</li>
      <li><strong>Installation:</strong> They will arrange delivery and installation at your specified address</li>
      <li><strong>Support:</strong> You'll receive all the technical details and support information you need</li>
    </ul>
  </div>

  <div style="background: #fef2f2; border-left: 4px solid #ef4444; padding: 15px; margin-bottom: 20px;">
    <h4 style="color: #dc2626; margin: 0 0 10px 0;">💝 Your Impact</h4>
    <p style="margin: 0; color: #7f1d1d;">With your order, <strong>R200 will be donated</strong> to LoveMedia Foundation to support South African communities. Thank you for making a difference!</p>
  </div>

  <div style="border: 1px solid #e5e7eb; padding: 20px; border-radius: 8px; margin-bottom: 20px;">
    <h3 style="color: #374151; margin-top: 0;">Your Order Details</h3>
    <table style="width: 100%; border-collapse: collapse;">
      <tr>
        <td style="padding: 8px 0; border-bottom: 1px solid #f3f4f6;"><strong>Package:</strong></td>
        <td style="padding: 8px 0; border-bottom: 1px solid #f3f4f6;">{{packageName}}</td>
      </tr>
      <tr>
        <td style="padding: 8px 0; border-bottom: 1px solid #f3f4f6;"><strong>Monthly Cost:</strong></td>
        <td style="padding: 8px 0; border-bottom: 1px solid #f3f4f6;">R{{totalPrice}}</td>
      </tr>
      <tr>
        <td style="padding: 8px 0; border-bottom: 1px solid #f3f4f6;"><strong>Installation Address:</strong></td>
        <td style="padding: 8px 0; border-bottom: 1px solid #f3f4f6;">{{installationAddress}}</td>
      </tr>
      <tr>
        <td style="padding: 8px 0;"><strong>Contact Mobile:</strong></td>
        <td style="padding: 8px 0;">{{customerMobile}}</td>
      </tr>
    </table>
  </div>

  <div style="text-align: center; margin: 30px 0;">
    <p style="color: #6b7280; margin-bottom: 15px;">Need help or have questions?</p>
    <p style="margin: 5px 0;"><strong>Email:</strong> <a href="mailto:info@lovemedia.foundation" style="color: #3b82f6;">info@lovemedia.foundation</a></p>
    <p style="margin: 5px 0;"><strong>WhatsApp:</strong> Join our community groups for support</p>
  </div>

  <div style="text-align: center; padding: 20px; background: #f9fafb; border-radius: 8px; margin-top: 30px;">
    <p style="color: #6b7280; margin: 0; font-size: 14px;">Thank you for supporting LoveMedia Foundation and choosing premium data solutions!</p>
  </div>
</div>`),
  
  // Available Variables Documentation
  availableVariables: text("available_variables").default("orderNumber,orderDate,customerName,customerEmail,customerMobile,customerIdNumber,packageName,basePrice,premiumPrice,totalPrice,installationAddress,deliveryAddress,donationAmount"),
  
  // Metadata
  createdBy: integer("created_by").references(() => users.id),
  createdAt: timestamp("created_at").defaultNow(),
  updatedAt: timestamp("updated_at").defaultNow(),
});

// Define TypeScript types for the page sections
export type PageSection = {
  id: string;
  type: 'hero' | 'community' | 'about' | 'features' | 'contact' | 'business' | 'custom';
  title: string;
  content: any;
  styles: {
    background?: string;
    textColor?: string;
    padding?: string;
    margin?: string;
    customCSS?: string;
  };
  isVisible: boolean;
  order: number;
};

// Relations
export const usersRelations = relations(users, ({ many }) => ({
  businesses: many(businesses),
  content: many(content),
}));

export const businessesRelations = relations(businesses, ({ one }) => ({
  createdBy: one(users, {
    fields: [businesses.createdBy],
    references: [users.id],
  }),
}));

export const contentRelations = relations(content, ({ one }) => ({
  author: one(users, {
    fields: [content.authorId],
    references: [users.id],
  }),
}));

// Insert schemas
export const insertUserSchema = createInsertSchema(users, {
  email: z.string().email("Please enter a valid email address"),
  mobile: z.string().min(10, "Please enter a valid mobile number"),
  firstName: z.string().min(2, "First name must be at least 2 characters"),
  lastName: z.string().min(2, "Last name must be at least 2 characters"),
  province: z.string().min(1, "Please select a province"),
}).omit({ id: true, createdAt: true, lastLogin: true, isActive: true, role: true });

export const insertBusinessSchema = createInsertSchema(businesses).omit({
  id: true,
  createdAt: true,
  updatedAt: true,
  createdBy: true,
  isVerified: true,
  rating: true,
  reviewCount: true,
});

export const insertContentSchema = createInsertSchema(content).omit({
  id: true,
  createdAt: true,
  updatedAt: true,
  authorId: true,
});

export const insertContactSchema = createInsertSchema(contactSubmissions).omit({
  id: true,
  createdAt: true,
  status: true,
});

export const insertProductSchema = createInsertSchema(products).omit({
  id: true,
  createdAt: true,
  updatedAt: true,
  finalPrice: true,
});

export const insertEmailNewsletterSchema = createInsertSchema(emailNewsletters, {
  email: z.string().email("Please enter a valid email address"),
}).omit({
  id: true,
  createdAt: true,
  unsubscribedAt: true,
  isActive: true,
});

export const insertSmtpConfigSchema = createInsertSchema(smtpConfig).omit({
  id: true,
  createdAt: true,
  updatedAt: true,
});

export const insertEmailCampaignSchema = createInsertSchema(emailCampaigns).omit({
  id: true,
  createdAt: true,
  sentAt: true,
  sentBy: true,
  sentCount: true,
  failedCount: true,
});

export const insertPageLayoutSchema = createInsertSchema(pageLayouts).omit({
  id: true,
  createdAt: true,
  updatedAt: true,
  createdBy: true,
});

export const insertHeaderTagSchema = createInsertSchema(headerTags).omit({
  id: true,
  createdAt: true,
  updatedAt: true,
  createdBy: true,
});

export const insertSeoSettingsSchema = createInsertSchema(seoSettings).omit({
  id: true,
  createdAt: true,
  updatedAt: true,
  createdBy: true,
});

export const insertBusinessListingPaymentSchema = createInsertSchema(businessListingPayments).omit({
  id: true,
  createdAt: true,
  completedAt: true,
});

export const insertBusinessCategorySchema = createInsertSchema(businessCategories).omit({
  id: true,
  createdAt: true,
});

export const insertDataDealsOrderSchema = createInsertSchema(dataDealsOrders, {
  email: z.string().email("Valid email address is required"),
  mobile: z.string().min(10, "Valid mobile number is required"),
  firstName: z.string().min(1, "First name is required"),
  surname: z.string().min(1, "Surname is required"),
  idNumber: z.string().min(13, "Valid ID/Passport number is required"),
  installationAddress: z.string().min(10, "Full installation address is required"),
  deliveryAddress: z.string().min(10, "Full delivery address is required"),
  accountNumber: z.string().min(8, "Bank account number is required"),
  branchCode: z.string().min(6, "Branch code is required"),
  mandateSignature: z.string().min(1, "Electronic signature is required"),
}).omit({
  id: true,
  createdAt: true,
  processedAt: true,
  status: true,
  orderNumber: true,
  sentToOpenWeb: true,
});

export const insertDataDealsEmailSettingsSchema = createInsertSchema(dataDealsEmailSettings, {
  toEmail: z.string().email("Please enter a valid email address"),
  fromEmail: z.string().email("Please enter a valid from email address"),
  ccEmail: z.string().email("Please enter a valid CC email address").optional().or(z.literal("")),
  subject: z.string().min(5, "Subject must be at least 5 characters"),
  emailTemplate: z.string().min(50, "Email template must be at least 50 characters"),
}).omit({
  id: true,
  createdAt: true,
  updatedAt: true,
  createdBy: true,
  availableVariables: true,
});

export const insertDataDealsCustomerEmailSettingsSchema = createInsertSchema(dataDealsCustomerEmailSettings, {
  fromEmail: z.string().email("Please enter a valid from email address"),
  subject: z.string().min(5, "Subject must be at least 5 characters"),
  emailTemplate: z.string().min(50, "Email template must be at least 50 characters"),
}).omit({
  id: true,
  createdAt: true,
  updatedAt: true,
  createdBy: true,
  availableVariables: true,
});

// Registration schema with Terms of Service agreement and enhanced validation
export const registrationSchema = insertUserSchema.extend({
  confirmPassword: z.string(),
  agreeToTerms: z.boolean().refine(val => val === true, {
    message: "You must agree to the Terms of Service"
  })
}).refine(data => data.password === data.confirmPassword, {
  message: "Passwords do not match",
  path: ["confirmPassword"]
});

// South African provinces
export const southAfricanProvinces = [
  "Eastern Cape",
  "Free State", 
  "Gauteng",
  "KwaZulu-Natal",
  "Limpopo",
  "Mpumalanga",
  "Northern Cape",
  "North West",
  "Western Cape"
] as const;

// Types
export type User = typeof users.$inferSelect;
export type InsertUser = z.infer<typeof insertUserSchema>;
export type Business = typeof businesses.$inferSelect;
export type InsertBusiness = z.infer<typeof insertBusinessSchema>;
export type Content = typeof content.$inferSelect;
export type InsertContent = z.infer<typeof insertContentSchema>;
export type WhatsAppGroup = typeof whatsappGroups.$inferSelect;
export type ContactSubmission = typeof contactSubmissions.$inferSelect;
export type InsertContact = z.infer<typeof insertContactSchema>;
export type Product = typeof products.$inferSelect;
export type InsertProduct = z.infer<typeof insertProductSchema>;
export type HeaderTag = typeof headerTags.$inferSelect;
export type InsertHeaderTag = z.infer<typeof insertHeaderTagSchema>;
export type SeoSettings = typeof seoSettings.$inferSelect;
export type InsertSeoSettings = z.infer<typeof insertSeoSettingsSchema>;
export type EmailNewsletter = typeof emailNewsletters.$inferSelect;
export type InsertEmailNewsletter = z.infer<typeof insertEmailNewsletterSchema>;
export type SmtpConfig = typeof smtpConfig.$inferSelect;
export type InsertSmtpConfig = z.infer<typeof insertSmtpConfigSchema>;
export type EmailCampaign = typeof emailCampaigns.$inferSelect;
export type InsertEmailCampaign = z.infer<typeof insertEmailCampaignSchema>;
export type PageLayout = typeof pageLayouts.$inferSelect;
export type InsertPageLayout = z.infer<typeof insertPageLayoutSchema>;
export type RegistrationData = z.infer<typeof registrationSchema>;
export type Province = typeof southAfricanProvinces[number];

// System settings table
export const systemSettings = pgTable("system_settings", {
  id: serial("id").primaryKey(),
  organization: text("organization").notNull().default("LoveMedia Foundation NPC"),
  location: text("location").notNull().default("Ballito, North Coast, KwaZulu-Natal"),
  adminFee: text("admin_fee").notNull().default("5% on wholesale prices"),
  businessModel: text("business_model").notNull().default("Non-profit with transparent pricing"),
  contactEmail: text("contact_email").default("info@lovemedia.foundation"),
  contactPhone: text("contact_phone"),
  website: text("website").default("https://lovemedia.foundation"),
  updatedAt: timestamp("updated_at").defaultNow().notNull(),
  updatedBy: integer("updated_by").references(() => users.id),
});

// System settings schemas
export const insertSystemSettingsSchema = createInsertSchema(systemSettings).omit({
  id: true,
  updatedAt: true,
});

export type InsertSystemSettings = z.infer<typeof insertSystemSettingsSchema>;
export type SystemSettings = typeof systemSettings.$inferSelect;
export type DataDealsOrder = typeof dataDealsOrders.$inferSelect;
export type InsertDataDealsOrder = z.infer<typeof insertDataDealsOrderSchema>;
export type DataDealsEmailSettings = typeof dataDealsEmailSettings.$inferSelect;
export type InsertDataDealsEmailSettings = z.infer<typeof insertDataDealsEmailSettingsSchema>;
export type BusinessListingPayment = typeof businessListingPayments.$inferSelect;
export type InsertBusinessListingPayment = z.infer<typeof insertBusinessListingPaymentSchema>;
export type BusinessCategory = typeof businessCategories.$inferSelect;
export type InsertBusinessCategory = z.infer<typeof insertBusinessCategorySchema>;
