import { useQuery } from "@tanstack/react-query";
import { Card, CardContent } from "@/components/ui/card";
import { Badge } from "@/components/ui/badge";
import { AlertTriangle, CheckCircle, Info, AlertCircle, Bell } from "lucide-react";
import type { StaffNotification } from "@shared/schema";

export default function StaffNotifications() {

  const { data: notifications, isLoading } = useQuery<StaffNotification[]>({
    queryKey: ["/api/staff-notifications"],
  });

  const getTypeIcon = (type: string) => {
    switch (type) {
      case "success":
        return CheckCircle;
      case "warning":
        return AlertTriangle;
      case "error":
        return AlertCircle;
      default:
        return Info;
    }
  };

  const getTypeColor = (type: string) => {
    switch (type) {
      case "success":
        return "text-green-600";
      case "warning":
        return "text-yellow-600";
      case "error":
        return "text-red-600";
      default:
        return "text-blue-600";
    }
  };

  const getTypeBgColor = (type: string) => {
    switch (type) {
      case "success":
        return "bg-green-50";
      case "warning":
        return "bg-yellow-50";
      case "error":
        return "bg-red-50";
      default:
        return "bg-blue-50";
    }
  };

  const getPriorityBadge = (priority: string) => {
    switch (priority) {
      case "high":
        return "bg-red-100 text-red-800";
      case "medium":
        return "bg-yellow-100 text-yellow-800";
      case "low":
        return "bg-green-100 text-green-800";
      default:
        return "bg-gray-100 text-gray-800";
    }
  };

  if (isLoading) {
    return (
      <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6 mb-8">
        {[...Array(3)].map((_, i) => (
          <Card key={i} className="animate-pulse">
            <CardContent className="p-6">
              <div className="h-4 bg-gray-200 rounded w-3/4 mb-2"></div>
              <div className="h-3 bg-gray-200 rounded w-1/2"></div>
            </CardContent>
          </Card>
        ))}
      </div>
    );
  }

  if (!notifications || notifications.length === 0) {
    return (
      <div className="grid grid-cols-1 gap-6 mb-8">
        <Card className="border-dashed border-gray-300">
          <CardContent className="p-8 text-center">
            <Bell className="w-12 h-12 text-gray-400 mx-auto mb-4" />
            <h3 className="text-lg font-medium text-gray-900 mb-2">No Staff Notifications</h3>
            <p className="text-gray-600">There are currently no active staff notifications to display.</p>
          </CardContent>
        </Card>
      </div>
    );
  }

  return (
    <div className="space-y-4 mb-8">
      <div className="flex items-center gap-2 mb-4">
        <Bell className="w-5 h-5 text-blue-600" />
        <h2 className="text-xl font-semibold text-gray-900">Staff Notifications</h2>
      </div>
      
      <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-4">
        {notifications.map((notification) => {
          const Icon = getTypeIcon(notification.type);
          return (
            <Card key={notification.id} className={`shadow-lg border-l-4 hover:shadow-xl transition-all duration-300 ${
              notification.type === 'error' ? 'border-l-red-500 bg-gradient-to-r from-red-50 to-white' :
              notification.type === 'warning' ? 'border-l-yellow-500 bg-gradient-to-r from-yellow-50 to-white' :
              notification.type === 'success' ? 'border-l-green-500 bg-gradient-to-r from-green-50 to-white' :
              'border-l-blue-500 bg-gradient-to-r from-blue-50 to-white'
            }`}>
              <CardContent className="p-6">
                <div className="flex items-start gap-4">
                  <div className={`w-12 h-12 ${getTypeBgColor(notification.type)} rounded-xl flex items-center justify-center flex-shrink-0 shadow-md`}>
                    <Icon className={`${getTypeColor(notification.type)} w-6 h-6`} />
                  </div>
                  <div className="flex-1 min-w-0">
                    <div className="flex items-center gap-3 mb-3">
                      <h3 className="font-bold text-gray-900 text-lg">{notification.title}</h3>
                      <Badge className={`text-sm font-semibold ${getPriorityBadge(notification.priority)}`}>
                        {notification.priority}
                      </Badge>
                    </div>
                    <p className="text-gray-700 leading-relaxed whitespace-pre-wrap mb-4">{notification.message}</p>
                    <div className="text-sm text-gray-500 font-medium">
                      {new Date(notification.createdAt).toLocaleDateString()} at {new Date(notification.createdAt).toLocaleTimeString([], { hour: '2-digit', minute: '2-digit' })}
                    </div>
                  </div>
                </div>
              </CardContent>
            </Card>
          );
        })}
      </div>


    </div>
  );
}