import { useState } from "react";
import { useQuery, useMutation } from "@tanstack/react-query";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Badge } from "@/components/ui/badge";
import { Card, CardContent } from "@/components/ui/card";
import {
  AlertDialog,
  AlertDialogAction,
  AlertDialogCancel,
  AlertDialogContent,
  AlertDialogDescription,
  AlertDialogFooter,
  AlertDialogHeader,
  AlertDialogTitle,
} from "@/components/ui/alert-dialog";
import { Plus, Search, ChevronLeft, ChevronRight, Edit, Trash2, Wifi, BarChart3, Database, Settings, TestTube, TrendingUp, RefreshCw, Calendar } from "lucide-react";
import SimEditModal from "@/components/sim-edit-modal";
import ClientEditModal from "@/components/client-edit-modal";
import ClientViewModal from "@/components/client-view-modal";
import AdminToolsModal from "@/components/admin-tools-modal";
import Top15Modal from "@/components/top15-modal";
import RoleGuard from "@/components/role-guard";
import { useToast } from "@/hooks/use-toast";
import { apiRequest } from "@/lib/queryClient";
import { queryClient } from "@/lib/queryClient";
import type { Client } from "@shared/schema";

export default function MTNFixed() {
  const { toast } = useToast();
  const [searchTerm, setSearchTerm] = useState("");
  const [editingSim, setEditingSim] = useState<Client | null>(null);
  const [isSimEditModalOpen, setIsSimEditModalOpen] = useState(false);
  const [isAddClientModalOpen, setIsAddClientModalOpen] = useState(false);
  const [isClientViewModalOpen, setIsClientViewModalOpen] = useState(false);
  const [isAdminToolsModalOpen, setIsAdminToolsModalOpen] = useState(false);
  const [selectedClient, setSelectedClient] = useState<Client | null>(null);
  const [currentPage, setCurrentPage] = useState(1);
  const [clientToDelete, setClientToDelete] = useState<Client | null>(null);
  const [isDeleteDialogOpen, setIsDeleteDialogOpen] = useState(false);
  const [isTop15ModalOpen, setIsTop15ModalOpen] = useState(false);
  const [isBottom30ModalOpen, setIsBottom30ModalOpen] = useState(false);
  const clientsPerPage = 20;

  const { data: clientsData, isLoading } = useQuery({
    queryKey: ["/api/clients", "MTN Fixed", currentPage],
    queryFn: async () => {
      const offset = (currentPage - 1) * clientsPerPage;
      const params = new URLSearchParams({
        category: "MTN Fixed",
        limit: clientsPerPage.toString(),
        offset: offset.toString(),
      });
      
      const response = await fetch(`/api/clients?${params}`, {
        credentials: "include",
      });
      if (!response.ok) throw new Error("Failed to fetch MTN Fixed clients");
      return response.json();
    },
  });

  const clients = clientsData?.clients || [];
  const totalCount = clientsData?.total || 0;
  const totalPages = Math.ceil(totalCount / clientsPerPage);

  // Top 15 query
  const { data: top15ApiData, isLoading: isTop15Loading, refetch: refetchTop15 } = useQuery({
    queryKey: ["/api/admin/mtn-fixed-top15"],
    queryFn: async () => {
      const response = await fetch("/api/admin/mtn-fixed-top15", {
        credentials: "include",
      });
      if (!response.ok) {
        throw new Error("Failed to fetch top 15 data");
      }
      return response.json();
    },
    enabled: isTop15ModalOpen, // Load when modal is open
  });

  // Transform Top 15 data to match expected format
  const top15Data = top15ApiData ? {
    ...top15ApiData,
    users: top15ApiData.users.map((user: any) => ({
      username: user.accountNumber, // Map accountNumber to username for display
      name: user.clientName,
      msisdn: user.msisdn,
      totalBytes: user.totalBytes,
      connectedTimeMinutes: user.connectedTimeMinutes
    })),
    count: top15ApiData.users.length
  } : null;

  // Bottom 30 query  
  const { data: bottom30ApiData, isLoading: isBottom30Loading, refetch: refetchBottom30 } = useQuery({
    queryKey: ["/api/admin/mtn-fixed-bottom30"],
    queryFn: async () => {
      const response = await fetch("/api/admin/mtn-fixed-bottom30", {
        credentials: "include",
      });
      if (!response.ok) {
        throw new Error("Failed to fetch bottom 30 data");
      }
      return response.json();
    },
    enabled: isBottom30ModalOpen, // Load when modal is open
  });

  // Transform Bottom 30 data to match expected format
  const bottom30Data = bottom30ApiData ? {
    ...bottom30ApiData,
    users: bottom30ApiData.users.map((user: any) => ({
      username: user.accountNumber, // Map accountNumber to username for display
      name: user.clientName,
      msisdn: user.msisdn,
      totalBytes: user.totalBytes,
      connectedTimeMinutes: user.connectedTimeMinutes
    })),
    count: bottom30ApiData.users.length
  } : null;

  // Delete mutation
  const deleteMutation = useMutation({
    mutationFn: async (clientId: number) => {
      return apiRequest("DELETE", `/api/clients/${clientId}`);
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/clients"] });
      queryClient.invalidateQueries({ queryKey: ["/api/stats"] });
      queryClient.invalidateQueries({ queryKey: ["/api/recently-viewed"] });
      toast({
        title: "Success",
        description: "MTN Fixed client deleted successfully",
      });
      setIsDeleteDialogOpen(false);
      setClientToDelete(null);
    },
    onError: (error: any) => {
      toast({
        title: "Error",
        description: error.message || "Failed to delete client",
        variant: "destructive",
      });
    },
  });

  // Sync & Top 15 mutation
  const syncMutation = useMutation({
    mutationFn: async () => {
      console.log("[MTN Fixed] Starting sync & top 15...");
      const response = await apiRequest("POST", "/api/admin/mtn-fixed-sync");
      console.log("[MTN Fixed] Sync result:", response);
      return response;
    },
    onSuccess: (data) => {
      console.log("[MTN Fixed] Sync completed successfully:", data);
      toast({
        title: "Success",
        description: `MTN Fixed data synced successfully${data?.processed !== undefined ? ` (${data.processed} accounts processed)` : ''}`,
      });
      // Auto-show Top 15 modal after sync
      queryClient.invalidateQueries({ queryKey: ["/api/admin/mtn-fixed-top15"] });
      setIsTop15ModalOpen(true);
    },
    onError: (error: any) => {
      console.error("[MTN Fixed] Sync error:", error);
      toast({
        title: "Error",
        description: error.message || "Failed to sync data",
        variant: "destructive",
      });
    },
  });

  // Sync 3 Months mutation
  const sync3MonthsMutation = useMutation({
    mutationFn: async () => {
      console.log("[MTN Fixed] Starting 3 months sync...");
      const response = await apiRequest("POST", "/api/admin/mtn-fixed-sync-3months");
      console.log("[MTN Fixed] 3 months sync result:", response);
      return response;
    },
    onSuccess: (data) => {
      console.log("[MTN Fixed] 3 months sync completed successfully:", data);
      toast({
        title: "Success",
        description: `3 months MTN Fixed data synced successfully${data?.processed !== undefined ? ` (${data.processed} accounts processed)` : ''}`,
      });
      queryClient.invalidateQueries({ queryKey: ["/api/admin/mtn-fixed-top15"] });
    },
    onError: (error: any) => {
      console.error("[MTN Fixed] 3 months sync error:", error);
      toast({
        title: "Error",
        description: error.message || "Failed to sync 3 months data",
        variant: "destructive",
      });
    },
  });

  // Sync Daily Usage mutation
  const syncDailyUsageMutation = useMutation({
    mutationFn: async () => {
      console.log("[MTN Fixed] Starting daily usage sync...");
      const response = await apiRequest("POST", "/api/admin/mtn-fixed-sync-daily");
      console.log("[MTN Fixed] Daily sync result:", response);
      return response;
    },
    onSuccess: (data) => {
      console.log("[MTN Fixed] Daily sync completed successfully:", data);
      toast({
        title: "Success",
        description: `MTN Fixed daily usage data synced successfully${data?.processed !== undefined ? ` (${data.processed} accounts processed)` : ''}`,
      });
    },
    onError: (error: any) => {
      console.error("[MTN Fixed] Daily sync error:", error);
      toast({
        title: "Error",
        description: error.message || "Failed to sync daily usage data",
        variant: "destructive",
      });
    },
  });

  // Test API mutation
  const testApiMutation = useMutation({
    mutationFn: async () => {
      console.log("[MTN Fixed] Starting API test...");
      const response = await apiRequest("POST", "/api/admin/mtn-fixed-test");
      console.log("[MTN Fixed] API test result:", response);
      return response;
    },
    onSuccess: (data) => {
      console.log("[MTN Fixed] API test completed successfully:", data);
      toast({
        title: "API Test Success",
        description: data?.message || "MTN Fixed API test completed successfully",
      });
    },
    onError: (error: any) => {
      console.error("[MTN Fixed] API test error:", error);
      toast({
        title: "API Test Error",
        description: error.message || "Failed to test MTN Fixed API",
        variant: "destructive",
      });
    },
  });

  const handleEditSim = (sim: Client) => {
    setEditingSim(sim);
    setIsSimEditModalOpen(true);
  };

  const handleDeleteClient = (client: Client) => {
    setClientToDelete(client);
    setIsDeleteDialogOpen(true);
  };

  const confirmDelete = () => {
    if (clientToDelete) {
      deleteMutation.mutate(clientToDelete.id);
    }
  };

  const handleViewUsageStats = (client: Client) => {
    setSelectedClient(client);
    setIsClientViewModalOpen(true);
  };

  const handleAddClient = () => {
    setIsAddClientModalOpen(true);
  };

  const handleCloseAddClientModal = () => {
    setIsAddClientModalOpen(false);
  };

  // Handler functions
  const handleTestAPI = () => {
    testApiMutation.mutate();
  };

  const handleShowTop15 = () => {
    queryClient.invalidateQueries({ queryKey: ["/api/admin/mtn-fixed-top15"] });
    setIsTop15ModalOpen(true);
  };

  const handleShowBottom30 = () => {
    setIsBottom30ModalOpen(true);
  };

  const handleSyncAndShowTop15 = () => {
    syncMutation.mutate();
  };

  const getStatusColor = (status: string) => {
    switch (status.toLowerCase()) {
      case "active":
        return "bg-green-600 text-white border-0";
      case "suspended":
        return "bg-red-600 text-white border-0";
      case "cancelled":
        return "bg-gray-700 text-white border-0";
      case "debit order unpaid":
        return "bg-orange-600 text-white border-0";
      case "pending":
        return "bg-yellow-600 text-white border-0";
      default:
        return "bg-gray-600 text-white border-0";
    }
  };

  const filteredClients = clients.filter((client: Client) =>
    client.name?.toLowerCase().includes(searchTerm.toLowerCase()) ||
    client.email?.toLowerCase().includes(searchTerm.toLowerCase()) ||
    client.accountNumber?.toLowerCase().includes(searchTerm.toLowerCase())
  );

  return (
    <>
      {/* Top Bar */}
      <div className="bg-white shadow-sm border-b border-gray-200 p-6">
        <div className="flex flex-col lg:flex-row lg:items-center lg:justify-between gap-6">
          <div className="flex items-center space-x-4">
            <div className="w-12 h-12 bg-orange-500 rounded-lg flex items-center justify-center">
              <Wifi className="text-white" size={24} />
            </div>
            <div>
              <h1 className="text-xl lg:text-2xl font-bold text-gray-900">MTN Fixed</h1>
              <p className="text-sm lg:text-base text-gray-600">Manage MTN Fixed client SIMs</p>
            </div>
          </div>
          
          {/* Search and Actions Section */}
          <div className="flex flex-col sm:flex-row lg:flex-row items-stretch sm:items-center gap-3 lg:ml-auto">
            {/* Search Input */}
            <div className="relative order-1 sm:order-1">
              <Search className="absolute left-3 top-1/2 transform -translate-y-1/2 text-gray-400" size={20} />
              <Input
                placeholder="Search clients..."
                value={searchTerm}
                onChange={(e) => setSearchTerm(e.target.value)}
                className="pl-10 w-full sm:w-64 lg:w-80"
              />
            </div>
            
            {/* Action Buttons Container */}
            <div className="flex flex-wrap items-center gap-2 order-2 sm:order-2">
              <RoleGuard requiredRole="admin">
                <Button 
                  variant="outline"
                  size="sm"
                  onClick={handleTestAPI}
                  disabled={testApiMutation.isPending}
                  className="border-blue-200 text-blue-700 hover:bg-blue-50 text-xs lg:text-sm"
                >
                  <TestTube className={`w-3 h-3 lg:w-4 lg:h-4 mr-1 lg:mr-2 ${testApiMutation.isPending ? 'animate-bounce' : ''}`} />
                  <span className="hidden sm:inline">{testApiMutation.isPending ? 'Testing...' : 'Test API'}</span>
                  <span className="sm:hidden">Test</span>
                </Button>
              </RoleGuard>
              
              <RoleGuard requiredRole="admin">
                <Button 
                  variant="outline"
                  size="sm"
                  onClick={handleShowTop15}
                  className="border-amber-200 text-amber-700 hover:bg-amber-50 text-xs lg:text-sm"
                >
                  <TrendingUp className="w-3 h-3 lg:w-4 lg:h-4 mr-1 lg:mr-2" />
                  <span className="hidden sm:inline">Top 15 Users</span>
                  <span className="sm:hidden">Top 15</span>
                </Button>
              </RoleGuard>

              <RoleGuard requiredRole="admin">
                <Button 
                  variant="outline"
                  size="sm"
                  onClick={handleShowBottom30}
                  className="border-red-200 text-red-700 hover:bg-red-50 text-xs lg:text-sm"
                >
                  <TrendingUp className="w-3 h-3 lg:w-4 lg:h-4 mr-1 lg:mr-2 rotate-180" />
                  <span className="hidden sm:inline">Bottom 30 Users</span>
                  <span className="sm:hidden">Bottom 30</span>
                </Button>
              </RoleGuard>
              
              <RoleGuard requiredRole="admin">
                <Button 
                  variant="outline"
                  size="sm"
                  onClick={handleSyncAndShowTop15}
                  disabled={syncMutation.isPending}
                  className="border-green-200 text-green-700 hover:bg-green-50 text-xs lg:text-sm"
                >
                  <RefreshCw className={`w-3 h-3 lg:w-4 lg:h-4 mr-1 lg:mr-2 ${syncMutation.isPending ? 'animate-spin' : ''}`} />
                  <span className="hidden lg:inline">{syncMutation.isPending ? 'Syncing...' : 'Sync & Top 15'}</span>
                  <span className="lg:hidden">{syncMutation.isPending ? 'Sync...' : 'Sync'}</span>
                </Button>
              </RoleGuard>
              
              <RoleGuard requiredRole="admin">
                <Button 
                  variant="outline"
                  size="sm"
                  onClick={() => sync3MonthsMutation.mutate()}
                  disabled={sync3MonthsMutation.isPending}
                  className="border-blue-200 text-blue-700 hover:bg-blue-50 text-xs lg:text-sm"
                >
                  <Calendar className={`w-3 h-3 lg:w-4 lg:h-4 mr-1 lg:mr-2 ${sync3MonthsMutation.isPending ? 'animate-spin' : ''}`} />
                  <span className="hidden lg:inline">{sync3MonthsMutation.isPending ? 'Syncing 3 Months...' : 'Sync 3 Months'}</span>
                  <span className="lg:hidden">{sync3MonthsMutation.isPending ? '3M...' : '3M'}</span>
                </Button>
              </RoleGuard>
              
              <RoleGuard requiredRole="admin">
                <Button 
                  variant="outline"
                  size="sm"
                  onClick={() => syncDailyUsageMutation.mutate()}
                  disabled={syncDailyUsageMutation.isPending}
                  className="border-purple-200 text-purple-700 hover:bg-purple-50 text-xs lg:text-sm"
                >
                  <BarChart3 className={`w-3 h-3 lg:w-4 lg:h-4 mr-1 lg:mr-2 ${syncDailyUsageMutation.isPending ? 'animate-pulse' : ''}`} />
                  <span className="hidden lg:inline">{syncDailyUsageMutation.isPending ? 'Syncing Daily Usage...' : 'Sync Daily Usage'}</span>
                  <span className="lg:hidden">{syncDailyUsageMutation.isPending ? 'Daily...' : 'Daily'}</span>
                </Button>
              </RoleGuard>

              <RoleGuard requiredRole="admin">
                <Button 
                  variant="outline"
                  size="sm"
                  onClick={() => setIsAdminToolsModalOpen(true)}
                  className="border-gray-200 text-gray-700 hover:bg-gray-50 text-xs lg:text-sm"
                >
                  <Settings className="w-3 h-3 lg:w-4 lg:h-4 mr-1 lg:mr-2" />
                  <span className="hidden sm:inline">Admin Tools</span>
                  <span className="sm:hidden">Admin</span>
                </Button>
              </RoleGuard>

              <RoleGuard requiredRole="admin">
                <Button 
                  size="sm"
                  onClick={handleAddClient}
                  className="text-xs lg:text-sm"
                >
                  <Plus className="w-3 h-3 lg:w-4 lg:h-4 mr-1 lg:mr-2" />
                  <span className="hidden sm:inline">Add Client</span>
                  <span className="sm:hidden">Add</span>
                </Button>
              </RoleGuard>
            </div>
          </div>
        </div>
      </div>

      {/* Content Area */}
      <div className="flex-1 overflow-auto p-6">
        {isLoading ? (
          <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 xl:grid-cols-4 gap-4">
            {[1, 2, 3, 4, 5, 6, 7, 8].map((i) => (
              <div key={i} className="bg-white rounded-xl shadow-sm border border-gray-200 p-4 animate-pulse">
                <div className="h-32 bg-gray-200 rounded"></div>
              </div>
            ))}
          </div>
        ) : filteredClients && filteredClients.length > 0 ? (
          <>
            <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 xl:grid-cols-4 gap-4">
              {filteredClients.map((client: Client) => (
                <Card key={client.id} className={`shadow-sm border hover:shadow-md transition-shadow ${client.isReseller ? 'bg-blue-50 border-blue-300 ring-2 ring-blue-200' : 'border-gray-200'}`}>
                  <CardContent className="p-4">
                    <div className="flex items-center justify-between mb-3">
                      <div className="flex items-center space-x-2">
                        <Wifi className={`${client.isReseller ? 'text-blue-600' : 'text-blue-500'}`} size={16} />
                        <Badge className="bg-blue-100 text-blue-800 hover:bg-blue-100">
                          MTN Fixed
                        </Badge>
                        {client.isReseller && (
                          <Badge className="bg-blue-600 text-white text-xs px-2 py-1 hover:bg-blue-600">
                            Reseller
                          </Badge>
                        )}
                      </div>
                      <div className="flex items-center space-x-1">
                        <Button
                          variant="ghost"
                          size="sm"
                          onClick={() => handleViewUsageStats(client)}
                          className="h-8 w-8 p-0 text-blue-600 hover:text-blue-700"
                          title="View Usage Statistics"
                        >
                          <BarChart3 className="h-4 w-4" />
                        </Button>
                        <RoleGuard requiredRole="admin">
                          <Button
                            variant="ghost"
                            size="sm"
                            onClick={() => handleEditSim(client)}
                            className="h-8 w-8 p-0"
                          >
                            <Edit className="h-4 w-4" />
                          </Button>
                        </RoleGuard>
                        <RoleGuard requiredRole="admin">
                          <Button
                            variant="ghost"
                            size="sm"
                            onClick={() => handleDeleteClient(client)}
                            className="h-8 w-8 p-0 text-red-500 hover:text-red-700"
                            disabled={deleteMutation.isPending}
                          >
                            <Trash2 className="h-4 w-4" />
                          </Button>
                        </RoleGuard>
                      </div>
                    </div>

                    <div className="space-y-2">
                      <div>
                        <h4 className="font-semibold text-gray-900 mb-1">{client.name}</h4>
                        <p className="text-sm text-gray-600">{client.email || 'No email'}</p>
                      </div>

                      <div className="space-y-1">
                        <div className="flex justify-between">
                          <span className="text-xs text-gray-500">Account:</span>
                          <span className="text-xs font-medium">{client.accountNumber || 'N/A'}</span>
                        </div>
                        <div className="flex justify-between">
                          <span className="text-xs text-gray-500">Status:</span>
                          <Badge className={`${getStatusColor(client.status || '')} text-xs`}>
                            {client.status || 'Unknown'}
                          </Badge>
                        </div>
                        {client.status2 && (
                          <div className="flex justify-between">
                            <span className="text-xs text-gray-500">Status 2:</span>
                            <span className="text-xs font-medium">{client.status2}</span>
                          </div>
                        )}
                      </div>

                      {client.serviceDetails && (
                        <div className="mt-2 p-2 bg-gray-50 rounded text-xs text-gray-600 line-clamp-3">
                          {client.serviceDetails}
                        </div>
                      )}
                    </div>
                  </CardContent>
                </Card>
              ))}
            </div>

            {/* Pagination Controls */}
            <div className="mt-6 flex items-center justify-between">
              <div className="text-sm text-gray-700">
                Page {currentPage} - {filteredClients.length} clients shown
              </div>
              <div className="flex items-center space-x-2">
                <Button
                  variant="outline"
                  size="sm"
                  onClick={() => setCurrentPage(prev => Math.max(1, prev - 1))}
                  disabled={currentPage === 1}
                >
                  <ChevronLeft className="h-4 w-4" />
                  Previous
                </Button>
                <Button
                  variant="outline"
                  size="sm"
                  onClick={() => setCurrentPage(prev => prev + 1)}
                  disabled={!clients || clients.length < clientsPerPage}
                >
                  Next
                  <ChevronRight className="h-4 w-4" />
                </Button>
              </div>
            </div>
          </>
        ) : (
          <div className="bg-white rounded-xl shadow-sm border border-gray-200 p-12 text-center">
            <Wifi className="mx-auto h-12 w-12 text-gray-400 mb-4" />
            <p className="text-gray-500">
              {searchTerm ? "No MTN Fixed clients found matching your search" : "No MTN Fixed clients found"}
            </p>
          </div>
        )}
      </div>

      <SimEditModal
        sim={editingSim}
        simType="primary"
        isOpen={isSimEditModalOpen}
        onClose={() => setIsSimEditModalOpen(false)}
      />
      
      <ClientEditModal
        client={null}
        isOpen={isAddClientModalOpen}
        onClose={handleCloseAddClientModal}
        defaultCategory="MTN Fixed"
      />

      <ClientViewModal
        client={selectedClient}
        isOpen={isClientViewModalOpen}
        onClose={() => setIsClientViewModalOpen(false)}
      />

      <AdminToolsModal
        isOpen={isAdminToolsModalOpen}
        onClose={() => setIsAdminToolsModalOpen(false)}
      />

      {/* Top 15 Modal */}
      <Top15Modal
        isOpen={isTop15ModalOpen}
        onClose={() => setIsTop15ModalOpen(false)}
        data={top15Data}
        isLoading={isTop15Loading}
        serviceType="MTN Fixed"
      />

      {/* Bottom 30 Modal */}
      <Top15Modal
        isOpen={isBottom30ModalOpen}
        onClose={() => setIsBottom30ModalOpen(false)}
        data={bottom30Data}
        isLoading={isBottom30Loading}
        isBottomModal={true}
        serviceType="MTN Fixed"
      />

      {/* Delete Confirmation Dialog */}
      <AlertDialog open={isDeleteDialogOpen} onOpenChange={setIsDeleteDialogOpen}>
        <AlertDialogContent>
          <AlertDialogHeader>
            <AlertDialogTitle>Delete MTN Fixed Client</AlertDialogTitle>
            <AlertDialogDescription>
              Are you sure you want to permanently delete the client "{clientToDelete?.name}"? 
              This action will completely remove all data associated with this account, including:
              <br /><br />
              • Client information and contact details
              <br />
              • All SIM swap history and records
              <br />
              • MTN Fixed usage statistics and data
              <br />
              • Recently viewed tracking data
              <br /><br />
              <strong>This action cannot be undone.</strong>
            </AlertDialogDescription>
          </AlertDialogHeader>
          <AlertDialogFooter>
            <AlertDialogCancel 
              onClick={() => {
                setIsDeleteDialogOpen(false);
                setClientToDelete(null);
              }}
            >
              Cancel
            </AlertDialogCancel>
            <AlertDialogAction
              onClick={confirmDelete}
              disabled={deleteMutation.isPending}
              className="bg-red-600 hover:bg-red-700 text-white"
            >
              {deleteMutation.isPending ? "Deleting..." : "Delete Client"}
            </AlertDialogAction>
          </AlertDialogFooter>
        </AlertDialogContent>
      </AlertDialog>
    </>
  );
}