import { useQuery } from "@tanstack/react-query";
import { Button } from "@/components/ui/button";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Download, FileText, BarChart3, TrendingUp } from "lucide-react";
import { useToast } from "@/hooks/use-toast";

export default function Reports() {
  const { toast } = useToast();

  const { data: stats } = useQuery({
    queryKey: ["/api/stats"],
  });

  const { data: clientsData } = useQuery({
    queryKey: ["/api/clients"],
  });

  const clients = clientsData?.clients || [];

  const handleExport = (type: string) => {
    toast({
      title: "Export Started",
      description: `${type} export functionality will be implemented`,
    });
  };

  const reportTypes = [
    {
      title: "Client Summary Report",
      description: "Overview of all clients with basic information and SIM counts",
      icon: FileText,
      color: "bg-blue-50 text-blue-600",
    },
    {
      title: "Telkom Detailed Report",
      description: "Comprehensive report of Telkom clients with Primary/Secondary SIM relationships",
      icon: BarChart3,
      color: "bg-green-50 text-green-600",
    },
    {
      title: "SIM Inventory Report",
      description: "Complete inventory of all SIMs with status and data plan information",
      icon: TrendingUp,
      color: "bg-purple-50 text-purple-600",
    },
    {
      title: "Usage Analytics Report",
      description: "Data usage analytics and trends across all clients and categories",
      icon: BarChart3,
      color: "bg-orange-50 text-orange-600",
    },
  ];

  return (
    <>
      {/* Top Bar */}
      <div className="bg-white border-b border-gray-200 px-6 py-4">
        <div className="flex items-center justify-between">
          <div>
            <h2 className="text-2xl font-bold text-gray-900">Reports & Analytics</h2>
            <p className="text-gray-600 mt-1">Generate and export comprehensive reports</p>
          </div>
        </div>
      </div>

      {/* Content Area */}
      <div className="flex-1 overflow-auto p-6">
        {/* Quick Stats Overview */}
        <div className="grid grid-cols-1 md:grid-cols-3 gap-6 mb-8">
          <Card>
            <CardContent className="p-6">
              <div className="text-center">
                <p className="text-2xl font-bold text-gray-900">{stats?.totalClients || 0}</p>
                <p className="text-sm text-gray-600">Total Clients</p>
              </div>
            </CardContent>
          </Card>
          <Card>
            <CardContent className="p-6">
              <div className="text-center">
                <p className="text-2xl font-bold text-gray-900">{stats?.telkomClients || 0}</p>
                <p className="text-sm text-gray-600">Telkom Clients</p>
              </div>
            </CardContent>
          </Card>
          <Card>
            <CardContent className="p-6">
              <div className="text-center">
                <p className="text-2xl font-bold text-gray-900">{stats?.activeSims || 0}</p>
                <p className="text-sm text-gray-600">Active SIMs</p>
              </div>
            </CardContent>
          </Card>
        </div>

        {/* Development Message */}
        <div className="flex items-center justify-center min-h-[400px]">
          <Card className="max-w-lg mx-auto">
            <CardContent className="p-8 text-center">
              <div className="w-16 h-16 mx-auto mb-4 rounded-full bg-blue-100 flex items-center justify-center">
                <FileText className="w-8 h-8 text-blue-600" />
              </div>
              <h3 className="text-xl font-semibold text-gray-900 mb-3">Reports Coming Soon</h3>
              <p className="text-gray-600 leading-relaxed">
                Reports is a feature that Keoma is currently busy developing and should be ready soon.
              </p>
            </CardContent>
          </Card>
        </div>
      </div>
    </>
  );
}
