"""
Version management and changelog for SIM Analytics Dashboard
"""

# Current version
VERSION = "1.1.3"

# Changelog entries
CHANGELOG = {
    "1.1.3": {
        "date": "2025-08-14",
        "features": [
            "Comprehensive REST API with two-tier authentication system",
            "Admin API Keys with full CRUD access to all platform features",
            "User API Keys with read-only access to analytics and reports",
            "Bulk client and SIM card creation endpoints for external system integration",
            "Interactive API documentation with integration examples",
            "API key management interface with usage monitoring and rate limiting"
        ],
        "improvements": [
            "External system integration capabilities for automated provisioning",
            "Real-time analytics access for external dashboards and reporting systems",
            "Comprehensive API documentation with Python, JavaScript, and cURL examples",
            "Secure API key generation with SHA-256 hashing and usage tracking",
            "Rate limiting and usage monitoring for API security and performance"
        ],
        "fixes": [
            "Enhanced authentication system with proper API key validation",
            "Improved database connection handling for API operations",
            "Better error handling and response formatting for API endpoints"
        ]
    },
    "1.1.2": {
        "date": "2025-07-06",
        "features": [
            "Automatic daily usage generation system with 1-38GB configurable range",
            "Monthly usage reset functionality for automatic data lifecycle management",
            "Enhanced SIM provisioning with automatic usage data population",
            "Real-time usage tracking with automatic daily updates"
        ],
        "improvements": [
            "Streamlined data population workflow with intelligent gap detection",
            "Enhanced dashboard displaying auto-generated usage statistics",
            "Improved user experience with automated data management",
            "Optimized usage generation algorithms for realistic data patterns"
        ],
        "fixes": [
            "Resolved data consistency issues in automatic usage generation",
            "Improved database transaction handling for bulk operations",
            "Enhanced error handling for daily usage update processes"
        ]
    },
    "1.1.1": {
        "date": "2025-06-20",
        "features": [
            "Individual SIM PDF Export - Generate comprehensive protocol analysis reports for specific SIM cards",
            "Client-ready PDF reports with professional formatting and detailed usage insights",
            "Database cleanup functionality for fresh monthly starts",
            "Direct SQL integration for advanced data management operations"
        ],
        "improvements": [
            "Enhanced Individual Usage Records section with export capabilities",
            "Professional PDF layout with client information, usage summary, and protocol breakdown",
            "Clean filename generation for easy client distribution",
            "Comprehensive usage insights and recommendations in PDF reports"
        ],
        "fixes": [
            "Data type conversion issues in Data Population feature resolved",
            "Improved error handling for decimal/float arithmetic operations",
            "Enhanced database connection stability for bulk operations"
        ]
    },
    "1.1.0": {
        "date": "2025-06-20",
        "features": [
            "Intelligent Data Population - Admin-only feature that detects last day with data for each SIM",
            "Smart gap detection that adds usage data only to days without existing records",
            "Configurable usage ranges (min/max GB per day) with random distribution",
            "Automatic protocol analysis generation for newly populated data",
            "Client-specific or system-wide data population options",
            "Real-time progress tracking with success/error metrics"
        ],
        "improvements": [
            "Enhanced admin interface with additional data management capabilities",
            "Intelligent date detection to prevent data conflicts",
            "Comprehensive validation to protect existing usage records"
        ],
        "fixes": [
            "Database cursor handling improvements for better reliability",
            "Memory optimization for bulk data operations"
        ]
    },
    "1.0.0": {
        "date": "2025-06-19",
        "features": [
            "Initial release of 4G/5G SIM Card Usage Analytics Dashboard",
            "Custom authentication system with admin and staff roles",
            "Client and SIM management with FUP tracking",
            "Automatic usage generation with protocol analysis",
            "Beautiful daily usage visualizations with color coding",
            "Comprehensive PDF report generation",
            "Protocol analysis with interactive charts and export",
            "Admin-only client deletion with cascading data removal",
            "Individual usage records with detailed breakdowns",
            "PostgreSQL database integration for production deployment"
        ],
        "improvements": [],
        "fixes": []
    }
}

def get_version():
    """Get current version"""
    return VERSION

def get_changelog():
    """Get full changelog"""
    return CHANGELOG

def get_latest_changelog():
    """Get changelog for current version"""
    return CHANGELOG.get(VERSION, {})

def format_version_display(include_emoji=True):
    """Format version for display"""
    if include_emoji:
        return f"🔄 v{VERSION}"
    return f"v{VERSION}"

def get_version_badge():
    """Get HTML version badge"""
    return f"""
    <div style="
        display: inline-block;
        background: linear-gradient(45deg, #667eea, #764ba2);
        color: white;
        padding: 4px 12px;
        border-radius: 15px;
        font-size: 12px;
        font-weight: bold;
        margin-left: 10px;
    ">
        v{VERSION}
    </div>
    """

def get_dashboard_version_card():
    """Get beautiful version card for dashboard"""
    latest = get_latest_changelog()
    feature_count = len(latest.get('features', []))
    
    return f"""
    <div style="
        background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
        padding: 20px;
        border-radius: 15px;
        color: white;
        margin: 20px 0;
        box-shadow: 0 4px 15px rgba(0,0,0,0.1);
    ">
        <div style="display: flex; justify-content: space-between; align-items: center;">
            <div>
                <h3 style="margin: 0; font-size: 24px;">📊 SIM Analytics Dashboard</h3>
                <p style="margin: 5px 0 0 0; opacity: 0.9;">Version {VERSION} • Released {latest.get('date', 'Unknown')}</p>
            </div>
            <div style="text-align: right;">
                <div style="font-size: 32px; font-weight: bold;">v{VERSION}</div>
                <div style="font-size: 14px; opacity: 0.8;">{feature_count} Features</div>
            </div>
        </div>
    </div>
    """