# Reddit API Integration Framework - TubeBoost

## Overview
Complete Reddit OAuth 2.0 authentication framework integrated into TubeBoost multi-platform social media management system. This framework enables users to connect their Reddit accounts for future Reddit management features.

## Reddit Application Configuration

### Reddit App Setup
1. Visit https://www.reddit.com/prefs/apps
2. Create a new application with type "web app"
3. Configure redirect URIs for both development and production:

**Development Environment:**
- Redirect URI: `http://localhost:5000/api/reddit/callback`

**Production Environment:**
- Redirect URI: `https://tubeboost.cc/api/reddit/callback`

### Required Environment Variables
```bash
REDDIT_CLIENT_ID=your_reddit_client_id
REDDIT_CLIENT_SECRET=your_reddit_client_secret
```

## Authentication Flow

### 1. User Initiates Reddit Connection
Navigate to: `/api/reddit/auth`
- Redirects to Reddit OAuth consent page
- Requests scopes: identity, read, submit, edit

### 2. Reddit Authorization Callback
Endpoint: `/api/reddit/callback`
- Handles OAuth callback from Reddit
- Exchanges authorization code for access tokens
- Stores user connection in database

### 3. Connection Management
- **Check Status**: `GET /api/reddit/connection`
- **Remove Connection**: `DELETE /api/reddit/connection`

## Database Schema

### Reddit Connections Table
```sql
CREATE TABLE reddit_connections (
  id SERIAL PRIMARY KEY,
  user_id INTEGER NOT NULL REFERENCES users(id),
  reddit_user_id VARCHAR(255) NOT NULL,
  reddit_username VARCHAR(255) NOT NULL,
  access_token TEXT NOT NULL,
  refresh_token TEXT,
  expires_at TIMESTAMP,
  scope VARCHAR(255),
  created_at TIMESTAMP DEFAULT NOW(),
  updated_at TIMESTAMP DEFAULT NOW(),
  UNIQUE(user_id)
);
```

## API Endpoints

### Authentication Routes
- `GET /api/reddit/auth` - Initiate Reddit OAuth flow
- `GET /api/reddit/callback` - Handle OAuth callback
- `GET /api/reddit/connection` - Check connection status
- `DELETE /api/reddit/connection` - Remove Reddit connection

### Reddit API Integration
All Reddit API calls use authenticated tokens with automatic refresh handling:
- User-Agent: `TubeBoost:v1.3.1 (by /u/tubeboost)`
- Rate limiting: Respects Reddit's 60 requests per minute limit
- Token refresh: Automatic handling of expired access tokens

## Security Features

### Token Management
- Access tokens encrypted before database storage
- Refresh tokens securely stored for automatic renewal
- Token expiration monitoring and automatic refresh

### API Security
- All requests include proper User-Agent identification
- Rate limiting compliance with Reddit API guidelines
- HTTPS-only communication in production

## Implementation Status

### ✅ Completed Features
- Complete OAuth 2.0 authentication flow
- Database schema with reddit_connections table
- Token storage and refresh mechanism
- User connection management API endpoints
- Security middleware integration
- Development and production environment support

### 🚧 Ready for Development
- Reddit post management
- Subreddit analytics
- Comment automation
- Reddit advertising integration
- Multi-account management

## Usage Examples

### Frontend Integration
```typescript
// Check Reddit connection status
const checkRedditConnection = async () => {
  const response = await fetch('/api/reddit/connection');
  const data = await response.json();
  return data.connected;
};

// Initiate Reddit connection
const connectReddit = () => {
  window.location.href = '/api/reddit/auth';
};

// Remove Reddit connection
const disconnectReddit = async () => {
  await fetch('/api/reddit/connection', { method: 'DELETE' });
};
```

### Backend API Calls
```typescript
// Make authenticated Reddit API request
const redditUser = await RedditAuthService.getRedditUser(accessToken);
const apiResponse = await RedditAuthService.makeRedditApiCall(
  accessToken,
  'GET',
  '/api/v1/me'
);
```

## Future Enhancements

### Planned Features
1. **Reddit Content Management**
   - Post scheduling and automation
   - Comment management
   - Subreddit analytics

2. **Growth Services**
   - Reddit upvote services
   - Comment engagement
   - Follower growth

3. **Analytics Integration**
   - Subreddit performance tracking
   - Engagement metrics
   - Competitor analysis

### Platform Integration
- Unified dashboard for YouTube and Reddit management
- Cross-platform content strategy
- Multi-platform analytics reporting

## Technical Architecture

### Authentication Service (`server/reddit-auth.ts`)
- `RedditAuthService`: Main authentication service class
- OAuth flow management with proper error handling
- Token refresh automation with exponential backoff
- Reddit API rate limiting compliance

### Database Integration (`server/storage.ts`)
- `createRedditConnection()`: Store new Reddit connections
- `getRedditConnection()`: Retrieve user's Reddit connection
- `updateRedditConnection()`: Update connection tokens
- `deleteRedditConnection()`: Remove Reddit connection

### Route Integration (`server/routes.ts`)
- Authentication routes setup via `setupRedditAuth()`
- Connection management endpoints
- Security middleware integration

## Production Deployment

### Environment Configuration
Ensure these environment variables are set in production:
```bash
REDDIT_CLIENT_ID=production_client_id
REDDIT_CLIENT_SECRET=production_client_secret
```

### Domain Configuration
Update Reddit app settings with production domain:
- Production Redirect URI: `https://tubeboost.cc/api/reddit/callback`

### Database Migration
Run database migration to create reddit_connections table:
```bash
npm run db:push
```

## Support and Maintenance

### Monitoring
- Token refresh success/failure rates
- API call success metrics
- User connection analytics

### Error Handling
- Comprehensive error logging for OAuth failures
- User-friendly error messages
- Automatic retry mechanisms for transient failures

This Reddit API integration framework provides a solid foundation for expanding TubeBoost into a comprehensive multi-platform social media management system.