import { useState } from "react";
import { Elements, CardElement, useStripe, useElements } from "@stripe/react-stripe-js";
import { loadStripe } from "@stripe/stripe-js";
import { Dialog, DialogContent, DialogHeader, DialogTitle } from "@/components/ui/dialog";
import { Button } from "@/components/ui/button";
import { Badge } from "@/components/ui/badge";
import { Separator } from "@/components/ui/separator";
import { useToast } from "@/hooks/use-toast";
import { apiRequest } from "@/lib/queryClient";
import { Zap, Star, Crown, Sparkles, CreditCard, DollarSign } from "lucide-react";

// Load Stripe
const stripePromise = loadStripe(import.meta.env.VITE_STRIPE_PUBLIC_KEY);

// Point packages configuration
const POINT_PACKAGES = {
  starter: { 
    points: 1500, 
    price: 5.00, 
    icon: Zap, 
    color: "bg-blue-500",
    badge: "Popular"
  },
  value: { 
    points: 2500, 
    price: 7.50, 
    icon: Star, 
    color: "bg-green-500",
    badge: "Best Value"
  },
  popular: { 
    points: 5000, 
    price: 10.00, 
    icon: Crown, 
    color: "bg-purple-500",
    badge: "Most Popular"
  },
  pro: { 
    points: 10000, 
    price: 20.00, 
    icon: Sparkles, 
    color: "bg-orange-500",
    badge: "Pro Choice"
  }
};

interface InsufficientPointsModalProps {
  isOpen: boolean;
  onClose: () => void;
  requiredPoints: number;
  currentPoints: number;
  onSuccess: () => void;
}

const CheckoutForm = ({ packageType, onSuccess, onCancel }: { 
  packageType: keyof typeof POINT_PACKAGES; 
  onSuccess: () => void;
  onCancel: () => void;
}) => {
  const stripe = useStripe();
  const elements = useElements();
  const { toast } = useToast();
  const [isProcessing, setIsProcessing] = useState(false);
  const [clientSecret, setClientSecret] = useState<string | null>(null);

  const pkg = POINT_PACKAGES[packageType];

  const initializePayment = async () => {
    try {
      const response = await apiRequest("/api/create-payment-intent", "POST", {
        packageType
      });
      setClientSecret(response.clientSecret);
    } catch (error: any) {
      toast({
        title: "Payment Error",
        description: error.message || "Failed to initialize payment",
        variant: "destructive",
      });
    }
  };

  // Initialize payment when component mounts
  useState(() => {
    initializePayment();
  });

  const handleSubmit = async (e: React.FormEvent) => {
    e.preventDefault();

    if (!stripe || !elements || !clientSecret) {
      return;
    }

    setIsProcessing(true);

    try {
      const cardElement = elements.getElement(CardElement);
      if (!cardElement) {
        throw new Error("Card element not found");
      }

      const { error, paymentIntent } = await stripe.confirmCardPayment(clientSecret, {
        payment_method: {
          card: cardElement,
        },
      });

      if (error) {
        throw new Error(error.message);
      }

      if (paymentIntent.status === 'succeeded') {
        // Complete the payment on backend
        await apiRequest("/api/complete-payment", "POST", {
          paymentIntentId: paymentIntent.id
        });

        toast({
          title: "Payment Successful!",
          description: `${pkg.points} points have been added to your account.`,
        });

        onSuccess();
      }
    } catch (error: any) {
      toast({
        title: "Payment Failed",
        description: error.message || "Payment could not be processed",
        variant: "destructive",
      });
    } finally {
      setIsProcessing(false);
    }
  };

  return (
    <div className="space-y-6">
      <div className="text-center">
        <div className={`w-16 h-16 mx-auto rounded-full ${pkg.color} flex items-center justify-center mb-4`}>
          <pkg.icon className="w-8 h-8 text-white" />
        </div>
        <h3 className="text-xl font-semibold mb-2">
          {pkg.points.toLocaleString()} Points
        </h3>
        <div className="text-3xl font-bold text-primary">
          ${pkg.price.toFixed(2)}
        </div>
        <p className="text-sm text-muted-foreground mt-2">
          ${(pkg.price / pkg.points * 1000).toFixed(2)} per 1000 points
        </p>
      </div>

      <form onSubmit={handleSubmit} className="space-y-4">
        <div className="border rounded-lg p-4">
          <CardElement
            options={{
              style: {
                base: {
                  fontSize: '16px',
                  color: '#424770',
                  '::placeholder': {
                    color: '#aab7c4',
                  },
                },
              },
            }}
          />
        </div>

        <div className="flex gap-3">
          <Button 
            type="button" 
            variant="outline" 
            className="flex-1"
            onClick={onCancel}
          >
            Cancel
          </Button>
          <Button 
            type="submit" 
            className="flex-1"
            disabled={!stripe || isProcessing}
          >
            {isProcessing ? (
              <div className="flex items-center gap-2">
                <div className="w-4 h-4 border-2 border-white border-t-transparent rounded-full animate-spin" />
                Processing...
              </div>
            ) : (
              <div className="flex items-center gap-2">
                <CreditCard className="w-4 h-4" />
                Pay ${pkg.price.toFixed(2)}
              </div>
            )}
          </Button>
        </div>
      </form>
    </div>
  );
};

export default function InsufficientPointsModal({ 
  isOpen, 
  onClose, 
  requiredPoints, 
  currentPoints, 
  onSuccess 
}: InsufficientPointsModalProps) {
  const [selectedPackage, setSelectedPackage] = useState<keyof typeof POINT_PACKAGES | null>(null);
  const [showCheckout, setShowCheckout] = useState(false);

  const pointsNeeded = requiredPoints - currentPoints;

  const handlePackageSelect = (packageType: keyof typeof POINT_PACKAGES) => {
    setSelectedPackage(packageType);
    setShowCheckout(true);
  };

  const handleSuccess = () => {
    setShowCheckout(false);
    setSelectedPackage(null);
    onSuccess();
    onClose();
  };

  const handleCancel = () => {
    setShowCheckout(false);
    setSelectedPackage(null);
  };

  return (
    <Dialog open={isOpen} onOpenChange={onClose}>
      <DialogContent className="max-w-2xl">
        <DialogHeader>
          <DialogTitle className="text-center text-2xl font-bold">
            {showCheckout ? "Complete Purchase" : "Insufficient Points"}
          </DialogTitle>
        </DialogHeader>

        {!showCheckout ? (
          <div className="space-y-6">
            {/* Current Status */}
            <div className="text-center bg-red-50 dark:bg-red-950/20 border border-red-200 dark:border-red-800 rounded-lg p-4">
              <div className="flex items-center justify-center gap-2 mb-2">
                <DollarSign className="w-5 h-5 text-red-600" />
                <span className="font-semibold text-red-700 dark:text-red-400">
                  You need {pointsNeeded} more points
                </span>
              </div>
              <p className="text-sm text-red-600 dark:text-red-400">
                Required: {requiredPoints} points • Available: {currentPoints} points
              </p>
            </div>

            <div className="text-center">
              <h3 className="text-lg font-semibold mb-2">Choose Your Points Package</h3>
              <p className="text-muted-foreground mb-6">
                Purchase points to continue using TubeBoost services. Points never expire!
              </p>
            </div>

            {/* Package Grid */}
            <div className="grid grid-cols-2 gap-4">
              {Object.entries(POINT_PACKAGES).map(([key, pkg]) => {
                const packageKey = key as keyof typeof POINT_PACKAGES;
                const Icon = pkg.icon;
                const isRecommended = pkg.points >= pointsNeeded;
                
                return (
                  <div
                    key={key}
                    className={`relative border rounded-xl p-4 cursor-pointer transition-all hover:scale-105 hover:shadow-lg ${
                      isRecommended ? 'border-primary bg-primary/5' : 'border-border'
                    }`}
                    onClick={() => handlePackageSelect(packageKey)}
                  >
                    {pkg.badge && (
                      <Badge 
                        className={`absolute -top-2 left-4 ${
                          isRecommended ? 'bg-primary' : 'bg-secondary'
                        }`}
                      >
                        {pkg.badge}
                      </Badge>
                    )}
                    
                    <div className="text-center">
                      <div className={`w-12 h-12 mx-auto rounded-full ${pkg.color} flex items-center justify-center mb-3`}>
                        <Icon className="w-6 h-6 text-white" />
                      </div>
                      
                      <div className="space-y-1">
                        <div className="text-lg font-bold">
                          {pkg.points.toLocaleString()}
                        </div>
                        <div className="text-sm text-muted-foreground">
                          points
                        </div>
                        <div className="text-xl font-bold text-primary">
                          ${pkg.price.toFixed(2)}
                        </div>
                        <div className="text-xs text-muted-foreground">
                          ${(pkg.price / pkg.points * 1000).toFixed(2)}/1k points
                        </div>
                      </div>

                      {isRecommended && (
                        <div className="mt-2">
                          <Badge variant="outline" className="text-xs bg-green-50 text-green-700 border-green-200">
                            ✓ Covers your needs
                          </Badge>
                        </div>
                      )}
                    </div>
                  </div>
                );
              })}
            </div>

            <Separator />
            
            <div className="text-center space-y-2">
              <p className="text-sm text-muted-foreground">
                ✓ Points never expire  ✓ Instant delivery  ✓ Secure payment
              </p>
              <Button variant="outline" onClick={onClose}>
                Cancel
              </Button>
            </div>
          </div>
        ) : selectedPackage ? (
          <Elements stripe={stripePromise}>
            <CheckoutForm 
              packageType={selectedPackage} 
              onSuccess={handleSuccess}
              onCancel={handleCancel}
            />
          </Elements>
        ) : null}
      </DialogContent>
    </Dialog>
  );
}