import { useState, useEffect } from "react";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Alert, AlertDescription } from "@/components/ui/alert";
import { AlertTriangle, Copy, Eye, EyeOff, X } from "lucide-react";
import { useAuth } from "@/hooks/use-auth";
import { useToast } from "@/hooks/use-toast";

export default function AnonymousCredentialsDisplay() {
  const { user } = useAuth();
  const { toast } = useToast();
  const [showPassword, setShowPassword] = useState(true);
  const [dismissed, setDismissed] = useState(false);
  const [tempPassword, setTempPassword] = useState<string | null>(null);

  // Get the temporary password from localStorage if available
  useEffect(() => {
    if (user && user.accountType === "anonymous") {
      const storedPassword = localStorage.getItem(`temp_password_${user.username}`);
      setTempPassword(storedPassword);
    }
  }, [user]);

  const copyToClipboard = async (text: string, label: string) => {
    try {
      await navigator.clipboard.writeText(text);
      toast({
        title: "Copied!",
        description: `${label} copied to clipboard`,
        variant: "default"
      });
    } catch (error) {
      toast({
        title: "Copy Failed",
        description: "Could not copy to clipboard",
        variant: "destructive"
      });
    }
  };

  // Don't show if user is not anonymous, component was dismissed, or no temp password available
  if (!user || user.accountType !== "anonymous" || dismissed || !tempPassword) {
    return null;
  }

  return (
    <Card className="border-red-200 bg-red-50 dark:bg-red-900/20 mb-6">
      <CardHeader className="pb-3">
        <div className="flex items-center justify-between">
          <CardTitle className="text-lg text-red-800 dark:text-red-200 flex items-center gap-2">
            <AlertTriangle className="h-5 w-5" />
            Save Your Anonymous Account Details
          </CardTitle>
          <Button
            variant="ghost"
            size="sm"
            onClick={() => setDismissed(true)}
            className="text-red-600 hover:bg-red-100 dark:text-red-400"
          >
            <X className="h-4 w-4" />
          </Button>
        </div>
      </CardHeader>
      <CardContent>
        <Alert className="border-red-300 bg-red-100 dark:bg-red-800/30 mb-4">
          <AlertDescription className="text-red-800 dark:text-red-200">
            <strong>IMPORTANT:</strong> Write down these credentials immediately! No password recovery is available for anonymous accounts.
          </AlertDescription>
        </Alert>
        
        <div className="space-y-3">
          <div className="flex items-center justify-between bg-white dark:bg-slate-700 p-3 rounded-lg border">
            <div className="flex-1">
              <label className="text-sm font-medium text-gray-600 dark:text-gray-300">Username</label>
              <div className="font-mono text-lg font-bold text-gray-900 dark:text-white">
                {user.username}
              </div>
            </div>
            <Button
              variant="outline"
              size="sm"
              onClick={() => copyToClipboard(user.username, "Username")}
              className="ml-2"
            >
              <Copy className="h-4 w-4" />
            </Button>
          </div>
          
          <div className="flex items-center justify-between bg-white dark:bg-slate-700 p-3 rounded-lg border">
            <div className="flex-1">
              <label className="text-sm font-medium text-gray-600 dark:text-gray-300">Password</label>
              <div className="font-mono text-lg font-bold text-gray-900 dark:text-white">
                {showPassword && tempPassword ? tempPassword : "••••••••••••"}
              </div>
            </div>
            <div className="flex items-center gap-2 ml-2">
              <Button
                variant="outline"
                size="sm"
                onClick={() => setShowPassword(!showPassword)}
              >
                {showPassword ? <EyeOff className="h-4 w-4" /> : <Eye className="h-4 w-4" />}
              </Button>
              <Button
                variant="outline"
                size="sm"
                onClick={() => tempPassword && copyToClipboard(tempPassword, "Password")}
                disabled={!tempPassword}
              >
                <Copy className="h-4 w-4" />
              </Button>
            </div>
          </div>
        </div>
        
        <div className="mt-4 p-3 bg-yellow-50 dark:bg-yellow-900/20 rounded-lg border border-yellow-200 dark:border-yellow-800">
          <p className="text-sm text-yellow-800 dark:text-yellow-200">
            💡 <strong>Tip:</strong> Add your email address using the banner above to enable password recovery and notifications.
          </p>
        </div>
      </CardContent>
    </Card>
  );
}