import { useState } from "react";
import { useMutation } from "@tanstack/react-query";
import { Dialog, DialogContent, DialogHeader, DialogTitle, DialogTrigger } from "@/components/ui/dialog";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import { useToast } from "@/hooks/use-toast";
import { Settings, Key } from "lucide-react";
import { apiRequest } from "@/lib/queryClient";

interface ApiKeyManagerProps {
  trigger?: React.ReactNode;
}

export function ApiKeyManager({ trigger }: ApiKeyManagerProps) {
  const [isOpen, setIsOpen] = useState(false);
  const [smmcostKey, setSmmcostKey] = useState("");
  const { toast } = useToast();

  const { mutate: updateApiKey, isPending } = useMutation({
    mutationFn: async (newKey: string) => {
      return await apiRequest("/api/update-smmcost-key", "POST", {
        apiKey: newKey
      });
    },
    onSuccess: () => {
      toast({
        title: "API Key Updated",
        description: "SMMCost API key has been successfully updated",
      });
      setIsOpen(false);
      setSmmcostKey("");
    },
    onError: (error: any) => {
      toast({
        title: "Error",
        description: error.message || "Failed to update API key",
        variant: "destructive"
      });
    }
  });

  const handleSubmit = (e: React.FormEvent) => {
    e.preventDefault();
    if (!smmcostKey.trim()) {
      toast({
        title: "Error",
        description: "Please enter a valid API key",
        variant: "destructive"
      });
      return;
    }
    updateApiKey(smmcostKey.trim());
  };

  return (
    <Dialog open={isOpen} onOpenChange={setIsOpen}>
      <DialogTrigger asChild>
        {trigger || (
          <Button variant="outline" size="sm">
            <Key className="w-4 h-4 mr-2" />
            Update API Key
          </Button>
        )}
      </DialogTrigger>
      <DialogContent className="sm:max-w-md">
        <DialogHeader>
          <DialogTitle className="flex items-center space-x-2">
            <Settings className="w-5 h-5" />
            <span>Update SMMCost API Key</span>
          </DialogTitle>
        </DialogHeader>
        <form onSubmit={handleSubmit} className="space-y-4">
          <div className="space-y-2">
            <Label htmlFor="smmcost-key">SMMCost API Key</Label>
            <Input
              id="smmcost-key"
              type="password"
              placeholder="Enter your SMMCost API key"
              value={smmcostKey}
              onChange={(e) => setSmmcostKey(e.target.value)}
              disabled={isPending}
            />
            <p className="text-sm text-gray-500">
              This key is used for subscriber and view boosting services
            </p>
          </div>
          <div className="flex justify-end space-x-2">
            <Button
              type="button"
              variant="outline"
              onClick={() => setIsOpen(false)}
              disabled={isPending}
            >
              Cancel
            </Button>
            <Button type="submit" disabled={isPending}>
              {isPending ? "Updating..." : "Update Key"}
            </Button>
          </div>
        </form>
      </DialogContent>
    </Dialog>
  );
}