import { useState } from "react";
import { useQuery, useMutation, useQueryClient } from "@tanstack/react-query";
import { apiRequest } from "@/lib/queryClient";
import { Button } from "@/components/ui/button";
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select";
import { Dialog, DialogContent, DialogHeader, DialogTitle, DialogTrigger } from "@/components/ui/dialog";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import { Card, CardContent } from "@/components/ui/card";
import { Badge } from "@/components/ui/badge";
import { toast } from "@/hooks/use-toast";
import { Plus, Youtube, Users, Eye, Clock, RefreshCw } from "lucide-react";
import { Link } from "wouter";

interface Channel {
  id: number;
  channelName: string;
  channelHandle: string;
  customUrl?: string;
  subscriberCount: number;
  viewCount: number;
  videoCount: number;
  isActive: boolean;
}

export function ChannelSelector() {
  const [addChannelOpen, setAddChannelOpen] = useState(false);
  const [channelUrl, setChannelUrl] = useState("");
  const queryClient = useQueryClient();

  // Fetch all channels
  const { data: channels, isLoading: channelsLoading } = useQuery<Channel[]>({
    queryKey: ["/api/channels"],
  });

  // Fetch current active channel
  const { data: activeChannel } = useQuery<Channel>({
    queryKey: ["/api/channel"],
  });

  // Add new channel mutation
  const { mutate: addChannel, isPending: isAddingChannel } = useMutation({
    mutationFn: (url: string) => apiRequest("/api/add-channel", "POST", { channelUrl: url }),
    onSuccess: () => {
      toast({
        title: "Success",
        description: "Channel added successfully",
      });
      setAddChannelOpen(false);
      setChannelUrl("");
      queryClient.invalidateQueries({ queryKey: ["/api/channels"] });
      queryClient.invalidateQueries({ queryKey: ["/api/channel"] });
    },
    onError: (error: any) => {
      toast({
        title: "Error",
        description: error.message || "Failed to add channel",
        variant: "destructive",
      });
    }
  });

  // Switch channel mutation
  const { mutate: switchChannel } = useMutation({
    mutationFn: (channelId: number) => apiRequest("/api/switch-channel", "POST", { channelId }),
    onSuccess: () => {
      toast({
        title: "Success",
        description: "Channel switched successfully",
      });
      queryClient.invalidateQueries({ queryKey: ["/api/channel"] });
      queryClient.invalidateQueries({ queryKey: ["/api/videos"] });
      queryClient.invalidateQueries({ queryKey: ["/api/youtube-stats"] });
    },
    onError: (error: any) => {
      toast({
        title: "Error",
        description: error.message || "Failed to switch channel",
        variant: "destructive",
      });
    }
  });

  // Sync YouTube data mutation
  const { mutate: syncYouTubeData, isPending: isSyncing } = useMutation({
    mutationFn: () => apiRequest("/api/sync-youtube-data", "POST"),
    onSuccess: (data: any) => {
      toast({
        title: "Success",
        description: `Synced ${data.videosUpdated || 0} videos with YouTube API`,
      });
      queryClient.invalidateQueries({ queryKey: ["/api/channel"] });
      queryClient.invalidateQueries({ queryKey: ["/api/videos"] });
      queryClient.invalidateQueries({ queryKey: ["/api/youtube-stats"] });
      queryClient.invalidateQueries({ queryKey: ["/api/channels"] });
    },
    onError: (error: any) => {
      console.error("YouTube sync error:", error);
      toast({
        title: "Sync Failed",
        description: error.message || "Failed to sync YouTube data. Check console for details.",
        variant: "destructive",
      });
    }
  });

  const handleAddChannel = () => {
    if (!channelUrl.trim()) {
      toast({
        title: "Error",
        description: "Please enter a valid YouTube channel URL",
        variant: "destructive",
      });
      return;
    }
    addChannel(channelUrl);
  };

  const formatNumber = (num: number) => {
    if (num >= 1000000) {
      return (num / 1000000).toFixed(1) + 'M';
    } else if (num >= 1000) {
      return (num / 1000).toFixed(1) + 'K';
    }
    return num.toString();
  };

  if (channelsLoading) {
    return (
      <div className="flex items-center justify-center p-4">
        <div className="w-6 h-6 border-2 border-primary border-t-transparent rounded-full animate-spin"></div>
      </div>
    );
  }

  return (
    <div className="space-y-4">
      {/* Channel Selection */}
      <div className="flex items-center gap-4">
        <div className="flex-1">
          <Label htmlFor="channel-select" className="text-sm font-medium">Active Channel</Label>
          <Select 
            value={activeChannel?.id?.toString() || ""} 
            onValueChange={(value) => switchChannel(parseInt(value))}
          >
            <SelectTrigger id="channel-select" className="mt-1">
              <SelectValue placeholder="Select a channel..." />
            </SelectTrigger>
            <SelectContent>
              {channels?.map((channel) => (
                <SelectItem key={channel.id} value={channel.id.toString()}>
                  <div className="flex items-center gap-2">
                    <Youtube className="h-4 w-4 text-red-600" />
                    <div className="flex flex-col">
                      <span className="font-medium">{channel.channelName}</span>
                      <span className="text-xs text-muted-foreground">@{channel.channelHandle}</span>
                    </div>
                    {channel.isActive && (
                      <Badge variant="secondary" className="ml-2">Active</Badge>
                    )}
                  </div>
                </SelectItem>
              ))}
            </SelectContent>
          </Select>
        </div>

        {/* Action Buttons */}
        <div className="flex gap-2 mt-6">
          <Dialog open={addChannelOpen} onOpenChange={setAddChannelOpen}>
            <DialogTrigger asChild>
              <Button variant="outline">
                <Plus className="h-4 w-4 mr-2" />
                Add Channel
              </Button>
            </DialogTrigger>
            <DialogContent>
              <DialogHeader>
                <DialogTitle>Add YouTube Channel</DialogTitle>
              </DialogHeader>
              <div className="space-y-4">
                <div>
                  <Label htmlFor="channel-url">YouTube Channel URL</Label>
                  <Input
                    id="channel-url"
                    placeholder="https://www.youtube.com/@channelname"
                    value={channelUrl}
                    onChange={(e) => setChannelUrl(e.target.value)}
                  />
                </div>
                <div className="flex justify-end gap-2">
                  <Button variant="outline" onClick={() => setAddChannelOpen(false)}>
                    Cancel
                  </Button>
                  <Button onClick={handleAddChannel} disabled={isAddingChannel}>
                    {isAddingChannel ? "Adding..." : "Add Channel"}
                  </Button>
                </div>
              </div>
            </DialogContent>
          </Dialog>
          
          {/* Sync YouTube Button */}
          <Button 
            variant="default" 
            onClick={() => syncYouTubeData()}
            disabled={isSyncing}
          >
            <RefreshCw className={`h-4 w-4 mr-2 ${isSyncing ? 'animate-spin' : ''}`} />
            {isSyncing ? "Syncing..." : "Sync YouTube"}
          </Button>
        </div>
      </div>

      {/* Active Channel Info */}
      {activeChannel && (
        <Card>
          <CardContent className="p-4">
            <div className="flex items-center gap-4">
              <div className="w-12 h-12 bg-gradient-to-br from-red-500 to-red-600 rounded-full flex items-center justify-center">
                <Youtube className="h-6 w-6 text-white" />
              </div>
              <div className="flex-1">
                <h3 className="font-semibold">{activeChannel.channelName}</h3>
                <p className="text-sm text-muted-foreground">@{activeChannel.channelHandle}</p>
              </div>
              <div className="grid grid-cols-3 gap-4 text-center">
                <div className="flex flex-col items-center">
                  <Users className="h-4 w-4 text-muted-foreground mb-1" />
                  <span className="text-sm font-medium">{formatNumber(activeChannel.subscriberCount || 0)}</span>
                  <span className="text-xs text-muted-foreground">Subscribers</span>
                </div>
                <div className="flex flex-col items-center">
                  <Eye className="h-4 w-4 text-muted-foreground mb-1" />
                  <span className="text-sm font-medium">{formatNumber(activeChannel.viewCount || 0)}</span>
                  <span className="text-xs text-muted-foreground">Views</span>
                </div>
                <div className="flex flex-col items-center">
                  <Clock className="h-4 w-4 text-muted-foreground mb-1" />
                  <span className="text-sm font-medium">{activeChannel.videoCount || 0}</span>
                  <span className="text-xs text-muted-foreground">Videos</span>
                </div>
              </div>
            </div>
          </CardContent>
        </Card>
      )}

      {/* No Channel Warning */}
      {!activeChannel && channels && channels.length === 0 && (
        <Card>
          <CardContent className="p-6 text-center">
            <Youtube className="h-12 w-12 text-muted-foreground mx-auto mb-4" />
            <h3 className="text-lg font-semibold mb-2">No YouTube Channel Connected</h3>
            <p className="text-muted-foreground mb-4">
              Connect your YouTube channel to start managing your videos and analytics.
            </p>
            <Button onClick={() => setAddChannelOpen(true)}>
              <Plus className="h-4 w-4 mr-2" />
              Connect YouTube Channel
            </Button>
          </CardContent>
        </Card>
      )}

      {/* Channel Manager Link */}
      {channels && channels.length > 0 && (
        <div className="flex justify-end">
          <Link href="/channel-manager">
            <Button variant="outline" size="sm">
              Manage All Channels
            </Button>
          </Link>
        </div>
      )}
    </div>
  );
}