import { useState } from "react";
import { Alert, AlertDescription } from "@/components/ui/alert";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Mail, Gift, CheckCircle2 } from "lucide-react";
import { useMutation, useQueryClient } from "@tanstack/react-query";
import { apiRequest } from "@/lib/queryClient";
import { useToast } from "@/hooks/use-toast";
import { useAuth } from "@/hooks/use-auth";

export default function EmailIncentiveBanner() {
  const { user } = useAuth();
  const [email, setEmail] = useState("");
  const [isUpdating, setIsUpdating] = useState(false);
  const { toast } = useToast();
  const queryClient = useQueryClient();

  const updateEmailMutation = useMutation({
    mutationFn: async (email: string) => {
      const response = await apiRequest("/api/update-email-with-bonus", "PATCH", { email });
      if (!response.ok) {
        const error = await response.json();
        throw new Error(error.message || "Failed to update email");
      }
      return response.json();
    },
    onSuccess: (data) => {
      toast({
        title: "Email Updated Successfully!",
        description: `Email saved and ${data.bonusPoints || 300} bonus points added to your account!`,
        variant: "default"
      });
      queryClient.invalidateQueries({ queryKey: ["/api/me"] });
      queryClient.invalidateQueries({ queryKey: ["/api/points"] });
      setIsUpdating(false);
    },
    onError: (error: any) => {
      toast({
        title: "Update Failed",
        description: error.message,
        variant: "destructive"
      });
      setIsUpdating(false);
    }
  });

  const handleUpdateEmail = () => {
    if (!email || !email.includes("@")) {
      toast({
        title: "Invalid Email",
        description: "Please enter a valid email address",
        variant: "destructive"
      });
      return;
    }
    setIsUpdating(true);
    updateEmailMutation.mutate(email);
  };

  // Don't show banner if user has email or is updating
  if (!user || user.email || isUpdating) {
    return null;
  }

  return (
    <Alert className="border-red-500 bg-red-600 text-white mb-6 shadow-lg">
      <div className="flex items-center justify-between">
        <div className="flex items-center gap-3">
          <div className="flex items-center gap-2">
            <Mail className="h-5 w-5 text-white" />
            <Gift className="h-5 w-5 text-yellow-300 animate-pulse" />
          </div>
          <div className="flex-1">
            <AlertDescription className="text-white font-medium">
              <span className="font-bold">EARN 300 BONUS POINTS!</span> Add your email address for account security and notifications.
            </AlertDescription>
          </div>
        </div>
        
        <div className="flex items-center gap-3 ml-4">
          <Input
            type="email"
            placeholder="Enter your email address"
            value={email}
            onChange={(e) => setEmail(e.target.value)}
            className="w-64 bg-white text-black border-white placeholder:text-gray-500"
            onKeyPress={(e) => e.key === 'Enter' && handleUpdateEmail()}
          />
          <Button
            onClick={handleUpdateEmail}
            disabled={updateEmailMutation.isPending || !email}
            className="bg-white text-red-600 hover:bg-gray-100 font-semibold px-6"
          >
            {updateEmailMutation.isPending ? (
              <>
                <div className="animate-spin rounded-full h-4 w-4 border-b-2 border-red-600 mr-2"></div>
                Saving...
              </>
            ) : (
              <>
                <CheckCircle2 className="h-4 w-4 mr-2" />
                Claim 300 Points
              </>
            )}
          </Button>
        </div>
      </div>
    </Alert>
  );
}