import { useQuery } from "@tanstack/react-query";
import { apiRequest } from "@/lib/queryClient";
import { useEffect } from "react";
import type { HeaderCode } from "@shared/schema";

export default function HeaderCodesInjector() {
  const { data: headerCodes = [] } = useQuery<HeaderCode[]>({
    queryKey: ["/api/active-header-codes"],
    queryFn: () => apiRequest("/api/active-header-codes"),
  });

  useEffect(() => {
    // Clear existing header codes to avoid duplicates
    const existingCodes = document.querySelectorAll('[data-header-code]');
    existingCodes.forEach(el => el.remove());

    // Add new header codes to the document head
    headerCodes.forEach((headerCode) => {
      try {
        // Create a container element to safely parse and inject the code
        const container = document.createElement('div');
        container.innerHTML = headerCode.code;
        container.setAttribute('data-header-code', headerCode.id.toString());
        
        // Extract and properly handle script tags
        const scripts = container.querySelectorAll('script');
        scripts.forEach((originalScript) => {
          const newScript = document.createElement('script');
          
          // Copy all attributes
          Array.from(originalScript.attributes).forEach(attr => {
            newScript.setAttribute(attr.name, attr.value);
          });
          
          // Set the script content properly
          if (originalScript.src) {
            newScript.src = originalScript.src;
          } else {
            newScript.textContent = originalScript.textContent || originalScript.innerHTML;
          }
          
          newScript.setAttribute('data-header-code', headerCode.id.toString());
          document.head.appendChild(newScript);
        });

        // Handle non-script elements (like meta tags, styles, etc.)
        const nonScriptElements = container.querySelectorAll(':not(script)');
        nonScriptElements.forEach((element) => {
          const clonedElement = element.cloneNode(true) as Element;
          clonedElement.setAttribute('data-header-code', headerCode.id.toString());
          document.head.appendChild(clonedElement);
        });

        // If there are no child elements, treat the entire code as a script
        if (container.children.length === 0 && headerCode.code.trim()) {
          const script = document.createElement('script');
          script.textContent = headerCode.code;
          script.setAttribute('data-header-code', headerCode.id.toString());
          document.head.appendChild(script);
        }
      } catch (error) {
        console.error(`❌ [HEADER CODE] Failed to inject header code ${headerCode.id}:`, error);
        console.error(`❌ [HEADER CODE] Problematic code:`, headerCode.code);
      }
    });
  }, [headerCodes]);

  return null;
}