import { useState } from 'react';
import jsPDF from 'jspdf';
import { Button } from '@/components/ui/button';
import { Download } from 'lucide-react';

interface PDFExportProps {
  analytics: any;
  subredditName: string;
  onExport?: () => void;
}

export function PDFExport({ analytics, subredditName, onExport }: PDFExportProps) {
  const [isGenerating, setIsGenerating] = useState(false);

  const generatePDF = async () => {
    if (!analytics) return;

    setIsGenerating(true);
    try {
      const pdf = new jsPDF('p', 'mm', 'a4');
      const pageWidth = pdf.internal.pageSize.getWidth();
      const pageHeight = pdf.internal.pageSize.getHeight();
      let yPosition = 20;

      // Helper function to add text with word wrapping
      const addText = (text: string, x: number, y: number, maxWidth: number = pageWidth - 40) => {
        const lines = pdf.splitTextToSize(text, maxWidth);
        pdf.text(lines, x, y);
        return y + (lines.length * 7);
      };

      // Header with blue background
      pdf.setFillColor(59, 130, 246);
      pdf.rect(0, 0, pageWidth, 50, 'F');
      
      pdf.setTextColor(255, 255, 255);
      pdf.setFontSize(28);
      pdf.setFont('helvetica', 'bold');
      pdf.text('TubeBoost Analytics Report', pageWidth / 2, 25, { align: 'center' });
      
      pdf.setFontSize(18);
      pdf.text(`r/${subredditName}`, pageWidth / 2, 35, { align: 'center' });
      
      pdf.setFontSize(12);
      pdf.text(`Generated on ${new Date().toLocaleDateString()}`, pageWidth / 2, 42, { align: 'center' });

      // Reset for content
      pdf.setTextColor(0, 0, 0);
      yPosition = 70;

      // Executive Summary
      pdf.setFillColor(59, 130, 246);
      pdf.rect(20, yPosition - 5, pageWidth - 40, 8, 'F');
      pdf.setTextColor(255, 255, 255);
      pdf.setFontSize(16);
      pdf.setFont('helvetica', 'bold');
      pdf.text('Executive Summary', 25, yPosition);
      yPosition += 15;

      pdf.setTextColor(0, 0, 0);
      pdf.setFontSize(12);
      pdf.setFont('helvetica', 'normal');

      const summaryData = [
        `Subreddit: r/${subredditName}`,
        `Title: ${analytics.title || 'N/A'}`,
        `Subscribers: ${analytics.subscribers?.toLocaleString() || 'N/A'}`,
        `Active Users: ${analytics.activeUsers?.toLocaleString() || 'N/A'}`,
        `Posts Analyzed: ${analytics.totalPostsAnalyzed || 0}`,
        `Average Engagement Rate: ${analytics.engagementRate?.toFixed(2) || 0}%`,
        `Growth Estimate: ${analytics.estimatedGrowthRate?.toFixed(2) || 0}%`,
        `Subreddit Type: ${analytics.subredditType || 'Public'}`,
        `Language: ${analytics.language || 'English'}`,
        `NSFW: ${analytics.isNsfw ? 'Yes' : 'No'}`,
        `Created: ${analytics.createdUtc ? new Date(analytics.createdUtc * 1000).toLocaleDateString() : 'N/A'}`
      ];

      summaryData.forEach(line => {
        yPosition = addText(line, 25, yPosition);
        yPosition += 2;
      });

      yPosition += 10;

      // Engagement Metrics Section
      pdf.setFillColor(16, 185, 129);
      pdf.rect(20, yPosition - 5, pageWidth - 40, 8, 'F');
      pdf.setTextColor(255, 255, 255);
      pdf.setFontSize(16);
      pdf.setFont('helvetica', 'bold');
      pdf.text('Engagement Metrics', 25, yPosition);
      yPosition += 15;

      pdf.setTextColor(0, 0, 0);
      pdf.setFontSize(12);
      pdf.setFont('helvetica', 'normal');

      const engagementData = [
        `Average Upvotes per Post: ${analytics.averageUpvotes?.toLocaleString() || 0}`,
        `Average Comments per Post: ${analytics.averageComments?.toLocaleString() || 0}`,
        `Engagement Rate: ${analytics.engagementRate?.toFixed(2) || 0}%`,
        `Daily Posting Rate: ${analytics.dailyPostingRate || 0} posts/day`,
        `Growth Rate Estimate: ${analytics.estimatedGrowthRate?.toFixed(2) || 0}%`
      ];

      engagementData.forEach(line => {
        yPosition = addText(line, 25, yPosition);
        yPosition += 2;
      });

      yPosition += 10;

      // Content Analysis Section
      pdf.setFillColor(245, 158, 11);
      pdf.rect(20, yPosition - 5, pageWidth - 40, 8, 'F');
      pdf.setTextColor(255, 255, 255);
      pdf.setFontSize(16);
      pdf.setFont('helvetica', 'bold');
      pdf.text('Content Analysis', 25, yPosition);
      yPosition += 15;

      pdf.setTextColor(0, 0, 0);
      pdf.setFontSize(12);
      pdf.setFont('helvetica', 'normal');

      if (analytics.postTypes) {
        Object.entries(analytics.postTypes).forEach(([type, count]: [string, any]) => {
          yPosition = addText(`${type.charAt(0).toUpperCase() + type.slice(1)}: ${count} posts`, 25, yPosition);
          yPosition += 2;
        });
      }

      // Add subreddit features
      yPosition += 5;
      const features = [
        `Allows Images: ${analytics.allowImages ? 'Yes' : 'No'}`,
        `Allows Videos: ${analytics.allowVideos ? 'Yes' : 'No'}`,
        `Allows Polls: ${analytics.allowPolls ? 'Yes' : 'No'}`,
        `Allows Galleries: ${analytics.allowGalleries ? 'Yes' : 'No'}`,
        `Has Rules: ${analytics.hasRules ? 'Yes' : 'No'} (${analytics.rulesCount || 0} rules)`
      ];

      features.forEach(line => {
        yPosition = addText(line, 25, yPosition);
        yPosition += 2;
      });

      yPosition += 15;

      // Top Keywords Section
      if (analytics.topKeywords?.length > 0) {
        pdf.setFillColor(139, 92, 246);
        pdf.rect(20, yPosition - 5, pageWidth - 40, 8, 'F');
        pdf.setTextColor(255, 255, 255);
        pdf.setFontSize(16);
        pdf.setFont('helvetica', 'bold');
        pdf.text('Top Keywords', 25, yPosition);
        yPosition += 15;

        pdf.setTextColor(0, 0, 0);
        pdf.setFontSize(10);
        pdf.setFont('helvetica', 'normal');

        const keywords = analytics.topKeywords.slice(0, 20).join(', ');
        yPosition = addText(`Keywords: ${keywords}`, 25, yPosition);
        yPosition += 15;
      }

      // Best Posting Hours Section
      if (analytics.bestPostingHours?.length > 0) {
        pdf.setFillColor(168, 85, 247);
        pdf.rect(20, yPosition - 5, pageWidth - 40, 8, 'F');
        pdf.setTextColor(255, 255, 255);
        pdf.setFontSize(16);
        pdf.setFont('helvetica', 'bold');
        pdf.text('Optimal Posting Times', 25, yPosition);
        yPosition += 15;

        pdf.setTextColor(0, 0, 0);
        pdf.setFontSize(12);
        pdf.setFont('helvetica', 'normal');

        yPosition = addText(`Best posting hours: ${analytics.bestPostingHours.join(', ')}`, 25, yPosition);
        yPosition += 15;
      }

      // Subreddit Description Section
      if (analytics.description || analytics.publicDescription) {
        if (yPosition > pageHeight - 60) {
          pdf.addPage();
          yPosition = 20;
        }

        pdf.setFillColor(34, 197, 94);
        pdf.rect(20, yPosition - 5, pageWidth - 40, 8, 'F');
        pdf.setTextColor(255, 255, 255);
        pdf.setFontSize(16);
        pdf.setFont('helvetica', 'bold');
        pdf.text('Subreddit Description', 25, yPosition);
        yPosition += 15;

        pdf.setTextColor(0, 0, 0);
        pdf.setFontSize(10);
        pdf.setFont('helvetica', 'normal');

        if (analytics.publicDescription) {
          yPosition = addText(`Public Description: ${analytics.publicDescription}`, 25, yPosition);
          yPosition += 5;
        }

        if (analytics.description && analytics.description !== analytics.publicDescription) {
          yPosition = addText(`Full Description: ${analytics.description}`, 25, yPosition);
          yPosition += 5;
        }
        yPosition += 10;
      }

      // Check if we need a new page
      if (yPosition > pageHeight - 80) {
        pdf.addPage();
        yPosition = 20;
      }

      // Top Performing Posts Section
      if (analytics.topPosts?.length > 0) {
        pdf.setFillColor(239, 68, 68);
        pdf.rect(20, yPosition - 5, pageWidth - 40, 8, 'F');
        pdf.setTextColor(255, 255, 255);
        pdf.setFontSize(16);
        pdf.setFont('helvetica', 'bold');
        pdf.text('Top Performing Posts', 25, yPosition);
        yPosition += 15;

        pdf.setTextColor(0, 0, 0);
        pdf.setFontSize(10);
        pdf.setFont('helvetica', 'normal');

        analytics.topPosts.slice(0, 5).forEach((post: any, index: number) => {
          if (yPosition > pageHeight - 40) {
            pdf.addPage();
            yPosition = 20;
          }

          pdf.setFont('helvetica', 'bold');
          const title = post.title.length > 80 ? post.title.substring(0, 80) + '...' : post.title;
          yPosition = addText(`${index + 1}. ${title}`, 25, yPosition);
          
          pdf.setFont('helvetica', 'normal');
          const stats = `   Author: u/${post.author} | Score: ${post.score} | Upvotes: ${post.upvotes} | Comments: ${post.comments} | Date: ${new Date(post.created_utc * 1000).toLocaleDateString()}`;
          yPosition = addText(stats, 25, yPosition);
          
          if (post.url && !post.is_self) {
            pdf.setFontSize(9);
            yPosition = addText(`   URL: ${post.url}`, 25, yPosition);
            pdf.setFontSize(10);
          }
          yPosition += 5;
        });
      }

      // Additional Analytics Summary
      yPosition += 10;
      if (yPosition > pageHeight - 80) {
        pdf.addPage();
        yPosition = 20;
      }

      pdf.setFillColor(99, 102, 241);
      pdf.rect(20, yPosition - 5, pageWidth - 40, 8, 'F');
      pdf.setTextColor(255, 255, 255);
      pdf.setFontSize(16);
      pdf.setFont('helvetica', 'bold');
      pdf.text('Analytics Summary', 25, yPosition);
      yPosition += 15;

      pdf.setTextColor(0, 0, 0);
      pdf.setFontSize(12);
      pdf.setFont('helvetica', 'normal');

      const summaryStats = [
        `Analysis completed on: ${new Date(analytics.analysisTimestamp).toLocaleString()}`,
        `Processing time: ${analytics.processingTime}ms`,
        `Display name: ${analytics.displayName || analytics.subredditName}`,
        `Community age: ${analytics.createdUtc ? `${Math.floor((Date.now() / 1000 - analytics.createdUtc) / (365.25 * 24 * 3600))} years` : 'Unknown'}`
      ];

      summaryStats.forEach(line => {
        yPosition = addText(line, 25, yPosition);
        yPosition += 2;
      });

      // Footer
      pdf.setFontSize(10);
      pdf.setTextColor(128, 128, 128);
      pdf.text('Generated by TubeBoost Analytics Platform', 20, pageHeight - 30);
      pdf.text('https://tubeboost.cc', pageWidth - 60, pageHeight - 30);
      pdf.text(`Report Date: ${new Date().toLocaleString()}`, 20, pageHeight - 20);
      pdf.text(`Analysis: ${analytics.totalPostsAnalyzed} posts in ${analytics.processingTime}ms`, 20, pageHeight - 10);
      pdf.text(`Subreddit: r/${analytics.subredditName} | Type: ${analytics.subredditType || 'Public'}`, pageWidth - 120, pageHeight - 20);
      pdf.text(`Language: ${analytics.language || 'en'} | NSFW: ${analytics.isNsfw ? 'Yes' : 'No'}`, pageWidth - 120, pageHeight - 10);

      // Save the PDF
      pdf.save(`${subredditName}_analytics_report.pdf`);
      onExport?.();

    } catch (error) {
      console.error('PDF generation error:', error);
    } finally {
      setIsGenerating(false);
    }
  };

  if (!analytics) {
    return null;
  }

  return (
    <div>
      <Button 
        onClick={generatePDF}
        disabled={isGenerating}
        className="w-full bg-blue-600 hover:bg-blue-700 text-white font-medium"
      >
        <Download className="w-4 h-4 mr-2" />
        {isGenerating ? 'Generating Professional PDF...' : 'Export Analytics as PDF'}
      </Button>
    </div>
  );
}