import { useState, useEffect } from "react";
import { useQuery } from "@tanstack/react-query";
import { apiRequest } from "@/lib/queryClient";
import { Button } from "@/components/ui/button";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Badge } from "@/components/ui/badge";
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select";
import { Eye, ThumbsUp, Clock, Play, MoreVertical, RefreshCw, Search, MessageSquare, Star } from "lucide-react";
import { Input } from "@/components/ui/input";

interface YouTubeVideoStats {
  videoId: string;
  // Statistics
  likeCount: string;
  viewCount: string;
  commentCount: string;
  favoriteCount: string;
  // Snippet
  publishedAt: string;
  channelId: string;
  channelTitle: string;
  title: string;
  description: string;
  tags: string[];
  categoryId: string;
  liveBroadcastContent: string;
  defaultLanguage: string;
  defaultAudioLanguage: string;
  // Content Details
  duration: string;
  durationSeconds: number;
  dimension: string;
  definition: string;
  caption: string;
  licensedContent: boolean;
  projection: string;
  // Status
  uploadStatus: string;
  privacyStatus: string;
  license: string;
  embeddable: boolean;
  publicStatsViewable: boolean;
  madeForKids: boolean;
  // Recording Details
  recordingDate: string;
  // Topic Details
  topicCategories: string[];
  // Thumbnails
  thumbnails: {
    default?: { url: string; width: number; height: number };
    medium?: { url: string; width: number; height: number };
    high?: { url: string; width: number; height: number };
    standard?: { url: string; width: number; height: number };
    maxres?: { url: string; width: number; height: number };
  };
}

interface Video {
  id: number;
  videoId: string;
  title: string;
  views: number;
  channelId: number;
}

interface VideoManagerProps {
  selectedVideo: number | null;
  setSelectedVideo: (videoId: number | null) => void;
  isLoadingViews: boolean;
  loadVideoViews: any;
  setLoadViewsModal: (open: boolean) => void;
  setAdminLoadViewsModal: (open: boolean) => void;
  setLikesDialogOpen: (open: boolean) => void;
  userPoints: any;
  ordersMap: Record<number, any[]>;
  user: any;
}

export function VideoManager({
  selectedVideo,
  setSelectedVideo,
  isLoadingViews,
  loadVideoViews,
  setLoadViewsModal,
  setAdminLoadViewsModal,
  setLikesDialogOpen,
  userPoints,
  ordersMap,
  user
}: VideoManagerProps) {
  const [ordersMapState, setOrdersMapState] = useState<Record<number, any[]>>({});
  const [isSyncing, setIsSyncing] = useState(false);
  const [searchQuery, setSearchQuery] = useState("");
  const [isSearching, setIsSearching] = useState(false);

  const { data: videos } = useQuery<Video[]>({
    queryKey: ["/api/videos"],
  });

  // Fetch YouTube stats for all videos in active channel
  const { data: youtubeStats, isLoading: youtubeStatsLoading, error: youtubeStatsError } = useQuery<YouTubeVideoStats[]>({
    queryKey: ["/api/youtube-stats"],
    refetchInterval: 10 * 60 * 1000, // Refresh every 10 minutes
    staleTime: 5 * 60 * 1000, // Consider data stale after 5 minutes
    retry: 3,
    retryDelay: (attemptIndex) => Math.min(1000 * 2 ** attemptIndex, 30000),
  });

  // Log YouTube stats status for debugging
  useEffect(() => {
    if (youtubeStatsError) {
      console.error("❌ YouTube Stats Error:", youtubeStatsError);
    }
    if (youtubeStats && Array.isArray(youtubeStats)) {
      console.log("✅ YouTube Stats loaded:", youtubeStats.length, "videos");
    }
  }, [youtubeStats, youtubeStatsError]);

  // Fetch orders for each video
  useEffect(() => {
    if (videos && Array.isArray(videos) && videos.length > 0) {
      const fetchOrders = async () => {
        const newOrdersMap: Record<number, any[]> = {};
        
        for (const video of videos) {
          try {
            const orders = await apiRequest(`/api/video-orders/${video.id}`, "GET");
            newOrdersMap[video.id] = orders || [];
            console.log(`Video ${video.id} orders:`, orders);
          } catch (error) {
            console.error(`Failed to fetch orders for video ${video.id}:`, error);
            newOrdersMap[video.id] = [];
          }
        }
        
        console.log('Final orders map:', newOrdersMap);
        setOrdersMapState(newOrdersMap);
      };

      fetchOrders();
    }
  }, [videos]);

  const formatNumber = (num: number) => {
    if (num >= 1000000) {
      return (num / 1000000).toFixed(1) + 'M';
    } else if (num >= 1000) {
      return (num / 1000).toFixed(1) + 'K';
    }
    return num.toString();
  };

  const getVideoStats = (videoId: string) => {
    if (!youtubeStats || youtubeStatsLoading) return null;
    return youtubeStats.find((stat: any) => stat.videoId === videoId);
  };

  const formatYouTubeNumber = (numStr: string | number) => {
    const num = parseInt(numStr?.toString() || '0') || 0;
    
    if (num >= 1000000) {
      return (num / 1000000).toFixed(1) + 'M';
    } else if (num >= 1000) {
      return (num / 1000).toFixed(1) + 'K';
    }
    
    return num.toString();
  };

  const handleSyncYouTube = async () => {
    setIsSyncing(true);
    try {
      // Ensure we stay on YouTube platform during sync
      localStorage.setItem('selectedPlatform', 'youtube');
      
      // Force refresh of YouTube stats with cache busting
      const timestamp = Date.now();
      const response = await apiRequest(`/api/youtube-stats?refresh=${timestamp}`, "GET");
      console.log("✅ YouTube stats refreshed:", response);
      
      // Also sync database with fresh data
      const syncResponse = await apiRequest("/api/sync-youtube-data", "POST");
      console.log("✅ YouTube sync successful:", syncResponse);
      
      // Force refresh the page to show updated data while staying on YouTube
      setTimeout(() => {
        window.location.reload();
      }, 500);
    } catch (error) {
      console.error("❌ YouTube sync error:", error);
    } finally {
      setIsSyncing(false);
    }
  };

  const handleSearchVideos = async () => {
    if (!searchQuery.trim()) return;
    
    setIsSearching(true);
    try {
      // Ensure we stay on YouTube platform during search
      localStorage.setItem('selectedPlatform', 'youtube');
      
      const response = await apiRequest("/api/search-channel-videos", "POST", {
        query: searchQuery
      });
      
      console.log("✅ Search results:", response);
      
      // Refresh videos list to show newly found videos while staying on YouTube
      setTimeout(() => {
        window.location.reload();
      }, 500);
    } catch (error) {
      console.error("❌ Search error:", error);
    } finally {
      setIsSearching(false);
    }
  };

  const getServiceTypeDisplay = (serviceId: string) => {
    switch (serviceId) {
      case "940":
        return "views";
      case "1800":
        return "likes";
      case "1523":
        return "views";
      default:
        return "views";
    }
  };

  if (!videos || videos.length === 0) {
    return (
      <Card>
        <CardHeader>
          <CardTitle>Video Manager</CardTitle>
        </CardHeader>
        <CardContent>
          <div className="text-center py-8">
            <p className="text-muted-foreground">No videos found. Connect your YouTube channel to see videos.</p>
          </div>
        </CardContent>
      </Card>
    );
  }

  return (
    <Card>
      <CardHeader>
        <CardTitle className="flex flex-col sm:flex-row sm:items-center sm:justify-between gap-4">
          <div className="flex items-center gap-2">
            <Play className="h-5 w-5" />
            Video Manager
          </div>
          <div className="flex items-center gap-2">
            <Button 
              onClick={handleSyncYouTube}
              disabled={isSyncing}
              variant="outline"
              size="sm"
              className="flex items-center gap-2 whitespace-nowrap"
            >
              <RefreshCw className={`h-4 w-4 ${isSyncing ? 'animate-spin' : ''}`} />
              {isSyncing ? 'Syncing...' : 'Sync YouTube'}
            </Button>
          </div>
        </CardTitle>
      </CardHeader>
      <CardContent>
        <div className="space-y-4">
          {/* Video Search */}
          <div className="space-y-2">
            <label className="text-sm font-medium">Search for Videos</label>
            <div className="flex flex-col sm:flex-row gap-2">
              <Input
                placeholder="e.g., Yoga 7 & 7i: Worth Buying?"
                value={searchQuery}
                onChange={(e) => setSearchQuery(e.target.value)}
                onKeyPress={(e) => e.key === 'Enter' && handleSearchVideos()}
                className="flex-1"
              />
              <Button 
                onClick={handleSearchVideos}
                disabled={isSearching || !searchQuery.trim()}
                size="sm"
                className="flex items-center gap-2 whitespace-nowrap"
              >
                <Search className={`h-4 w-4 ${isSearching ? 'animate-spin' : ''}`} />
                {isSearching ? 'Searching...' : 'Search'}
              </Button>
            </div>
            <p className="text-xs text-muted-foreground">
              Search for specific videos from this channel and add them to your library
            </p>
          </div>

          {/* Video Selection */}
          <div className="space-y-2">
            <label className="text-sm font-medium">Select Video for Actions</label>
            <Select value={selectedVideo?.toString() || ""} onValueChange={(value) => setSelectedVideo(value ? parseInt(value) : null)}>
              <SelectTrigger>
                <SelectValue placeholder="Choose a video..." />
              </SelectTrigger>
              <SelectContent>
                {videos.map((video: any) => (
                  <SelectItem key={video.id} value={video.id.toString()}>
                    <div className="flex items-center gap-2">
                      <span className="truncate max-w-[300px]">{video.title}</span>
                      {selectedVideo === video.id && (
                        <Badge variant="secondary" className="ml-2">Selected</Badge>
                      )}
                    </div>
                  </SelectItem>
                ))}
              </SelectContent>
            </Select>
          </div>

          {/* Video Grid */}
          <div className="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-3 gap-3 sm:gap-4">
            {videos.map((video: any) => {
              const stats = getVideoStats(video.videoId);
              const videoOrders = ordersMapState[video.id] || [];
              
              return (
                <Card 
                  key={video.id} 
                  className={`cursor-pointer transition-all ${selectedVideo === video.id ? 'ring-2 ring-blue-500' : ''}`}
                  onClick={() => setSelectedVideo(video.id)}
                >
                  <div className="relative">
                    <img 
                      src={video.thumbnailUrl} 
                      alt={video.title}
                      className="w-full h-32 object-cover rounded-t-lg"
                      onError={(e) => {
                        const target = e.target as HTMLImageElement;
                        target.src = '/placeholder-video.png';
                      }}
                    />
                    {stats?.duration && (
                      <div className="absolute bottom-2 right-2 bg-black bg-opacity-75 text-white px-2 py-1 rounded text-xs">
                        {stats.duration}
                      </div>
                    )}
                  </div>
                  
                  <CardContent className="p-4">
                    <h3 className="font-medium text-sm line-clamp-2 mb-3">{video.title}</h3>
                    
                    {/* Core YouTube Stats */}
                    <div className="grid grid-cols-2 gap-2 text-xs text-muted-foreground mb-3">
                      <div className="flex items-center gap-1">
                        <Eye className="h-3 w-3" />
                        <span>{stats?.viewCount ? formatNumber(parseInt(stats.viewCount)) : 'N/A'}</span>
                      </div>
                      <div className="flex items-center gap-1">
                        <ThumbsUp className="h-3 w-3" />
                        <span>{stats?.likeCount ? formatNumber(parseInt(stats.likeCount)) : 'N/A'}</span>
                      </div>
                      <div className="flex items-center gap-1">
                        <MessageSquare className="h-3 w-3" />
                        <span>{stats?.commentCount ? formatNumber(parseInt(stats.commentCount)) : 'N/A'}</span>
                      </div>
                      <div className="flex items-center gap-1">
                        <Star className="h-3 w-3" />
                        <span>{stats?.favoriteCount ? formatNumber(parseInt(stats.favoriteCount)) : 'N/A'}</span>
                      </div>
                    </div>

                    {/* Comprehensive YouTube Data - Expandable */}
                    {stats && (
                      <details className="mb-3">
                        <summary className="text-xs font-medium text-muted-foreground cursor-pointer hover:text-foreground transition-colors">
                          📊 Complete YouTube Data
                        </summary>
                        <div className="mt-2 space-y-3 text-xs">
                          
                          {/* Video Details */}
                          <div className="bg-blue-50 dark:bg-blue-950 rounded-lg p-2">
                            <h5 className="font-medium text-blue-900 dark:text-blue-100 mb-1">Video Details</h5>
                            <div className="grid grid-cols-2 gap-1 text-blue-800 dark:text-blue-200">
                              <div>Status: <span className="font-medium">{stats.privacyStatus}</span></div>
                              <div>Upload: <span className="font-medium">{stats.uploadStatus}</span></div>
                              <div>Definition: <span className="font-medium">{stats.definition?.toUpperCase()}</span></div>
                              <div>Dimension: <span className="font-medium">{stats.dimension}</span></div>
                              <div>Captions: <span className="font-medium">{stats.caption === 'true' ? 'Yes' : 'No'}</span></div>
                              <div>Embeddable: <span className="font-medium">{stats.embeddable ? 'Yes' : 'No'}</span></div>
                              <div>Made for Kids: <span className="font-medium">{stats.madeForKids ? 'Yes' : 'No'}</span></div>
                              <div>Licensed: <span className="font-medium">{stats.licensedContent ? 'Yes' : 'No'}</span></div>
                            </div>
                            {stats.publishedAt && (
                              <div className="mt-1 text-blue-700 dark:text-blue-300">
                                Published: {new Date(stats.publishedAt).toLocaleDateString()} at {new Date(stats.publishedAt).toLocaleTimeString()}
                              </div>
                            )}
                          </div>

                          {/* Technical Info */}
                          <div className="bg-green-50 dark:bg-green-950 rounded-lg p-2">
                            <h5 className="font-medium text-green-900 dark:text-green-100 mb-1">Technical Info</h5>
                            <div className="grid grid-cols-2 gap-1 text-green-800 dark:text-green-200">
                              <div>License: <span className="font-medium">{stats.license}</span></div>
                              <div>Projection: <span className="font-medium">{stats.projection}</span></div>
                              <div>Live Content: <span className="font-medium">{stats.liveBroadcastContent}</span></div>
                              <div>Public Stats: <span className="font-medium">{stats.publicStatsViewable ? 'Yes' : 'No'}</span></div>
                            </div>
                            {stats.defaultLanguage && (
                              <div className="mt-1 text-green-700 dark:text-green-300">
                                Language: {stats.defaultLanguage}
                                {stats.defaultAudioLanguage && stats.defaultAudioLanguage !== stats.defaultLanguage && 
                                  ` | Audio: ${stats.defaultAudioLanguage}`}
                              </div>
                            )}
                          </div>

                          {/* Channel Info */}
                          <div className="bg-purple-50 dark:bg-purple-950 rounded-lg p-2">
                            <h5 className="font-medium text-purple-900 dark:text-purple-100 mb-1">Channel Info</h5>
                            <div className="text-purple-800 dark:text-purple-200">
                              <div>Channel: <span className="font-medium">{stats.channelTitle}</span></div>
                              <div className="text-xs text-purple-600 dark:text-purple-400 mt-1">ID: {stats.channelId}</div>
                            </div>
                          </div>

                          {/* Video Tags */}
                          {stats.tags && stats.tags.length > 0 && (
                            <div className="bg-orange-50 dark:bg-orange-950 rounded-lg p-2">
                              <h5 className="font-medium text-orange-900 dark:text-orange-100 mb-1">Tags ({stats.tags.length})</h5>
                              <div className="flex flex-wrap gap-1">
                                {stats.tags.slice(0, 6).map((tag: string, idx: number) => (
                                  <span key={idx} className="bg-orange-200 dark:bg-orange-800 text-orange-800 dark:text-orange-200 px-1 py-0.5 rounded text-xs">
                                    {tag}
                                  </span>
                                ))}
                                {stats.tags.length > 6 && (
                                  <span className="text-orange-600 dark:text-orange-400 text-xs">
                                    +{stats.tags.length - 6} more
                                  </span>
                                )}
                              </div>
                            </div>
                          )}

                          {/* Topic Categories */}
                          {stats.topicCategories && stats.topicCategories.length > 0 && (
                            <div className="bg-indigo-50 dark:bg-indigo-950 rounded-lg p-2">
                              <h5 className="font-medium text-indigo-900 dark:text-indigo-100 mb-1">Topics</h5>
                              <div className="text-indigo-800 dark:text-indigo-200">
                                {stats.topicCategories.map((topic: string, idx: number) => (
                                  <div key={idx} className="text-xs">
                                    {topic.replace('https://en.wikipedia.org/wiki/', '').replace(/_/g, ' ')}
                                  </div>
                                ))}
                              </div>
                            </div>
                          )}

                          {/* Recording Date */}
                          {stats.recordingDate && (
                            <div className="bg-gray-50 dark:bg-gray-900 rounded-lg p-2">
                              <h5 className="font-medium text-gray-900 dark:text-gray-100 mb-1">Recording Date</h5>
                              <div className="text-gray-800 dark:text-gray-200 text-xs">
                                {new Date(stats.recordingDate).toLocaleDateString()}
                              </div>
                            </div>
                          )}

                          {/* Video Description Preview */}
                          {stats.description && (
                            <div className="bg-slate-50 dark:bg-slate-900 rounded-lg p-2">
                              <h5 className="font-medium text-slate-900 dark:text-slate-100 mb-1">Description</h5>
                              <div className="text-slate-800 dark:text-slate-200 text-xs line-clamp-3">
                                {stats.description}
                              </div>
                            </div>
                          )}

                        </div>
                      </details>
                    )}

                    {/* Detailed Order History */}
                    {videoOrders.length > 0 && (
                      <div className="space-y-2 mb-3">
                        <p className="text-xs font-medium text-muted-foreground">Order History ({videoOrders.length}):</p>
                        {videoOrders.map((order: any, index: number) => (
                          <div key={index} className="bg-gray-50 rounded-lg p-2 space-y-1">
                            <div className="flex justify-between items-center">
                              <span className="text-xs font-medium text-gray-900">
                                Order #{order.orderId}
                              </span>
                              <Badge 
                                variant={
                                  order.status === 'Completed' ? 'default' : 
                                  order.status === 'In progress' ? 'secondary' : 
                                  order.status === 'Processing' ? 'secondary' :
                                  order.status === 'Pending' ? 'outline' : 'destructive'
                                } 
                                className="text-xs"
                              >
                                {order.status}
                              </Badge>
                            </div>
                            
                            <div className="grid grid-cols-2 gap-1 text-xs text-gray-600">
                              <div>Service: {getServiceTypeDisplay(order.service)}</div>
                              <div>Qty: {order.quantity.toLocaleString()}</div>
                              <div>Cost: ${parseFloat(order.charge || '0').toFixed(4)}</div>
                              <div>Remaining: {order.remains || '0'}</div>
                            </div>
                            
                            {order.startCount && (
                              <div className="text-xs text-gray-500">
                                Start Count: {parseInt(order.startCount).toLocaleString()}
                              </div>
                            )}
                            
                            <div className="text-xs text-gray-400">
                              Created: {new Date(order.createdAt).toLocaleDateString()}
                              {order.updatedAt !== order.createdAt && (
                                <span> • Updated: {new Date(order.updatedAt).toLocaleDateString()}</span>
                              )}
                            </div>
                          </div>
                        ))}
                      </div>
                    )}

                    {/* Action Buttons */}
                    <div className="flex flex-col sm:flex-row gap-2">
                      <Button
                        size="sm"
                        variant="outline"
                        onClick={() => {
                          setSelectedVideo(video.id);
                          setLoadViewsModal(true);
                        }}
                        className="flex-1 text-xs"
                      >
                        Add Views
                      </Button>
                      <Button
                        size="sm"
                        variant="outline"
                        onClick={() => {
                          setSelectedVideo(video.id);
                          setLikesDialogOpen(true);
                        }}
                        className="flex-1 text-xs"
                      >
                        Add Likes
                      </Button>
                      {user?.role === "admin" && (
                        <Button
                          size="sm"
                          variant="outline"
                          onClick={() => {
                            setSelectedVideo(video.id);
                            setAdminLoadViewsModal(true);
                          }}
                          className="text-xs"
                        >
                          <MoreVertical className="h-3 w-3" />
                        </Button>
                      )}
                    </div>
                  </CardContent>
                </Card>
              );
            })}
          </div>
        </div>
      </CardContent>
    </Card>
  );
}