import { createContext, useContext, useEffect, useState } from "react";
import { useQuery, useMutation, useQueryClient } from "@tanstack/react-query";
import { useLocation } from "wouter";
import { apiRequest, getQueryFn } from "@/lib/queryClient";

interface User {
  id: number;
  username: string;
  email: string | null;
  role: string;
  accountType?: string;
  subscriptionTier?: string;
  mobileNumber?: string | null;
  country?: string | null;
}

interface AuthContextType {
  user: User | null;
  isLoading: boolean;
  login: (emailOrUsername: string, password: string) => Promise<void>;
  logout: () => void;
  isAuthenticated: boolean;
}

const AuthContext = createContext<AuthContextType | undefined>(undefined);

export function AuthProvider({ children }: { children: React.ReactNode }) {
  const [user, setUser] = useState<User | null>(null);
  const [, setLocation] = useLocation();
  const queryClient = useQueryClient();

  // Check if user is authenticated
  const { data: authData, isLoading } = useQuery({
    queryKey: ["/api/me"],
    queryFn: getQueryFn({ on401: "returnNull" }),
    retry: false,
    staleTime: 5 * 60 * 1000, // 5 minutes
  });

  useEffect(() => {
    if (authData) {
      setUser(authData as User);
    } else {
      setUser(null);
    }
  }, [authData]);

  const { mutate: loginMutation } = useMutation({
    mutationFn: async ({ emailOrUsername, password }: { emailOrUsername: string; password: string }) => {
      const response = await apiRequest("/api/login", "POST", { emailOrUsername, password });
      return response; // apiRequest already returns parsed JSON
    },
    onSuccess: (data) => {
      setUser(data.user);
      queryClient.invalidateQueries({ queryKey: ["/api/me"] });
    },
  });

  const { mutate: logoutMutation } = useMutation({
    mutationFn: async () => {
      const response = await apiRequest("/api/logout", "POST");
      return response; // apiRequest already returns parsed JSON
    },
    onSuccess: () => {
      setUser(null);
      queryClient.clear();
      setLocation("/");
    },
  });

  const login = async (emailOrUsername: string, password: string) => {
    return new Promise<void>((resolve, reject) => {
      loginMutation(
        { emailOrUsername, password },
        {
          onSuccess: () => resolve(),
          onError: (error) => reject(error),
        }
      );
    });
  };

  const logout = () => {
    logoutMutation();
  };

  const value: AuthContextType = {
    user,
    isLoading,
    login,
    logout,
    isAuthenticated: !!user,
  };

  return <AuthContext.Provider value={value}>{children}</AuthContext.Provider>;
}

export function useAuth() {
  const context = useContext(AuthContext);
  if (context === undefined) {
    throw new Error("useAuth must be used within an AuthProvider");
  }
  return context;
}