import { useQuery } from "@tanstack/react-query";
import { apiRequest } from "@/lib/queryClient";

export interface YouTubeVideoStats {
  videoId: string;
  likeCount: string;
  viewCount: string;
  commentCount: string;
  publishedAt: string;
  duration: string;
  durationSeconds: number;
}

export function useYouTubeStats(videoIds: string[], enabled = true) {
  return useQuery({
    queryKey: ["youtube-stats", videoIds, Date.now()], // Add timestamp for cache busting
    queryFn: async () => {
      if (!videoIds || videoIds.length === 0) return [];
      
      const response = await apiRequest("/api/youtube-stats", "POST", {
        videoIds
      });
      
      return await response.json() as YouTubeVideoStats[];
    },
    enabled: enabled && videoIds && videoIds.length > 0,
    staleTime: 0, // Always consider data stale to ensure fresh fetches
    cacheTime: 1 * 60 * 1000, // Keep cache for 1 minute only
    refetchInterval: 5 * 60 * 1000, // Refresh every 5 minutes for live data
    refetchOnMount: true, // Always refetch when component mounts
    refetchOnWindowFocus: true, // Refetch when user returns to window
  });
}

export function formatYouTubeNumber(numStr: string): string {
  const num = parseInt(numStr) || 0;
  
  if (num >= 1000000) {
    return (num / 1000000).toFixed(1) + 'M';
  } else if (num >= 1000) {
    return (num / 1000).toFixed(1) + 'K';
  }
  
  return num.toString();
}