import { QueryClient, QueryFunction } from "@tanstack/react-query";
import { errorLogger, logApiError, validatePlatformApi } from "./error-logger";

async function throwIfResNotOk(res: Response) {
  if (!res.ok) {
    const text = (await res.text()) || res.statusText;
    throw new Error(`${res.status}: ${text}`);
  }
}

export async function apiRequest(
  url: string,
  method: string,
  data?: unknown | undefined,
): Promise<any> {
  try {
    // Get current platform for validation
    const currentPlatform = localStorage.getItem('selectedPlatform') || 'youtube';
    
    // Validate platform-specific API calls
    if (!validatePlatformApi(url, currentPlatform)) {
      errorLogger.warning(`Platform mismatch: ${method} ${url} called on ${currentPlatform} platform`, {
        apiEndpoint: url,
        httpMethod: method,
        platform: currentPlatform,
        action: 'blocked-cross-platform-call'
      });
      // Return empty array for blocked cross-platform calls
      return [];
    }

    const res = await fetch(url, {
      method,
      headers: data ? { "Content-Type": "application/json" } : {},
      body: data ? JSON.stringify(data) : undefined,
      credentials: "include",
    });

    if (!res.ok) {
      const text = (await res.text()) || res.statusText;
      const error = new Error(`${res.status}: ${text}`);
      logApiError(url, method, error, {
        statusCode: res.status,
        platform: currentPlatform
      });
      throw error;
    }

    errorLogger.debug(`API request successful: ${method} ${url}`, {
      apiEndpoint: url,
      httpMethod: method,
      statusCode: res.status,
      platform: currentPlatform
    });

    // Parse and return JSON data
    const contentType = res.headers.get('content-type') || '';
    
    if (!contentType.includes('application/json')) {
      const text = await res.text();
      throw new Error(`Server returned non-JSON response: ${text.substring(0, 100)}...`);
    }
    
    try {
      return await res.json();
    } catch (parseError: any) {
      throw new Error(`Failed to parse JSON response from ${url}: ${parseError.message}`);
    }
  } catch (error: any) {
    logApiError(url, method, error, {
      platform: localStorage.getItem('selectedPlatform') || 'youtube'
    });
    throw error;
  }
}

type UnauthorizedBehavior = "returnNull" | "throw";
export const getQueryFn: <T>(options: {
  on401: UnauthorizedBehavior;
}) => QueryFunction<T> =
  ({ on401: unauthorizedBehavior }) =>
  async ({ queryKey }) => {
    const res = await fetch(queryKey[0] as string, {
      credentials: "include",
    });

    if (unauthorizedBehavior === "returnNull" && res.status === 401) {
      return null;
    }

    await throwIfResNotOk(res);
    return await res.json();
  };

export const queryClient = new QueryClient({
  defaultOptions: {
    queries: {
      queryFn: getQueryFn({ on401: "throw" }),
      refetchInterval: false,
      refetchOnWindowFocus: false,
      staleTime: Infinity,
      retry: false,
    },
    mutations: {
      retry: false,
    },
  },
});
