import { useState } from "react";
import { useQuery, useMutation, useQueryClient } from "@tanstack/react-query";
import { apiRequest } from "@/lib/queryClient";
import { useToast } from "@/hooks/use-toast";
import { Settings, Users, Coins, Save, RefreshCw, Calendar, TrendingUp, Edit3, Check, X, BarChart3, Shield, Database, History, Home } from "lucide-react";

interface User {
  id: number;
  username: string;
  email: string;
  role: string;
  points: {
    total: number;
    used: number;
    remaining: number;
    isCustom: boolean;
  } | null;
}

interface PointSettings {
  defaultDailyPoints: number;
}

export default function AdminPoints() {
  const { toast } = useToast();
  const queryClient = useQueryClient();
  const [editingUserId, setEditingUserId] = useState<number | null>(null);
  const [editingPoints, setEditingPoints] = useState<string>("");
  const [newDefaultPoints, setNewDefaultPoints] = useState<string>("");
  const [activeSection, setActiveSection] = useState<string>("overview");

  // Fetch point settings
  const { data: settings, isLoading: settingsLoading } = useQuery<PointSettings>({
    queryKey: ["/api/admin/points/settings"],
  });

  // Fetch users with their points
  const { data: users = [], isLoading: usersLoading } = useQuery<User[]>({
    queryKey: ["/api/admin/points/users"],
    refetchInterval: 30000,
  });

  // Update settings mutation
  const updateSettingsMutation = useMutation({
    mutationFn: async (data: { defaultDailyPoints: number }) => {
      const response = await fetch("/api/admin/points/settings", {
        method: "POST",
        body: JSON.stringify(data),
        headers: { "Content-Type": "application/json" },
        credentials: "include"
      });
      if (!response.ok) {
        const errorData = await response.json();
        throw new Error(errorData.message || "Failed to update settings");
      }
      return response.json();
    },
    onSuccess: () => {
      toast({
        title: "Settings Updated",
        description: "Default daily points have been updated successfully.",
      });
      queryClient.invalidateQueries({ queryKey: ["/api/admin/points/settings"] });
      queryClient.invalidateQueries({ queryKey: ["/api/admin/points/users"] });
      setNewDefaultPoints("");
    },
    onError: (error: Error) => {
      toast({
        title: "Error",
        description: error.message,
        variant: "destructive",
      });
    },
  });

  // Update user points mutation
  const updateUserPointsMutation = useMutation({
    mutationFn: async (data: { userId: number; points: number }) => {
      const response = await fetch(`/api/admin/points/users/${data.userId}`, {
        method: "POST",
        body: JSON.stringify({ points: data.points }),
        headers: { "Content-Type": "application/json" },
        credentials: "include"
      });
      if (!response.ok) {
        const errorData = await response.json();
        throw new Error(errorData.message || "Failed to update user points");
      }
      return response.json();
    },
    onSuccess: () => {
      toast({
        title: "User Points Updated",
        description: "User points have been updated successfully.",
      });
      queryClient.invalidateQueries({ queryKey: ["/api/admin/points/users"] });
      setEditingUserId(null);
      setEditingPoints("");
    },
    onError: (error: Error) => {
      toast({
        title: "Error",
        description: error.message,
        variant: "destructive",
      });
    },
  });

  // Daily reset mutation
  const dailyResetMutation = useMutation({
    mutationFn: async () => {
      const response = await fetch("/api/admin/points/reset-daily", {
        method: "POST",
        headers: { "Content-Type": "application/json" },
        credentials: "include"
      });
      if (!response.ok) {
        const errorData = await response.json();
        throw new Error(errorData.message || "Failed to reset daily points");
      }
      return response.json();
    },
    onSuccess: (data: any) => {
      toast({
        title: "Daily Reset Complete",
        description: `Points reset for ${data.resetCount} users with ${data.defaultPoints} points each.`,
      });
      queryClient.invalidateQueries({ queryKey: ["/api/admin/points/users"] });
    },
    onError: (error: Error) => {
      toast({
        title: "Error",
        description: error.message,
        variant: "destructive",
      });
    },
  });

  const handleUpdateSettings = () => {
    const points = parseInt(newDefaultPoints);
    if (isNaN(points) || points < 0) {
      toast({
        title: "Invalid Input",
        description: "Please enter a valid number of points.",
        variant: "destructive",
      });
      return;
    }
    updateSettingsMutation.mutate({ defaultDailyPoints: points });
  };

  const handleEditUser = (userId: number, currentPoints: number) => {
    setEditingUserId(userId);
    setEditingPoints(currentPoints.toString());
  };

  const handleSaveUserPoints = () => {
    const points = parseInt(editingPoints);
    if (isNaN(points) || points < 0) {
      toast({
        title: "Invalid Input",
        description: "Please enter a valid number of points.",
        variant: "destructive",
      });
      return;
    }
    if (editingUserId) {
      updateUserPointsMutation.mutate({ userId: editingUserId, points });
    }
  };

  const handleCancelEdit = () => {
    setEditingUserId(null);
    setEditingPoints("");
  };

  const totalUsers = users.length;
  const usersWithPoints = users.filter(user => user.points !== null).length;
  const totalPointsAllocated = users.reduce((sum, user) => sum + (user.points?.total || 0), 0);
  const totalPointsUsed = users.reduce((sum, user) => sum + (user.points?.used || 0), 0);

  const sidebarItems = [
    { id: "overview", label: "Overview", icon: BarChart3 },
    { id: "settings", label: "System Settings", icon: Settings },
    { id: "users", label: "User Management", icon: Users },
    { id: "history", label: "Point History", icon: History },
    { id: "analytics", label: "Analytics", icon: TrendingUp },
  ];

  if (settingsLoading || usersLoading) {
    return (
      <div className="flex items-center justify-center h-64">
        <RefreshCw className="w-8 h-8 animate-spin text-blue-600" />
      </div>
    );
  }

  const renderOverviewSection = () => (
    <div className="space-y-8">
      {/* Statistics Cards */}
      <div className="grid grid-cols-1 md:grid-cols-4 gap-4">
        <div className="bg-white dark:bg-gray-800 p-6 rounded-lg border border-gray-200 dark:border-gray-700">
          <div className="flex items-center space-x-3">
            <div className="p-2 bg-blue-100 dark:bg-blue-900 rounded-lg">
              <Users className="w-5 h-5 text-blue-600 dark:text-blue-400" />
            </div>
            <div>
              <p className="text-sm text-gray-600 dark:text-gray-400">Total Users</p>
              <p className="text-2xl font-bold text-gray-900 dark:text-white">{totalUsers}</p>
            </div>
          </div>
        </div>

        <div className="bg-white dark:bg-gray-800 p-6 rounded-lg border border-gray-200 dark:border-gray-700">
          <div className="flex items-center space-x-3">
            <div className="p-2 bg-green-100 dark:bg-green-900 rounded-lg">
              <Coins className="w-5 h-5 text-green-600 dark:text-green-400" />
            </div>
            <div>
              <p className="text-sm text-gray-600 dark:text-gray-400">Users with Points</p>
              <p className="text-2xl font-bold text-gray-900 dark:text-white">{usersWithPoints}</p>
            </div>
          </div>
        </div>

        <div className="bg-white dark:bg-gray-800 p-6 rounded-lg border border-gray-200 dark:border-gray-700">
          <div className="flex items-center space-x-3">
            <div className="p-2 bg-purple-100 dark:bg-purple-900 rounded-lg">
              <TrendingUp className="w-5 h-5 text-purple-600 dark:text-purple-400" />
            </div>
            <div>
              <p className="text-sm text-gray-600 dark:text-gray-400">Points Allocated</p>
              <p className="text-2xl font-bold text-gray-900 dark:text-white">{totalPointsAllocated.toLocaleString()}</p>
            </div>
          </div>
        </div>

        <div className="bg-white dark:bg-gray-800 p-6 rounded-lg border border-gray-200 dark:border-gray-700">
          <div className="flex items-center space-x-3">
            <div className="p-2 bg-orange-100 dark:bg-orange-900 rounded-lg">
              <TrendingUp className="w-5 h-5 text-orange-600 dark:text-orange-400" />
            </div>
            <div>
              <p className="text-sm text-gray-600 dark:text-gray-400">Points Used</p>
              <p className="text-2xl font-bold text-gray-900 dark:text-white">{totalPointsUsed.toLocaleString()}</p>
            </div>
          </div>
        </div>
      </div>
    </div>
  );

  const renderSettingsSection = () => (
    <div className="bg-white dark:bg-gray-800 rounded-lg border border-gray-200 dark:border-gray-700">
      <div className="p-6 border-b border-gray-200 dark:border-gray-700">
        <div className="flex items-center space-x-2">
          <Settings className="w-5 h-5 text-gray-600 dark:text-gray-400" />
          <h2 className="text-lg font-semibold text-gray-900 dark:text-white">System Settings</h2>
        </div>
      </div>
      <div className="p-6">
        <div className="space-y-4">
          <div>
            <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
              Default Daily Points
            </label>
            <div className="flex space-x-3">
              <input
                type="number"
                value={newDefaultPoints || settings?.defaultDailyPoints || ""}
                onChange={(e) => setNewDefaultPoints(e.target.value)}
                placeholder={settings?.defaultDailyPoints?.toString()}
                className="flex-1 px-3 py-2 border border-gray-300 dark:border-gray-600 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent bg-white dark:bg-gray-700 text-gray-900 dark:text-white"
              />
              <button
                onClick={handleUpdateSettings}
                disabled={updateSettingsMutation.isPending || !newDefaultPoints}
                className="flex items-center space-x-2 px-4 py-2 bg-blue-600 text-white rounded-lg hover:bg-blue-700 disabled:opacity-50 transition-colors"
              >
                {updateSettingsMutation.isPending ? (
                  <RefreshCw className="w-4 h-4 animate-spin" />
                ) : (
                  <Save className="w-4 h-4" />
                )}
                <span>Update</span>
              </button>
            </div>
            <p className="text-sm text-gray-500 dark:text-gray-400 mt-1">
              Current default: {settings?.defaultDailyPoints || 600} points per day
            </p>
          </div>
        </div>
      </div>
    </div>
  );

  const renderUsersSection = () => (
    <div className="bg-white dark:bg-gray-800 rounded-lg border border-gray-200 dark:border-gray-700">
      <div className="p-6 border-b border-gray-200 dark:border-gray-700">
        <div className="flex items-center space-x-2">
          <Users className="w-5 h-5 text-gray-600 dark:text-gray-400" />
          <h2 className="text-lg font-semibold text-gray-900 dark:text-white">User Point Management</h2>
        </div>
      </div>
      <div className="overflow-x-auto">
        <table className="w-full">
          <thead className="bg-gray-50 dark:bg-gray-700">
            <tr>
              <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 dark:text-gray-400 uppercase tracking-wider">
                User
              </th>
              <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 dark:text-gray-400 uppercase tracking-wider">
                Role
              </th>
              <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 dark:text-gray-400 uppercase tracking-wider">
                Total Points
              </th>
              <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 dark:text-gray-400 uppercase tracking-wider">
                Used Points
              </th>
              <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 dark:text-gray-400 uppercase tracking-wider">
                Remaining
              </th>
              <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 dark:text-gray-400 uppercase tracking-wider">
                Type
              </th>
              <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 dark:text-gray-400 uppercase tracking-wider">
                Actions
              </th>
            </tr>
          </thead>
          <tbody className="divide-y divide-gray-200 dark:divide-gray-600">
            {users.map((user) => (
              <tr key={user.id} className="hover:bg-gray-50 dark:hover:bg-gray-700">
                <td className="px-6 py-4 whitespace-nowrap">
                  <div>
                    <div className="text-sm font-medium text-gray-900 dark:text-white">
                      {user.username}
                    </div>
                    <div className="text-sm text-gray-500 dark:text-gray-400">
                      {user.email}
                    </div>
                  </div>
                </td>
                <td className="px-6 py-4 whitespace-nowrap">
                  <span className={`inline-flex px-2 py-1 text-xs font-semibold rounded-full ${
                    user.role === 'admin' 
                      ? 'bg-purple-100 text-purple-800 dark:bg-purple-900 dark:text-purple-200'
                      : 'bg-green-100 text-green-800 dark:bg-green-900 dark:text-green-200'
                  }`}>
                    {user.role}
                  </span>
                </td>
                <td className="px-6 py-4 whitespace-nowrap">
                  {editingUserId === user.id ? (
                    <input
                      type="number"
                      value={editingPoints}
                      onChange={(e) => setEditingPoints(e.target.value)}
                      className="w-20 px-2 py-1 text-sm border border-gray-300 dark:border-gray-600 rounded focus:ring-2 focus:ring-blue-500 bg-white dark:bg-gray-700 text-gray-900 dark:text-white"
                    />
                  ) : (
                    <span className="text-sm text-gray-900 dark:text-white">
                      {user.points?.total?.toLocaleString() || "Not set"}
                    </span>
                  )}
                </td>
                <td className="px-6 py-4 whitespace-nowrap text-sm text-gray-900 dark:text-white">
                  {user.points?.used?.toLocaleString() || "0"}
                </td>
                <td className="px-6 py-4 whitespace-nowrap">
                  <span className={`text-sm font-medium ${
                    (user.points?.remaining || 0) < (user.points?.total || 0) * 0.2
                      ? 'text-red-600 dark:text-red-400'
                      : 'text-gray-900 dark:text-white'
                  }`}>
                    {user.points?.remaining?.toLocaleString() || "0"}
                  </span>
                </td>
                <td className="px-6 py-4 whitespace-nowrap">
                  <span className={`inline-flex px-2 py-1 text-xs font-semibold rounded-full ${
                    user.points?.isCustom
                      ? 'bg-yellow-100 text-yellow-800 dark:bg-yellow-900 dark:text-yellow-200'
                      : 'bg-gray-100 text-gray-800 dark:bg-gray-700 dark:text-gray-200'
                  }`}>
                    {user.points?.isCustom ? 'Custom' : 'Default'}
                  </span>
                </td>
                <td className="px-6 py-4 whitespace-nowrap">
                  {editingUserId === user.id ? (
                    <div className="flex space-x-2">
                      <button
                        onClick={handleSaveUserPoints}
                        disabled={updateUserPointsMutation.isPending}
                        className="text-green-600 hover:text-green-700 dark:text-green-400 dark:hover:text-green-300"
                      >
                        <Check className="w-4 h-4" />
                      </button>
                      <button
                        onClick={handleCancelEdit}
                        className="text-red-600 hover:text-red-700 dark:text-red-400 dark:hover:text-red-300"
                      >
                        <X className="w-4 h-4" />
                      </button>
                    </div>
                  ) : (
                    <button
                      onClick={() => handleEditUser(user.id, user.points?.total || 0)}
                      className="text-blue-600 hover:text-blue-700 dark:text-blue-400 dark:hover:text-blue-300"
                    >
                      <Edit3 className="w-4 h-4" />
                    </button>
                  )}
                </td>
              </tr>
            ))}
          </tbody>
        </table>
      </div>
    </div>
  );

  const renderComingSoonSection = (title: string) => (
    <div className="bg-white dark:bg-gray-800 rounded-lg border border-gray-200 dark:border-gray-700 p-12 text-center">
      <div className="p-4 bg-gray-100 dark:bg-gray-700 rounded-full inline-flex mb-4">
        <Database className="w-8 h-8 text-gray-400" />
      </div>
      <h3 className="text-xl font-semibold text-gray-900 dark:text-white mb-2">{title}</h3>
      <p className="text-gray-600 dark:text-gray-400">This feature is coming soon. Stay tuned for updates!</p>
    </div>
  );

  return (
    <div className="flex h-screen bg-gray-50 dark:bg-gray-900">
      {/* Sidebar */}
      <div className="w-64 bg-white dark:bg-gray-800 border-r border-gray-200 dark:border-gray-700 flex flex-col">
        {/* Sidebar Header */}
        <div className="p-6 border-b border-gray-200 dark:border-gray-700">
          <div className="flex items-center space-x-3">
            <div className="p-2 bg-blue-100 dark:bg-blue-900 rounded-lg">
              <Shield className="w-6 h-6 text-blue-600 dark:text-blue-400" />
            </div>
            <div>
              <h1 className="text-lg font-bold text-gray-900 dark:text-white">Admin Panel</h1>
              <p className="text-sm text-gray-600 dark:text-gray-400">Point Management</p>
            </div>
          </div>
        </div>

        {/* Navigation Items */}
        <nav className="flex-1 p-4 space-y-2">
          {sidebarItems.map((item) => {
            const Icon = item.icon;
            const isActive = activeSection === item.id;
            
            return (
              <button
                key={item.id}
                onClick={() => setActiveSection(item.id)}
                className={`w-full flex items-center space-x-3 px-4 py-3 rounded-lg transition-colors ${
                  isActive
                    ? 'bg-blue-100 dark:bg-blue-900 text-blue-700 dark:text-blue-300 border border-blue-200 dark:border-blue-700'
                    : 'text-gray-700 dark:text-gray-300 hover:bg-gray-100 dark:hover:bg-gray-700'
                }`}
              >
                <Icon className="w-5 h-5" />
                <span className="font-medium">{item.label}</span>
              </button>
            );
          })}
        </nav>

        {/* Back to Dashboard */}
        <div className="p-4 border-t border-gray-200 dark:border-gray-700">
          <a
            href="/"
            className="w-full flex items-center space-x-3 px-4 py-3 text-gray-700 dark:text-gray-300 hover:bg-gray-100 dark:hover:bg-gray-700 rounded-lg transition-colors"
          >
            <Home className="w-5 h-5" />
            <span className="font-medium">Back to Dashboard</span>
          </a>
        </div>
      </div>

      {/* Main Content */}
      <div className="flex-1 overflow-auto">
        <div className="p-8">
          {/* Header */}
          <div className="flex items-center justify-between mb-8">
            <div>
              <h1 className="text-3xl font-bold text-gray-900 dark:text-white">
                {sidebarItems.find(item => item.id === activeSection)?.label || "Point Management"}
              </h1>
              <p className="text-gray-600 dark:text-gray-400 mt-2">
                {activeSection === "overview" && "Monitor system-wide point allocation and usage"}
                {activeSection === "settings" && "Configure default point allocations and system preferences"}
                {activeSection === "users" && "Manage individual user point allocations and custom settings"}
                {activeSection === "history" && "View detailed point transaction history and audit logs"}
                {activeSection === "analytics" && "Analyze point usage patterns and system performance"}
              </p>
            </div>
            {activeSection === "overview" && (
              <button
                onClick={() => dailyResetMutation.mutate()}
                disabled={dailyResetMutation.isPending}
                className="flex items-center space-x-2 px-6 py-3 bg-blue-600 text-white rounded-lg hover:bg-blue-700 disabled:opacity-50 transition-colors shadow-sm"
              >
                {dailyResetMutation.isPending ? (
                  <RefreshCw className="w-5 h-5 animate-spin" />
                ) : (
                  <Calendar className="w-5 h-5" />
                )}
                <span>Reset Daily Points</span>
              </button>
            )}
          </div>

          {/* Content based on active section */}
          {activeSection === "overview" && renderOverviewSection()}
          {activeSection === "settings" && renderSettingsSection()}
          {activeSection === "users" && renderUsersSection()}
          {activeSection === "history" && renderComingSoonSection("Point History")}
          {activeSection === "analytics" && renderComingSoonSection("Analytics Dashboard")}
        </div>
      </div>
    </div>
  );
}