import { useState, useEffect } from "react";
import { useQuery, useMutation, useQueryClient } from "@tanstack/react-query";
import { apiRequest } from "@/lib/queryClient";
import { DashboardLayout } from "@/components/dashboard-layout";
import { Button } from "@/components/ui/button";
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card";
import { Badge } from "@/components/ui/badge";
import { Dialog, DialogContent, DialogDescription, DialogFooter, DialogHeader, DialogTitle, DialogTrigger } from "@/components/ui/dialog";
import { Form, FormControl, FormDescription, FormField, FormItem, FormLabel, FormMessage } from "@/components/ui/form";
import { Input } from "@/components/ui/input";
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select";
import { Switch } from "@/components/ui/switch";
import { Textarea } from "@/components/ui/textarea";
import { Tabs, TabsContent, TabsList, TabsTrigger } from "@/components/ui/tabs";
import { Table, TableBody, TableCell, TableHead, TableHeader, TableRow } from "@/components/ui/table";
import { useForm } from "react-hook-form";
import { zodResolver } from "@hookform/resolvers/zod";
import { z } from "zod";
import { Plus, Settings, Activity, AlertTriangle, CheckCircle, Trash2, Power } from "lucide-react";
import { useToast } from "@/hooks/use-toast";

// Form schema for AI configuration
const aiConfigSchema = z.object({
  name: z.string().min(1, "Name is required"),
  provider: z.enum(["anthropic", "openai", "azure"], {
    required_error: "Please select a provider",
  }),
  model: z.string().min(1, "Model is required"),
  apiEndpoint: z.string().optional(),
  maxTokens: z.number().min(1).max(8000).default(1000),
  temperature: z.number().min(0).max(2).default(0.3),
  topP: z.number().min(0).max(1).default(0.1),
});

type AiConfigFormData = z.infer<typeof aiConfigSchema>;

interface AiConfiguration {
  id: number;
  name: string;
  provider: string;
  model: string;
  apiEndpoint?: string;
  maxTokens: number;
  temperature: number;
  topP: number;
  isActive: boolean;
  isDefault: boolean;
  createdBy: number;
  createdAt: string;
  updatedAt: string;
}

interface AiErrorLog {
  id: number;
  configurationId?: number;
  provider: string;
  model: string;
  endpoint?: string;
  errorMessage: string;
  errorCode?: string;
  httpStatus?: number;
  requestDuration?: number;
  retryAttempt: number;
  severity: string;
  isResolved: boolean;
  resolvedBy?: number;
  resolvedAt?: string;
  createdAt: string;
}

interface AiUsageStats {
  id: number;
  configurationId: number;
  provider: string;
  model: string;
  requestCount: number;
  successCount: number;
  errorCount: number;
  totalTokensUsed: number;
  averageResponseTime: number;
  dateRecord: string;
  createdAt: string;
}

export default function AiConfigurationPage() {
  const [dialogOpen, setDialogOpen] = useState(false);
  const [editingConfig, setEditingConfig] = useState<AiConfiguration | null>(null);
  const [activeTab, setActiveTab] = useState("configurations");
  const { toast } = useToast();
  const queryClient = useQueryClient();

  // Fetch AI configurations
  const { data: configurations = [], isLoading: configsLoading } = useQuery({
    queryKey: ["/api/ai-configurations"],
    queryFn: () => apiRequest("/api/ai-configurations"),
  });

  // Fetch AI error logs
  const { data: errorLogs = [], isLoading: logsLoading } = useQuery({
    queryKey: ["/api/ai-error-logs"],
    queryFn: () => apiRequest("/api/ai-error-logs", "GET"),
    enabled: activeTab === "errors",
  });

  // Fetch AI usage stats
  const { data: usageStats = [], isLoading: statsLoading } = useQuery({
    queryKey: ["/api/ai-usage-stats"],
    queryFn: () => apiRequest("/api/ai-usage-stats", "GET"),
    enabled: activeTab === "analytics",
  });

  // Create configuration mutation
  const createConfigMutation = useMutation({
    mutationFn: (data: AiConfigFormData) => apiRequest("/api/ai-configurations", "POST", data),
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/ai-configurations"] });
      setDialogOpen(false);
      toast({ title: "Success", description: "AI configuration created successfully" });
    },
    onError: () => {
      toast({ title: "Error", description: "Failed to create AI configuration", variant: "destructive" });
    },
  });

  // Update configuration mutation
  const updateConfigMutation = useMutation({
    mutationFn: ({ id, data }: { id: number; data: Partial<AiConfigFormData> }) =>
      apiRequest(`/api/ai-configurations/${id}`, "PUT", data),
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/ai-configurations"] });
      setDialogOpen(false);
      setEditingConfig(null);
      toast({ title: "Success", description: "AI configuration updated successfully" });
    },
    onError: () => {
      toast({ title: "Error", description: "Failed to update AI configuration", variant: "destructive" });
    },
  });

  // Activate configuration mutation
  const activateConfigMutation = useMutation({
    mutationFn: (id: number) => apiRequest(`/api/ai-configurations/${id}/activate`, "PUT"),
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/ai-configurations"] });
      toast({ title: "Success", description: "AI configuration activated successfully" });
    },
    onError: () => {
      toast({ title: "Error", description: "Failed to activate AI configuration", variant: "destructive" });
    },
  });

  // Delete configuration mutation
  const deleteConfigMutation = useMutation({
    mutationFn: (id: number) => apiRequest(`/api/ai-configurations/${id}`, "DELETE"),
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/ai-configurations"] });
      toast({ title: "Success", description: "AI configuration deleted successfully" });
    },
    onError: () => {
      toast({ title: "Error", description: "Failed to delete AI configuration", variant: "destructive" });
    },
  });

  // Resolve error log mutation
  const resolveErrorMutation = useMutation({
    mutationFn: (id: number) => apiRequest(`/api/ai-error-logs/${id}/resolve`, "PUT"),
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/ai-error-logs"] });
      toast({ title: "Success", description: "Error marked as resolved" });
    },
    onError: () => {
      toast({ title: "Error", description: "Failed to resolve error", variant: "destructive" });
    },
  });

  const form = useForm<AiConfigFormData>({
    resolver: zodResolver(aiConfigSchema),
    defaultValues: {
      name: "",
      provider: "anthropic",
      model: "",
      apiEndpoint: "",
      maxTokens: 1000,
      temperature: 0.3,
      topP: 0.1,
    },
  });

  // Reset form when dialog opens/closes
  useEffect(() => {
    if (!dialogOpen) {
      form.reset();
      setEditingConfig(null);
    }
  }, [dialogOpen, form]);

  // Set form values when editing
  useEffect(() => {
    if (editingConfig) {
      form.reset({
        name: editingConfig.name,
        provider: editingConfig.provider as any,
        model: editingConfig.model,
        apiEndpoint: editingConfig.apiEndpoint || "",
        maxTokens: editingConfig.maxTokens,
        temperature: editingConfig.temperature,
        topP: editingConfig.topP,
      });
    }
  }, [editingConfig, form]);

  const onSubmit = (data: AiConfigFormData) => {
    if (editingConfig) {
      updateConfigMutation.mutate({ id: editingConfig.id, data });
    } else {
      createConfigMutation.mutate(data);
    }
  };

  const getProviderColor = (provider: string) => {
    switch (provider) {
      case "anthropic": return "bg-purple-100 text-purple-800";
      case "openai": return "bg-green-100 text-green-800";
      case "azure": return "bg-blue-100 text-blue-800";
      default: return "bg-gray-100 text-gray-800";
    }
  };

  const getSeverityColor = (severity: string) => {
    switch (severity) {
      case "critical": return "bg-red-100 text-red-800";
      case "error": return "bg-orange-100 text-orange-800";
      case "warning": return "bg-yellow-100 text-yellow-800";
      default: return "bg-gray-100 text-gray-800";
    }
  };

  const getModelOptions = (provider: string) => {
    switch (provider) {
      case "anthropic":
        return [
          { value: "claude-sonnet-4", label: "Claude Sonnet 4" },
          { value: "claude-opus-4", label: "Claude Opus 4" },
        ];
      case "openai":
        return [
          { value: "gpt-4o", label: "GPT-4o" },
          { value: "gpt-4", label: "GPT-4" },
        ];
      case "azure":
        return [
          { value: "gpt-4.1", label: "GPT-4.1" },
          { value: "gpt-4o", label: "GPT-4o" },
        ];
      default:
        return [];
    }
  };

  const selectedProvider = form.watch("provider");

  return (
    <DashboardLayout>
      <div className="container mx-auto p-6 space-y-6">
        <div className="flex justify-between items-center">
          <div>
            <h1 className="text-3xl font-bold">AI Configuration Management</h1>
            <p className="text-muted-foreground">
              Manage AI providers and models for Reddit moderation system
            </p>
          </div>
          <Dialog open={dialogOpen} onOpenChange={setDialogOpen}>
            <DialogTrigger asChild>
              <Button>
                <Plus className="h-4 w-4 mr-2" />
                Add Configuration
              </Button>
            </DialogTrigger>
            <DialogContent className="max-w-2xl">
              <DialogHeader>
                <DialogTitle>
                  {editingConfig ? "Edit AI Configuration" : "Create New AI Configuration"}
                </DialogTitle>
                <DialogDescription>
                  Configure AI provider settings for Reddit moderation analysis
                </DialogDescription>
              </DialogHeader>
              <Form {...form}>
                <form onSubmit={form.handleSubmit(onSubmit)} className="space-y-4">
                  <FormField
                    control={form.control}
                    name="name"
                    render={({ field }) => (
                      <FormItem>
                        <FormLabel>Configuration Name</FormLabel>
                        <FormControl>
                          <Input placeholder="e.g., Default Reddit Moderation" {...field} />
                        </FormControl>
                        <FormMessage />
                      </FormItem>
                    )}
                  />
                  
                  <FormField
                    control={form.control}
                    name="provider"
                    render={({ field }) => (
                      <FormItem>
                        <FormLabel>AI Provider</FormLabel>
                        <Select onValueChange={field.onChange} defaultValue={field.value}>
                          <FormControl>
                            <SelectTrigger>
                              <SelectValue placeholder="Select provider" />
                            </SelectTrigger>
                          </FormControl>
                          <SelectContent>
                            <SelectItem value="anthropic">Anthropic</SelectItem>
                            <SelectItem value="openai">OpenAI</SelectItem>
                            <SelectItem value="azure">Azure AI</SelectItem>
                          </SelectContent>
                        </Select>
                        <FormMessage />
                      </FormItem>
                    )}
                  />

                  <FormField
                    control={form.control}
                    name="model"
                    render={({ field }) => (
                      <FormItem>
                        <FormLabel>Model</FormLabel>
                        <Select onValueChange={field.onChange} value={field.value}>
                          <FormControl>
                            <SelectTrigger>
                              <SelectValue placeholder="Select model" />
                            </SelectTrigger>
                          </FormControl>
                          <SelectContent>
                            {getModelOptions(selectedProvider).map((option) => (
                              <SelectItem key={option.value} value={option.value}>
                                {option.label}
                              </SelectItem>
                            ))}
                          </SelectContent>
                        </Select>
                        <FormMessage />
                      </FormItem>
                    )}
                  />

                  {selectedProvider === "azure" && (
                    <FormField
                      control={form.control}
                      name="apiEndpoint"
                      render={({ field }) => (
                        <FormItem>
                          <FormLabel>API Endpoint</FormLabel>
                          <FormControl>
                            <Input 
                              placeholder="https://your-resource.cognitiveservices.azure.com/openai/deployments/..." 
                              {...field} 
                            />
                          </FormControl>
                          <FormDescription>
                            Azure OpenAI endpoint URL for your deployment
                          </FormDescription>
                          <FormMessage />
                        </FormItem>
                      )}
                    />
                  )}

                  <div className="grid grid-cols-3 gap-4">
                    <FormField
                      control={form.control}
                      name="maxTokens"
                      render={({ field }) => (
                        <FormItem>
                          <FormLabel>Max Tokens</FormLabel>
                          <FormControl>
                            <Input 
                              type="number" 
                              {...field} 
                              onChange={e => field.onChange(parseInt(e.target.value))}
                            />
                          </FormControl>
                          <FormMessage />
                        </FormItem>
                      )}
                    />

                    <FormField
                      control={form.control}
                      name="temperature"
                      render={({ field }) => (
                        <FormItem>
                          <FormLabel>Temperature</FormLabel>
                          <FormControl>
                            <Input 
                              type="number" 
                              step="0.1"
                              min="0"
                              max="2"
                              {...field} 
                              onChange={e => field.onChange(parseFloat(e.target.value))}
                            />
                          </FormControl>
                          <FormMessage />
                        </FormItem>
                      )}
                    />

                    <FormField
                      control={form.control}
                      name="topP"
                      render={({ field }) => (
                        <FormItem>
                          <FormLabel>Top P</FormLabel>
                          <FormControl>
                            <Input 
                              type="number" 
                              step="0.1"
                              min="0"
                              max="1"
                              {...field} 
                              onChange={e => field.onChange(parseFloat(e.target.value))}
                            />
                          </FormControl>
                          <FormMessage />
                        </FormItem>
                      )}
                    />
                  </div>

                  <DialogFooter>
                    <Button type="button" variant="outline" onClick={() => setDialogOpen(false)}>
                      Cancel
                    </Button>
                    <Button 
                      type="submit" 
                      disabled={createConfigMutation.isPending || updateConfigMutation.isPending}
                    >
                      {editingConfig ? "Update" : "Create"} Configuration
                    </Button>
                  </DialogFooter>
                </form>
              </Form>
            </DialogContent>
          </Dialog>
        </div>

        <Tabs value={activeTab} onValueChange={setActiveTab}>
          <TabsList>
            <TabsTrigger value="configurations">
              <Settings className="h-4 w-4 mr-2" />
              Configurations
            </TabsTrigger>
            <TabsTrigger value="errors">
              <AlertTriangle className="h-4 w-4 mr-2" />
              Error Logs
            </TabsTrigger>
            <TabsTrigger value="analytics">
              <Activity className="h-4 w-4 mr-2" />
              Usage Analytics
            </TabsTrigger>
          </TabsList>

          <TabsContent value="configurations" className="space-y-4">
            {configsLoading ? (
              <div className="text-center py-8">Loading configurations...</div>
            ) : configurations.length === 0 ? (
              <Card>
                <CardContent className="text-center py-8">
                  <p className="text-muted-foreground">No AI configurations found.</p>
                  <Button 
                    onClick={() => setDialogOpen(true)} 
                    className="mt-4"
                  >
                    Create First Configuration
                  </Button>
                </CardContent>
              </Card>
            ) : (
              <div className="grid gap-4">
                {configurations.map((config: AiConfiguration) => (
                  <Card key={config.id} className={config.isActive ? "border-green-500" : ""}>
                    <CardHeader>
                      <div className="flex justify-between items-start">
                        <div>
                          <CardTitle className="flex items-center gap-2">
                            {config.name}
                            {config.isActive && (
                              <Badge variant="default" className="bg-green-500">
                                Active
                              </Badge>
                            )}
                          </CardTitle>
                          <CardDescription>
                            <Badge className={getProviderColor(config.provider)}>
                              {config.provider}
                            </Badge>
                            <span className="ml-2">{config.model}</span>
                          </CardDescription>
                        </div>
                        <div className="flex gap-2">
                          {!config.isActive && (
                            <Button
                              size="sm"
                              variant="outline"
                              onClick={() => activateConfigMutation.mutate(config.id)}
                              disabled={activateConfigMutation.isPending}
                            >
                              <Power className="h-4 w-4 mr-2" />
                              Activate
                            </Button>
                          )}
                          <Button
                            size="sm"
                            variant="outline"
                            onClick={() => {
                              setEditingConfig(config);
                              setDialogOpen(true);
                            }}
                          >
                            Edit
                          </Button>
                          {!config.isActive && (
                            <Button
                              size="sm"
                              variant="destructive"
                              onClick={() => deleteConfigMutation.mutate(config.id)}
                              disabled={deleteConfigMutation.isPending}
                            >
                              <Trash2 className="h-4 w-4" />
                            </Button>
                          )}
                        </div>
                      </div>
                    </CardHeader>
                    <CardContent>
                      <div className="grid grid-cols-3 gap-4 text-sm">
                        <div>
                          <span className="font-medium">Max Tokens:</span> {config.maxTokens}
                        </div>
                        <div>
                          <span className="font-medium">Temperature:</span> {config.temperature}
                        </div>
                        <div>
                          <span className="font-medium">Top P:</span> {config.topP}
                        </div>
                      </div>
                      {config.apiEndpoint && (
                        <div className="mt-2 text-sm">
                          <span className="font-medium">Endpoint:</span> {config.apiEndpoint}
                        </div>
                      )}
                    </CardContent>
                  </Card>
                ))}
              </div>
            )}
          </TabsContent>

          <TabsContent value="errors" className="space-y-4">
            {logsLoading ? (
              <div className="text-center py-8">Loading error logs...</div>
            ) : errorLogs.length === 0 ? (
              <Card>
                <CardContent className="text-center py-8">
                  <CheckCircle className="h-12 w-12 text-green-500 mx-auto mb-4" />
                  <p className="text-muted-foreground">No error logs found. System is running smoothly!</p>
                </CardContent>
              </Card>
            ) : (
              <Card>
                <CardHeader>
                  <CardTitle>AI Error Logs</CardTitle>
                  <CardDescription>
                    Monitor and resolve AI system errors
                  </CardDescription>
                </CardHeader>
                <CardContent>
                  <Table>
                    <TableHeader>
                      <TableRow>
                        <TableHead>Date</TableHead>
                        <TableHead>Provider/Model</TableHead>
                        <TableHead>Error</TableHead>
                        <TableHead>Severity</TableHead>
                        <TableHead>Status</TableHead>
                        <TableHead>Actions</TableHead>
                      </TableRow>
                    </TableHeader>
                    <TableBody>
                      {errorLogs.map((log: AiErrorLog) => (
                        <TableRow key={log.id}>
                          <TableCell>
                            {new Date(log.createdAt).toLocaleString()}
                          </TableCell>
                          <TableCell>
                            <div className="flex flex-col">
                              <Badge className={getProviderColor(log.provider)} size="sm">
                                {log.provider}
                              </Badge>
                              <span className="text-sm text-muted-foreground">{log.model}</span>
                            </div>
                          </TableCell>
                          <TableCell className="max-w-xs">
                            <div className="truncate" title={log.errorMessage}>
                              {log.errorMessage}
                            </div>
                            {log.httpStatus && (
                              <span className="text-xs text-muted-foreground">
                                HTTP {log.httpStatus}
                              </span>
                            )}
                          </TableCell>
                          <TableCell>
                            <Badge className={getSeverityColor(log.severity)}>
                              {log.severity}
                            </Badge>
                          </TableCell>
                          <TableCell>
                            {log.isResolved ? (
                              <Badge variant="outline" className="text-green-600">
                                Resolved
                              </Badge>
                            ) : (
                              <Badge variant="destructive">
                                Open
                              </Badge>
                            )}
                          </TableCell>
                          <TableCell>
                            {!log.isResolved && (
                              <Button
                                size="sm"
                                variant="outline"
                                onClick={() => resolveErrorMutation.mutate(log.id)}
                                disabled={resolveErrorMutation.isPending}
                              >
                                Mark Resolved
                              </Button>
                            )}
                          </TableCell>
                        </TableRow>
                      ))}
                    </TableBody>
                  </Table>
                </CardContent>
              </Card>
            )}
          </TabsContent>

          <TabsContent value="analytics" className="space-y-4">
            {statsLoading ? (
              <div className="text-center py-8">Loading usage statistics...</div>
            ) : usageStats.length === 0 ? (
              <Card>
                <CardContent className="text-center py-8">
                  <Activity className="h-12 w-12 text-blue-500 mx-auto mb-4" />
                  <p className="text-muted-foreground">No usage data available yet.</p>
                </CardContent>
              </Card>
            ) : (
              <div className="grid gap-4">
                <div className="grid grid-cols-1 md:grid-cols-4 gap-4">
                  {usageStats.map((stat: AiUsageStats) => {
                    const successRate = stat.requestCount > 0 ? 
                      ((stat.successCount / stat.requestCount) * 100).toFixed(1) : "0";
                    
                    return (
                      <Card key={stat.id}>
                        <CardHeader className="pb-2">
                          <CardTitle className="text-lg">
                            {new Date(stat.dateRecord).toLocaleDateString()}
                          </CardTitle>
                          <CardDescription>
                            <Badge className={getProviderColor(stat.provider)}>
                              {stat.provider}
                            </Badge>
                          </CardDescription>
                        </CardHeader>
                        <CardContent>
                          <div className="space-y-2 text-sm">
                            <div className="flex justify-between">
                              <span>Requests:</span>
                              <span className="font-medium">{stat.requestCount}</span>
                            </div>
                            <div className="flex justify-between">
                              <span>Success Rate:</span>
                              <span className="font-medium">{successRate}%</span>
                            </div>
                            <div className="flex justify-between">
                              <span>Tokens Used:</span>
                              <span className="font-medium">{stat.totalTokensUsed.toLocaleString()}</span>
                            </div>
                            <div className="flex justify-between">
                              <span>Avg Response:</span>
                              <span className="font-medium">{stat.averageResponseTime.toFixed(0)}ms</span>
                            </div>
                          </div>
                        </CardContent>
                      </Card>
                    );
                  })}
                </div>
              </div>
            )}
          </TabsContent>
        </Tabs>
      </div>
    </DashboardLayout>
  );
}