import { useState } from "react";
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card";
import { Badge } from "@/components/ui/badge";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import { useMutation } from "@tanstack/react-query";
import { useToast } from "@/hooks/use-toast";
import { apiRequest } from "@/lib/queryClient";
import { Settings, Key, Shield, ExternalLink, CheckCircle, AlertCircle } from "lucide-react";
import DashboardLayout from "@/components/dashboard-layout";

export default function ApiServices() {
  const [smmcostKey, setSmmcostKey] = useState("");
  const { toast } = useToast();

  const { mutate: updateSmmcostKey, isPending: isUpdatingSmmcost } = useMutation({
    mutationFn: async (newKey: string) => {
      return await apiRequest("/api/update-smmcost-key", "POST", {
        apiKey: newKey
      });
    },
    onSuccess: () => {
      toast({
        title: "SMMCost API Key Updated",
        description: "Your API key has been successfully updated and is now active",
      });
      setSmmcostKey("");
    },
    onError: (error: any) => {
      toast({
        title: "Error",
        description: error.message || "Failed to update API key",
        variant: "destructive"
      });
    }
  });

  const handleSmmcostSubmit = (e: React.FormEvent) => {
    e.preventDefault();
    if (!smmcostKey.trim()) {
      toast({
        title: "Error",
        description: "Please enter a valid API key",
        variant: "destructive"
      });
      return;
    }
    updateSmmcostKey(smmcostKey.trim());
  };

  return (
    <DashboardLayout>
      <div className="p-6 max-w-4xl mx-auto">
        <div className="mb-8">
          <h1 className="text-3xl font-bold text-gray-900 dark:text-white mb-2">API Services</h1>
          <p className="text-gray-600 dark:text-gray-400">
            Manage your external API keys and service integrations
          </p>
        </div>

        <div className="grid gap-6">
          {/* SMMCost API Configuration */}
          <Card>
            <CardHeader>
              <div className="flex items-center justify-between">
                <div className="flex items-center space-x-3">
                  <div className="p-2 bg-blue-100 dark:bg-blue-900 rounded-lg">
                    <Key className="w-5 h-5 text-blue-600 dark:text-blue-400" />
                  </div>
                  <div>
                    <CardTitle className="text-xl">SMMCost API</CardTitle>
                    <CardDescription>
                      Required for subscriber and view boosting services
                    </CardDescription>
                  </div>
                </div>
                <Badge variant="secondary" className="flex items-center space-x-1">
                  <AlertCircle className="w-3 h-3" />
                  <span>Required</span>
                </Badge>
              </div>
            </CardHeader>
            <CardContent className="space-y-4">
              <div className="bg-yellow-50 dark:bg-yellow-900/20 border border-yellow-200 dark:border-yellow-800 rounded-lg p-4">
                <div className="flex items-start space-x-3">
                  <AlertCircle className="w-5 h-5 text-yellow-600 dark:text-yellow-400 mt-0.5" />
                  <div>
                    <h4 className="font-medium text-yellow-800 dark:text-yellow-200">API Key Required</h4>
                    <p className="text-sm text-yellow-700 dark:text-yellow-300 mt-1">
                      Your current SMMCost API key is invalid or expired. Please update it to restore service functionality.
                    </p>
                  </div>
                </div>
              </div>

              <form onSubmit={handleSmmcostSubmit} className="space-y-4">
                <div className="space-y-2">
                  <Label htmlFor="smmcost-key">SMMCost API Key</Label>
                  <Input
                    id="smmcost-key"
                    type="password"
                    placeholder="Enter your SMMCost API key"
                    value={smmcostKey}
                    onChange={(e) => setSmmcostKey(e.target.value)}
                    disabled={isUpdatingSmmcost}
                    className="font-mono"
                  />
                  <p className="text-sm text-gray-500 dark:text-gray-400">
                    Get your API key from{" "}
                    <Button
                      variant="link"
                      className="p-0 h-auto text-sm"
                      onClick={() => window.open("https://smmcost.com/api", "_blank")}
                    >
                      SMMCost API Dashboard
                      <ExternalLink className="w-3 h-3 ml-1" />
                    </Button>
                  </p>
                </div>
                <Button type="submit" disabled={isUpdatingSmmcost} className="w-full sm:w-auto">
                  {isUpdatingSmmcost ? "Updating..." : "Update API Key"}
                </Button>
              </form>
            </CardContent>
          </Card>

          {/* YouTube API Status */}
          <Card>
            <CardHeader>
              <div className="flex items-center justify-between">
                <div className="flex items-center space-x-3">
                  <div className="p-2 bg-red-100 dark:bg-red-900 rounded-lg">
                    <Shield className="w-5 h-5 text-red-600 dark:text-red-400" />
                  </div>
                  <div>
                    <CardTitle className="text-xl">YouTube Data API</CardTitle>
                    <CardDescription>
                      Used for channel analytics and video statistics
                    </CardDescription>
                  </div>
                </div>
                <Badge variant="outline" className="flex items-center space-x-1 text-green-600 border-green-600">
                  <CheckCircle className="w-3 h-3" />
                  <span>Active</span>
                </Badge>
              </div>
            </CardHeader>
            <CardContent>
              <div className="bg-green-50 dark:bg-green-900/20 border border-green-200 dark:border-green-800 rounded-lg p-4">
                <div className="flex items-start space-x-3">
                  <CheckCircle className="w-5 h-5 text-green-600 dark:text-green-400 mt-0.5" />
                  <div>
                    <h4 className="font-medium text-green-800 dark:text-green-200">Service Active</h4>
                    <p className="text-sm text-green-700 dark:text-green-300 mt-1">
                      YouTube API is configured and working properly. No action required.
                    </p>
                  </div>
                </div>
              </div>
            </CardContent>
          </Card>

          {/* Anthropic AI Status */}
          <Card>
            <CardHeader>
              <div className="flex items-center justify-between">
                <div className="flex items-center space-x-3">
                  <div className="p-2 bg-purple-100 dark:bg-purple-900 rounded-lg">
                    <Settings className="w-5 h-5 text-purple-600 dark:text-purple-400" />
                  </div>
                  <div>
                    <CardTitle className="text-xl">Anthropic Claude AI</CardTitle>
                    <CardDescription>
                      Powers AI-powered content analysis and recommendations
                    </CardDescription>
                  </div>
                </div>
                <Badge variant="outline" className="flex items-center space-x-1 text-green-600 border-green-600">
                  <CheckCircle className="w-3 h-3" />
                  <span>Active</span>
                </Badge>
              </div>
            </CardHeader>
            <CardContent>
              <div className="bg-green-50 dark:bg-green-900/20 border border-green-200 dark:border-green-800 rounded-lg p-4">
                <div className="flex items-start space-x-3">
                  <CheckCircle className="w-5 h-5 text-green-600 dark:text-green-400 mt-0.5" />
                  <div>
                    <h4 className="font-medium text-green-800 dark:text-green-200">Service Active</h4>
                    <p className="text-sm text-green-700 dark:text-green-300 mt-1">
                      Claude Sonnet 4 AI is configured and operational. All AI features are available.
                    </p>
                  </div>
                </div>
              </div>
            </CardContent>
          </Card>

          {/* Service Usage Information */}
          <Card>
            <CardHeader>
              <CardTitle className="text-xl flex items-center space-x-2">
                <Shield className="w-5 h-5" />
                <span>Service Information</span>
              </CardTitle>
              <CardDescription>
                Important information about API usage and billing
              </CardDescription>
            </CardHeader>
            <CardContent className="space-y-4">
              <div className="grid md:grid-cols-2 gap-4">
                <div className="space-y-2">
                  <h4 className="font-medium text-gray-900 dark:text-white">SMMCost Services</h4>
                  <ul className="text-sm text-gray-600 dark:text-gray-400 space-y-1">
                    <li>• YouTube subscriber boosting</li>
                    <li>• Video view enhancement</li>
                    <li>• Like count boosting</li>
                    <li>• Charged using your points balance</li>
                  </ul>
                </div>
                <div className="space-y-2">
                  <h4 className="font-medium text-gray-900 dark:text-white">AI Services</h4>
                  <ul className="text-sm text-gray-600 dark:text-gray-400 space-y-1">
                    <li>• Content analysis and recommendations</li>
                    <li>• Reddit moderation assistance</li>
                    <li>• 5 points per AI analysis</li>
                    <li>• 2 points per AI suggestion acceptance</li>
                  </ul>
                </div>
              </div>
            </CardContent>
          </Card>
        </div>
      </div>
    </DashboardLayout>
  );
}