import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card";
import { Badge } from "@/components/ui/badge";
import { Button } from "@/components/ui/button";
import { ArrowLeft, Calendar, Shield, Zap, Bug, Plus, Settings } from "lucide-react";
import { Link } from "wouter";

interface ChangelogEntry {
  version: string;
  date: string;
  type: 'major' | 'minor' | 'patch';
  changes: Array<{
    type: 'added' | 'improved' | 'fixed' | 'security' | 'changed';
    description: string;
  }>;
}

const changelog: ChangelogEntry[] = [
  {
    version: "1.2.4",
    date: "2025-06-26",
    type: "minor",
    changes: [
      {
        type: "added",
        description: "System Operations dashboard showing real-time queue progress with live status updates"
      },
      {
        type: "added", 
        description: "Comprehensive queue management system for algorithm tools with pause/resume/cancel controls"
      },
      {
        type: "improved",
        description: "Enhanced points display to show only remaining available points without total allocation ratio"
      },
      {
        type: "improved",
        description: "Channel name display in queue operations now shows actual channel names instead of 'Unknown Channel'"
      },
      {
        type: "fixed",
        description: "Resolved Error 500 when adding algorithm tool simulations to processing queue"
      },
      {
        type: "fixed",
        description: "Fixed duplicate queue declaration issues causing application build failures"
      }
    ]
  },
  {
    version: "1.2.3",
    date: "2025-06-25",
    type: "patch",
    changes: [
      {
        type: "improved",
        description: "Updated Add Subscribers API Service ID to #3108 with reduced minimum quantity to 50 subscribers"
      },
      {
        type: "improved",
        description: "Updated Load Video Views API Service ID to #940 for proper service integration"
      },
      {
        type: "fixed",
        description: "Resolved database schema issues in security service preventing anonymous account creation"
      }
    ]
  },
  {
    version: "1.2.2",
    date: "2025-06-24",
    type: "minor",
    changes: [
      {
        type: "added",
        description: "Role-based navigation menu system hiding admin-only items from client users"
      },
      {
        type: "added",
        description: "Platform selector dropdown supporting YouTube with Reddit as 'Coming Soon'"
      },
      {
        type: "added",
        description: "Comprehensive About page with company mission, features, and technology stack"
      },
      {
        type: "changed",
        description: "Complete rebranding from SMMCost to TubeBoost throughout entire codebase"
      },
      {
        type: "changed",
        description: "Updated platform description to 'Multi-Platform Growth Management'"
      },
      {
        type: "improved",
        description: "Migrated YouTube advice generation from OpenAI to Claude Sonnet for enhanced AI recommendations"
      },
      {
        type: "fixed",
        description: "Resolved duplicate navigation menu issues across all dashboard pages"
      }
    ]
  },
  {
    version: "1.2.1",
    date: "2025-06-23",
    type: "major",
    changes: [
      {
        type: "added",
        description: "Enterprise-grade security system with device fingerprinting and fraud detection"
      },
      {
        type: "added",
        description: "Security administration dashboard for monitoring violations and managing fraud detection"
      },
      {
        type: "added",
        description: "Points Management admin navigation for easier access to point economy controls"
      },
      {
        type: "security",
        description: "Multi-factor household detection using IP geolocation and hardware fingerprinting"
      },
      {
        type: "security",
        description: "Comprehensive logging and admin monitoring for security violations"
      }
    ]
  },
  {
    version: "1.2.0",
    date: "2025-06-22",
    type: "major",
    changes: [
      {
        type: "added",
        description: "Multi-channel management supporting unlimited YouTube channel connections"
      },
      {
        type: "added",
        description: "AI-powered optimization with OpenWeb 3.2 AI generating personalized improvement strategies"
      },
      {
        type: "added",
        description: "Real-time order tracking and status updates for all growth services"
      },
      {
        type: "added",
        description: "Point economy system with credit-based service usage"
      },
      {
        type: "improved",
        description: "Enhanced dashboard with comprehensive analytics and performance tracking"
      }
    ]
  }
];

const getTypeIcon = (type: string) => {
  switch (type) {
    case 'added':
      return <Plus className="w-4 h-4" />;
    case 'improved':
      return <Zap className="w-4 h-4" />;
    case 'fixed':
      return <Bug className="w-4 h-4" />;
    case 'security':
      return <Shield className="w-4 h-4" />;
    case 'changed':
      return <Settings className="w-4 h-4" />;
    default:
      return <Calendar className="w-4 h-4" />;
  }
};

const getTypeColor = (type: string) => {
  switch (type) {
    case 'added':
      return 'bg-green-100 text-green-800 border-green-200';
    case 'improved':
      return 'bg-blue-100 text-blue-800 border-blue-200';
    case 'fixed':
      return 'bg-orange-100 text-orange-800 border-orange-200';
    case 'security':
      return 'bg-red-100 text-red-800 border-red-200';
    case 'changed':
      return 'bg-purple-100 text-purple-800 border-purple-200';
    default:
      return 'bg-gray-100 text-gray-800 border-gray-200';
  }
};

const getVersionBadgeColor = (type: string) => {
  switch (type) {
    case 'major':
      return 'bg-red-500 text-white';
    case 'minor':
      return 'bg-blue-500 text-white';
    case 'patch':
      return 'bg-green-500 text-white';
    default:
      return 'bg-gray-500 text-white';
  }
};

export default function Changelog() {
  return (
    <div className="min-h-screen bg-gray-50">
      <div className="max-w-4xl mx-auto px-4 py-8">
        {/* Header */}
        <div className="mb-8">
          <Link href="/">
            <Button variant="ghost" className="mb-4">
              <ArrowLeft className="w-4 h-4 mr-2" />
              Back to Home
            </Button>
          </Link>
          
          <div className="text-center">
            <h1 className="text-4xl font-bold text-gray-900 mb-4">
              TubeBoost Changelog
            </h1>
            <p className="text-xl text-gray-600">
              Track the latest updates, improvements, and new features
            </p>
          </div>
        </div>

        {/* Current Version Highlight */}
        <Card className="mb-8 border-2 border-blue-200 bg-gradient-to-r from-blue-50 to-indigo-50">
          <CardHeader>
            <div className="flex items-center justify-between">
              <CardTitle className="text-2xl text-blue-900">
                Current Version: {changelog[0].version}
              </CardTitle>
              <Badge className={getVersionBadgeColor(changelog[0].type)}>
                {changelog[0].type.toUpperCase()}
              </Badge>
            </div>
            <CardDescription className="text-blue-700">
              Released on {new Date(changelog[0].date).toLocaleDateString('en-US', { 
                year: 'numeric', 
                month: 'long', 
                day: 'numeric' 
              })}
            </CardDescription>
          </CardHeader>
        </Card>

        {/* Changelog Entries */}
        <div className="space-y-6">
          {changelog.map((entry, index) => (
            <Card key={entry.version} className="shadow-sm hover:shadow-md transition-shadow">
              <CardHeader>
                <div className="flex items-center justify-between">
                  <CardTitle className="text-xl">
                    Version {entry.version}
                  </CardTitle>
                  <div className="flex items-center space-x-2">
                    <Badge className={getVersionBadgeColor(entry.type)}>
                      {entry.type}
                    </Badge>
                    <span className="text-sm text-gray-500 flex items-center">
                      <Calendar className="w-4 h-4 mr-1" />
                      {new Date(entry.date).toLocaleDateString('en-US', { 
                        year: 'numeric', 
                        month: 'short', 
                        day: 'numeric' 
                      })}
                    </span>
                  </div>
                </div>
              </CardHeader>
              
              <CardContent>
                <div className="space-y-3">
                  {entry.changes.map((change, changeIndex) => (
                    <div key={changeIndex} className="flex items-start space-x-3">
                      <div className={`flex items-center justify-center w-6 h-6 rounded-full border ${getTypeColor(change.type)}`}>
                        {getTypeIcon(change.type)}
                      </div>
                      <div className="flex-1">
                        <div className="flex items-center space-x-2 mb-1">
                          <Badge variant="outline" className={`text-xs ${getTypeColor(change.type)}`}>
                            {change.type.toUpperCase()}
                          </Badge>
                        </div>
                        <p className="text-gray-700">{change.description}</p>
                      </div>
                    </div>
                  ))}
                </div>
              </CardContent>
            </Card>
          ))}
        </div>

        {/* Footer */}
        <div className="mt-12 text-center">
          <Card className="bg-gradient-to-r from-purple-50 to-pink-50 border-purple-200">
            <CardContent className="py-6">
              <h3 className="text-lg font-semibold text-purple-900 mb-2">
                Stay Updated
              </h3>
              <p className="text-purple-700 mb-4">
                Follow our changelog to stay informed about the latest features and improvements.
              </p>
              <div className="flex justify-center space-x-4">
                <Link href="/dashboard">
                  <Button>
                    Go to Dashboard
                  </Button>
                </Link>
                <Link href="/about">
                  <Button variant="outline">
                    Learn More
                  </Button>
                </Link>
              </div>
            </CardContent>
          </Card>
        </div>
      </div>
    </div>
  );
}