import { useState } from "react";
import { useQuery, useMutation, useQueryClient } from "@tanstack/react-query";
import { Card, CardHeader, CardTitle, CardContent } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import { Badge } from "@/components/ui/badge";
import { Dialog, DialogContent, DialogHeader, DialogTitle, DialogTrigger } from "@/components/ui/dialog";
import { DashboardLayout } from "@/components/dashboard-layout";
import { Youtube, Plus, Settings, Trash2, Users, Video, Eye } from "lucide-react";
import { useToast } from "@/hooks/use-toast";
import { apiRequest } from "@/lib/queryClient";

interface Channel {
  id: number;
  channelName: string;
  channelHandle: string;
  customUrl?: string;
  subscriberCount?: number;
  viewCount?: number;
  videoCount?: number;
  isActive: boolean;
}

export default function ChannelManager() {
  const [addChannelOpen, setAddChannelOpen] = useState(false);
  const [channelUrl, setChannelUrl] = useState("");
  const queryClient = useQueryClient();
  const { toast } = useToast();

  // Fetch all channels
  const { data: channels, isLoading: channelsLoading } = useQuery<Channel[]>({
    queryKey: ["/api/channels"],
  });

  // Add new channel mutation
  const { mutate: addChannel, isPending: isAddingChannel } = useMutation({
    mutationFn: (url: string) => apiRequest("/api/add-channel", "POST", { channelUrl: url }),
    onSuccess: () => {
      toast({
        title: "Success",
        description: "Channel added successfully",
      });
      setAddChannelOpen(false);
      setChannelUrl("");
      queryClient.invalidateQueries({ queryKey: ["/api/channels"] });
    },
    onError: (error: any) => {
      toast({
        title: "Error",
        description: error.message || "Failed to add channel",
        variant: "destructive",
      });
    }
  });

  // Switch active channel mutation
  const { mutate: switchChannel } = useMutation({
    mutationFn: (channelId: number) => apiRequest("/api/switch-channel", "POST", { channelId }),
    onSuccess: () => {
      toast({
        title: "Success",
        description: "Active channel switched successfully",
      });
      queryClient.invalidateQueries({ queryKey: ["/api/channels"] });
      queryClient.invalidateQueries({ queryKey: ["/api/channel"] });
    },
    onError: (error: any) => {
      toast({
        title: "Error",
        description: error.message || "Failed to switch channel",
        variant: "destructive",
      });
    }
  });

  // Delete channel mutation
  const { mutate: deleteChannel } = useMutation({
    mutationFn: (channelId: number) => apiRequest(`/api/channels/${channelId}`, "DELETE"),
    onSuccess: () => {
      toast({
        title: "Success",
        description: "Channel removed successfully",
      });
      queryClient.invalidateQueries({ queryKey: ["/api/channels"] });
    },
    onError: (error: any) => {
      toast({
        title: "Error",
        description: error.message || "Failed to remove channel",
        variant: "destructive",
      });
    }
  });

  const handleAddChannel = () => {
    if (!channelUrl.trim()) {
      toast({
        title: "Error",
        description: "Please enter a channel URL",
        variant: "destructive",
      });
      return;
    }
    addChannel(channelUrl);
  };

  const formatNumber = (num: number | undefined | null) => {
    if (!num || num === 0) return "0";
    if (num >= 1000000) {
      return (num / 1000000).toFixed(1) + "M";
    } else if (num >= 1000) {
      return (num / 1000).toFixed(1) + "K";
    }
    return num.toString();
  };

  return (
    <DashboardLayout>
      <div className="p-6">
        <div className="flex justify-between items-center mb-6">
          <div>
            <h1 className="text-3xl font-bold">Channel Manager</h1>
            <p className="text-muted-foreground">Manage your connected YouTube channels</p>
          </div>
          
          <Dialog open={addChannelOpen} onOpenChange={setAddChannelOpen}>
            <DialogTrigger asChild>
              <Button>
                <Plus className="h-4 w-4 mr-2" />
                Add Channel
              </Button>
            </DialogTrigger>
            <DialogContent>
              <DialogHeader>
                <DialogTitle>Add YouTube Channel</DialogTitle>
              </DialogHeader>
              <div className="space-y-4">
                <div>
                  <Label htmlFor="channel-url">Channel URL</Label>
                  <Input
                    id="channel-url"
                    placeholder="https://www.youtube.com/@username or https://www.youtube.com/c/channelname"
                    value={channelUrl}
                    onChange={(e) => setChannelUrl(e.target.value)}
                  />
                </div>
                <div className="flex justify-end gap-2">
                  <Button variant="outline" onClick={() => setAddChannelOpen(false)}>
                    Cancel
                  </Button>
                  <Button onClick={handleAddChannel} disabled={isAddingChannel}>
                    {isAddingChannel ? "Adding..." : "Add Channel"}
                  </Button>
                </div>
              </div>
            </DialogContent>
          </Dialog>
        </div>

        {channelsLoading ? (
          <div className="flex items-center justify-center h-64">
            <div className="w-16 h-16 border-4 border-primary border-t-transparent rounded-full animate-spin"></div>
          </div>
        ) : channels && channels.length > 0 ? (
          <div className="grid gap-6 md:grid-cols-2 lg:grid-cols-3">
            {channels.map((channel) => (
              <Card key={channel.id} className="relative">
                <CardHeader className="pb-3">
                  <div className="flex items-center justify-between">
                    <div className="flex items-center gap-3">
                      <div className="w-12 h-12 bg-red-100 rounded-full flex items-center justify-center">
                        <Youtube className="w-6 h-6 text-red-600" />
                      </div>
                      <div>
                        <CardTitle className="text-lg">{channel.channelName}</CardTitle>
                        <p className="text-sm text-muted-foreground">@{channel.channelHandle}</p>
                      </div>
                    </div>
                    {channel.isActive && (
                      <Badge variant="default">Active</Badge>
                    )}
                  </div>
                </CardHeader>
                <CardContent className="space-y-4">
                  <div className="grid grid-cols-3 gap-4 text-center">
                    <div>
                      <div className="flex items-center justify-center mb-1">
                        <Users className="h-4 w-4 text-muted-foreground" />
                      </div>
                      <div className="text-sm font-medium">{formatNumber(channel.subscriberCount)}</div>
                      <div className="text-xs text-muted-foreground">Subscribers</div>
                    </div>
                    <div>
                      <div className="flex items-center justify-center mb-1">
                        <Eye className="h-4 w-4 text-muted-foreground" />
                      </div>
                      <div className="text-sm font-medium">{formatNumber(channel.viewCount)}</div>
                      <div className="text-xs text-muted-foreground">Views</div>
                    </div>
                    <div>
                      <div className="flex items-center justify-center mb-1">
                        <Video className="h-4 w-4 text-muted-foreground" />
                      </div>
                      <div className="text-sm font-medium">{channel.videoCount || 0}</div>
                      <div className="text-xs text-muted-foreground">Videos</div>
                    </div>
                  </div>
                  
                  <div className="flex gap-2">
                    {!channel.isActive && (
                      <Button
                        variant="outline"
                        size="sm"
                        className="flex-1"
                        onClick={() => switchChannel(channel.id)}
                      >
                        <Settings className="h-4 w-4 mr-2" />
                        Set Active
                      </Button>
                    )}
                    <Button
                      variant="outline"
                      size="sm"
                      className={channel.isActive ? "flex-1" : ""}
                      onClick={() => deleteChannel(channel.id)}
                    >
                      <Trash2 className="h-4 w-4 mr-2" />
                      Remove
                    </Button>
                  </div>
                </CardContent>
              </Card>
            ))}
          </div>
        ) : (
          <Card>
            <CardContent className="p-12 text-center">
              <Youtube className="h-16 w-16 text-muted-foreground mx-auto mb-4" />
              <h3 className="text-xl font-semibold mb-2">No Channels Connected</h3>
              <p className="text-muted-foreground mb-6">
                Connect your first YouTube channel to start managing your content and analytics.
              </p>
              <Button onClick={() => setAddChannelOpen(true)}>
                <Plus className="h-4 w-4 mr-2" />
                Add Your First Channel
              </Button>
            </CardContent>
          </Card>
        )}
      </div>
    </DashboardLayout>
  );
}