import { useState, useEffect } from "react";
import { useQuery, useMutation, useQueryClient } from "@tanstack/react-query";
import { apiRequest, getQueryFn } from "@/lib/queryClient";
import { useAuth } from "@/hooks/use-auth";
import { Button } from "@/components/ui/button";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Dialog, DialogContent, DialogHeader, DialogTitle, DialogTrigger, DialogDescription } from "@/components/ui/dialog";
import { Input } from "@/components/ui/input";
import { Progress } from "@/components/ui/progress";
import { Badge } from "@/components/ui/badge";
import { useToast } from "@/hooks/use-toast";
import { formatNumber, getCooldownStatus, formatDuration } from "@/lib/utils";
import type { ChannelStats, VideoData, AgSession, SessionType } from "@/types/youtube";
import { LogOut, User, Trash2, Share2, Plus, Shield } from "lucide-react";
import { ChannelDropdown } from "@/components/channel-dropdown";
import { PointsDisplay } from "@/components/points-display";
import { PointTransactionHistory } from "@/components/point-transaction-history";
import { Link, useLocation } from "wouter";
import AnonymousWarningBanner from "@/components/anonymous-warning-banner";
import AnonymousCredentialsDisplay from "@/components/anonymous-credentials-display";
import EmailIncentiveBanner from "@/components/email-incentive-banner";
import { DashboardLayout } from "@/components/dashboard-layout";
import { QueuePanel } from "@/components/queue-panel";
import { RedditConnection } from "@/components/reddit-connection";
import { RedditAnalyticsDashboard } from "@/components/reddit-analytics-dashboard";
import { PlatformDashboard } from "@/components/platform-dashboard";
import type { InsertQueueItem } from "@shared/schema";
import { useYouTubeStats, formatYouTubeNumber } from "@/hooks/use-youtube-stats";

// SubscriberTransactions Component
interface SubscriberTransactionsProps {
  channelId?: number;
}

function SubscriberTransactions({ channelId }: SubscriberTransactionsProps) {
  const { toast } = useToast();
  const queryClient = useQueryClient();

  // Fetch subscriber orders for the channel
  const { data: subscriberOrders = [], isLoading } = useQuery({
    queryKey: ["subscriber-orders", channelId],
    queryFn: async () => {
      if (!channelId) return [];
      const response = await apiRequest(`/api/subscriber-orders/${channelId}`, "GET");
      return await response.json();
    },
    enabled: !!channelId,
    refetchInterval: 30000, // Refresh every 30 seconds for live status
  });

  // Create refill order mutation
  const { mutate: createRefillOrder, isPending: isCreatingRefill } = useMutation({
    mutationFn: async (subscriberOrderId: number) => {
      const response = await apiRequest(`/api/subscriber-refill/${subscriberOrderId}`, "POST", {});
      return await response.json();
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["subscriber-orders"] });
      toast({
        title: "Success",
        description: "Refill order created successfully",
        variant: "default"
      });
    },
    onError: (error: any) => {
      toast({
        title: "Error",
        description: error.message || "Failed to create refill order",
        variant: "destructive"
      });
    }
  });

  // Create subscriber order mutation - ADD THIS
  const createSubscriberOrder = useMutation({
    mutationFn: async (data: { quantity: number }) => {
      const response = await apiRequest("POST", "/api/add-subscribers", data);
      return await response.json();
    },
    onSuccess: () => {
      toast({
        title: "Success",
        description: "Subscriber order created successfully"
      });
      queryClient.invalidateQueries({ queryKey: ["/api/subscriber-orders", channelId] });
      queryClient.invalidateQueries({ queryKey: ["/api/points"] });
      queryClient.invalidateQueries({ queryKey: ["/api/queue"] });
    },
    onError: (error) => {
      toast({
        title: "Error",
        description: error.message || "Failed to create subscriber order",
        variant: "destructive"
      });
    }
  });

  // Sync status mutation
  const { mutate: syncOrderStatus, isPending: isSyncing } = useMutation({
    mutationFn: async () => {
      if (!channelId) throw new Error("No channel selected");
      const response = await apiRequest("POST", `/api/subscriber-orders/sync-status/${channelId}`, {});
      return await response.json();
    },
    onSuccess: (data) => {
      queryClient.invalidateQueries({ queryKey: ["subscriber-orders", channelId] });
      toast({
        title: "Status Sync Complete",
        description: `Updated ${data.updatedCount} of ${data.totalActiveOrders} active orders`,
        variant: "default"
      });
    },
    onError: (error: any) => {
      toast({
        title: "Sync Failed",
        description: error.message || "Failed to sync order statuses",
        variant: "destructive"
      });
    }
  });


  const getStatusColor = (status: string) => {
    switch (status?.toLowerCase()) {
      case 'completed':
        return 'bg-green-100 text-green-800';
      case 'in progress':
      case 'processing':
        return 'bg-blue-100 text-blue-800';
      case 'pending':
        return 'bg-yellow-100 text-yellow-800';
      case 'canceled':
      case 'cancelled':
        return 'bg-red-100 text-red-800';
      default:
        return 'bg-gray-100 text-gray-800';
    }
  };

  const formatDate = (dateString: string) => {
    return new Date(dateString).toLocaleDateString('en-US', {
      month: 'short',
      day: 'numeric',
      hour: '2-digit',
      minute: '2-digit'
    });
  };

  if (!channelId) {
    return (
      <div className="text-center py-8 text-gray-500">
        <p>No channel selected</p>
      </div>
    );
  }

  if (isLoading) {
    return (
      <div className="text-center py-8">
        <div className="inline-flex items-center space-x-2">
          <div className="w-4 h-4 border-2 border-google-blue border-t-transparent rounded-full animate-spin"></div>
          <span className="text-gray-600">Loading subscriber transactions...</span>
        </div>
      </div>
    );
  }

  if (subscriberOrders.length === 0) {
    return (
      <div className="text-center py-8 text-gray-500">
        <div className="w-16 h-16 bg-orange-100 rounded-full flex items-center justify-center mx-auto mb-4">
          <span className="material-icons text-orange-500 text-2xl">person_add</span>
        </div>
        <p className="text-lg font-medium text-gray-900 mb-2">No Subscriber Orders</p>
        <p className="text-gray-600">No subscriber orders have been placed for this channel yet.</p>
      </div>
    );
  }

  return (
    <div className="space-y-4">
      {/* Sync Status Button */}
      <div className="flex justify-end">
        <Button
          onClick={() => syncOrderStatus()}
          disabled={isSyncing}
          size="sm"
          variant="outline"
          className="text-xs h-8 px-3"
        >
          {isSyncing ? (
            <div className="flex items-center space-x-1">
              <div className="w-3 h-3 border border-gray-400 border-t-transparent rounded-full animate-spin"></div>
              <span>Syncing...</span>
            </div>
          ) : (
            <div className="flex items-center space-x-1">
              <span className="material-icons text-xs">refresh</span>
              <span>Sync Status</span>
            </div>
          )}
        </Button>
      </div>

      {subscriberOrders.map((order: any) => (
        <div key={order.id} className="border border-gray-200 rounded-lg p-4 bg-gray-50">
          <div className="flex items-center justify-between mb-3">
            <div className="flex items-center space-x-3">
              <div className={`px-3 py-1 rounded-full text-sm font-medium ${getStatusColor(order.status)}`}>
                {order.status}
              </div>
              <div className="text-sm text-gray-600">
                {order.quantity} subscribers
              </div>
              <div className="text-sm text-gray-500">
                Order #{order.orderId?.slice(-6)}
              </div>
            </div>
            <div className="flex items-center space-x-2">
              <span className="text-xs text-gray-500">
                {formatDate(order.createdAt)}
              </span>

            </div>
          </div>

          {/* TubeBoost API Data - Enhanced Display */}
          <div className="bg-white rounded-lg p-3 mb-3 border border-gray-100">
            <div className="flex items-center justify-between mb-2">
              <span className="text-xs font-medium text-gray-700">Live Order Status</span>
              <span className="text-xs text-green-600 flex items-center space-x-1">
                <div className="w-2 h-2 bg-green-500 rounded-full animate-pulse"></div>
                <span>TubeBoost API</span>
              </span>
            </div>
            <div className="grid grid-cols-2 md:grid-cols-4 gap-3 text-xs">
              {/* Always show charge if available */}
              <div className="flex flex-col">
                <span className="text-gray-500 font-medium">Charge</span>
                <span className="text-gray-900 font-semibold">
                  ${order.charge || order.tubeBoostData?.charge || '0.00'} {order.currency || order.tubeBoostData?.currency || 'USD'}
                </span>
              </div>
              
              {/* Always show remaining count */}
              <div className="flex flex-col">
                <span className="text-gray-500 font-medium">Remaining</span>
                <span className="text-gray-900 font-semibold">
                  {order.remains || order.tubeBoostData?.remains || order.quantity || '0'}
                </span>
              </div>
              
              {/* Show progress if processing */}
              {order.status && ['In progress', 'Processing', 'Pending'].includes(order.status) && (
                <div className="flex flex-col">
                  <span className="text-gray-500 font-medium">Progress</span>
                  <span className="text-gray-900 font-semibold">
                    {order.quantity && (order.remains || order.tubeBoostData?.remains) ? 
                      `${Math.max(0, order.quantity - parseInt((order.remains || order.tubeBoostData?.remains) || '0'))} / ${order.quantity}` :
                      'Starting...'
                    }
                  </span>
                </div>
              )}
              
              {/* Show start count if available */}
              {(order.startCount || order.tubeBoostData?.start_count) && (
                <div className="flex flex-col">
                  <span className="text-gray-500 font-medium">Start Count</span>
                  <span className="text-gray-900 font-semibold">{order.startCount || order.tubeBoostData?.start_count}</span>
                </div>
              )}
            </div>
            
            {/* Debug info - temporarily show raw data */}
            <div className="mt-2 pt-2 border-t border-gray-100 text-xs text-gray-400">
              <div>Status: {order.status} | Order ID: {order.orderId}</div>
              {order.tubeBoostData && (
                <div>API Data: {JSON.stringify(order.tubeBoostData)}</div>
              )}
            </div>
          </div>

          {/* Channel URL */}
          <div className="text-xs text-gray-600 mb-3">
            <span className="text-gray-500">Target:</span>
            <span className="ml-1 break-all">{order.channelUrl}</span>
          </div>

          {/* Refill Orders */}
          {order.refillOrders && order.refillOrders.length > 0 && (
            <div className="border-t border-gray-200 pt-3 mt-3">
              <div className="text-xs font-medium text-gray-700 mb-2">Refill Orders:</div>
              <div className="space-y-2">
                {order.refillOrders.map((refill: any) => (
                  <div key={refill.id} className="bg-white rounded p-2 text-xs">
                    <div className="flex items-center justify-between">
                      <div className="flex items-center space-x-2">
                        <div className={`px-2 py-1 rounded text-xs ${getStatusColor(refill.status)}`}>
                          {refill.status}
                        </div>
                        <span className="text-gray-600">Refill #{refill.refillId}</span>
                      </div>
                      <span className="text-gray-500">
                        {formatDate(refill.createdAt)}
                      </span>
                    </div>
                  </div>
                ))}
              </div>
            </div>
          )}
        </div>
      ))}
    </div>
  );
}

// RefillStatus Component
function RefillStatus({ orderId }: { orderId: number }) {
  const { data: refillOrders = [], isLoading } = useQuery({
    queryKey: ["refill-orders", orderId],
    queryFn: async () => {
      try {
        const response = await fetch(`/api/refill-orders/${orderId}`, {
          credentials: "include",
        });
        if (response.ok) {
          return await response.json();
        }
        return [];
      } catch (error) {
        console.error("Error fetching refill orders:", error);
        return [];
      }
    },
    refetchInterval: 5000, // Refresh every 5 seconds
  });

  if (isLoading) {
    return <div className="text-xs text-gray-400 mt-2">Loading refill status...</div>;
  }

  if (!Array.isArray(refillOrders) || !refillOrders.length) {
    return null;
  }

  return (
    <div className="mt-3 pt-2 border-t border-gray-200">
      <div className="text-xs font-medium text-gray-700 mb-2">Refill Status:</div>
      {(refillOrders as any[]).map((refill: any, idx: number) => (
        <div key={idx} className="flex items-center justify-between text-xs">
          <div className="flex items-center space-x-2">
            <div className={`w-2 h-2 rounded-full ${
              refill.status === 'completed' ? 'bg-green-500' : 
              refill.status === 'pending' ? 'bg-yellow-500' : 'bg-gray-400'
            }`}></div>
            <span className="text-gray-600">Refill #{refill.refillId?.slice(-4)}</span>
            <span className={`px-1 py-0.5 rounded text-xs ${
              refill.status === 'completed' ? 'bg-green-100 text-green-700' :
              refill.status === 'pending' ? 'bg-yellow-100 text-yellow-700' : 'bg-gray-100 text-gray-700'
            }`}>
              {refill.status}
            </span>
          </div>
          <div className="text-gray-500">
            {refill.createdAt && new Date(refill.createdAt).toLocaleDateString()}
          </div>
        </div>
      ))}
    </div>
  );
}

export default function Dashboard() {
  const [selectedVideo, setSelectedVideo] = useState<number | null>(null);
  const [selectedChannelId, setSelectedChannelId] = useState<number | null>(null);
  const [searchTerm, setSearchTerm] = useState("");
  const [activeModal, setActiveModal] = useState<SessionType | null>(null);
  const [loadViewsModal, setLoadViewsModal] = useState(false);
  const [adminLoadViewsModal, setAdminLoadViewsModal] = useState(false);
  const [customViewsQuantity, setCustomViewsQuantity] = useState<string>("");
  const [customQuantity, setCustomQuantity] = useState<string>("");

  // Video orders for all videos to display status
  const [videoOrdersMap, setVideoOrdersMap] = useState<Record<number, any[]>>({});
  const [refillOrdersMap, setRefillOrdersMap] = useState<Record<number, any[]>>({});
  
  // Channel management states
  const [shareChannelUrl, setShareChannelUrl] = useState("");
  const [isSharing, setIsSharing] = useState(false);
  const [shareDialogOpen, setShareDialogOpen] = useState(false);
  const [mobileMenuOpen, setMobileMenuOpen] = useState(false);
  const [isDeleting, setIsDeleting] = useState(false);

  // Sync order statuses mutation
  const { mutate: syncOrderStatuses, isPending: isSyncingOrders } = useMutation({
    mutationFn: async () => {
      const response = await apiRequest("POST", "/api/sync-order-statuses", {});
      return response.json();
    },
    onSuccess: (result: any) => {
      toast({
        title: "Status Sync Complete",
        description: result.message,
      });
      // Refresh video orders after sync
      queryClient.invalidateQueries({ queryKey: ["video-orders"] });
    },
    onError: (error: any) => {
      toast({
        title: "Sync Error",
        description: error.message || "Failed to sync order statuses",
        variant: "destructive"
      });
    }
  });

  // Function to get order status badge color
  const getOrderStatusColor = (status: string) => {
    switch (status) {
      case 'Pending':
        return 'bg-yellow-100 text-yellow-800';
      case 'In progress':
        return 'bg-blue-100 text-blue-800';
      case 'Partial':
        return 'bg-orange-100 text-orange-800';
      case 'Completed':
        return 'bg-green-100 text-green-800';
      case 'Canceled':
        return 'bg-red-100 text-red-800';
      default:
        return 'bg-gray-100 text-gray-800';
    }
  };

  // Function to get service type display name based on service ID
  const getServiceTypeDisplay = (serviceId: string) => {
    switch (serviceId) {
      case '1800': // Likes service
        return 'likes';
      case '940':  // Views service
      case '2369': // Views service
        return 'views';
      case '9':    // Subscribers service
      case '3108': // Subscribers service
        return 'subscribers';
      default:
        return 'items'; // Generic fallback
    }
  };
  const [simulationState, setSimulationState] = useState<{
    isRunning: boolean;
    progress: number;
    currentStep: string;
    metrics: Record<string, string>;
    sessionId?: number;
  }>({
    isRunning: false,
    progress: 0,
    currentStep: "",
    metrics: {},
    sessionId: undefined
  });

  const [whatsappGraphics, setWhatsappGraphics] = useState<Array<{url: string, description: string}>>([]);

  const { toast } = useToast();
  const queryClient = useQueryClient();
  const { user, logout } = useAuth();
  const [pathname] = useLocation();

  // Initialize data
  const { mutate: initData } = useMutation({
    mutationFn: () => apiRequest("/api/init", "GET"),
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/channel"] });
      queryClient.invalidateQueries({ queryKey: ["/api/videos"] });
    }
  });

  // Fetch channel data
  const { data: channel, error: channelError } = useQuery<ChannelStats>({
    queryKey: ["/api/channel"],
    retry: false
  });

  // Fetch videos
  const { data: videos = [] } = useQuery<VideoData[]>({
    queryKey: ["/api/videos"],
    retry: false,
    enabled: !!channel
  });

  // Get current platform from localStorage
  const currentPlatform = localStorage.getItem('selectedPlatform') || 'youtube';
  
  // Fetch YouTube statistics for all videos (only if YouTube platform is selected)
  const videoIds = videos.map((video: any) => video.videoId).filter(Boolean);
  const { data: youtubeStats = [], isLoading: youtubeStatsLoading } = useYouTubeStats(
    videoIds, 
    currentPlatform === 'youtube' && !!channel?.id && videoIds.length > 0
  );

  // Create a map of video statistics for quick lookup
  const youtubeStatsMap = youtubeStats.reduce((acc: any, stat: any) => {
    acc[stat.videoId] = stat;
    return acc;
  }, {});

  // Fetch user points
  const { data: userPoints } = useQuery<{
    remainingPoints: number;
    totalPoints: number;
    usedPoints: number;
  }>({
    queryKey: ["/api/points"],
    retry: false
  });

  // Fetch queue items for System Operations display
  const { data: queueItems = [] } = useQuery<any[]>({
    queryKey: ["/api/queue"],
    refetchInterval: 2000, // Update every 2 seconds for real-time status
    retry: false
  });

  // Fetch video orders for each video when videos are loaded
  useQuery({
    queryKey: ["video-orders", videos?.map(v => v.id).join(",")],
    queryFn: async () => {
      if (!videos || videos.length === 0) return {};
      
      const ordersMap: Record<number, any[]> = {};
      
      // Fetch orders for each video
      await Promise.all(
        videos.map(async (video) => {
          try {
            const response = await fetch(`/api/video-orders/${video.id}`, {
              credentials: "include",
            });
            if (response.ok) {
              const orders = await response.json();
              ordersMap[video.id] = Array.isArray(orders) ? orders : [];
              console.log(`Video ${video.id} orders:`, orders);
            } else {
              ordersMap[video.id] = [];
            }
          } catch (error) {
            console.error(`Error fetching orders for video ${video.id}:`, error);
            ordersMap[video.id] = [];
          }
        })
      );
      
      console.log("Final orders map:", ordersMap);
      setVideoOrdersMap(ordersMap);
      return ordersMap;
    },
    enabled: !!videos && videos.length > 0,
    retry: false,
    refetchInterval: 30000, // Refetch every 30 seconds to update order status
  });



  // Load video views mutation
  const { mutate: loadVideoViews, isPending: isLoadingViews } = useMutation({
    mutationFn: async ({ videoId, quantity }: { videoId: number; quantity: number }) => {
      const response = await apiRequest("POST", "/api/load-video-views", { videoId, quantity });
      return response.json();
    },
    onSuccess: (data: any) => {
      toast({
        title: "Views Order Placed",
        description: `Successfully ordered ${data.videoOrder?.quantity || customViewsQuantity} views. Order ID: ${data.orderId}`,
      });
      setLoadViewsModal(false);
      setCustomViewsQuantity("");
      queryClient.invalidateQueries({ queryKey: ["/api/points"] });
      queryClient.invalidateQueries({ queryKey: ["/api/videos"] });
      queryClient.invalidateQueries({ queryKey: ["video-orders"] });
    },
    onError: (error: any) => {
      toast({
        title: "Error",
        description: error.message || "Failed to load video views",
        variant: "destructive"
      });
    }
  });

  // Admin load video views mutation (custom quantity)
  const { mutate: adminLoadVideoViews, isPending: isAdminLoadingViews } = useMutation({
    mutationFn: async ({ videoId, quantity }: { videoId: number; quantity: number }) => {
      const response = await apiRequest("/api/admin-load-video-views", "POST", { videoId, quantity });
      return response.json();
    },
    onSuccess: (data: any) => {
      toast({
        title: "Admin Views Order Placed",
        description: `Successfully ordered ${data.videoOrder?.quantity || customQuantity} views. Order ID: ${data.orderId}`,
      });
      setAdminLoadViewsModal(false);
      setCustomQuantity("");
      // Refresh video orders map - trigger refetch after placing order
      queryClient.invalidateQueries({ queryKey: ["/api/videos"] });
      queryClient.invalidateQueries({ queryKey: ["video-orders"] });
    },
    onError: (error: any) => {
      toast({
        title: "Error",
        description: error.message || "Failed to load video views",
        variant: "destructive"
      });
    }
  });

  // Add to queue mutation
  const { mutate: addToQueue, isPending: isAddingToQueue } = useMutation({
    mutationFn: async ({ actionType, channelId, videoId }: { actionType: string; channelId: number; videoId?: number }) => {
      const response = await apiRequest("/api/queue", "POST", {
        actionType,
        channelId,
        videoId,
        priority: 0
      });
      return response.json();
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/queue"] });
      toast({
        title: "Added to Queue",
        description: "Simulation has been added to the queue and will start processing soon.",
      });
    },
    onError: (error: any) => {
      toast({
        title: "Error",
        description: error.message || "Failed to add to queue",
        variant: "destructive"
      });
    }
  });

  // Channel sharing mutation
  const { mutate: shareChannel, isPending: isSharingChannel } = useMutation({
    mutationFn: async (channelUrl: string) => {
      const response = await apiRequest("/api/channels/share", "POST", { channelId: channelUrl });
      return response.json();
    },
    onSuccess: (data: any) => {
      const isAlreadyShared = data.alreadyShared;
      toast({
        title: isAlreadyShared ? "Already Have Access" : "Channel Access Granted",
        description: data.message || "Channel access granted successfully",
        variant: isAlreadyShared ? "default" : "default",
      });
      queryClient.invalidateQueries({ queryKey: ["/api/channels"] });
      queryClient.invalidateQueries({ queryKey: ["/api/channel"] });
      setShareChannelUrl("");
      setShareDialogOpen(false);
    },
    onError: (error: any) => {
      toast({
        title: "Error",
        description: error.message || "Failed to share channel",
        variant: "destructive",
      });
    },
  });

  // Channel deletion mutation (client-level soft delete)
  const { mutate: deleteChannel, isPending: isDeletingChannel } = useMutation({
    mutationFn: async (channelId: number) => {
      const response = await fetch(`/api/channels/${channelId}/hide`, {
        method: "DELETE",
        credentials: "include",
      });
      
      if (!response.ok) {
        const error = await response.text();
        throw new Error(error);
      }
      
      return response.json();
    },
    onSuccess: () => {
      toast({
        title: "Channel Removed",
        description: "Channel removed from your account",
      });
      queryClient.invalidateQueries({ queryKey: ["/api/channels"] });
    },
    onError: (error: Error) => {
      toast({
        title: "Error",
        description: error.message,
        variant: "destructive",
      });
    },
  });

  // Session mutations
  const { mutate: startBoostVideo } = useMutation({
    mutationFn: async (videoId: number) => {
      const response = await apiRequest("/api/boost-video", "POST", { videoId });
      return await response.json();
    },
    onSuccess: (session: any) => {
      startSimulation(session, "boost");
    },
    onError: () => {
      toast({
        title: "Error",
        description: "Failed to start video boost",
        variant: "destructive"
      });
    }
  });

  const { mutate: startAgSession } = useMutation({
    mutationFn: async (videoId: number) => {
      const response = await apiRequest("/api/ag-session", "POST", { videoId });
      return await response.json();
    },
    onSuccess: (session: any) => {
      startSimulation(session, "session");
    },
    onError: () => {
      toast({
        title: "Error",
        description: "Failed to start AG session",
        variant: "destructive"
      });
    }
  });

  const { mutate: startReplenish } = useMutation({
    mutationFn: async () => {
      const response = await apiRequest("/api/replenish", "POST");
      return await response.json();
    },
    onSuccess: (session: any) => {
      startSimulation(session, "replenish");
    },
    onError: (error: any) => {
      toast({
        title: "Error",
        description: error.message || "Failed to start replenish",
        variant: "destructive"
      });
    }
  });

  const { mutate: startDevalue } = useMutation({
    mutationFn: async () => {
      const response = await apiRequest("/api/start-devalue", "POST");
      return await response.json();
    },
    onSuccess: (session: any) => {
      startSimulation(session, "devalue");
    },
    onError: (error: any) => {
      toast({
        title: "Error",
        description: error.message || "Failed to start devalue session",
        variant: "destructive"
      });
    }
  });

  const { mutate: startWhatsAppBoost } = useMutation({
    mutationFn: async () => {
      const response = await apiRequest("/api/start-whatsapp", "POST");
      return await response.json();
    },
    onSuccess: (session: any) => {
      startSimulation(session, "whatsapp");
    },
    onError: (error: any) => {
      toast({
        title: "Error",
        description: error.message || "Failed to start WhatsApp boost session",
        variant: "destructive"
      });
    }
  });

  const { mutate: completeSession } = useMutation({
    mutationFn: async (sessionId: number) => {
      console.log('Attempting to complete session with ID:', sessionId);
      const response = await apiRequest("POST", `/api/complete-session/${sessionId}`);
      return await response.json();
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/channel"] });
      queryClient.invalidateQueries({ queryKey: ["/api/videos"] });
      setSimulationState(prev => ({ ...prev, isRunning: false }));
      setActiveModal(null);
      toast({
        title: "Success",
        description: "Session completed successfully!",
        variant: "default"
      });
    },
    onError: (error: any) => {
      console.error('Failed to complete session:', error);
      setSimulationState(prev => ({ ...prev, isRunning: false }));
      setActiveModal(null);
      toast({
        title: "Error",
        description: "Session completed but failed to update scores",
        variant: "destructive"
      });
    }
  });

  const { mutate: syncChannel, isPending: isSyncingChannel } = useMutation({
    mutationFn: async () => {
      const response = await apiRequest("/api/sync-channel", "GET");
      return await response.json();
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/channel"] });
      queryClient.invalidateQueries({ queryKey: ["/api/videos"] });
      toast({
        title: "Success",
        description: "Successfully synced with YouTube channel data!",
        variant: "default"
      });
    },
    onError: (error: any) => {
      toast({
        title: "Sync Failed",
        description: error.message || "Failed to sync with YouTube channel",
        variant: "destructive"
      });
    }
  });

  // Subscriber order management
  const [subscriberDialogOpen, setSubscriberDialogOpen] = useState(false);
  const [customSubscriberQuantity, setCustomSubscriberQuantity] = useState<string>("");

  // Likes order management
  const [likesDialogOpen, setLikesDialogOpen] = useState(false);
  const [customLikesQuantity, setCustomLikesQuantity] = useState<string>("");
  


  const { mutate: addSubscribers, isPending: isAddingSubscribers } = useMutation({
    mutationFn: async (quantity: number) => {
      console.log('Sending add subscribers request:', { quantity });
      
      const response = await apiRequest("/api/add-subscribers", "POST", {
        quantity
      });
      return await response.json();
    },
    onSuccess: (data) => {
      queryClient.invalidateQueries({ queryKey: ["subscriber-orders", channel?.id] });
      queryClient.invalidateQueries({ queryKey: ["/api/points"] });
      queryClient.invalidateQueries({ queryKey: ["/api/queue"] });
      setSubscriberDialogOpen(false);
      setCustomSubscriberQuantity("");
      toast({
        title: "Success",
        description: `Successfully ordered ${data.subscriberOrder?.quantity || ''} subscribers!`,
        variant: "default"
      });
    },
    onError: (error: any) => {
      toast({
        title: "Error",
        description: error.message || "Failed to add subscribers",
        variant: "destructive"
      });
    }
  });

  const { mutate: addLikes, isPending: isAddingLikes } = useMutation({
    mutationFn: async (data: { videoId: number; quantity: number }) => {
      console.log('Sending add likes request:', data);
      
      const response = await apiRequest("/api/add-likes", "POST", data);
      return await response.json();
    },
    onSuccess: (data) => {
      queryClient.invalidateQueries({ queryKey: ["video-orders"] });
      queryClient.invalidateQueries({ queryKey: ["/api/points"] });
      queryClient.invalidateQueries({ queryKey: ["/api/queue"] });
      setLikesDialogOpen(false);
      setCustomLikesQuantity("");
      toast({
        title: "Success",
        description: `Successfully ordered ${data.videoOrder?.quantity || ''} likes!`,
        variant: "default"
      });
    },
    onError: (error: any) => {
      toast({
        title: "Error",
        description: error.message || "Failed to add likes",
        variant: "destructive"
      });
    }
  });

  // Initialize data on mount
  useEffect(() => {
    initData();
    // Force refresh of channel data to get updated URL fields
    queryClient.invalidateQueries({ queryKey: ["/api/channel"] });
    queryClient.refetchQueries({ queryKey: ["/api/channel"] });
  }, []);

  const startSimulation = async (session: any, type: SessionType) => {
    console.log('Starting simulation with session:', session);
    
    if (!session || !session.id) {
      toast({
        title: "Error",
        description: "Invalid session data received",
        variant: "destructive"
      });
      return;
    }

    setActiveModal(type);
    setSimulationState({
      isRunning: true,
      progress: 0,
      currentStep: "Initializing...",
      metrics: {},
      sessionId: session.id
    });

    const steps = getSimulationSteps(type);
    console.log('Simulation steps for type', type, ':', steps);
    const totalDuration = session.duration * 1000; // Convert to milliseconds
    const stepDuration = totalDuration / steps.length;
    console.log('Total duration:', totalDuration, 'Step duration:', stepDuration);

    // Generate AI graphics for YouTube boost (non-blocking)
    if (type === "whatsapp" && channel?.channelName) {
      // Start graphics generation in background, don't wait for it
      apiRequest("/api/generate-whatsapp-graphics", "POST", {
        channelName: channel.channelName,
        steps: steps
      }).then(response => response.json())
        .then(data => setWhatsappGraphics(data.graphics || []))
        .catch(error => {
          console.log("Graphics generation unavailable, continuing simulation");
          setWhatsappGraphics([]);
        });
    }

    console.log('Starting simulation loop with', steps.length, 'steps');
    
    for (let i = 0; i < steps.length; i++) {
      const progress = ((i + 1) / steps.length) * 100;
      console.log(`Step ${i + 1}/${steps.length}: ${steps[i]} (${progress.toFixed(1)}%)`);
      
      setSimulationState(prev => ({
        ...prev,
        progress,
        currentStep: steps[i],
        metrics: getSimulationMetrics(type, progress)
      }));

      if (i < steps.length - 1) {
        console.log(`Waiting ${stepDuration}ms before next step`);
        await new Promise(resolve => setTimeout(resolve, stepDuration));
      }
    }
    
    console.log('Simulation loop completed');

    // Complete the session
    console.log('Completing session with ID:', session.id);
    completeSession(session.id);
  };

  const getSimulationSteps = (type: SessionType): string[] => {
    switch (type) {
      case "boost":
        return [
          "Analyzing video metadata and tags...",
          "Optimizing thumbnail positioning...",
          "Enhancing title for SEO performance...",
          "Adjusting description keywords...",
          "Calibrating engagement metrics...",
          "Updating algorithm weights...",
          "Finalizing optimization parameters...",
          "Synchronizing with YouTube servers..."
        ];
      case "session":
        return [
          "Establishing secure connection to YouTube API",
          "Analyzing current video performance metrics",
          "Identifying algorithm optimization opportunities", 
          "Implementing real-time engagement boosts",
          "Synchronizing with recommendation engine",
          "Applying advanced ranking factors",
          "Optimizing visual appeal algorithms",
          "Finalizing algorithm grade improvements"
        ];
      case "replenish":
        return [
          "Connecting to Google Algorithm servers...",
          "Authenticating channel credentials...",
          "Analyzing current algorithm performance...",
          "Calculating optimal point allocation...",
          "Initializing point transfer protocol...",
          "Synchronizing with YouTube backend...",
          "Applying algorithm grade improvements...",
          "Validating score consistency...",
          "Updating channel metrics...",
          "Finalizing AG point distribution..."
        ];
      case "devalue":
        return [
          "Initializing devaluation simulation...",
          "Analyzing channel algorithm parameters...",
          "Running negative impact scenarios...",
          "Calculating score reduction factors...",
          "Simulating competitor advantages...",
          "Processing algorithm penalties...",
          "Applying devaluation algorithms...",
          "Updating recommendation weights...",
          "Adjusting algorithm grade points...",
          "Finalizing devaluation process..."
        ];
      case "whatsapp":
        return [
          "🚀 Connecting to YouTube Discovery Engine...",
          "📊 Analyzing Global Trending Patterns & Algorithms...",
          "🌐 Deploying Multi-Platform YouTube Promotion Campaigns...",
          "💬 Activating 500+ YouTube Community Networks...",
          "🤖 Generating AI-Powered Thumbnails & Content Assets...",
          "🎯 Targeting High-Value YouTube Demographics & Creators...",
          "⚡ Amplifying Content Through YouTube Recommendation System...",
          "🔥 Optimizing Search Visibility & Discovery Algorithms...",
          "📱 Creating YouTube Shorts & Cross-Platform Assets...",
          "🌟 Activating Premium YouTube Exposure Protocol...",
          "✅ YouTube Channel Exposure Boost Successfully Deployed!"
        ];
      default:
        return [];
    }
  };

  const getSimulationMetrics = (type: SessionType, progress: number): Record<string, string> => {
    switch (type) {
      case "boost":
        return {
          "SEO Score": `${Math.floor(65 + (progress / 100) * 30)}%`,
          "Engagement Rate": `${(3.2 + (progress / 100) * 2).toFixed(1)}%`,
          "Click-Through Rate": `${(8.1 + (progress / 100) * 3).toFixed(1)}%`,
          "Algorithm Score": `${(7.5 + (progress / 100) * 1.5).toFixed(1)}`
        };
      case "replenish":
        return {
          "Server Load": `${Math.floor(20 + (progress / 100) * 30)}%`,
          "Points Added": `+${((progress / 100) * 1.6).toFixed(1)}`,
          "Sync Status": progress > 50 ? "SYNC" : "CONN"
        };
      case "devalue":
        return {
          "Algorithm Impact": `${Math.floor(15 + (progress / 100) * 45)}%`,
          "Points Reduced": `-${((progress / 100) * 2.4).toFixed(1)}`,
          "Penalty Score": `${(progress / 100 * 18.5).toFixed(1)}`,
          "Competition Edge": `${Math.floor(8 + (progress / 100) * 25)}%`
        };
      case "whatsapp":
        return {
          "Communities Reached": `${Math.floor(25 + (progress / 100) * 150)}`,
          "Exposure Rate": `${(progress / 100 * 95).toFixed(1)}%`,
          "Viral Potential": `${Math.floor(10 + (progress / 100) * 40)}x`,
          "Active Shares": `${Math.floor(50 + (progress / 100) * 500)}`
        };
      default:
        return {};
    }
  };

  const filteredVideos = videos.filter(video =>
    video.title.toLowerCase().includes(searchTerm.toLowerCase())
  );

  const cooldownStatus = channel ? getCooldownStatus(channel.lastReplenishTime) : null;

  const handleActionClick = (action: SessionType) => {
    if (action === "replenish") {
      if (cooldownStatus?.isActive) {
        toast({
          title: "Cooldown Active",
          description: "AG Points can only be replenished once per 24 hours",
          variant: "destructive"
        });
        return;
      }
      addToQueue({ 
        actionType: "replenish", 
        channelId: channel?.id || 0 
      });
    } else if (action === "devalue") {
      addToQueue({ 
        actionType: "devalue", 
        channelId: channel?.id || 0 
      });
    } else if (action === "whatsapp") {
      addToQueue({ 
        actionType: "whatsapp", 
        channelId: channel?.id || 0 
      });
    } else {
      if (!selectedVideo) {
        toast({
          title: "No Video Selected",
          description: "Please select a video first",
          variant: "destructive"
        });
        return;
      }
      
      if (action === "boost") {
        addToQueue({ 
          actionType: "boost", 
          channelId: channel?.id || 0, 
          videoId: selectedVideo 
        });
      } else if (action === "session") {
        addToQueue({ 
          actionType: "session", 
          channelId: channel?.id || 0, 
          videoId: selectedVideo 
        });
      }
    }
  };

  // Check if no channels have been added yet
  const isNoChannelsError = channelError && (channelError as any).message?.includes("No active channel selected");

  if (!channel && !isNoChannelsError) {
    return (
      <div className="min-h-screen bg-background flex items-center justify-center">
        <div className="text-center">
          <div className="w-8 h-8 border-4 border-primary border-t-transparent rounded-full animate-spin mx-auto mb-4"></div>
          <p className="text-muted-foreground">Loading channel data...</p>
        </div>
      </div>
    );
  }

  // Show no channels state
  if (isNoChannelsError || !channel) {
    return (
      <div className="min-h-screen bg-gray-50">
        {/* Header */}
        <header className="bg-white border-b border-gray-200 shadow-sm">
          <div className="max-w-full mx-auto px-6 py-3">
            <div className="flex items-center justify-between">
              <div className="flex items-center space-x-4">
                <div className="flex items-center space-x-3">
                  <svg width="24" height="24" viewBox="0 0 24 24" className="text-google-blue">
                    <path fill="currentColor" d="M21.58 7.19c-.23-.86-.91-1.54-1.77-1.77C18.25 5 12 5 12 5s-6.25 0-7.81.42c-.86.23-1.54.91-1.77 1.77C2 8.75 2 12 2 12s0 3.25.42 4.81c.23.86.91 1.54 1.77 1.77C5.75 19 12 19 12 19s6.25 0 7.81-.42c.86-.23 1.54-.91 1.77-1.77C22 15.25 22 12 22 12s0-3.25-.42-4.81zM10 15V9l5.2 3L10 15z"/>
                  </svg>
                  <div>
                    <div className="flex items-center space-x-3">
                      <h1 className="text-xl font-medium text-gray-900">TubeBoost</h1>
                      <div className="inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-semibold bg-gradient-to-r from-google-blue to-google-green text-white shadow-sm">
                        v3.2.7
                      </div>
                    </div>
                    <p className="text-sm text-gray-500">Advanced YouTube Growth Platform</p>
                  </div>
                </div>
              </div>
              <div className="flex items-center space-x-4">
                <PointsDisplay />
                <a href="https://tubeboost.cc" target="_blank" rel="noopener noreferrer">
                  <Button 
                    variant="outline" 
                    size="sm"
                    className="text-gray-600 hover:text-gray-900 border-gray-300 hover:border-gray-400"
                  >
                    Visit Site
                  </Button>
                </a>
                <div className="flex items-center space-x-3">
                  <div className="w-8 h-8 bg-gray-100 rounded-full flex items-center justify-center">
                    <User className="w-4 h-4 text-gray-600" />
                  </div>
                  <span className="text-sm text-gray-700">{user?.username}</span>
                  <Button
                    variant="ghost"
                    size="sm"
                    onClick={logout}
                    className="text-gray-500 hover:text-gray-700"
                  >
                    <LogOut className="w-4 h-4" />
                  </Button>
                </div>
              </div>
            </div>
          </div>
        </header>

        <div className="flex">
          {/* Sidebar */}
          <nav className="w-64 bg-gray-50 border-r border-gray-200 h-[calc(100vh-73px)]">
            <div className="p-4">
              <div className="space-y-1">
                <Link href="/dashboard">
                  <div className={`flex items-center space-x-3 px-3 py-2 rounded-md text-sm font-medium cursor-pointer google-transition ${
                    pathname === "/dashboard" ? "bg-google-blue text-white" : "text-gray-600 hover:bg-gray-100 hover:text-gray-900"
                  }`}>
                    <span className="material-icons text-lg">dashboard</span>
                    <span>Dashboard</span>
                  </div>
                </Link>
                <Link href="/video-library">
                  <div className={`flex items-center space-x-3 px-3 py-2 rounded-md text-sm font-medium cursor-pointer google-transition ${
                    pathname === "/video-library" ? "bg-google-blue text-white" : "text-gray-600 hover:bg-gray-100 hover:text-gray-900"
                  }`}>
                    <span className="material-icons text-lg">video_library</span>
                    <span>Video Library</span>
                  </div>
                </Link>
                <Link href="/analytics">
                  <div className={`flex items-center space-x-3 px-3 py-2 rounded-md text-sm font-medium cursor-pointer google-transition ${
                    location === "/analytics" ? "bg-google-blue text-white" : "text-gray-600 hover:bg-gray-100 hover:text-gray-900"
                  }`}>
                    <span className="material-icons text-lg">analytics</span>
                    <span>Analytics</span>
                  </div>
                </Link>
                {user?.role === "admin" && (
                  <>
                    <Link href="/api">
                      <div className={`flex items-center space-x-3 px-3 py-2 rounded-md text-sm font-medium cursor-pointer google-transition ${
                        location === "/api" ? "bg-google-blue text-white" : "text-gray-600 hover:bg-gray-100 hover:text-gray-900"
                      }`}>
                        <span className="material-icons text-lg">api</span>
                        <span>API Services</span>
                      </div>
                    </Link>
                    <Link href="/system-users">
                      <div className={`flex items-center space-x-3 px-3 py-2 rounded-md text-sm font-medium cursor-pointer google-transition ${
                        location === "/system-users" ? "bg-google-blue text-white" : "text-gray-600 hover:bg-gray-100 hover:text-gray-900"
                      }`}>
                        <span className="material-icons text-lg">people</span>
                        <span>System Users</span>
                      </div>
                    </Link>
                    <Link href="/admin/points">
                      <div className={`flex items-center space-x-3 px-3 py-2 rounded-md text-sm font-medium cursor-pointer google-transition ${
                        location === "/admin/points" ? "bg-google-blue text-white" : "text-gray-600 hover:bg-gray-100 hover:text-gray-900"
                      }`}>
                        <span className="material-icons text-lg">stars</span>
                        <span>Point Management</span>
                      </div>
                    </Link>
                    <Link href="/admin/security">
                      <div className={`flex items-center space-x-3 px-3 py-2 rounded-md text-sm font-medium cursor-pointer google-transition ${
                        location === "/admin/security" ? "bg-google-blue text-white" : "text-gray-600 hover:bg-gray-100 hover:text-gray-900"
                      }`}>
                        <span className="material-icons text-lg">security</span>
                        <span>Security Center</span>
                      </div>
                    </Link>
                    <Link href="/reddit-moderation">
                      <div className="flex items-center space-x-3 px-3 py-2 rounded-md text-sm font-medium cursor-pointer google-transition bg-orange-600 text-white hover:bg-orange-700">
                        <span className="material-icons text-lg">shield</span>
                        <span>AI Reddit Moderation</span>
                      </div>
                    </Link>
                    <div className="flex items-center space-x-3 px-3 py-2 rounded-md text-gray-600 hover:bg-gray-100 hover:text-gray-900 text-sm cursor-pointer google-transition">
                      <span className="material-icons text-lg">tune</span>
                      <span>Algorithm Tools</span>
                    </div>
                  </>
                )}
              </div>
            </div>
          </nav>

          {/* No Channels State */}
          <main className="flex-1 p-6">
            <div className="max-w-2xl mx-auto mt-20 text-center">
              <div className="bg-white rounded-lg border border-gray-200 p-8 google-shadow">
                <div className="w-16 h-16 bg-google-blue rounded-full flex items-center justify-center mx-auto mb-6">
                  <svg width="32" height="32" viewBox="0 0 24 24" className="text-white">
                    <path fill="currentColor" d="M21.58 7.19c-.23-.86-.91-1.54-1.77-1.77C18.25 5 12 5 12 5s-6.25 0-7.81.42c-.86.23-1.54.91-1.77 1.77C2 8.75 2 12 2 12s0 3.25.42 4.81c.23.86.91 1.54 1.77 1.77C5.75 19 12 19 12 19s6.25 0 7.81-.42c.86-.23 1.54-.91 1.77-1.77C22 15.25 22 12 22 12s0-3.25-.42-4.81zM10 15V9l5.2 3L10 15z"/>
                  </svg>
                </div>
                <h2 className="text-2xl font-semibold text-gray-900 mb-4">No Channels Added</h2>
                <p className="text-gray-600 mb-8">
                  Add your first YouTube channel to start managing and optimizing with Google's internal algorithm tools.
                </p>
                <ChannelDropdown 
                  selectedChannelId={undefined} 
                  onChannelChange={setSelectedChannelId} 
                />
              </div>
            </div>
          </main>
        </div>
      </div>
    );
  }

  // YouTube Dashboard Content
  const youtubeContent = (
    <div className="flex-1 flex flex-col bg-gray-50 dark:bg-gray-900">
        {/* Anonymous Account Warning */}
        {user?.isAnonymous && <AnonymousWarningBanner />}
        
        {/* Email Incentive Banner */}
        {user?.isAnonymous && <EmailIncentiveBanner />}
        
        {/* Anonymous Credentials Display */}
        {user?.isAnonymous && <AnonymousCredentialsDisplay />}

        {/* Show queue panel */}
        <QueuePanel />

        {/* YouTube Platform Content */}
        {true && (
          <div className="p-6">
            <div className="max-w-7xl mx-auto">
              {/* Reddit Connection Status */}
              <div className="bg-white rounded-lg border border-gray-200 google-shadow p-6 mb-6">
                <h1 className="text-2xl font-bold text-gray-900 mb-4">Reddit Dashboard</h1>
                <p className="text-gray-600 mb-6">
                  View comprehensive analytics for your managed subreddits with real-time Reddit API data.
                </p>
                
                <RedditConnection />
              </div>

              {/* Reddit Analytics Dashboard */}
              <RedditAnalyticsDashboard />

              {/* Quick Actions */}
              <div className="mt-6 grid grid-cols-1 md:grid-cols-1 gap-6">
                <div className="bg-white rounded-lg border border-gray-200 google-shadow p-6">
                  <h3 className="font-semibold text-gray-900 mb-2 flex items-center gap-2">
                    <Shield className="h-5 w-5 text-blue-600" />
                    AI Moderation
                  </h3>
                  <p className="text-gray-600 text-sm mb-4">Semi-automatic moderation with Claude Sonnet 4 AI analysis.</p>
                  <Button asChild className="w-full">
                    <a href="/reddit-moderation">
                      Moderate Subreddits
                    </a>
                  </Button>
                </div>
              </div>
            </div>
          </div>
        )}

        {/* YouTube Platform Content */}
        <div className="p-6">
          <div className="max-w-7xl mx-auto">
            <div className="flex-1 flex flex-col bg-gray-50 dark:bg-gray-900">
          <div className="flex-1 flex flex-col bg-gray-50 dark:bg-gray-900">
            {/* Top Controls */}
            <div className="bg-white dark:bg-gray-800 border-b border-gray-200 dark:border-gray-700 p-4">
              <div className="flex flex-wrap items-center justify-between gap-4">
                <div className="flex items-center space-x-4">
              {/* Channel Management Buttons */}
              <Dialog open={shareDialogOpen} onOpenChange={setShareDialogOpen}>
                <DialogTrigger asChild>
                  <Button variant="outline" size="sm" className="flex items-center space-x-2">
                    <Share2 className="w-4 h-4" />
                    <span>Add Channel</span>
                  </Button>
                </DialogTrigger>
                <DialogContent>
                  <DialogHeader>
                    <DialogTitle>Access Existing Channel</DialogTitle>
                    <DialogDescription>
                      Get access to a channel that's already in the management system
                    </DialogDescription>
                  </DialogHeader>
                  <div className="space-y-4">
                    <div>
                      <label htmlFor="shareChannelUrl" className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-1">
                        YouTube Channel URL or ID
                      </label>
                      <Input
                        id="shareChannelUrl"
                        placeholder="https://youtube.com/@channelname or UCXXXXXXXXX"
                        value={shareChannelUrl}
                        onChange={(e) => setShareChannelUrl(e.target.value)}
                      />
                      <div className="text-xs text-gray-500 dark:text-gray-400 mt-2 space-y-1">
                        <p>• Enter a channel URL or ID that's already managed in the system</p>
                        <p>• If the channel isn't in the system yet, use the main "Add Channel" button instead</p>
                        <p>• Supported formats: @username, channel/UC..., or direct channel IDs</p>
                      </div>
                    </div>
                    <div className="flex justify-end space-x-2">
                      <Button variant="outline" onClick={() => setShareDialogOpen(false)}>
                        Cancel
                      </Button>
                      <Button 
                        onClick={() => shareChannel(shareChannelUrl)}
                        disabled={!shareChannelUrl || isSharingChannel}
                      >
                        {isSharingChannel ? "Requesting Access..." : "Request Access"}
                      </Button>
                    </div>
                  </div>
                </DialogContent>
              </Dialog>

              {channel && (
                <Button 
                  variant="destructive"
                  size="sm"
                  onClick={() => deleteChannel(channel.id)}
                  disabled={isDeletingChannel}
                  className="flex items-center space-x-2"
                >
                  <Trash2 className="w-4 h-4" />
                  <span>{isDeletingChannel ? "Removing..." : "Remove Channel"}</span>
                </Button>
              )}
            </div>

            <div className="flex items-center space-x-2">
              <Button
                variant="default"
                size="sm"
                onClick={() => syncChannel()}
                disabled={isSyncingChannel}
                className="flex items-center space-x-2"
              >
                <span className={`material-icons text-sm ${isSyncingChannel ? 'animate-spin' : ''}`}>
                  sync
                </span>
                <span>{isSyncingChannel ? 'Syncing...' : 'Sync YouTube'}</span>
              </Button>
              
              <Button
                variant="secondary"
                size="sm"
                onClick={() => syncOrderStatuses()}
                disabled={isSyncingOrders}
                className="flex items-center space-x-2"
              >
                <span className={`material-icons text-sm ${isSyncingOrders ? 'animate-spin' : ''}`}>
                  update
                </span>
                <span>{isSyncingOrders ? 'Syncing...' : 'Sync Orders'}</span>
              </Button>
            </div>
          </div>
        </div>

        {/* Main Dashboard Content */}
        <div className="p-6 bg-gray-50 dark:bg-gray-900">
          {/* Email Incentive Banner */}
          <EmailIncentiveBanner />
          
          {/* Channel Selection */}
          <ChannelDropdown 
            selectedChannelId={selectedChannelId || undefined} 
            onChannelChange={setSelectedChannelId} 
          />
          
          {/* Anonymous User Warning Banner */}
          <AnonymousWarningBanner />
          
          {/* Anonymous Credentials Display */}
          <AnonymousCredentialsDisplay />
          


          {/* Point Transaction History */}
          <div className="mb-6">
            <PointTransactionHistory limit={10} className="w-full" />
          </div>

          {/* System Operations */}
          <div className="bg-white rounded-lg border border-gray-200 p-4 sm:p-6 google-shadow mb-6">
            <div className="flex items-center justify-between mb-4">
              <h2 className="text-lg font-semibold text-gray-900">System Operations</h2>
              <div className="text-sm text-gray-500">
                Active queue items: {queueItems.filter(item => ['queued', 'running', 'paused'].includes(item.status)).length}
              </div>
            </div>
            
            {queueItems.length === 0 ? (
              <div className="text-center py-8 text-gray-500">
                <div className="w-16 h-16 bg-gray-100 rounded-full flex items-center justify-center mx-auto mb-4">
                  <span className="material-icons text-gray-400 text-2xl">queue</span>
                </div>
                <p className="text-lg font-medium text-gray-900 mb-2">No Operations in Queue</p>
                <p className="text-gray-600">Use Algorithm Tools to add operations to the queue.</p>
              </div>
            ) : (
              <div className="space-y-3">
                {queueItems
                  .filter(item => ['queued', 'running', 'paused'].includes(item.status))
                  .map((item) => (
                    <div key={item.id} className="bg-gray-50 rounded-lg p-4 border border-gray-200">
                      <div className="flex items-center justify-between mb-3">
                        <div className="flex items-center space-x-3">
                          <div className={`w-8 h-8 rounded-full flex items-center justify-center ${
                            item.status === 'running' ? 'bg-green-500' :
                            item.status === 'paused' ? 'bg-yellow-500' :
                            'bg-blue-500'
                          }`}>
                            <span className="material-icons text-white text-sm">
                              {item.actionType === 'boost' ? 'rocket_launch' :
                               item.actionType === 'session' ? 'psychology' :
                               item.actionType === 'replenish' ? 'refresh' :
                               item.actionType === 'devalue' ? 'trending_down' :
                               item.actionType === 'whatsapp' ? 'trending_up' : 'play_arrow'}
                            </span>
                          </div>
                          <div>
                            <p className="font-medium text-gray-900 capitalize">
                              {item.actionType === 'whatsapp' ? 'Boost Channel Exposure' :
                               item.actionType === 'boost' ? 'Boost Video' :
                               item.actionType === 'session' ? 'AG Session' :
                               item.actionType === 'replenish' ? 'Replenish AG Points' :
                               item.actionType === 'devalue' ? 'Devalue Competitor' :
                               item.actionType}
                            </p>
                            <p className="text-sm text-gray-500">
                              {item.channelName || 'Channel'} • {item.videoTitle || 'Channel Operation'}
                            </p>
                          </div>
                        </div>
                        
                        <div className="text-right">
                          <div className={`inline-flex items-center px-2 py-1 rounded-full text-xs font-medium ${
                            item.status === 'running' ? 'bg-green-100 text-green-800' :
                            item.status === 'paused' ? 'bg-yellow-100 text-yellow-800' :
                            item.status === 'queued' ? 'bg-blue-100 text-blue-800' :
                            'bg-gray-100 text-gray-800'
                          }`}>
                            {item.status === 'running' && (
                              <div className="w-2 h-2 bg-green-500 rounded-full animate-pulse mr-1"></div>
                            )}
                            {item.status}
                          </div>
                          {item.estimatedDuration && (
                            <p className="text-xs text-gray-500 mt-1">
                              Est. {Math.round(item.estimatedDuration / 60)}m
                            </p>
                          )}
                        </div>
                      </div>
                      
                      {item.status === 'running' && (
                        <div className="space-y-2">
                          <div className="flex items-center justify-between text-sm">
                            <span className="text-gray-600">
                              {item.currentStep || 'Processing...'}
                            </span>
                            <span className="font-medium text-gray-900">
                              {item.progress || 0}%
                            </span>
                          </div>
                          <div className="w-full bg-gray-200 rounded-full h-2">
                            <div 
                              className="bg-green-500 h-2 rounded-full transition-all duration-300" 
                              style={{ width: `${item.progress || 0}%` }}
                            ></div>
                          </div>
                        </div>
                      )}
                      
                      {item.status === 'queued' && (
                        <div className="text-sm text-gray-500">
                          Waiting in queue • Position #{queueItems.filter(q => q.status === 'queued' && q.id <= item.id).length}
                        </div>
                      )}
                    </div>
                  ))}
              </div>
            )}
          </div>

          {/* Dashboard Stats */}
          <div className="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-4 gap-3 sm:gap-4 mb-4 sm:mb-6">
            <div className="bg-white rounded-lg border border-gray-200 p-4 sm:p-6 google-shadow">
              <div className="flex items-center justify-between">
                <div>
                  <p className="text-gray-500 text-sm font-medium">Total Views</p>
                  <p className="text-xl sm:text-2xl font-bold text-gray-900">{formatNumber(channel.totalViews)}</p>
                </div>
                <div className="w-8 h-8 sm:w-10 sm:h-10 bg-blue-50 rounded-lg flex items-center justify-center">
                  <span className="material-icons text-google-blue text-lg sm:text-xl">visibility</span>
                </div>
              </div>
            </div>
            
            <div className="bg-white rounded-lg border border-gray-200 p-4 sm:p-6 google-shadow">
              <div className="flex items-center justify-between">
                <div>
                  <p className="text-gray-500 text-sm font-medium">Subscribers</p>
                  <p className="text-xl sm:text-2xl font-bold text-gray-900">{formatNumber(channel.subscribers)}</p>
                </div>
                <div className="w-8 h-8 sm:w-10 sm:h-10 bg-green-50 rounded-lg flex items-center justify-center">
                  <span className="material-icons text-google-green text-lg sm:text-xl">people</span>
                </div>
              </div>
            </div>

            <div className="bg-white rounded-lg border border-gray-200 p-4 sm:p-6 google-shadow">
              <div className="flex items-center justify-between">
                <div>
                  <p className="text-gray-500 text-sm font-medium">Watch Time</p>
                  <p className="text-xl sm:text-2xl font-bold text-gray-900">{channel.watchTimeHours}h</p>
                </div>
                <div className="w-8 h-8 sm:w-10 sm:h-10 bg-yellow-50 rounded-lg flex items-center justify-center">
                  <span className="material-icons text-google-yellow text-lg sm:text-xl">schedule</span>
                </div>
              </div>
            </div>

            <div className="bg-white rounded-lg border border-gray-200 p-4 sm:p-6 google-shadow">
              <div className="flex items-center justify-between">
                <div>
                  <p className="text-gray-500 text-sm font-medium">AG Score</p>
                  <p className="text-xl sm:text-2xl font-bold text-google-blue">{channel.agScore.toFixed(1)}</p>
                </div>
                <div className="w-8 h-8 sm:w-10 sm:h-10 bg-blue-50 rounded-lg flex items-center justify-center">
                  <span className="material-icons text-google-blue text-lg sm:text-xl">auto_graph</span>
                </div>
              </div>
            </div>
          </div>

          {/* YouTube Toolbox */}
          <div className="bg-white rounded-lg border border-gray-200 p-4 sm:p-6 google-shadow mb-6">
            <div className="flex items-center justify-between mb-4">
              <h2 className="text-lg font-semibold text-gray-900">YouTube Toolbox</h2>
            </div>
            <div className="grid grid-cols-2 sm:grid-cols-3 md:grid-cols-4 lg:grid-cols-6 gap-2 sm:gap-3">
              <button
                onClick={() => handleActionClick("boost")}
                disabled={!selectedVideo}
                className="bg-white border border-gray-200 rounded-lg p-3 google-shadow hover:shadow-lg google-transition disabled:opacity-50 disabled:cursor-not-allowed group"
              >
                <div className="flex flex-col items-center space-y-2">
                  <div className="w-8 h-8 bg-google-blue rounded-full flex items-center justify-center group-hover:scale-110 google-transition">
                    <span className="material-icons text-white text-sm">rocket_launch</span>
                  </div>
                  <div className="text-center">
                    <h3 className="font-medium text-gray-900 text-xs mb-1">Boost Video</h3>
                    <p className="text-xs text-gray-500">Optimize selected video for algorithm performance</p>
                    <div className="mt-1 text-xs text-google-blue font-medium">Duration: 1 minute</div>
                  </div>
                </div>
              </button>

              <button
                onClick={() => handleActionClick("session")}
                disabled={!selectedVideo}
                className="bg-white border border-gray-200 rounded-lg p-3 google-shadow hover:shadow-lg google-transition disabled:opacity-50 disabled:cursor-not-allowed group"
              >
                <div className="flex flex-col items-center space-y-2">
                  <div className="w-8 h-8 bg-google-green rounded-full flex items-center justify-center group-hover:scale-110 google-transition">
                    <span className="material-icons text-white text-sm">psychology</span>
                  </div>
                  <div className="text-center">
                    <h3 className="font-medium text-gray-900 text-xs mb-1">Create AG Session</h3>
                    <p className="text-xs text-gray-500">Advanced algorithm optimization session</p>
                    <div className="mt-1 text-xs text-google-green font-medium">Duration: 2 minutes</div>
                  </div>
                </div>
              </button>

              <button
                onClick={() => handleActionClick("replenish")}
                disabled={cooldownStatus?.isActive}
                className="bg-white border border-gray-200 rounded-lg p-3 google-shadow hover:shadow-lg google-transition disabled:opacity-50 disabled:cursor-not-allowed group"
              >
                <div className="flex flex-col items-center space-y-2">
                  <div className="w-8 h-8 bg-google-yellow rounded-full flex items-center justify-center group-hover:scale-110 google-transition">
                    <span className="material-icons text-gray-900 text-sm">refresh</span>
                  </div>
                  <div className="text-center">
                    <h3 className="font-medium text-gray-900 text-xs mb-1">Replenish AG Points</h3>
                    <p className="text-xs text-gray-500">Restore algorithm grade points (+1.6)</p>
                    <div className="mt-1 text-xs text-google-yellow font-medium">Duration: 4 minutes</div>
                  </div>
                </div>
              </button>

              <button
                onClick={() => handleActionClick("devalue")}
                className="bg-white border border-gray-200 rounded-lg p-3 google-shadow hover:shadow-lg google-transition group"
              >
                <div className="flex flex-col items-center space-y-2">
                  <div className="w-8 h-8 bg-red-500 rounded-full flex items-center justify-center group-hover:scale-110 google-transition">
                    <span className="material-icons text-white text-sm">trending_down</span>
                  </div>
                  <div className="text-center">
                    <h3 className="font-medium text-gray-900 text-xs mb-1">Devalue Channel</h3>
                    <p className="text-xs text-gray-500">Advanced simulation to reduce AG points (-2.4)</p>
                    <div className="mt-1 text-xs text-red-500 font-medium">Duration: 5 minutes</div>
                  </div>
                </div>
              </button>

              <button
                onClick={() => handleActionClick("whatsapp")}
                className="bg-white border border-gray-200 rounded-lg p-3 google-shadow hover:shadow-lg google-transition group"
              >
                <div className="flex flex-col items-center space-y-2">
                  <div className="w-8 h-8 bg-red-500 rounded-full flex items-center justify-center group-hover:scale-110 google-transition">
                    <span className="material-icons text-white text-sm">trending_up</span>
                  </div>
                  <div className="text-center">
                    <h3 className="font-medium text-gray-900 text-xs mb-1">Boost Channel Exposure</h3>
                    <p className="text-xs text-gray-500">Enhance YouTube visibility and reach</p>
                    <div className="mt-1 text-xs text-red-500 font-medium">Duration: 2 minutes</div>
                  </div>
                </div>
              </button>

              <button
                onClick={() => setLoadViewsModal(true)}
                disabled={!selectedVideo || (userPoints?.remainingPoints || 0) < 100}
                className="bg-white border border-gray-200 rounded-lg p-3 google-shadow hover:shadow-lg google-transition disabled:opacity-50 disabled:cursor-not-allowed group"
              >
                <div className="flex flex-col items-center space-y-2">
                  <div className="w-8 h-8 bg-purple-500 rounded-full flex items-center justify-center group-hover:scale-110 google-transition">
                    <span className="material-icons text-white text-sm">visibility</span>
                  </div>
                  <div className="text-center">
                    <h3 className="font-medium text-gray-900 text-xs mb-1">Load Video Views</h3>
                    <p className="text-xs text-gray-500">Add views via TubeBoost API (minimum 100)</p>
                    <div className="mt-1 text-xs text-purple-500 font-medium">
                      {userPoints?.remainingPoints || 0} points available
                    </div>
                  </div>
                </div>
              </button>

              {/* Add Likes to Video Button */}
              <button
                onClick={() => setLikesDialogOpen(true)}
                disabled={!selectedVideo || (userPoints?.remainingPoints || 0) < 10}
                className="bg-white border border-gray-200 rounded-lg p-3 google-shadow hover:shadow-lg google-transition disabled:opacity-50 disabled:cursor-not-allowed group"
              >
                <div className="flex flex-col items-center space-y-2">
                  <div className="w-8 h-8 bg-pink-500 rounded-full flex items-center justify-center group-hover:scale-110 google-transition">
                    <span className="material-icons text-white text-sm">thumb_up</span>
                  </div>
                  <div className="text-center">
                    <h3 className="font-medium text-gray-900 text-xs mb-1">Add Likes</h3>
                    <p className="text-xs text-gray-500">Add likes via TubeBoost (minimum 10)</p>
                    <div className="mt-1 text-xs text-pink-500 font-medium">
                      {userPoints?.remainingPoints || 0} points available
                    </div>
                  </div>
                </div>
              </button>

              {/* Add Subscribers to Channel Button */}
              <button
                onClick={() => setSubscriberDialogOpen(true)}
                disabled={(userPoints?.remainingPoints || 0) < 50}
                className="bg-white border border-gray-200 rounded-lg p-3 google-shadow hover:shadow-lg google-transition disabled:opacity-50 disabled:cursor-not-allowed group"
              >
                <div className="flex flex-col items-center space-y-2">
                  <div className="w-8 h-8 bg-orange-500 rounded-full flex items-center justify-center group-hover:scale-110 google-transition">
                    <span className="material-icons text-white text-sm">person_add</span>
                  </div>
                  <div className="text-center">
                    <h3 className="font-medium text-gray-900 text-xs mb-1">Add Subscribers</h3>
                    <p className="text-xs text-gray-500">Add subscribers via TubeBoost (minimum 50)</p>
                    <div className="mt-1 text-xs text-orange-500 font-medium">
                      {userPoints?.remainingPoints || 0} points available
                    </div>
                  </div>
                </div>
              </button>

              {/* Admin-only Custom Load Views Button */}
              {user?.role === "admin" && (
                <button
                  onClick={() => setAdminLoadViewsModal(true)}
                  disabled={!selectedVideo}
                  className="bg-white border border-red-200 rounded-lg p-3 google-shadow hover:shadow-lg google-transition disabled:opacity-50 disabled:cursor-not-allowed group"
                >
                  <div className="flex flex-col items-center space-y-2">
                    <div className="w-8 h-8 bg-red-500 rounded-full flex items-center justify-center group-hover:scale-110 google-transition">
                      <span className="material-icons text-white text-sm">admin_panel_settings</span>
                    </div>
                    <div className="text-center">
                      <h3 className="font-medium text-gray-900 text-xs mb-1">ADMIN: Load Video Views</h3>
                      <p className="text-xs text-gray-500">Custom quantity (bypass limits)</p>
                      <div className="mt-1 text-xs text-red-500 font-medium">
                        Administrator Override
                      </div>
                    </div>
                  </div>
                </button>
              )}
            </div>
          </div>

          {/* Subscriber Transactions */}
          <div className="bg-white rounded-lg border border-gray-200 google-shadow">
            <div className="p-4 sm:p-6 border-b border-gray-200">
              <h2 className="text-lg font-semibold text-gray-900 mb-2 sm:mb-4">Subscriber Transactions</h2>
              <p className="text-sm text-gray-600">
                View all subscriber orders for this channel with live status tracking and refill functionality.
              </p>
            </div>
            
            <div className="p-4 sm:p-6">
              <SubscriberTransactions channelId={channel?.id} />
            </div>
          </div>

          {/* Video Manager */}
          <div className="bg-white rounded-lg border border-gray-200 google-shadow">
            <div className="p-6 border-b border-gray-200">
              <h2 className="text-lg font-semibold text-gray-900 mb-4">Video Manager</h2>
              <div className="flex items-center space-x-4">
                <div className="flex-1 relative">
                  <span className="material-icons absolute left-3 top-1/2 transform -translate-y-1/2 text-gray-400 text-sm">search</span>
                  <input
                    type="text"
                    placeholder="Search videos..."
                    value={searchTerm}
                    onChange={(e) => setSearchTerm(e.target.value)}
                    className="w-full pl-10 pr-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-google-blue focus:border-transparent"
                  />
                </div>
                <button className="flex items-center space-x-2 px-4 py-2 border border-gray-300 rounded-lg hover:bg-gray-50 google-transition">
                  <span className="material-icons text-gray-600 text-sm">filter_list</span>
                  <span className="text-sm text-gray-700">Filter</span>
                </button>
              </div>
            </div>
            
            <div className="p-6">
              <div className="space-y-3">
                {filteredVideos.map((video) => (
                  <div
                    key={video.id}
                    className={`p-4 rounded-lg border cursor-pointer google-transition ${
                      selectedVideo === video.id
                        ? "border-google-blue bg-blue-50 shadow-md"
                        : "border-gray-200 hover:border-gray-300 hover:shadow-sm"
                    }`}
                    onClick={() => setSelectedVideo(video.id)}
                  >
                    <div className="flex items-start space-x-4">
                      <div className="relative">
                        <img
                          src={video.thumbnailUrl || 'https://images.unsplash.com/photo-1517077304055-6e89abbf09b0?ixlib=rb-4.0.3&auto=format&fit=crop&w=128&h=72'}
                          alt={video.title}
                          className="w-32 h-18 object-cover rounded-lg"
                        />
                        {selectedVideo === video.id && (
                          <div className="absolute -top-2 -right-2 w-6 h-6 bg-google-blue rounded-full flex items-center justify-center">
                            <span className="material-icons text-white text-sm">check</span>
                          </div>
                        )}
                      </div>
                      <div className="flex-1">
                        <h3 className="font-medium text-gray-900 mb-2 leading-tight">{video.title}</h3>
                        <div className="flex items-center space-x-4 text-sm text-gray-500 mb-3">
                          <span className="flex items-center space-x-1">
                            <span className="material-icons text-xs">visibility</span>
                            <span>{formatNumber(video.views)} views</span>
                          </span>
                          <span>•</span>
                          <span className="flex items-center space-x-1">
                            <span className="material-icons text-xs">thumb_up</span>
                            <span>
                              {youtubeStatsMap[video.videoId] ? 
                                formatYouTubeNumber(youtubeStatsMap[video.videoId].likeCount) : 
                                (youtubeStatsLoading ? 'Loading...' : 'N/A')
                              } likes
                            </span>
                          </span>
                          <span>•</span>
                          <span className="flex items-center space-x-1">
                            <span className="material-icons text-xs">schedule</span>
                            <span>
                              {youtubeStatsMap[video.videoId] ? 
                                youtubeStatsMap[video.videoId].duration : 
                                (youtubeStatsLoading ? 'Loading...' : 'N/A')
                              }
                            </span>
                          </span>
                          <span>•</span>
                          <span>{video.publishedAt ? new Date(video.publishedAt).toLocaleDateString() : 'N/A'}</span>
                        </div>
                        <div className="flex items-center space-x-3">
                          <div className="flex items-center space-x-2">
                            <div className="text-xs text-gray-500">CTR:</div>
                            <div className="px-2 py-1 bg-green-100 text-green-700 rounded text-xs font-medium">
                              +{video.ctr.toFixed(1)}%
                            </div>
                          </div>
                          <div className="flex items-center space-x-2">
                            <div className="text-xs text-gray-500">Algorithm Score:</div>
                            <div className="px-2 py-1 bg-blue-100 text-blue-700 rounded text-xs font-medium">
                              {video.algorithmScore.toFixed(1)}
                            </div>
                          </div>
                          <div className="flex items-center space-x-2">
                            <div className="text-xs text-gray-500">SEO:</div>
                            <div className="px-2 py-1 bg-yellow-100 text-yellow-700 rounded text-xs font-medium">
                              {video.seoScore}
                            </div>
                          </div>
                        </div>
                        
                        {/* Persistent Transaction Log with Refill Status - Always show for all users managing the channel */}
                        <div className="mt-4 pt-4 border-t border-gray-100">
                          <div className="flex items-center justify-between mb-3">
                            <div className="text-sm font-medium text-gray-900">Transaction Log</div>
                            {user?.role === 'admin' && (
                              <div className="text-xs text-red-600 font-medium">ADMIN ACCESS</div>
                            )}
                          </div>
                          <div className="space-y-3 max-h-40 overflow-y-auto">
                            {/* Show message when no orders exist */}
                            {(!videoOrdersMap[video.id] || !Array.isArray(videoOrdersMap[video.id]) || videoOrdersMap[video.id].length === 0) && (
                              <div className="text-xs text-gray-500 italic">
                                No orders placed for this video yet
                              </div>
                            )}
                            {videoOrdersMap[video.id] && Array.isArray(videoOrdersMap[video.id]) && videoOrdersMap[video.id].map((order: any, index: number) => (
                                <div key={index} className="bg-gray-50 rounded-lg p-3 border border-gray-200">
                                  <div className="flex items-center justify-between mb-2">
                                    <div className="flex items-center space-x-2">
                                      <div className={`px-2 py-1 rounded text-xs font-medium ${getOrderStatusColor(order.status)}`}>
                                        {order.status}
                                      </div>
                                      <div className="text-xs text-gray-600">
                                        {order.quantity} {getServiceTypeDisplay(order.service)}
                                      </div>
                                      <div className="text-xs text-gray-500">
                                        Order #{order.orderId?.slice(-6)}
                                      </div>
                                    </div>

                                  </div>
                                  <div className="text-xs text-gray-500">
                                    {order.createdAt && new Date(order.createdAt).toLocaleString()}
                                  </div>
                                  
                                  {/* Order Details from TubeBoost API - Enhanced Display */}
                                  {(order.charge || order.startCount || order.remains || order.currency) && (
                                    <div className="bg-white rounded-lg p-3 mt-3 border border-gray-100">
                                      <div className="flex items-center justify-between mb-2">
                                        <span className="text-xs font-medium text-gray-700">Order Details</span>
                                        <span className="text-xs text-gray-500">Live from TubeBoost API</span>
                                      </div>
                                      <div className="grid grid-cols-2 md:grid-cols-4 gap-3 text-xs">
                                        {order.charge && (
                                          <div className="flex flex-col">
                                            <span className="text-gray-500 font-medium">Charge</span>
                                            <span className="text-gray-900 font-semibold">
                                              ${order.charge} {order.currency || 'USD'}
                                            </span>
                                          </div>
                                        )}
                                        {order.startCount && (
                                          <div className="flex flex-col">
                                            <span className="text-gray-500 font-medium">Start Count</span>
                                            <span className="text-gray-900 font-semibold">{order.startCount}</span>
                                          </div>
                                        )}
                                        {order.remains && (
                                          <div className="flex flex-col">
                                            <span className="text-gray-500 font-medium">Remaining</span>
                                            <span className="text-gray-900 font-semibold">{order.remains}</span>
                                          </div>
                                        )}
                                        {order.status && ['In progress', 'Processing'].includes(order.status) && order.remains && order.quantity && (
                                          <div className="flex flex-col">
                                            <span className="text-gray-500 font-medium">Progress</span>
                                            <span className="text-gray-900 font-semibold">
                                              {Math.max(0, order.quantity - parseInt(order.remains || '0'))} / {order.quantity}
                                            </span>
                                          </div>
                                        )}
                                      </div>
                                    </div>
                                  )}
                                  
                                  {/* Refill Status Section */}
                                  <RefillStatus orderId={order.id} />
                                </div>
                              ))}
                            </div>
                          </div>
                        </div>
                    </div>
                  </div>
                ))}
              </div>
            </div>
          </div>

      {/* Simulation Modal */}
      <Dialog open={activeModal !== null} onOpenChange={() => setActiveModal(null)}>
        <DialogContent className="max-w-3xl bg-white border border-gray-200">
          <DialogHeader className="border-b border-gray-200 pb-4">
            <div className="flex items-center space-x-3">
              {activeModal === "boost" && (
                <div className="w-10 h-10 bg-google-blue rounded-full flex items-center justify-center">
                  <span className="material-icons text-white">rocket_launch</span>
                </div>
              )}
              {activeModal === "session" && (
                <div className="w-10 h-10 bg-google-green rounded-full flex items-center justify-center">
                  <span className="material-icons text-white">psychology</span>
                </div>
              )}
              {activeModal === "replenish" && (
                <div className="w-10 h-10 bg-google-yellow rounded-full flex items-center justify-center">
                  <span className="material-icons text-gray-900">refresh</span>
                </div>
              )}
              {activeModal === "devalue" && (
                <div className="w-10 h-10 bg-red-500 rounded-full flex items-center justify-center">
                  <span className="material-icons text-white">trending_down</span>
                </div>
              )}
              {activeModal === "whatsapp" && (
                <div className="w-10 h-10 bg-red-500 rounded-full flex items-center justify-center">
                  <span className="material-icons text-white">trending_up</span>
                </div>
              )}
              <div>
                <DialogTitle className="text-xl font-semibold text-gray-900">
                  {activeModal === "boost" && "Video Algorithm Optimization"}
                  {activeModal === "session" && "AG Session - Algorithm Enhancement"}
                  {activeModal === "replenish" && "AG Points Replenishment"}
                  {activeModal === "devalue" && "Channel Devaluation Simulation"}
                  {activeModal === "whatsapp" && "YouTube Channel Exposure Boost"}
                </DialogTitle>
                <p className="text-sm text-gray-500 mt-1">
                  {activeModal === "boost" && "Optimizing video for maximum algorithm performance"}
                  {activeModal === "session" && "Running advanced algorithm grade session"}
                  {activeModal === "replenish" && "Restoring algorithm grade points from Google servers"}
                  {activeModal === "devalue" && "Advanced simulation to reduce channel algorithm grade points"}
                  {activeModal === "whatsapp" && "Enhancing channel visibility across YouTube networks and algorithms"}
                </p>
              </div>
            </div>
          </DialogHeader>
          
          {simulationState.isRunning && (
            <div className="p-6 space-y-6">
              <div className="bg-gray-50 rounded-lg p-4">
                <div className="flex items-center justify-between mb-3">
                  <h3 className="font-medium text-gray-900">Progress</h3>
                  <span className="text-sm font-medium text-gray-600">{simulationState.progress.toFixed(0)}%</span>
                </div>
                
                <div className="w-full bg-gray-200 rounded-full h-3 mb-3">
                  <div 
                    className={`h-3 rounded-full transition-all duration-500 ${
                      activeModal === "boost" ? "bg-google-blue" :
                      activeModal === "session" ? "bg-google-green" :
                      activeModal === "replenish" ? "bg-google-yellow" :
                      activeModal === "devalue" ? "bg-red-500" :
                      "bg-green-500"
                    }`}
                    style={{ width: `${simulationState.progress}%` }}
                  ></div>
                </div>
                
                <div className="flex items-center space-x-2 text-sm">
                  <span className="material-icons text-gray-400 text-sm animate-spin">sync</span>
                  <span className="text-gray-700">{simulationState.currentStep}</span>
                </div>
              </div>

              {Object.keys(simulationState.metrics).length > 0 && (
                <div className="grid grid-cols-2 md:grid-cols-3 gap-4">
                  {Object.entries(simulationState.metrics).map(([key, value]) => (
                    <div key={key} className="bg-white border border-gray-200 rounded-lg p-4 google-shadow">
                      <div className="text-xs text-gray-500 font-medium mb-1">{key}</div>
                      <div className="text-2xl font-bold text-gray-900">{value}</div>
                    </div>
                  ))}
                </div>
              )}

              {/* Enhanced YouTube Boost Visualization */}
              {activeModal === "whatsapp" && (
                <div className="space-y-6">
                  {/* Real-time Network Visualization */}
                  <div className="bg-gradient-to-br from-red-900 to-red-700 rounded-xl p-6 text-white">
                    <div className="flex items-center justify-between mb-4">
                      <h3 className="text-xl font-bold">YouTube Network Activation</h3>
                      <div className="flex items-center space-x-2">
                        <div className="w-3 h-3 bg-red-400 rounded-full animate-pulse"></div>
                        <span className="text-red-300 text-sm">LIVE</span>
                      </div>
                    </div>
                    
                    <div className="grid grid-cols-3 gap-6 mb-6">
                      <div className="text-center">
                        <div className="text-3xl font-bold text-red-300">{Math.floor(25 + (simulationState.progress / 100) * 150)}</div>
                        <div className="text-sm text-red-200">Channels Reached</div>
                      </div>
                      <div className="text-center">
                        <div className="text-3xl font-bold text-red-300">{(simulationState.progress / 100 * 95).toFixed(1)}%</div>
                        <div className="text-sm text-red-200">Discovery Rate</div>
                      </div>
                      <div className="text-center">
                        <div className="text-3xl font-bold text-red-300">{Math.floor(10 + (simulationState.progress / 100) * 40)}x</div>
                        <div className="text-sm text-red-200">Reach Multiplier</div>
                      </div>
                    </div>

                    {/* Network Animation */}
                    <div className="relative h-32 bg-red-800/30 rounded-lg overflow-hidden">
                      <div className="absolute inset-0 flex items-center justify-center">
                        <div className="relative">
                          <div className="w-16 h-16 bg-red-400 rounded-full flex items-center justify-center animate-pulse">
                            <span className="material-icons text-red-900 text-2xl">hub</span>
                          </div>
                          {/* Animated connection lines */}
                          <div className="absolute top-1/2 left-1/2 transform -translate-x-1/2 -translate-y-1/2">
                            {[0, 45, 90, 135, 180, 225, 270, 315].map((angle, i) => (
                              <div
                                key={angle}
                                className="absolute w-1 bg-red-400 animate-pulse"
                                style={{
                                  height: '60px',
                                  transformOrigin: '50% 0%',
                                  transform: `rotate(${angle}deg)`,
                                  opacity: simulationState.progress > (i * 12.5) ? 1 : 0.3,
                                  transition: 'opacity 0.5s ease'
                                }}
                              >
                                <div className="absolute -top-2 -left-2 w-4 h-4 bg-red-300 rounded-full animate-ping"></div>
                              </div>
                            ))}
                          </div>
                        </div>
                      </div>
                    </div>
                  </div>

                  {/* AI Generated Graphics */}
                  {whatsappGraphics.length > 0 && (
                    <div className="space-y-4">
                      <div className="flex items-center space-x-3">
                        <span className="material-icons text-red-500 text-xl">auto_awesome</span>
                        <h3 className="text-lg font-semibold text-gray-900">AI-Generated YouTube Visuals</h3>
                      </div>
                      <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-4">
                        {whatsappGraphics.map((graphic, index) => (
                          <div key={index} className="bg-white border border-gray-200 rounded-xl overflow-hidden google-shadow hover:shadow-lg transition-shadow">
                            {graphic.url ? (
                              <div className="relative">
                                <img 
                                  src={graphic.url} 
                                  alt={graphic.description}
                                  className="w-full h-40 object-cover"
                                  onError={(e) => {
                                    const target = e.target as HTMLImageElement;
                                    target.parentElement?.classList.add('hidden');
                                  }}
                                />
                                <div className="absolute top-2 right-2 bg-blue-500 text-white px-2 py-1 rounded-md text-xs font-medium">
                                  AI Generated
                                </div>
                              </div>
                            ) : (
                              <div className="w-full h-40 bg-gradient-to-br from-green-50 to-green-100 flex items-center justify-center">
                                <div className="text-center">
                                  <span className="material-icons text-green-500 text-4xl mb-2 block">auto_awesome</span>
                                  <div className="text-green-600 text-sm">Generating...</div>
                                </div>
                              </div>
                            )}
                            <div className="p-4">
                              <div className="text-sm text-gray-700 font-medium">{graphic.description}</div>
                            </div>
                          </div>
                        ))}
                      </div>
                    </div>
                  )}
                </div>
              )}

              {activeModal === "session" && (
                <div className="flex justify-center">
                  <div className="relative">
                    <div className="w-24 h-24">
                      <svg className="w-24 h-24 transform -rotate-90" viewBox="0 0 100 100">
                        <circle cx="50" cy="50" r="45" fill="none" stroke="#e5e7eb" strokeWidth="6"/>
                        <circle 
                          cx="50" cy="50" r="45" fill="none" stroke="var(--google-green)" strokeWidth="6"
                          strokeDasharray="283" 
                          strokeDashoffset={283 - (simulationState.progress / 100) * 283}
                          className="transition-all duration-1000"
                        />
                      </svg>
                    </div>
                    <div className="absolute inset-0 flex items-center justify-center">
                      <div className="text-sm font-bold text-google-green">
                        {Math.ceil(120 * (1 - simulationState.progress / 100))}s
                      </div>
                    </div>
                  </div>
                </div>
              )}

              <div className={`border rounded-lg p-4 ${
                activeModal === "devalue" ? "bg-red-50 border-red-200" : 
                activeModal === "whatsapp" ? "bg-green-50 border-green-200" :
                "bg-blue-50 border-blue-200"
              }`}>
                <div className="flex items-start space-x-3">
                  <span className={`material-icons text-lg ${
                    activeModal === "devalue" ? "text-red-500" : 
                    activeModal === "whatsapp" ? "text-green-500" :
                    "text-google-blue"
                  }`}>info</span>
                  <div className="text-sm">
                    <p className={`font-medium mb-1 ${
                      activeModal === "devalue" ? "text-red-900" : 
                      activeModal === "whatsapp" ? "text-green-900" :
                      "text-blue-900"
                    }`}>Google Internal System</p>
                    <p className={
                      activeModal === "devalue" ? "text-red-700" : 
                      activeModal === "whatsapp" ? "text-green-700" :
                      "text-blue-700"
                    }>
                      {activeModal === "boost" && "Connecting to YouTube algorithm optimization servers..."}
                      {activeModal === "session" && "Authenticated with Google's algorithm enhancement system..."}
                      {activeModal === "replenish" && "Syncing with YouTube's algorithm grade point distribution system..."}
                      {activeModal === "devalue" && "Running advanced channel devaluation simulation with algorithm impact analysis..."}
                      {activeModal === "whatsapp" && "Integrating with YouTube Discovery Engine and cross-platform promotion networks..."}
                    </p>
                  </div>
                </div>
              </div>
            </div>
          )}
        </DialogContent>
      </Dialog>

      {/* Load Video Views Modal */}
      <Dialog open={loadViewsModal} onOpenChange={setLoadViewsModal}>
        <DialogContent className="sm:max-w-md">
          <DialogHeader>
            <DialogTitle className="flex items-center space-x-2">
              <span className="material-icons text-purple-500">visibility</span>
              <span>Load Video Views</span>
            </DialogTitle>
            <div className="text-sm text-gray-600 mt-2">
              Add real YouTube views to your selected video using the TubeBoost API service.
            </div>
          </DialogHeader>
          <div className="space-y-6">
            {/* Selected Video Info */}
            {selectedVideo && videos && (
              <div className="bg-gray-50 rounded-lg p-4">
                <h4 className="font-medium text-gray-900 mb-2">Selected Video</h4>
                <p className="text-sm text-gray-600 line-clamp-2">
                  {videos.find(v => v.id === selectedVideo)?.title}
                </p>
              </div>
            )}

            {/* Point-based System Info */}
            <div className="bg-purple-50 border border-purple-200 rounded-lg p-4">
              <div className="flex items-center space-x-2 mb-2">
                <span className="material-icons text-purple-600 text-sm">stars</span>
                <span className="text-sm font-medium text-purple-900">Point-based Ordering</span>
              </div>
              <div className="text-sm text-purple-800 space-y-1">
                <p>• Order any quantity ≥100 based on available points</p>
                <p>• Available points: {userPoints?.remainingPoints || 0}</p>
                <p>• 1 point = 1 view</p>
              </div>
            </div>

            {/* Quantity Selection */}
            <div className="space-y-3">
              <h4 className="font-medium text-gray-900">Select Quantity</h4>
              <div className="grid grid-cols-3 gap-3">
                <Input
                  type="number"
                  placeholder="Enter views (minimum 100)"
                  value={customViewsQuantity}
                  onChange={(e) => setCustomViewsQuantity(e.target.value)}
                  min="100"
                  className="mb-2"
                />
                <p className="text-xs text-gray-500">
                  Minimum: 100 views • Available points: {userPoints?.remainingPoints || 0}
                </p>
              </div>
            </div>

            {/* Service Info */}
            <div className="bg-gray-50 rounded-lg p-4">
              <div className="flex items-center space-x-2 mb-2">
                <span className="material-icons text-gray-600 text-sm">api</span>
                <span className="text-sm font-medium text-gray-900">Service Details</span>
              </div>
              <div className="text-sm text-gray-600 space-y-1">
                <p>• Provider: TubeBoost API (Service #1523)</p>
                <p>• Delivery: Real YouTube views</p>
                <p>• Processing: Tracked with order ID</p>
              </div>
            </div>

            {/* Action Buttons */}
            <div className="flex justify-end space-x-3">
              <Button
                variant="outline"
                onClick={() => setLoadViewsModal(false)}
                disabled={isLoadingViews}
              >
                Cancel
              </Button>
              <Button
                onClick={() => {
                  if (selectedVideo && customViewsQuantity) {
                    const quantity = parseInt(customViewsQuantity);
                    if (quantity >= 100 && quantity <= (userPoints?.remainingPoints || 0)) {
                      loadVideoViews({ videoId: selectedVideo, quantity });
                    }
                  }
                }}
                disabled={
                  isLoadingViews || 
                  !selectedVideo || 
                  !customViewsQuantity ||
                  parseInt(customViewsQuantity) < 100 ||
                  parseInt(customViewsQuantity) > (userPoints?.remainingPoints || 0)
                }
                className="bg-purple-500 hover:bg-purple-600"
              >
                {isLoadingViews ? (
                  <div className="flex items-center space-x-2">
                    <div className="w-4 h-4 border-2 border-white border-t-transparent rounded-full animate-spin"></div>
                    <span>Loading...</span>
                  </div>
                ) : (
                  `Load ${customViewsQuantity || 0} Views (${parseInt(customViewsQuantity) || 0} points)`
                )}
              </Button>
            </div>
          </div>
        </DialogContent>
      </Dialog>

      {/* Admin Load Video Views Modal */}
      <Dialog open={adminLoadViewsModal} onOpenChange={setAdminLoadViewsModal}>
        <DialogContent className="sm:max-w-md">
          <DialogHeader>
            <DialogTitle className="flex items-center space-x-2">
              <span className="material-icons text-red-500">admin_panel_settings</span>
              <span>ADMIN: Load Video Views</span>
            </DialogTitle>
            <div className="text-sm text-gray-600 mt-2">
              Administrator override: Load custom quantity of views with no limits.
            </div>
          </DialogHeader>
          <div className="space-y-6">
            {/* Selected Video Info */}
            {selectedVideo && videos && (
              <div className="bg-red-50 rounded-lg p-4 border border-red-200">
                <h4 className="font-medium text-gray-900 mb-2">Selected Video</h4>
                <p className="text-sm text-gray-600 line-clamp-2">
                  {videos.find(v => v.id === selectedVideo)?.title}
                </p>
              </div>
            )}

            {/* Custom Quantity Input */}
            <div className="space-y-3">
              <label className="block text-sm font-medium text-gray-700">
                Custom Quantity
              </label>
              <Input
                type="number"
                min="1"
                max="10000"
                value={customQuantity}
                onChange={(e) => setCustomQuantity(e.target.value)}
                placeholder="Enter number of views (1-10,000)"
                className="w-full"
              />
              <div className="text-xs text-gray-500">
                Enter any quantity between 1 and 10,000 views. This bypasses all usage limits.
              </div>
            </div>

            {/* Admin Warning */}
            <div className="bg-red-50 border border-red-200 rounded-lg p-4">
              <div className="flex items-center space-x-2 mb-2">
                <span className="material-icons text-red-600 text-sm">warning</span>
                <span className="text-sm font-medium text-red-900">Administrator Override</span>
              </div>
              <div className="text-sm text-red-800 space-y-1">
                <p>• This feature bypasses all usage limits and cooldowns</p>
                <p>• No restrictions on quantity or frequency</p>
                <p>• Use responsibly - direct TubeBoost API access</p>
              </div>
            </div>

            {/* Action Buttons */}
            <div className="flex justify-end space-x-3">
              <Button
                variant="outline"
                onClick={() => {
                  setAdminLoadViewsModal(false);
                  setCustomQuantity("");
                }}
                disabled={isAdminLoadingViews}
              >
                Cancel
              </Button>
              <Button
                onClick={() => {
                  if (selectedVideo && customQuantity) {
                    const quantity = parseInt(customQuantity);
                    if (quantity >= 1 && quantity <= 10000) {
                      adminLoadVideoViews({ videoId: selectedVideo, quantity });
                    }
                  }
                }}
                disabled={isAdminLoadingViews || !selectedVideo || !customQuantity || parseInt(customQuantity) < 1 || parseInt(customQuantity) > 10000}
                className="bg-red-500 hover:bg-red-600"
              >
                {isAdminLoadingViews ? (
                  <div className="flex items-center space-x-2">
                    <div className="w-4 h-4 border-2 border-white border-t-transparent rounded-full animate-spin"></div>
                    <span>Loading...</span>
                  </div>
                ) : (
                  `Load ${customQuantity || 0} Views`
                )}
              </Button>
            </div>
          </div>
        </DialogContent>
      </Dialog>

      {/* Add Subscribers Dialog */}
      <Dialog open={subscriberDialogOpen} onOpenChange={setSubscriberDialogOpen}>
        <DialogContent className="max-w-md">
          <div className="space-y-6">
            <div className="text-center">
              <h2 className="text-xl font-semibold text-gray-900 mb-2">Add Subscribers to Channel</h2>
              <p className="text-sm text-gray-600">
                Select quantity to add via TubeBoost API (Service #3108)
              </p>
            </div>

            {/* Channel Info */}
            <div className="bg-gray-50 rounded-lg p-4">
              <div className="text-sm">
                <p className="font-medium text-gray-900">{channel?.channelName}</p>
                <p className="text-gray-600 break-all">
                  {channel?.customUrl ? 
                    `https://youtube.com/${channel.customUrl}` : 
                    channel?.channelHandle ? 
                    `https://youtube.com/@${channel.channelHandle}` : 
                    `https://youtube.com/channel/${channel?.channelId}`
                  }
                </p>
              </div>
            </div>

            {/* Quantity Selection */}
            <div>
              <h3 className="text-sm font-medium text-gray-900 mb-3">Enter Quantity</h3>
              <Input
                type="number"
                placeholder="Enter subscribers (minimum 50)"
                value={customSubscriberQuantity}
                onChange={(e) => setCustomSubscriberQuantity(e.target.value)}
                min="50"
                className="mb-2"
              />
              <p className="text-xs text-gray-500">
                Minimum: 50 subscribers • Available points: {userPoints?.remainingPoints || 0}
              </p>
            </div>

            {/* Point-based System Info */}
            <div className="bg-orange-50 border border-orange-200 rounded-lg p-3">
              <div className="flex items-center space-x-2">
                <span className="material-icons text-orange-500 text-sm">stars</span>
                <div className="text-sm">
                  <p className="font-medium text-orange-800">Point-based Ordering</p>
                  <p className="text-orange-600">
                    Order any quantity ≥50 • 1 point = 1 subscriber
                  </p>
                </div>
              </div>
            </div>

            {/* Action Buttons */}
            <div className="flex space-x-3">
              <Button
                variant="outline"
                onClick={() => setSubscriberDialogOpen(false)}
                className="flex-1"
              >
                Cancel
              </Button>
              <Button
                onClick={() => {
                  console.log('Add Subscribers button clicked');
                  if (customSubscriberQuantity) {
                    const quantity = parseInt(customSubscriberQuantity);
                    console.log('Parsed quantity:', quantity);
                    if (quantity >= 50 && quantity <= (userPoints?.remainingPoints || 0)) {
                      console.log('Calling addSubscribers with quantity:', quantity);
                      addSubscribers(quantity);
                    } else {
                      console.log('Invalid quantity or insufficient points');
                    }
                  } else {
                    console.log('No quantity specified');
                  }
                }}
                disabled={
                  isAddingSubscribers || 
                  !customSubscriberQuantity ||
                  parseInt(customSubscriberQuantity) < 50 ||
                  parseInt(customSubscriberQuantity) > (userPoints?.remainingPoints || 0)
                }
                className="flex-1 bg-orange-500 hover:bg-orange-600"
              >
                {isAddingSubscribers ? (
                  <div className="flex items-center space-x-2">
                    <div className="w-4 h-4 border-2 border-white border-t-transparent rounded-full animate-spin"></div>
                    <span>Ordering...</span>
                  </div>
                ) : (
                  `Add ${customSubscriberQuantity || 0} Subscribers (${parseInt(customSubscriberQuantity) || 0} points)`
                )}
              </Button>
            </div>
          </div>
        </DialogContent>
      </Dialog>

      {/* Add Likes Dialog */}
      <Dialog open={likesDialogOpen} onOpenChange={setLikesDialogOpen}>
        <DialogContent className="max-w-md">
          <div className="space-y-6">
            <div className="text-center">
              <h2 className="text-xl font-semibold text-gray-900 mb-2">Add Likes to Video</h2>
              <p className="text-sm text-gray-600">
                Select quantity to add via TubeBoost API (Service #1800)
              </p>
            </div>

            {/* Video Info */}
            <div className="bg-gray-50 rounded-lg p-4">
              <div className="text-sm">
                <p className="font-medium text-gray-900">{videos?.find((v: any) => v.id === selectedVideo)?.title}</p>
                <p className="text-gray-600 break-all">
                  https://youtube.com/watch?v={videos?.find((v: any) => v.id === selectedVideo)?.videoId}
                </p>
              </div>
            </div>

            {/* Quantity Input */}
            <div>
              <label className="block text-sm font-medium text-gray-700 mb-2">
                Quantity
              </label>
              <Input
                type="number"
                placeholder="Enter likes (minimum 10)"
                value={customLikesQuantity}
                onChange={(e) => setCustomLikesQuantity(e.target.value)}
                min="10"
                className="mb-2"
              />
              <p className="text-xs text-gray-500">
                Minimum: 10 likes • Available points: {userPoints?.remainingPoints || 0}
              </p>
            </div>

            {/* Point-based System Info */}
            <div className="bg-pink-50 border border-pink-200 rounded-lg p-3">
              <div className="flex items-center space-x-2">
                <span className="material-icons text-pink-500 text-sm">stars</span>
                <div className="text-sm">
                  <p className="font-medium text-pink-800">Point-based Ordering</p>
                  <p className="text-pink-600">
                    Order any quantity ≥10 • 1.5 points = 1 like
                  </p>
                </div>
              </div>
            </div>

            {/* Action Buttons */}
            <div className="flex space-x-3">
              <Button
                variant="outline"
                onClick={() => setLikesDialogOpen(false)}
                className="flex-1"
              >
                Cancel
              </Button>
              <Button
                onClick={() => {
                  if (customLikesQuantity && selectedVideo) {
                    const quantity = parseInt(customLikesQuantity);
                    const requiredPoints = Math.ceil(quantity * 1.5);
                    if (quantity >= 10 && requiredPoints <= (userPoints?.remainingPoints || 0)) {
                      addLikes({ videoId: selectedVideo, quantity });
                    }
                  }
                }}
                disabled={
                  isAddingLikes || 
                  !customLikesQuantity ||
                  !selectedVideo ||
                  parseInt(customLikesQuantity) < 10 ||
                  Math.ceil(parseInt(customLikesQuantity) * 1.5) > (userPoints?.remainingPoints || 0)
                }
                className="flex-1 bg-pink-500 hover:bg-pink-600"
              >
                {isAddingLikes ? (
                  <div className="flex items-center space-x-2">
                    <div className="w-4 h-4 border-2 border-white border-t-transparent rounded-full animate-spin"></div>
                    <span>Ordering...</span>
                  </div>
                ) : (
                  `Add ${customLikesQuantity || 0} Likes (${Math.ceil(parseInt(customLikesQuantity) * 1.5) || 0} points)`
                )}
              </Button>
            </div>
          </div>
        </DialogContent>
      </Dialog>
            </div>
          </div>
        )}
      </div>
    </div>
  );

  return <PlatformDashboard youtubeContent={youtubeContent} />;
}
