import { useState, useEffect } from "react";
import { useAuth } from "@/hooks/use-auth";
import { useQuery, useMutation, useQueryClient } from "@tanstack/react-query";
import { apiRequest } from "@/lib/queryClient";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Dialog, DialogContent, DialogHeader, DialogTitle, DialogTrigger, DialogDescription } from "@/components/ui/dialog";
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Badge } from "@/components/ui/badge";
import { toast, useToast } from "@/hooks/use-toast";
import { AlertCircle, Share2, Shield, LogOut, User, Youtube, MessageSquare, RefreshCw, Users, Search, Plus, TrendingUp, BarChart3, Play } from "lucide-react";
import { Link, useLocation } from "wouter";
import AnonymousWarningBanner from "@/components/anonymous-warning-banner";
import AnonymousCredentialsDisplay from "@/components/anonymous-credentials-display";
import EmailIncentiveBanner from "@/components/email-incentive-banner";
import { QueuePanel } from "@/components/queue-panel";
import { RedditConnection } from "@/components/reddit-connection";
import { RedditAnalyticsDashboard } from "@/components/reddit-analytics-dashboard";
import { PlatformDashboard } from "@/components/platform-dashboard";
import { VideoManager } from "@/components/video-manager";
import { PointsDisplay } from "@/components/points-display";

type SessionType = "boost" | "session" | "replenish" | "devalue" | "whatsapp" | "expose";

interface SimulationState {
  isRunning: boolean;
  progress: number;
  currentStep: string;
  metrics: Record<string, string>;
  sessionId: string | null;
}

// Welcome Dashboard Component for new users
function WelcomeDashboard() {
  const { user, logout } = useAuth();
  const [showChannelSearch, setShowChannelSearch] = useState(false);
  const [searchQuery, setSearchQuery] = useState("");
  const [nameSearchQuery, setNameSearchQuery] = useState("");
  const [searchResults, setSearchResults] = useState<any[]>([]);
  const [isSearching, setIsSearching] = useState(false);
  const [currentPage, setCurrentPage] = useState(1);
  const [totalResults, setTotalResults] = useState(0);
  const { toast } = useToast();

  const resultsPerPage = 10;
  const maxPages = 5;

  // Search YouTube channels with pagination
  const searchChannels = async (query: string, page: number = 1, searchType: string = "channel") => {
    if (!query.trim()) return;
    
    setIsSearching(true);
    try {
      const response = await apiRequest(`/api/youtube/search?q=${encodeURIComponent(query)}&page=${page}&searchType=${searchType}`, "GET");
      setSearchResults(response.channels || []);
      setTotalResults(response.pagination?.totalResults || 0);
      setCurrentPage(page);
    } catch (error: any) {
      toast({
        title: "Search Failed",
        description: error.message || "Failed to search YouTube channels",
        variant: "destructive"
      });
      setSearchResults([]);
      setTotalResults(0);
    } finally {
      setIsSearching(false);
    }
  };

  const handleSearch = (e: React.FormEvent) => {
    e.preventDefault();
    setCurrentPage(1);
    searchChannels(searchQuery, 1, "channel");
  };

  const handleNameSearch = (e: React.FormEvent) => {
    e.preventDefault();
    setCurrentPage(1);
    searchChannels(nameSearchQuery, 1, "creator");
  };

  const handlePageChange = (page: number) => {
    // Use the current search query and determine search type
    const currentQuery = searchQuery || nameSearchQuery;
    const currentType = searchQuery ? "channel" : "creator";
    searchChannels(currentQuery, page, currentType);
  };

  const addChannel = async (channelData: any) => {
    try {
      // Enhanced error logging for channel addition
      const channelAddRequest = {
        channelId: channelData.id,
        channelName: channelData.title,
        channelHandle: channelData.customUrl || channelData.id,
        customUrl: channelData.customUrl || "",
        subscriberCount: channelData.subscriberCount,
        description: channelData.description,
        thumbnail: channelData.thumbnail
      };

      console.log("[ADD_CHANNEL] Starting channel addition process:", {
        channelData: channelAddRequest,
        timestamp: new Date().toISOString(),
        userAgent: navigator.userAgent,
        url: window.location.href
      });

      const response = await apiRequest("/api/channels", "POST", channelAddRequest);
      
      console.log("[ADD_CHANNEL] Channel addition successful:", {
        response,
        channelName: channelData.title,
        timestamp: new Date().toISOString()
      });
      
      toast({
        title: "Channel Added Successfully!",
        description: `${channelData.title} has been added to your account. Redirecting to YouTube Dashboard...`,
        variant: "default"
      });
      
      setShowChannelSearch(false);
      setSearchQuery("");
      setNameSearchQuery("");
      setSearchResults([]);
      
      // Set platform to YouTube and redirect to dashboard
      localStorage.setItem('selectedPlatform', 'youtube');
      
      // Refresh the page to load the normal YouTube dashboard
      setTimeout(() => {
        window.location.reload();
      }, 1500);
      
    } catch (error: any) {
      // Enhanced error logging
      console.error("[ADD_CHANNEL] Channel addition failed:", {
        error: error.message,
        stack: error.stack,
        channelData: {
          id: channelData.id,
          title: channelData.title,
          customUrl: channelData.customUrl
        },
        timestamp: new Date().toISOString(),
        userAgent: navigator.userAgent,
        url: window.location.href
      });

      // Send error to logging system
      try {
        await apiRequest("/api/error-logs", "POST", {
          level: "error",
          message: `Channel Addition Failed: ${error.message}`,
          context: {
            channelId: channelData.id,
            channelTitle: channelData.title,
            customUrl: channelData.customUrl,
            errorType: "ADD_CHANNEL_FAILURE",
            apiEndpoint: "/api/channels",
            httpMethod: "POST",
            timestamp: Date.now(),
            userAgent: navigator.userAgent,
            url: window.location.href,
            viewport: `${window.innerWidth}x${window.innerHeight}`
          },
          stack: error.stack,
          tags: ["error", "channel", "addition", "critical"]
        });
      } catch (logError) {
        console.error("[ADD_CHANNEL] Failed to log error:", logError);
      }

      toast({
        title: "Failed to Add Channel",
        description: error.message || "Could not add the channel to your account. Please try again.",
        variant: "destructive"
      });
    }
  };

  return (
    <div className="min-h-screen bg-gradient-to-br from-blue-50 via-white to-red-50">
      {/* Header */}
      <div className="bg-white border-b border-gray-200 shadow-sm">
        <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
          <div className="flex items-center justify-between h-16">
            <div className="flex items-center space-x-4">
              <Youtube className="h-8 w-8 text-red-500" />
              <h1 className="text-xl font-bold text-gray-900">TubeBoost Dashboard</h1>
            </div>
            
            <div className="flex items-center space-x-4">
              <PointsDisplay />
              
              {/* User Profile */}
              <div className="flex items-center space-x-3">
                <div className="text-right">
                  <p className="text-sm font-medium text-gray-900">{user?.username || user?.email}</p>
                  <p className="text-xs text-gray-500">Free Plan</p>
                </div>
                <div className="w-8 h-8 bg-gradient-to-r from-blue-500 to-red-500 rounded-full flex items-center justify-center">
                  <User className="h-4 w-4 text-white" />
                </div>
                <Button
                  variant="outline"
                  size="sm"
                  onClick={logout}
                  className="text-gray-600 hover:text-gray-900"
                >
                  <LogOut className="h-4 w-4 mr-1" />
                  Logout
                </Button>
              </div>
            </div>
          </div>
        </div>
      </div>

      {/* Welcome Content */}
      <div className="max-w-6xl mx-auto px-4 sm:px-6 lg:px-8 py-12">
        {/* Hero Section */}
        <div className="text-center mb-12">
          <div className="w-20 h-20 bg-gradient-to-r from-red-500 to-blue-600 rounded-full flex items-center justify-center mx-auto mb-6">
            <Youtube className="h-10 w-10 text-white" />
          </div>
          <h1 className="text-4xl font-bold text-gray-900 mb-4">
            Welcome to TubeBoost!
          </h1>
          <p className="text-xl text-gray-600 mb-8 max-w-2xl mx-auto">
            Your professional YouTube growth platform. Connect your first channel to start optimizing 
            your content, boosting views, and growing your subscriber base with AI-powered tools.
          </p>
        </div>

        {/* Action Cards */}
        <div className="grid md:grid-cols-2 gap-8 mb-12">
          {/* Add Channel Card */}
          <Card className="relative overflow-hidden border-2 border-blue-200 hover:border-blue-300 transition-colors">
            <div className="absolute inset-0 bg-gradient-to-br from-blue-50 to-blue-100 opacity-50"></div>
            <CardContent className="relative p-8 text-center">
              <div className="w-16 h-16 bg-blue-500 rounded-full flex items-center justify-center mx-auto mb-4">
                <Plus className="h-8 w-8 text-white" />
              </div>
              <h3 className="text-2xl font-bold text-gray-900 mb-3">Connect Your Channel</h3>
              <p className="text-gray-600 mb-6">
                Add your YouTube channel to access powerful analytics, AI optimization tools, and growth services.
              </p>
              <Link href="/channel-manager">
                <Button size="lg" className="w-full bg-blue-600 hover:bg-blue-700">
                  <Youtube className="h-5 w-5 mr-2" />
                  Add YouTube Channel
                </Button>
              </Link>
            </CardContent>
          </Card>

          {/* Search Channels Card */}
          <Card className="relative overflow-hidden border-2 border-green-200 hover:border-green-300 transition-colors">
            <div className="absolute inset-0 bg-gradient-to-br from-green-50 to-green-100 opacity-50"></div>
            <CardContent className="relative p-8 text-center">
              <div className="w-16 h-16 bg-green-500 rounded-full flex items-center justify-center mx-auto mb-4">
                <Search className="h-8 w-8 text-white" />
              </div>
              <h3 className="text-2xl font-bold text-gray-900 mb-3">Discover Channels</h3>
              <p className="text-gray-600 mb-6">
                Search and analyze any YouTube channel to see performance metrics and add them to your account.
              </p>
              <Button 
                size="lg" 
                variant="outline" 
                className="w-full border-green-600 text-green-600 hover:bg-green-50"
                onClick={() => setShowChannelSearch(true)}
              >
                <Search className="h-5 w-5 mr-2" />
                Search YouTube Channels
              </Button>
            </CardContent>
          </Card>
        </div>

        {/* Features Preview */}
        <div className="bg-white rounded-xl shadow-lg p-8">
          <h2 className="text-2xl font-bold text-gray-900 mb-6 text-center">
            What You'll Get Access To
          </h2>
          <div className="grid md:grid-cols-3 gap-6">
            <div className="text-center">
              <div className="w-12 h-12 bg-red-100 rounded-lg flex items-center justify-center mx-auto mb-3">
                <TrendingUp className="h-6 w-6 text-red-600" />
              </div>
              <h3 className="font-semibold text-gray-900 mb-2">Growth Services</h3>
              <p className="text-sm text-gray-600">Boost views, likes, and subscribers with legitimate growth services</p>
            </div>
            <div className="text-center">
              <div className="w-12 h-12 bg-blue-100 rounded-lg flex items-center justify-center mx-auto mb-3">
                <BarChart3 className="h-6 w-6 text-blue-600" />
              </div>
              <h3 className="font-semibold text-gray-900 mb-2">Advanced Analytics</h3>
              <p className="text-sm text-gray-600">Deep insights into your channel performance and optimization opportunities</p>
            </div>
            <div className="text-center">
              <div className="w-12 h-12 bg-purple-100 rounded-lg flex items-center justify-center mx-auto mb-3">
                <Play className="h-6 w-6 text-purple-600" />
              </div>
              <h3 className="font-semibold text-gray-900 mb-2">AI Optimization</h3>
              <p className="text-sm text-gray-600">OpenWeb 3.2 AI provides personalized recommendations for your content</p>
            </div>
          </div>
        </div>
      </div>

      {/* Channel Search Modal */}
      <Dialog open={showChannelSearch} onOpenChange={setShowChannelSearch}>
        <DialogContent className="max-w-2xl max-h-[80vh] overflow-y-auto">
          <DialogHeader>
            <DialogTitle className="flex items-center gap-2">
              <Search className="h-5 w-5" />
              Search YouTube Channels
            </DialogTitle>
            <DialogDescription>
              Search for YouTube channels by name or handle to add them to your TubeBoost account.
            </DialogDescription>
          </DialogHeader>
          
          <div className="space-y-4">
            {/* Channel Name/Handle Search */}
            <div className="space-y-2">
              <label className="text-sm font-medium text-gray-700">Search by Channel Name or @Handle</label>
              <form onSubmit={handleSearch} className="flex gap-2">
                <Input
                  placeholder="Search by channel name or @handle (e.g., 'PewDiePie' or '@mrbeast')..."
                  value={searchQuery}
                  onChange={(e) => setSearchQuery(e.target.value)}
                  className="flex-1"
                />
                <Button type="submit" disabled={isSearching || !searchQuery.trim()}>
                  {isSearching ? "Searching..." : "Search"}
                </Button>
              </form>
            </div>

            {/* Content/Topic Search */}
            <div className="space-y-2">
              <div className="flex items-center gap-2">
                <label className="text-sm font-medium text-gray-700">Search by Content Topic</label>
                <div className="text-xs text-green-600 bg-green-50 px-2 py-1 rounded border border-green-200">
                  Content Discovery
                </div>
              </div>
              <form onSubmit={handleNameSearch} className="flex gap-2">
                <Input
                  placeholder="Search by content type (e.g., 'gaming', 'cooking tutorials', 'tech reviews')..."
                  value={nameSearchQuery}
                  onChange={(e) => setNameSearchQuery(e.target.value)}
                  className="flex-1"
                />
                <Button type="submit" disabled={isSearching || !nameSearchQuery.trim()}>
                  {isSearching ? "Searching..." : "Search"}
                </Button>
              </form>
              <div className="text-xs text-gray-500 bg-gray-50 p-3 rounded border">
                <strong>Examples:</strong> Try searching by content themes like "minecraft", "cooking tutorials", "tech reviews", 
                "music covers", "fitness workouts", "travel vlogs", "educational", or "comedy sketches" to discover channels in specific niches.
              </div>
            </div>

            {searchResults.length > 0 && (
              <div className="space-y-3">
                <div className="flex items-center justify-between">
                  <h3 className="font-medium text-gray-900">Search Results</h3>
                  <div className="text-sm text-gray-500">
                    {totalResults} total results • Page {currentPage} of {Math.min(maxPages, Math.ceil(totalResults / resultsPerPage))}
                  </div>
                </div>
                
                {searchResults.map((channel) => (
                  <div key={channel.id} className="border rounded-lg p-4 hover:bg-gray-50">
                    <div className="flex items-center justify-between">
                      <div className="flex items-center space-x-3">
                        {channel.thumbnail && (
                          <img 
                            src={channel.thumbnail} 
                            alt={channel.title}
                            className="w-12 h-12 rounded-full"
                          />
                        )}
                        <div>
                          <h4 className="font-medium text-gray-900">{channel.title}</h4>
                          <p className="text-sm text-gray-500">
                            {channel.subscriberCount ? `${parseInt(channel.subscriberCount).toLocaleString()} subscribers` : 'Subscriber count hidden'}
                          </p>
                          {channel.customUrl && (
                            <p className="text-xs text-gray-400">@{channel.customUrl}</p>
                          )}
                        </div>
                      </div>
                      <Button 
                        size="sm"
                        onClick={() => addChannel(channel)}
                      >
                        Add Channel
                      </Button>
                    </div>
                    {channel.description && (
                      <p className="text-sm text-gray-600 mt-2 line-clamp-2">
                        {channel.description}
                      </p>
                    )}
                  </div>
                ))}

                {/* Pagination Controls */}
                {totalResults > resultsPerPage && (
                  <div className="flex items-center justify-center space-x-2 pt-4">
                    <Button
                      variant="outline"
                      size="sm"
                      disabled={currentPage === 1}
                      onClick={() => handlePageChange(currentPage - 1)}
                    >
                      Previous
                    </Button>
                    
                    {Array.from({ length: Math.min(maxPages, Math.ceil(totalResults / resultsPerPage)) }, (_, i) => i + 1).map((page) => (
                      <Button
                        key={page}
                        variant={currentPage === page ? "default" : "outline"}
                        size="sm"
                        onClick={() => handlePageChange(page)}
                      >
                        {page}
                      </Button>
                    ))}
                    
                    <Button
                      variant="outline"
                      size="sm"
                      disabled={currentPage >= Math.min(maxPages, Math.ceil(totalResults / resultsPerPage))}
                      onClick={() => handlePageChange(currentPage + 1)}
                    >
                      Next
                    </Button>
                  </div>
                )}
              </div>
            )}

            {searchResults.length === 0 && (searchQuery || nameSearchQuery) && !isSearching && (
              <div className="text-center py-8 text-gray-500">
                <Search className="h-12 w-12 mx-auto mb-3 text-gray-300" />
                <p>No channels found for "{searchQuery || nameSearchQuery}"</p>
                <div className="text-sm mt-3 space-y-1">
                  <p>Try these search strategies:</p>
                  <div className="text-left max-w-md mx-auto space-y-1">
                    <p>• Channel names: "MrBeast", "Markiplier", "MKBHD"</p>
                    <p>• @Handle format: "@pewdiepie", "@mrbeast"</p>
                    <p>• Content topics: "gaming", "cooking", "tech reviews"</p>
                    <p>• Niche keywords: "minecraft", "fitness", "tutorials"</p>
                  </div>
                </div>
              </div>
            )}
          </div>
        </DialogContent>
      </Dialog>
    </div>
  );
}

export default function Dashboard() {
  const { user, logout } = useAuth();
  const queryClient = useQueryClient();
  const [location] = useLocation();
  const pathname = location;

  // State for modals and dialogs
  const [shareDialogOpen, setShareDialogOpen] = useState(false);
  const [loadViewsModal, setLoadViewsModal] = useState(false);
  const [adminLoadViewsModal, setAdminLoadViewsModal] = useState(false);
  const [subscriberDialogOpen, setSubscriberDialogOpen] = useState(false);
  const [likesDialogOpen, setLikesDialogOpen] = useState(false);
  const [activeModal, setActiveModal] = useState<SessionType | null>(null);
  const [selectedVideo, setSelectedVideo] = useState<number | null>(null);
  const [customViewsQuantity, setCustomViewsQuantity] = useState<string>("");
  const [customQuantity, setCustomQuantity] = useState<string>("");
  const [customSubscriberQuantity, setCustomSubscriberQuantity] = useState<string>("");
  const [customLikesQuantity, setCustomLikesQuantity] = useState<string>("");
  const [simulationState, setSimulationState] = useState<SimulationState>({
    isRunning: false,
    progress: 0,
    currentStep: "",
    metrics: {},
    sessionId: null
  });
  const [whatsappGraphics, setWhatsappGraphics] = useState<Array<{url: string; description: string}>>([]);
  const [ordersMap, setOrdersMap] = useState<Record<number, any[]>>({});

  // API queries
  const { data: channel, error: channelError } = useQuery({
    queryKey: ["/api/channel"],
  });

  const { data: videos } = useQuery({
    queryKey: ["/api/videos"],
  });

  const { data: userPoints } = useQuery({
    queryKey: ["/api/points"],
  });

  const { data: cooldownStatus } = useQuery({
    queryKey: ["/api/cooldown-status"],
  });

  // Query for subscriber orders
  const { data: subscriberOrders = [], refetch: refetchSubscriberOrders } = useQuery({
    queryKey: ["/api/subscriber-orders"],
    refetchInterval: 30000, // Refresh every 30 seconds for real-time updates
  });

  // Query for recent orders
  const { data: recentOrders = [] } = useQuery({
    queryKey: ["/api/recent-orders"],
    refetchInterval: 30000, // Refresh every 30 seconds for real-time updates
  });

  // State for expandable recent orders
  const [showAllOrders, setShowAllOrders] = useState(false);

  // Mutations
  const { mutate: addToQueue } = useMutation({
    mutationFn: (data: any) => apiRequest("/api/queue", "POST", data),
    onSuccess: () => {
      toast({
        title: "Success",
        description: "Action added to queue successfully",
      });
      queryClient.invalidateQueries({ queryKey: ["/api/queue"] });
    },
    onError: (error: any) => {
      toast({
        title: "Error",
        description: error.message || "Failed to add action to queue",
        variant: "destructive",
      });
    }
  });

  const { mutate: loadVideoViews, isLoading: isLoadingViews } = useMutation({
    mutationFn: (data: { videoId: number; quantity: number }) => 
      apiRequest("/api/load-video-views", "POST", data),
    onSuccess: () => {
      toast({
        title: "Success",
        description: "Video views order placed successfully",
      });
      setLoadViewsModal(false);
      setAdminLoadViewsModal(false);
      setCustomViewsQuantity("");
      setCustomQuantity("");
      queryClient.invalidateQueries({ queryKey: ["/api/videos"] });
      queryClient.invalidateQueries({ queryKey: ["/api/points"] });
    },
    onError: (error: any) => {
      toast({
        title: "Error",
        description: error.message || "Failed to load video views",
        variant: "destructive"
      });
    }
  });

  const { mutate: addSubscribers, isLoading: isAddingSubscribers } = useMutation({
    mutationFn: (quantity: number) => 
      apiRequest("/api/add-subscribers", "POST", { quantity }),
    onSuccess: () => {
      toast({
        title: "Success",
        description: "Subscriber order placed successfully",
      });
      setSubscriberDialogOpen(false);
      setCustomSubscriberQuantity("");
      queryClient.invalidateQueries({ queryKey: ["/api/channel"] });
      queryClient.invalidateQueries({ queryKey: ["/api/points"] });
    },
    onError: (error: any) => {
      toast({
        title: "Error",
        description: error.message || "Failed to add subscribers",
        variant: "destructive"
      });
    }
  });

  const { mutate: addLikes, isLoading: isAddingLikes } = useMutation({
    mutationFn: (data: { videoId: number; quantity: number }) => 
      apiRequest("/api/add-likes", "POST", data),
    onSuccess: () => {
      toast({
        title: "Success",
        description: "Likes order placed successfully",
      });
      setLikesDialogOpen(false);
      setCustomLikesQuantity("");
      queryClient.invalidateQueries({ queryKey: ["/api/videos"] });
      queryClient.invalidateQueries({ queryKey: ["/api/points"] });
    },
    onError: (error: any) => {
      toast({
        title: "Error",
        description: error.message || "Failed to add likes",
        variant: "destructive"
      });
    }
  });

  // Channel exposure simulation mutation
  const { mutate: startChannelExposure, isPending: isStartingExposure } = useMutation({
    mutationFn: async () => {
      // Get the current active channel to ensure we have the latest data
      const currentChannel = await apiRequest("/api/channel", "GET");
      return apiRequest("/api/channel-exposure", "POST", { channelId: currentChannel.id });
    },
    onSuccess: (response: any) => {
      setActiveModal("expose");
      setSimulationState({
        isRunning: true,
        progress: 0,
        currentStep: "Initializing AI-powered exposure optimization...",
        metrics: {},
        sessionId: response.sessionId
      });
      // Start the AI simulation
      runExposureSimulation(response.sessionId);
      queryClient.invalidateQueries({ queryKey: ["/api/points"] });
    },
    onError: (error: any) => {
      toast({
        title: "Error",
        description: error.message || "Failed to start channel exposure",
        variant: "destructive"
      });
    }
  });

  // Channel exposure simulation function
  const runExposureSimulation = async (sessionId: string) => {
    const steps = [
      "Analyzing channel demographics and audience engagement patterns...",
      "Leveraging AI algorithms to identify optimal exposure opportunities...",
      "Scanning trending topics and hashtag effectiveness across platforms...",
      "Optimizing content visibility through machine learning insights...",
      "Cross-referencing with successful channel growth strategies...",
      "Implementing targeted recommendation algorithm adjustments...",
      "Expanding reach through strategic network effect amplification...",
      "Finalizing exposure optimization and updating channel metrics..."
    ];

    const totalDuration = 6 * 60 * 1000; // 6 minutes in milliseconds
    const stepDuration = totalDuration / steps.length;

    for (let i = 0; i < steps.length; i++) {
      const progress = ((i + 1) / steps.length) * 100;
      
      setSimulationState(prev => ({
        ...prev,
        progress,
        currentStep: steps[i],
        metrics: getExposureMetrics(progress)
      }));

      // Wait for the step duration
      await new Promise(resolve => setTimeout(resolve, stepDuration));
    }

    // Complete the simulation
    setSimulationState(prev => ({
      ...prev,
      isRunning: false,
      progress: 100,
      currentStep: "Channel exposure optimization completed successfully!",
      metrics: getExposureMetrics(100)
    }));

    // Auto-close modal after 3 seconds
    setTimeout(() => {
      setActiveModal(null);
      toast({
        title: "Success",
        description: "Channel exposure boost completed! Your reach has been optimized.",
        variant: "default"
      });
    }, 3000);
  };

  // Get metrics for exposure simulation
  const getExposureMetrics = (progress: number): Record<string, string> => {
    return {
      "AI Analysis Score": `${Math.floor(75 + (progress / 100) * 20)}%`,
      "Reach Multiplier": `${(2.1 + (progress / 100) * 1.4).toFixed(1)}x`,
      "Trend Alignment": `${Math.floor(60 + (progress / 100) * 35)}%`,
      "Exposure Networks": `${Math.floor(12 + (progress / 100) * 18)}`,
      "Optimization Level": `${Math.floor(progress)}%`
    };
  };

  // Initialize data on mount
  useEffect(() => {
    const initData = async () => {
      try {
        await apiRequest("/api/init", "GET");
        queryClient.invalidateQueries({ queryKey: ["/api/channel"] });
        queryClient.refetchQueries({ queryKey: ["/api/channel"] });
      } catch (error) {
        console.error("Failed to initialize data:", error);
      }
    };

    initData();
  }, [queryClient]);

  const handleActionClick = (action: SessionType) => {
    if (action === "replenish") {
      if (cooldownStatus?.isActive) {
        toast({
          title: "Cooldown Active",
          description: `Please wait ${cooldownStatus.remainingMinutes} minutes before using this action again`,
          variant: "destructive"
        });
        return;
      }
      addToQueue({ 
        actionType: "replenish", 
        channelId: channel?.id || 0 
      });
    } else if (action === "devalue") {
      addToQueue({ 
        actionType: "devalue", 
        channelId: channel?.id || 0 
      });
    } else if (action === "whatsapp") {
      addToQueue({ 
        actionType: "whatsapp", 
        channelId: channel?.id || 0 
      });
    } else if (action === "expose") {
      // Check if user has enough points (20 points required)
      if (userPoints?.remainingPoints < 20) {
        toast({
          title: "Insufficient Points",
          description: "You need 20 points to boost channel exposure",
          variant: "destructive"
        });
        return;
      }
      startChannelExposure();
    } else {
      if (!selectedVideo) {
        toast({
          title: "No Video Selected",
          description: "Please select a video first",
          variant: "destructive"
        });
        return;
      }
      
      if (action === "boost") {
        addToQueue({ 
          actionType: "boost", 
          channelId: channel?.id || 0, 
          videoId: selectedVideo 
        });
      } else if (action === "session") {
        addToQueue({ 
          actionType: "session", 
          channelId: channel?.id || 0, 
          videoId: selectedVideo 
        });
      }
    }
  };

  // Check if no channels have been added yet
  const isNoChannelsError = channelError && (channelError as any).message?.includes("No active channel selected");

  if (!channel && !isNoChannelsError) {
    return (
      <div className="min-h-screen bg-background flex items-center justify-center">
        <div className="text-center">
          <div className="w-16 h-16 border-4 border-primary border-t-transparent rounded-full animate-spin mx-auto mb-4"></div>
          <p className="text-muted-foreground">Loading channel data...</p>
        </div>
      </div>
    );
  }

  if (channelError) {
    return <WelcomeDashboard />;
  }

  // YouTube Dashboard Content
  const youtubeContent = (
    <>
      {/* Anonymous Account Warning */}
      {user?.isAnonymous && <AnonymousWarningBanner />}
      
      {/* Email Incentive Banner */}
      {user?.isAnonymous && <EmailIncentiveBanner />}
      
      {/* Anonymous Credentials Display */}
      {user?.isAnonymous && <AnonymousCredentialsDisplay />}

      {/* Show queue panel */}
      <QueuePanel />

      {/* Channel Header */}
      <div className="bg-white rounded-lg border border-gray-200 p-6 mb-6 google-shadow">
        <div className="flex items-center space-x-4">
          <div className="w-16 h-16 bg-gradient-to-br from-google-blue to-google-green rounded-full flex items-center justify-center">
            <span className="material-icons text-white text-2xl">account_circle</span>
          </div>
          <div className="flex-1">
            <h1 className="text-2xl font-bold text-gray-900">{channel.channelName}</h1>
            <div className="flex items-center space-x-4 text-sm text-gray-600">
              <span>@{channel.channelHandle}</span>
              <span>•</span>
              <span className="flex items-center space-x-1">
                <span className="material-icons text-sm">group</span>
                <span>{channel.subscriberCount?.toLocaleString() || 0} subscribers</span>
              </span>
            </div>
          </div>
          <div className="text-right">
            <p className="text-sm text-gray-500">Points Balance</p>
            <p className="text-xl font-bold text-google-blue">
              {userPoints?.remainingPoints || 0}
            </p>
          </div>
        </div>
      </div>

      {/* Key Metrics Grid */}
      <div className="grid grid-cols-2 lg:grid-cols-4 gap-3 sm:gap-4 mb-4 sm:mb-6">
        <div className="bg-white rounded-lg border border-gray-200 p-4 sm:p-6 google-shadow">
          <div className="flex items-center justify-between">
            <div>
              <p className="text-gray-500 text-sm font-medium">Subscribers</p>
              <p className="text-xl sm:text-2xl font-bold text-gray-900">{(channel.subscriberCount || 0).toLocaleString()}</p>
            </div>
            <div className="w-8 h-8 sm:w-10 sm:h-10 bg-red-50 rounded-lg flex items-center justify-center">
              <span className="material-icons text-google-red text-lg sm:text-xl">group</span>
            </div>
          </div>
        </div>

        <div className="bg-white rounded-lg border border-gray-200 p-4 sm:p-6 google-shadow">
          <div className="flex items-center justify-between">
            <div>
              <p className="text-gray-500 text-sm font-medium">Total Views</p>
              <p className="text-xl sm:text-2xl font-bold text-gray-900">{(channel.viewCount || 0).toLocaleString()}</p>
            </div>
            <div className="w-8 h-8 sm:w-10 sm:h-10 bg-green-50 rounded-lg flex items-center justify-center">
              <span className="material-icons text-google-green text-lg sm:text-xl">visibility</span>
            </div>
          </div>
        </div>

        <div className="bg-white rounded-lg border border-gray-200 p-4 sm:p-6 google-shadow">
          <div className="flex items-center justify-between">
            <div>
              <p className="text-gray-500 text-sm font-medium">Watch Time</p>
              <p className="text-xl sm:text-2xl font-bold text-gray-900">{channel.watchTimeHours}h</p>
            </div>
            <div className="w-8 h-8 sm:w-10 sm:h-10 bg-yellow-50 rounded-lg flex items-center justify-center">
              <span className="material-icons text-google-yellow text-lg sm:text-xl">schedule</span>
            </div>
          </div>
        </div>

        <div className="bg-white rounded-lg border border-gray-200 p-4 sm:p-6 google-shadow">
          <div className="flex items-center justify-between">
            <div>
              <p className="text-gray-500 text-sm font-medium">AG Score</p>
              <p className="text-xl sm:text-2xl font-bold text-google-blue">{channel.agScore.toFixed(1)}</p>
            </div>
            <div className="w-8 h-8 sm:w-10 sm:h-10 bg-blue-50 rounded-lg flex items-center justify-center">
              <span className="material-icons text-google-blue text-lg sm:text-xl">auto_graph</span>
            </div>
          </div>
        </div>
      </div>

      {/* YouTube Toolbox */}
      <div className="bg-white rounded-lg border border-gray-200 p-4 sm:p-6 google-shadow mb-6">
        <div className="flex items-center justify-between mb-4">
          <h2 className="text-lg font-semibold text-gray-900">YouTube Toolbox</h2>
        </div>
        <div className="grid grid-cols-2 sm:grid-cols-3 lg:grid-cols-6 gap-2 sm:gap-3">
          <button
            onClick={() => handleActionClick("boost")}
            disabled={!selectedVideo}
            className="bg-white border border-gray-200 rounded-lg p-3 google-shadow hover:shadow-lg google-transition disabled:opacity-50 disabled:cursor-not-allowed group"
          >
            <div className="flex flex-col items-center space-y-2">
              <div className="w-8 h-8 bg-google-blue rounded-full flex items-center justify-center group-hover:scale-110 google-transition">
                <span className="material-icons text-white text-sm">rocket_launch</span>
              </div>
              <div className="text-center">
                <h3 className="font-medium text-gray-900 text-xs mb-1">Boost Video</h3>
                <p className="text-xs text-gray-500">Optimize selected video for algorithm performance</p>
                <div className="mt-1 text-xs text-google-blue font-medium">Duration: 1 minute</div>
              </div>
            </div>
          </button>

          <button
            onClick={() => handleActionClick("session")}
            disabled={!selectedVideo}
            className="bg-white border border-gray-200 rounded-lg p-3 google-shadow hover:shadow-lg google-transition disabled:opacity-50 disabled:cursor-not-allowed group"
          >
            <div className="flex flex-col items-center space-y-2">
              <div className="w-8 h-8 bg-google-green rounded-full flex items-center justify-center group-hover:scale-110 google-transition">
                <span className="material-icons text-white text-sm">psychology</span>
              </div>
              <div className="text-center">
                <h3 className="font-medium text-gray-900 text-xs mb-1">Create AG Session</h3>
                <p className="text-xs text-gray-500">Advanced algorithm optimization session</p>
                <div className="mt-1 text-xs text-google-green font-medium">Duration: 2 minutes</div>
              </div>
            </div>
          </button>

          <button
            onClick={() => handleActionClick("replenish")}
            disabled={cooldownStatus?.isActive}
            className="bg-white border border-gray-200 rounded-lg p-3 google-shadow hover:shadow-lg google-transition disabled:opacity-50 disabled:cursor-not-allowed group"
          >
            <div className="flex flex-col items-center space-y-2">
              <div className="w-8 h-8 bg-google-yellow rounded-full flex items-center justify-center group-hover:scale-110 google-transition">
                <span className="material-icons text-gray-900 text-sm">refresh</span>
              </div>
              <div className="text-center">
                <h3 className="font-medium text-gray-900 text-xs mb-1">Replenish AG Points</h3>
                <p className="text-xs text-gray-500">Restore algorithm grade points (+1.6)</p>
                <div className="mt-1 text-xs text-google-yellow font-medium">Duration: 4 minutes</div>
              </div>
            </div>
          </button>

          <button
            onClick={() => handleActionClick("devalue")}
            className="bg-white border border-gray-200 rounded-lg p-3 google-shadow hover:shadow-lg google-transition group"
          >
            <div className="flex flex-col items-center space-y-2">
              <div className="w-8 h-8 bg-red-500 rounded-full flex items-center justify-center group-hover:scale-110 google-transition">
                <span className="material-icons text-white text-sm">trending_down</span>
              </div>
              <div className="text-center">
                <h3 className="font-medium text-gray-900 text-xs mb-1">Devalue Channel</h3>
                <p className="text-xs text-gray-500">Advanced simulation to reduce AG points (-2.4)</p>
                <div className="mt-1 text-xs text-red-500 font-medium">Duration: 5 minutes</div>
              </div>
            </div>
          </button>

          <button
            onClick={() => handleActionClick("whatsapp")}
            className="bg-white border border-gray-200 rounded-lg p-3 google-shadow hover:shadow-lg google-transition group"
          >
            <div className="flex flex-col items-center space-y-2">
              <div className="w-8 h-8 bg-green-500 rounded-full flex items-center justify-center group-hover:scale-110 google-transition">
                <span className="material-icons text-white text-sm">share</span>
              </div>
              <div className="text-center">
                <h3 className="font-medium text-gray-900 text-xs mb-1">WhatsApp Boost</h3>
                <p className="text-xs text-gray-500">Generate visual content for channel promotion</p>
                <div className="mt-1 text-xs text-green-500 font-medium">Duration: 3 minutes</div>
              </div>
            </div>
          </button>

          <button
            onClick={() => setLoadViewsModal(true)}
            disabled={!selectedVideo}
            className="bg-white border border-gray-200 rounded-lg p-3 google-shadow hover:shadow-lg google-transition disabled:opacity-50 disabled:cursor-not-allowed group"
          >
            <div className="flex flex-col items-center space-y-2">
              <div className="w-8 h-8 bg-blue-500 rounded-full flex items-center justify-center group-hover:scale-110 google-transition">
                <span className="material-icons text-white text-sm">visibility</span>
              </div>
              <div className="text-center">
                <h3 className="font-medium text-gray-900 text-xs mb-1">Add Video Views</h3>
                <p className="text-xs text-gray-500">Order real views via TubeBoost API</p>
                <div className="mt-1 text-xs text-blue-500 font-medium">Instant delivery</div>
              </div>
            </div>
          </button>

          <button
            onClick={() => setLikesDialogOpen(true)}
            disabled={!selectedVideo}
            className="bg-white border border-gray-200 rounded-lg p-3 google-shadow hover:shadow-lg google-transition disabled:opacity-50 disabled:cursor-not-allowed group"
          >
            <div className="flex flex-col items-center space-y-2">
              <div className="w-8 h-8 bg-red-500 rounded-full flex items-center justify-center group-hover:scale-110 google-transition">
                <span className="material-icons text-white text-sm">thumb_up</span>
              </div>
              <div className="text-center">
                <h3 className="font-medium text-gray-900 text-xs mb-1">Add Likes</h3>
                <p className="text-xs text-gray-500">Order real likes via TubeBoost API</p>
                <div className="mt-1 text-xs text-red-500 font-medium">Instant delivery</div>
              </div>
            </div>
          </button>

          <button
            onClick={() => setSubscriberDialogOpen(true)}
            className="bg-white border border-gray-200 rounded-lg p-3 google-shadow hover:shadow-lg google-transition group"
          >
            <div className="flex flex-col items-center space-y-2">
              <div className="w-8 h-8 bg-orange-500 rounded-full flex items-center justify-center group-hover:scale-110 google-transition">
                <span className="material-icons text-white text-sm">person_add</span>
              </div>
              <div className="text-center">
                <h3 className="font-medium text-gray-900 text-xs mb-1">Add Subscribers</h3>
                <p className="text-xs text-gray-500">Order real subscribers via TubeBoost API</p>
                <div className="mt-1 text-xs text-orange-500 font-medium">Instant delivery</div>
              </div>
            </div>
          </button>

          <button
            onClick={() => handleActionClick("expose")}
            className="bg-white border border-gray-200 rounded-lg p-3 google-shadow hover:shadow-lg google-transition group"
          >
            <div className="flex flex-col items-center space-y-2">
              <div className="w-8 h-8 bg-purple-500 rounded-full flex items-center justify-center group-hover:scale-110 google-transition">
                <span className="material-icons text-white text-sm">campaign</span>
              </div>
              <div className="text-center">
                <h3 className="font-medium text-gray-900 text-xs mb-1">Boost Channel Exposure</h3>
                <p className="text-xs text-gray-500">AI-powered channel exposure optimization</p>
                <div className="mt-1 text-xs text-purple-500 font-medium">Duration: 1.5 minutes • 20 points</div>
              </div>
            </div>
          </button>
        </div>
      </div>

      {/* Order Statistics Section */}
      <div className="bg-white rounded-lg google-shadow p-6 mb-6">
        <div className="flex items-center space-x-2 mb-4">
          <span className="material-icons text-blue-600">assessment</span>
          <h2 className="text-xl font-semibold text-gray-900">Order Statistics</h2>
          <div className="bg-green-100 text-green-800 text-xs font-medium px-2 py-1 rounded-full">
            30s Auto-Update
          </div>
        </div>
        
        <div className="grid grid-cols-1 md:grid-cols-3 gap-4">
          {/* Total Active Orders */}
          <div className="bg-gradient-to-r from-blue-50 to-blue-100 rounded-lg p-4">
            <div className="flex items-center space-x-3">
              <div className="w-10 h-10 bg-blue-500 rounded-full flex items-center justify-center">
                <span className="material-icons text-white text-sm">trending_up</span>
              </div>
              <div>
                <p className="text-sm text-blue-600 font-medium">Active Orders</p>
                <p className="text-2xl font-bold text-blue-900">12</p>
                <p className="text-xs text-blue-500">Being monitored</p>
              </div>
            </div>
          </div>

          {/* Completed Orders Today */}
          <div className="bg-gradient-to-r from-green-50 to-green-100 rounded-lg p-4">
            <div className="flex items-center space-x-3">
              <div className="w-10 h-10 bg-green-500 rounded-full flex items-center justify-center">
                <span className="material-icons text-white text-sm">check_circle</span>
              </div>
              <div>
                <p className="text-sm text-green-600 font-medium">Completed Today</p>
                <p className="text-2xl font-bold text-green-900">3</p>
                <p className="text-xs text-green-500">Videos boosted</p>
              </div>
            </div>
          </div>

          {/* SMMCost API Status */}
          <div className="bg-gradient-to-r from-orange-50 to-orange-100 rounded-lg p-4">
            <div className="flex items-center space-x-3">
              <div className="w-10 h-10 bg-orange-500 rounded-full flex items-center justify-center">
                <span className="material-icons text-white text-sm">api</span>
              </div>
              <div>
                <p className="text-sm text-orange-600 font-medium">SMMCost API</p>
                <p className="text-lg font-bold text-orange-900">ONLINE</p>
                <p className="text-xs text-orange-500">Real-time tracking</p>
              </div>
            </div>
          </div>
        </div>

        {/* Recent Order Updates */}
        <div className="mt-6">
          <div className="flex items-center justify-between mb-3">
            <h3 className="text-md font-medium text-gray-900">Recent Order Updates</h3>
            {recentOrders.length > 5 && (
              <button
                onClick={() => setShowAllOrders(!showAllOrders)}
                className="text-sm text-blue-600 hover:text-blue-800 font-medium"
              >
                {showAllOrders ? 'Show Less' : `Show All (${recentOrders.length})`}
              </button>
            )}
          </div>
          
          <div className="space-y-2">
            {recentOrders.length === 0 ? (
              <div className="bg-gray-50 rounded-lg p-4 text-center">
                <p className="text-sm text-gray-500">No recent orders found</p>
              </div>
            ) : (
              (showAllOrders ? recentOrders : recentOrders.slice(0, 5)).map((order: any) => (
                <div key={order.id} className="bg-gray-50 rounded-lg p-3">
                  <div className="flex items-center justify-between">
                    <div className="flex items-center space-x-3">
                      <div className={`w-2 h-2 rounded-full ${
                        order.status === 'Completed' ? 'bg-green-500' :
                        order.status === 'In progress' ? 'bg-blue-500 animate-pulse' :
                        order.status === 'Pending' ? 'bg-yellow-500 animate-pulse' :
                        order.status === 'Partial' ? 'bg-orange-500' :
                        'bg-gray-500'
                      }`}></div>
                      <span className="text-sm text-gray-900">Order #{order.orderId}</span>
                      <span className={`text-xs px-2 py-1 rounded ${
                        order.service === 'Views' ? 'bg-blue-100 text-blue-800' :
                        order.service === 'Likes' ? 'bg-red-100 text-red-800' :
                        order.service === 'Subscribers' ? 'bg-orange-100 text-orange-800' :
                        'bg-gray-100 text-gray-800'
                      }`}>
                        {order.service}
                      </span>
                    </div>
                    <div className="text-right">
                      <p className={`text-sm font-medium ${
                        order.status === 'Completed' ? 'text-green-600' :
                        order.status === 'In progress' ? 'text-blue-600' :
                        order.status === 'Pending' ? 'text-yellow-600' :
                        order.status === 'Partial' ? 'text-orange-600' :
                        'text-gray-600'
                      }`}>
                        {order.status || 'Pending'}
                      </p>
                      <p className="text-xs text-gray-500">
                        {order.remains ? `${order.remains} remaining` : 
                         order.status === 'Completed' ? `${order.quantity} delivered` :
                         `${order.quantity} ordered`}
                      </p>
                    </div>
                  </div>
                  
                  {/* Enhanced Details */}
                  <div className="mt-2 pl-5 space-y-1">
                    <div className="flex items-center text-xs text-gray-600">
                      <Youtube className="h-3 w-3 mr-1" />
                      <span className="font-medium">{order.channel?.name || 'Unknown Channel'}</span>
                      {order.channel?.handle && (
                        <span className="ml-1 text-gray-500">@{order.channel.handle}</span>
                      )}
                    </div>
                    
                    {order.video && (
                      <div className="flex items-center text-xs text-gray-600">
                        <svg className="h-3 w-3 mr-1" fill="currentColor" viewBox="0 0 20 20">
                          <path d="M10 12a2 2 0 100-4 2 2 0 000 4z"/>
                          <path fillRule="evenodd" d="M.458 10C1.732 5.943 5.522 3 10 3s8.268 2.943 9.542 7c-1.274 4.057-5.064 7-9.542 7S1.732 14.057.458 10zM14 10a4 4 0 11-8 0 4 4 0 018 0z" clipRule="evenodd"/>
                        </svg>
                        <span className="truncate max-w-[200px]">
                          {order.video.title || 'Unknown Video'}
                        </span>
                      </div>
                    )}
                    
                    <div className="flex items-center justify-between text-xs text-gray-500">
                      <span>Cost: ${parseFloat(order.charge || '0').toFixed(2)}</span>
                      <span>
                        {order.updatedAt ? 
                          `Updated ${new Date(order.updatedAt).toLocaleDateString()}` :
                          `Created ${new Date(order.createdAt).toLocaleDateString()}`
                        }
                      </span>
                    </div>
                  </div>
                </div>
              ))
            )}
          </div>
        </div>
      </div>

      {/* Video Manager */}
      <VideoManager 
        selectedVideo={selectedVideo}
        setSelectedVideo={setSelectedVideo}
        isLoadingViews={isLoadingViews}
        loadVideoViews={loadVideoViews}
        setLoadViewsModal={setLoadViewsModal}
        setAdminLoadViewsModal={setAdminLoadViewsModal}
        setLikesDialogOpen={setLikesDialogOpen}
        userPoints={userPoints}
        ordersMap={ordersMap}
        user={user}
      />

      {/* Subscriber Orders Section */}
      <div className="mt-8">
        <div className="flex items-center justify-between mb-6">
          <h2 className="text-2xl font-bold text-gray-900">Subscriber Orders</h2>
          <Button 
            onClick={() => refetchSubscriberOrders()}
            variant="outline"
            size="sm"
            className="flex items-center gap-2"
          >
            <RefreshCw className="h-4 w-4" />
            Refresh
          </Button>
        </div>
        
        <div className="bg-white rounded-lg shadow-sm border border-gray-200 overflow-hidden">
          {subscriberOrders && subscriberOrders.length > 0 ? (
            <div className="overflow-x-auto">
              <table className="w-full">
                <thead className="bg-gray-50 border-b border-gray-200">
                  <tr>
                    <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                      Order ID
                    </th>
                    <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                      Quantity
                    </th>
                    <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                      Status
                    </th>
                    {user?.role === "admin" && (
                      <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                        Cost
                      </th>
                    )}
                    <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                      Start Count
                    </th>
                    <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                      Remaining
                    </th>
                    <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                      Created
                    </th>
                  </tr>
                </thead>
                <tbody className="bg-white divide-y divide-gray-200">
                  {subscriberOrders.map((order: any) => (
                    <tr key={order.id} className="hover:bg-gray-50">
                      <td className="px-6 py-4 whitespace-nowrap text-sm font-medium text-gray-900">
                        #{order.orderId}
                      </td>
                      <td className="px-6 py-4 whitespace-nowrap text-sm text-gray-900">
                        {order.quantity?.toLocaleString() || 'N/A'}
                      </td>
                      <td className="px-6 py-4 whitespace-nowrap">
                        <span className={`inline-flex px-2 py-1 text-xs font-semibold rounded-full ${
                          order.status === 'Completed' ? 'bg-green-100 text-green-800' :
                          order.status === 'In progress' ? 'bg-blue-100 text-blue-800' :
                          order.status === 'Pending' ? 'bg-yellow-100 text-yellow-800' :
                          order.status === 'Partial' ? 'bg-orange-100 text-orange-800' :
                          'bg-gray-100 text-gray-800'
                        }`}>
                          {order.status || 'Pending'}
                        </span>
                      </td>
                      {user?.role === "admin" && (
                        <td className="px-6 py-4 whitespace-nowrap text-sm text-gray-900">
                          ${parseFloat(order.charge || '0').toFixed(2)}
                        </td>
                      )}
                      <td className="px-6 py-4 whitespace-nowrap text-sm text-gray-900">
                        {order.startCount?.toLocaleString() || 'N/A'}
                      </td>
                      <td className="px-6 py-4 whitespace-nowrap text-sm text-gray-900">
                        {order.remains?.toLocaleString() || '0'}
                      </td>
                      <td className="px-6 py-4 whitespace-nowrap text-sm text-gray-500">
                        {order.createdAt ? new Date(order.createdAt).toLocaleDateString() : 'N/A'}
                      </td>
                    </tr>
                  ))}
                </tbody>
              </table>
            </div>
          ) : (
            <div className="p-8 text-center">
              <Users className="mx-auto h-12 w-12 text-gray-400 mb-4" />
              <h3 className="text-lg font-medium text-gray-900 mb-2">No Subscriber Orders</h3>
              <p className="text-gray-500">
                You haven't placed any subscriber orders yet. Use the "Add Subscribers" button to boost your channel's subscriber count.
              </p>
            </div>
          )}
        </div>
      </div>

      {/* Add Video Views Modal */}
      <Dialog open={loadViewsModal} onOpenChange={setLoadViewsModal}>
        <DialogContent className="sm:max-w-md">
          <DialogHeader>
            <DialogTitle>Add Video Views</DialogTitle>
            <DialogDescription>
              Add real YouTube views to the selected video using TubeBoost API.
            </DialogDescription>
          </DialogHeader>
          <div className="space-y-4">
            <p className="text-sm text-gray-600">
              Add real YouTube views to the selected video using TubeBoost API.
            </p>
            <div>
              <label className="text-sm font-medium">Selected Video</label>
              <p className="text-sm text-gray-500">
                {selectedVideo ? `Video ID: ${selectedVideo}` : "No video selected"}
              </p>
            </div>
            <div>
              <label className="text-sm font-medium">Number of Views</label>
              <Input
                type="number"
                min="1"
                placeholder="Enter number of views (min: 1)"
                value={customViewsQuantity}
                onChange={(e) => setCustomViewsQuantity(e.target.value)}
              />
              <p className="text-xs text-gray-500 mt-1">
                Cost: {customViewsQuantity || 0} points (1 point per view)
              </p>
            </div>
            <div className="flex justify-end space-x-2">
              <Button
                variant="outline"
                onClick={() => setLoadViewsModal(false)}
              >
                Cancel
              </Button>
              <Button
                onClick={() => {
                  const quantity = parseInt(customViewsQuantity);
                  if (selectedVideo && quantity > 0) {
                    loadVideoViews({ videoId: selectedVideo, quantity });
                  }
                }}
                disabled={!selectedVideo || !customViewsQuantity || parseInt(customViewsQuantity) < 1 || isLoadingViews}
              >
                {isLoadingViews ? "Placing Order..." : "Add Views"}
              </Button>
            </div>
          </div>
        </DialogContent>
      </Dialog>

      {/* Add Likes Modal */}
      <Dialog open={likesDialogOpen} onOpenChange={setLikesDialogOpen}>
        <DialogContent className="sm:max-w-md">
          <DialogHeader>
            <DialogTitle>Add Video Likes</DialogTitle>
            <DialogDescription>
              Add real YouTube likes to the selected video using TubeBoost API.
            </DialogDescription>
          </DialogHeader>
          <div className="space-y-4">
            <p className="text-sm text-gray-600">
              Add real YouTube likes to the selected video using TubeBoost API.
            </p>
            <div>
              <label className="text-sm font-medium">Selected Video</label>
              <p className="text-sm text-gray-500">
                {selectedVideo ? `Video ID: ${selectedVideo}` : "No video selected"}
              </p>
            </div>
            <div>
              <label className="text-sm font-medium">Number of Likes</label>
              <Input
                type="number"
                min="10"
                placeholder="Enter number of likes (min: 10)"
                value={customLikesQuantity}
                onChange={(e) => setCustomLikesQuantity(e.target.value)}
              />
              <p className="text-xs text-gray-500 mt-1">
                Cost: {Math.ceil((parseInt(customLikesQuantity) || 0) * 1.5)} points (1.5 points per like)
              </p>
            </div>
            <div className="flex justify-end space-x-2">
              <Button
                variant="outline"
                onClick={() => setLikesDialogOpen(false)}
              >
                Cancel
              </Button>
              <Button
                onClick={() => {
                  const quantity = parseInt(customLikesQuantity);
                  if (selectedVideo && quantity >= 10) {
                    addLikes({ videoId: selectedVideo, quantity });
                  }
                }}
                disabled={!selectedVideo || !customLikesQuantity || parseInt(customLikesQuantity) < 10 || isAddingLikes}
              >
                {isAddingLikes ? "Placing Order..." : "Add Likes"}
              </Button>
            </div>
          </div>
        </DialogContent>
      </Dialog>

      {/* Add Subscribers Modal */}
      <Dialog open={subscriberDialogOpen} onOpenChange={setSubscriberDialogOpen}>
        <DialogContent className="sm:max-w-md">
          <DialogHeader>
            <DialogTitle>Add Channel Subscribers</DialogTitle>
            <DialogDescription>
              Add real YouTube subscribers to your channel using TubeBoost API.
            </DialogDescription>
          </DialogHeader>
          <div className="space-y-4">
            <p className="text-sm text-gray-600">
              Add real YouTube subscribers to your channel using TubeBoost API.
            </p>
            <div>
              <label className="text-sm font-medium">Channel</label>
              <p className="text-sm text-gray-500">
                {channel?.channelName || "No channel selected"}
              </p>
            </div>
            <div>
              <label className="text-sm font-medium">Number of Subscribers</label>
              <Input
                type="number"
                min="1"
                placeholder="Enter number of subscribers (min: 1)"
                value={customSubscriberQuantity}
                onChange={(e) => setCustomSubscriberQuantity(e.target.value)}
              />
              <p className="text-xs text-gray-500 mt-1">
                Cost: {customSubscriberQuantity || 0} points (1 point per subscriber)
              </p>
            </div>
            <div className="flex justify-end space-x-2">
              <Button
                variant="outline"
                onClick={() => setSubscriberDialogOpen(false)}
              >
                Cancel
              </Button>
              <Button
                onClick={() => {
                  const quantity = parseInt(customSubscriberQuantity);
                  if (quantity > 0) {
                    addSubscribers(quantity);
                  }
                }}
                disabled={!customSubscriberQuantity || parseInt(customSubscriberQuantity) < 1 || isAddingSubscribers}
              >
                {isAddingSubscribers ? "Placing Order..." : "Add Subscribers"}
              </Button>
            </div>
          </div>
        </DialogContent>
      </Dialog>

      {/* Channel Exposure Simulation Modal */}
      <Dialog open={activeModal === "expose"} onOpenChange={() => setActiveModal(null)}>
        <DialogContent className="max-w-3xl bg-white border border-gray-200">
          <DialogHeader className="border-b border-gray-200 pb-4">
            <div className="flex items-center space-x-3">
              <div className="w-10 h-10 bg-purple-500 rounded-full flex items-center justify-center">
                <span className="material-icons text-white">campaign</span>
              </div>
              <div>
                <DialogTitle className="text-xl font-semibold text-gray-900">
                  AI-Powered Channel Exposure Boost
                </DialogTitle>
                <p className="text-sm text-gray-500 mt-1">
                  Optimizing channel visibility through advanced AI algorithms and exposure networks
                </p>
              </div>
            </div>
          </DialogHeader>
          
          {simulationState.isRunning && (
            <div className="p-6 space-y-6">
              <div className="bg-gray-50 rounded-lg p-4">
                <div className="flex items-center justify-between mb-3">
                  <h3 className="font-medium text-gray-900">AI Optimization Progress</h3>
                  <span className="text-sm font-medium text-gray-600">{simulationState.progress.toFixed(0)}%</span>
                </div>
                
                <div className="w-full bg-gray-200 rounded-full h-3 mb-3">
                  <div 
                    className="h-3 rounded-full transition-all duration-500 bg-purple-500"
                    style={{ width: `${simulationState.progress}%` }}
                  ></div>
                </div>
                
                <div className="flex items-center space-x-2 text-sm">
                  <span className="material-icons text-purple-400 text-sm animate-spin">sync</span>
                  <span className="text-gray-700">{simulationState.currentStep}</span>
                </div>
              </div>

              {/* AI-Powered Exposure Visualization */}
              <div className="bg-gradient-to-br from-purple-900 to-purple-700 rounded-xl p-6 text-white">
                <div className="flex items-center justify-between mb-4">
                  <h3 className="text-xl font-bold">AI Network Activation</h3>
                  <div className="flex items-center space-x-2">
                    <div className="w-3 h-3 bg-purple-400 rounded-full animate-pulse"></div>
                    <span className="text-purple-300 text-sm">LIVE AI</span>
                  </div>
                </div>
                
                <div className="grid grid-cols-3 gap-6 mb-6">
                  <div className="text-center">
                    <div className="text-3xl font-bold text-purple-300">{Math.floor(12 + (simulationState.progress / 100) * 18)}</div>
                    <div className="text-sm text-purple-200">Exposure Networks</div>
                  </div>
                  <div className="text-center">
                    <div className="text-3xl font-bold text-purple-300">{(2.1 + (simulationState.progress / 100) * 1.4).toFixed(1)}x</div>
                    <div className="text-sm text-purple-200">Reach Multiplier</div>
                  </div>
                  <div className="text-center">
                    <div className="text-3xl font-bold text-purple-300">{Math.floor(75 + (simulationState.progress / 100) * 20)}%</div>
                    <div className="text-sm text-purple-200">AI Analysis Score</div>
                  </div>
                </div>
                
                <div className="bg-purple-800 bg-opacity-50 rounded-lg p-4">
                  <div className="flex items-center justify-between">
                    <span className="text-purple-200">Optimization Level</span>
                    <span className="text-purple-100 font-bold">{Math.floor(simulationState.progress)}%</span>
                  </div>
                  <div className="w-full bg-purple-800 rounded-full h-2 mt-2">
                    <div 
                      className="h-2 bg-purple-400 rounded-full transition-all duration-500"
                      style={{ width: `${simulationState.progress}%` }}
                    ></div>
                  </div>
                </div>
              </div>

              {Object.keys(simulationState.metrics).length > 0 && (
                <div className="grid grid-cols-2 md:grid-cols-3 gap-4">
                  {Object.entries(simulationState.metrics).map(([key, value]) => (
                    <div key={key} className="bg-white border border-gray-200 rounded-lg p-4 shadow-sm">
                      <div className="text-xs text-gray-500 font-medium mb-1">{key}</div>
                      <div className="text-2xl font-bold text-gray-900">{value}</div>
                    </div>
                  ))}
                </div>
              )}
            </div>
          )}
        </DialogContent>
      </Dialog>
    </>
  );

  return <PlatformDashboard youtubeContent={youtubeContent} />;
}