import { useState, useEffect } from "react";
import { useQuery, useMutation, useQueryClient } from "@tanstack/react-query";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import { Textarea } from "@/components/ui/textarea";
import { Card, CardContent, CardDescription, CardFooter, CardHeader, CardTitle } from "@/components/ui/card";
import { Badge } from "@/components/ui/badge";
import { Switch } from "@/components/ui/switch";
import { Separator } from "@/components/ui/separator";
import { Plus, Code, Edit2, Trash2, Save, X, Eye, EyeOff } from "lucide-react";
import { useToast } from "@/hooks/use-toast";
import { apiRequest } from "@/lib/queryClient";
import DashboardLayout from "@/components/dashboard-layout";
import type { HeaderCode } from "@shared/schema";

export default function HeaderCodesPage() {
  const { toast } = useToast();
  const queryClient = useQueryClient();
  const [isEditing, setIsEditing] = useState<number | null>(null);
  const [isAdding, setIsAdding] = useState(false);
  const [newCode, setNewCode] = useState({ name: "", code: "", isActive: true });
  const [editingCode, setEditingCode] = useState<Partial<HeaderCode>>({});

  const { data: headerCodes = [], isLoading } = useQuery({
    queryKey: ["/api/header-codes"],
    queryFn: () => apiRequest("/api/header-codes"),
  });

  const createMutation = useMutation({
    mutationFn: (code: { name: string; code: string; isActive: boolean }) =>
      apiRequest("/api/header-codes", "POST", code),
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/header-codes"] });
      toast({
        title: "Success",
        description: "Header code created successfully",
      });
      setIsAdding(false);
      setNewCode({ name: "", code: "", isActive: true });
    },
    onError: (error: any) => {
      console.error("❌ [HEADER CODE] Frontend create error:", error);
      
      // Enhanced error handling with specific messages
      let errorMessage = "Failed to create header code";
      
      if (error.message) {
        if (error.message.includes("Authentication required")) {
          errorMessage = "Authentication required. Please log in again.";
        } else if (error.message.includes("Name and code are required")) {
          errorMessage = "Please provide both name and code for the header code.";
        } else if (error.message.includes("NOT NULL constraint")) {
          errorMessage = "There was a database error. Please try again or contact support.";
        } else {
          errorMessage = error.message;
        }
      }
      
      toast({
        title: "Error Creating Header Code",
        description: errorMessage,
        variant: "destructive",
      });
    },
  });

  const updateMutation = useMutation({
    mutationFn: ({ id, updates }: { id: number; updates: Partial<HeaderCode> }) =>
      apiRequest(`/api/header-codes/${id}`, "PUT", updates),
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/header-codes"] });
      toast({
        title: "Success",
        description: "Header code updated successfully",
      });
      setIsEditing(null);
      setEditingCode({});
    },
    onError: (error: any) => {
      console.error("❌ [HEADER CODE] Frontend update error:", error);
      
      // Enhanced error handling with specific messages
      let errorMessage = "Failed to update header code";
      
      if (error.message) {
        if (error.message.includes("Authentication required")) {
          errorMessage = "Authentication required. Please log in again.";
        } else if (error.message.includes("Header code not found")) {
          errorMessage = "Header code not found. It may have been deleted.";
        } else if (error.message.includes("404")) {
          errorMessage = "Header code not found. Please refresh the page.";
        } else {
          errorMessage = error.message;
        }
      }
      
      toast({
        title: "Error Updating Header Code",
        description: errorMessage,
        variant: "destructive",
      });
    },
  });

  const deleteMutation = useMutation({
    mutationFn: (id: number) => apiRequest(`/api/header-codes/${id}`, "DELETE"),
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/header-codes"] });
      toast({
        title: "Success",
        description: "Header code deleted successfully",
      });
    },
    onError: (error: any) => {
      console.error("❌ [HEADER CODE] Frontend delete error:", error);
      
      // Enhanced error handling with specific messages
      let errorMessage = "Failed to delete header code";
      
      if (error.message) {
        if (error.message.includes("Authentication required")) {
          errorMessage = "Authentication required. Please log in again.";
        } else if (error.message.includes("Header code not found")) {
          errorMessage = "Header code not found. It may have already been deleted.";
        } else if (error.message.includes("404")) {
          errorMessage = "Header code not found. Please refresh the page.";
        } else {
          errorMessage = error.message;
        }
      }
      
      toast({
        title: "Error Deleting Header Code",
        description: errorMessage,
        variant: "destructive",
      });
    },
  });

  const handleCreate = () => {
    // Enhanced validation before submitting
    if (!newCode.name.trim() || !newCode.code.trim()) {
      toast({
        title: "Validation Error",
        description: "Please provide both name and code for the header code.",
        variant: "destructive",
      });
      return;
    }
    
    if (newCode.name.length < 3) {
      toast({
        title: "Validation Error",
        description: "Header code name must be at least 3 characters long.",
        variant: "destructive",
      });
      return;
    }
    
    if (newCode.code.length < 10) {
      toast({
        title: "Validation Error",
        description: "Header code must be at least 10 characters long.",
        variant: "destructive",
      });
      return;
    }
    
    // Log the submission for debugging
    console.log("🔍 [HEADER CODE] Frontend submitting:", {
      name: newCode.name,
      codeLength: newCode.code.length,
      isActive: newCode.isActive
    });
    
    createMutation.mutate(newCode);
  };

  const handleUpdate = (id: number) => {
    updateMutation.mutate({ id, updates: editingCode });
  };

  const handleEdit = (code: HeaderCode) => {
    setIsEditing(code.id);
    setEditingCode(code);
  };

  const handleCancelEdit = () => {
    setIsEditing(null);
    setEditingCode({});
  };

  const handleDelete = (id: number) => {
    if (confirm("Are you sure you want to delete this header code?")) {
      deleteMutation.mutate(id);
    }
  };

  const toggleActive = (id: number, isActive: boolean) => {
    updateMutation.mutate({ id, updates: { isActive } });
  };

  if (isLoading) {
    return (
      <DashboardLayout>
        <div className="p-6">
          <div className="animate-pulse space-y-4">
            <div className="h-8 bg-gray-200 rounded w-1/3"></div>
            <div className="h-32 bg-gray-200 rounded"></div>
            <div className="h-32 bg-gray-200 rounded"></div>
          </div>
        </div>
      </DashboardLayout>
    );
  }

  return (
    <DashboardLayout>
      <div className="p-6 max-w-7xl mx-auto">
        <div className="flex items-center justify-between mb-6">
          <div>
            <h1 className="text-3xl font-bold text-gray-900 dark:text-white">Header Codes</h1>
            <p className="text-gray-600 dark:text-gray-400 mt-1">
              Manage Facebook and Google pixel codes displayed on every page
            </p>
          </div>
          <Button onClick={() => setIsAdding(true)} className="flex items-center gap-2">
            <Plus className="w-4 h-4" />
            Add Header Code
          </Button>
        </div>

        {/* Add New Header Code Form */}
        {isAdding && (
          <Card className="mb-6 border-blue-200 bg-blue-50/50 dark:bg-blue-900/20 dark:border-blue-800">
            <CardHeader>
              <CardTitle className="text-blue-900 dark:text-blue-100">Add New Header Code</CardTitle>
              <CardDescription>Create a new tracking code to be displayed on every page</CardDescription>
            </CardHeader>
            <CardContent className="space-y-4">
              <div>
                <Label htmlFor="name">Name</Label>
                <Input
                  id="name"
                  placeholder="e.g., Facebook Pixel, Google Analytics"
                  value={newCode.name}
                  onChange={(e) => setNewCode({ ...newCode, name: e.target.value })}
                />
              </div>
              <div>
                <Label htmlFor="code">Code</Label>
                <Textarea
                  id="code"
                  placeholder="Paste your tracking code here..."
                  value={newCode.code}
                  onChange={(e) => setNewCode({ ...newCode, code: e.target.value })}
                  className="min-h-[120px] font-mono text-sm"
                />
              </div>
              <div className="flex items-center space-x-2">
                <Switch
                  id="active"
                  checked={newCode.isActive}
                  onCheckedChange={(checked) => setNewCode({ ...newCode, isActive: checked })}
                />
                <Label htmlFor="active">Active</Label>
              </div>
            </CardContent>
            <CardFooter className="flex items-center gap-2">
              <Button
                onClick={handleCreate}
                disabled={createMutation.isPending}
                className="flex items-center gap-2"
              >
                <Save className="w-4 h-4" />
                {createMutation.isPending ? "Creating..." : "Create"}
              </Button>
              <Button
                variant="outline"
                onClick={() => {
                  setIsAdding(false);
                  setNewCode({ name: "", code: "", isActive: true });
                }}
              >
                <X className="w-4 h-4" />
                Cancel
              </Button>
            </CardFooter>
          </Card>
        )}

        {/* Header Codes List */}
        <div className="space-y-4">
          {headerCodes.length === 0 ? (
            <Card>
              <CardContent className="text-center py-12">
                <Code className="w-16 h-16 text-gray-300 mx-auto mb-4" />
                <h3 className="text-lg font-semibold text-gray-900 dark:text-white mb-2">
                  No header codes yet
                </h3>
                <p className="text-gray-600 dark:text-gray-400 mb-4">
                  Add your first tracking code to get started
                </p>
                <Button onClick={() => setIsAdding(true)}>Add Header Code</Button>
              </CardContent>
            </Card>
          ) : (
            headerCodes.map((code: HeaderCode) => (
              <Card key={code.id} className="relative">
                <CardHeader className="pb-3">
                  <div className="flex items-center justify-between">
                    <div className="flex items-center gap-3">
                      <CardTitle className="text-lg">{code.name}</CardTitle>
                      <Badge variant={code.isActive ? "default" : "secondary"}>
                        {code.isActive ? "Active" : "Inactive"}
                      </Badge>
                    </div>
                    <div className="flex items-center gap-2">
                      <Switch
                        checked={code.isActive}
                        onCheckedChange={(checked) => toggleActive(code.id, checked)}
                        disabled={updateMutation.isPending}
                      />
                      <Button
                        variant="outline"
                        size="sm"
                        onClick={() => handleEdit(code)}
                        disabled={isEditing === code.id}
                      >
                        <Edit2 className="w-4 h-4" />
                      </Button>
                      <Button
                        variant="outline"
                        size="sm"
                        onClick={() => handleDelete(code.id)}
                        disabled={deleteMutation.isPending}
                      >
                        <Trash2 className="w-4 h-4" />
                      </Button>
                    </div>
                  </div>
                </CardHeader>
                <CardContent>
                  {isEditing === code.id ? (
                    <div className="space-y-4">
                      <div>
                        <Label htmlFor={`edit-name-${code.id}`}>Name</Label>
                        <Input
                          id={`edit-name-${code.id}`}
                          value={editingCode.name || ""}
                          onChange={(e) =>
                            setEditingCode({ ...editingCode, name: e.target.value })
                          }
                        />
                      </div>
                      <div>
                        <Label htmlFor={`edit-code-${code.id}`}>Code</Label>
                        <Textarea
                          id={`edit-code-${code.id}`}
                          value={editingCode.code || ""}
                          onChange={(e) =>
                            setEditingCode({ ...editingCode, code: e.target.value })
                          }
                          className="min-h-[120px] font-mono text-sm"
                        />
                      </div>
                      <div className="flex items-center gap-2">
                        <Button
                          onClick={() => handleUpdate(code.id)}
                          disabled={updateMutation.isPending}
                          className="flex items-center gap-2"
                        >
                          <Save className="w-4 h-4" />
                          {updateMutation.isPending ? "Saving..." : "Save"}
                        </Button>
                        <Button variant="outline" onClick={handleCancelEdit}>
                          <X className="w-4 h-4" />
                          Cancel
                        </Button>
                      </div>
                    </div>
                  ) : (
                    <div>
                      <pre className="bg-gray-50 dark:bg-gray-800 p-4 rounded-md text-sm font-mono max-h-40 overflow-y-auto">
                        {code.code}
                      </pre>
                      <div className="mt-3 text-sm text-gray-500 dark:text-gray-400">
                        Created: {new Date(code.createdAt).toLocaleDateString()}{" "}
                        {new Date(code.createdAt).toLocaleTimeString()}
                      </div>
                    </div>
                  )}
                </CardContent>
              </Card>
            ))
          )}
        </div>
      </div>
    </DashboardLayout>
  );
}