import { useState, useEffect } from 'react';
import { useQuery, useMutation, useQueryClient } from '@tanstack/react-query';
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from '@/components/ui/card';
import { Button } from '@/components/ui/button';
import { Input } from '@/components/ui/input';
import { Textarea } from '@/components/ui/textarea';
import { Badge } from '@/components/ui/badge';
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from '@/components/ui/select';
import { Dialog, DialogContent, DialogDescription, DialogHeader, DialogTitle } from '@/components/ui/dialog';
import { useToast } from '@/hooks/use-toast';
import DashboardLayout from '@/components/dashboard-layout';
import { 
  Mail, 
  Send, 
  User, 
  Clock, 
  AlertTriangle, 
  CheckCircle,
  RefreshCw,
  MessageSquare,
  Shield,
  Reply,
  Archive,
  Trash2,
  Search
} from 'lucide-react';
import { apiRequest } from '@/lib/queryClient';

interface ModmailMessage {
  id: string;
  subject: string;
  author: string;
  body: string;
  created: string;
  isFromModerator: boolean;
  distinguished: string | null;
}

interface ModmailConversation {
  id: string;
  subject: string;
  participant: string;
  numMessages: number;
  lastModUpdate: string;
  state: 'new' | 'inprogress' | 'archived' | 'highlighted' | 'mod';
  isAuto: boolean;
  isRepliable: boolean;
  messages: ModmailMessage[];
}

export default function Modmail() {
  const [selectedSubreddit, setSelectedSubreddit] = useState<string>('');
  const [selectedConversation, setSelectedConversation] = useState<ModmailConversation | null>(null);
  const [replyText, setReplyText] = useState('');
  const [searchQuery, setSearchQuery] = useState('');
  const [filterState, setFilterState] = useState<string>('all');
  const { toast } = useToast();
  const queryClient = useQueryClient();

  // Fetch moderated subreddits
  const { data: moderatedSubreddits, isLoading: isLoadingSubreddits } = useQuery({
    queryKey: ['/api/subreddits/moderated'],
    queryFn: async () => {
      return await apiRequest('/api/subreddits/moderated', 'GET');
    },
    enabled: true,
  });

  // Fetch modmail conversations for selected subreddit
  const { data: modmailData, isLoading: isLoadingModmail, refetch: refetchModmail, error: modmailError } = useQuery({
    queryKey: ['/api/reddit/modmail', selectedSubreddit],
    queryFn: async () => {
      console.log('🔍 [MODMAIL DEBUG] Fetching modmail for subreddit:', selectedSubreddit);
      return await apiRequest(`/api/reddit/modmail?subreddit=${selectedSubreddit}`, 'GET');
    },
    enabled: !!selectedSubreddit,
    retry: false, // Don't retry on auth errors
  });

  // Send modmail reply mutation
  const sendReplyMutation = useMutation({
    mutationFn: async ({ conversationId, body, isInternal }: { conversationId: string; body: string; isInternal: boolean }) => {
      return await apiRequest(`/api/reddit/modmail/reply`, 'POST', {
        conversationId,
        body,
        isInternal,
        subreddit: selectedSubreddit
      });
    },
    onSuccess: () => {
      toast({
        title: "Reply sent successfully",
        description: "Your modmail reply has been sent."
      });
      setReplyText('');
      setSelectedConversation(null);
      refetchModmail();
    },
    onError: (error: any) => {
      toast({
        title: "Failed to send reply",
        description: error.message || "An error occurred while sending the reply",
        variant: "destructive"
      });
    }
  });

  // Archive conversation mutation
  const archiveConversationMutation = useMutation({
    mutationFn: async (conversationId: string) => {
      return await apiRequest(`/api/reddit/modmail/archive`, 'POST', {
        conversationId,
        subreddit: selectedSubreddit
      });
    },
    onSuccess: () => {
      toast({
        title: "Conversation archived",
        description: "The modmail conversation has been archived."
      });
      setSelectedConversation(null);
      refetchModmail();
    },
    onError: (error: any) => {
      toast({
        title: "Failed to archive conversation",
        description: error.message || "An error occurred while archiving",
        variant: "destructive"
      });
    }
  });

  const conversations: ModmailConversation[] = modmailData?.conversations || [];
  
  const filteredConversations = conversations.filter((conv) => {
    const matchesSearch = !searchQuery || 
      conv.subject.toLowerCase().includes(searchQuery.toLowerCase()) ||
      conv.participant.toLowerCase().includes(searchQuery.toLowerCase());
    
    const matchesState = filterState === 'all' || conv.state === filterState;
    
    return matchesSearch && matchesState;
  });

  const getStateColor = (state: string) => {
    switch (state) {
      case 'new': return 'bg-red-100 text-red-700';
      case 'inprogress': return 'bg-yellow-100 text-yellow-700';
      case 'archived': return 'bg-gray-100 text-gray-700';
      case 'highlighted': return 'bg-blue-100 text-blue-700';
      case 'mod': return 'bg-green-100 text-green-700';
      default: return 'bg-gray-100 text-gray-700';
    }
  };

  const getStateIcon = (state: string) => {
    switch (state) {
      case 'new': return <AlertTriangle className="h-3 w-3" />;
      case 'inprogress': return <Clock className="h-3 w-3" />;
      case 'archived': return <Archive className="h-3 w-3" />;
      case 'highlighted': return <Shield className="h-3 w-3" />;
      case 'mod': return <CheckCircle className="h-3 w-3" />;
      default: return <MessageSquare className="h-3 w-3" />;
    }
  };

  const handleSendReply = async (isInternal: boolean = false) => {
    if (!selectedConversation || !replyText.trim()) return;
    
    sendReplyMutation.mutate({
      conversationId: selectedConversation.id,
      body: replyText,
      isInternal
    });
  };

  const handleArchiveConversation = async () => {
    if (!selectedConversation) return;
    archiveConversationMutation.mutate(selectedConversation.id);
  };

  return (
    <DashboardLayout>
      <div className="container mx-auto p-4 space-y-6">
        <div className="flex items-center justify-between">
          <div>
            <h1 className="text-3xl font-bold flex items-center gap-2">
              <Mail className="h-8 w-8 text-blue-600" />
              Modmail Management
            </h1>
            <p className="text-muted-foreground mt-1">
              Manage modmail conversations across your subreddits
            </p>
          </div>
          <Button 
            onClick={() => refetchModmail()} 
            disabled={isLoadingModmail || !selectedSubreddit}
          >
            <RefreshCw className={`h-4 w-4 mr-2 ${isLoadingModmail ? 'animate-spin' : ''}`} />
            Refresh
          </Button>
        </div>

        {/* Subreddit Selection */}
        <Card>
          <CardHeader>
            <CardTitle className="text-lg">Select Subreddit</CardTitle>
            <CardDescription>Choose a subreddit to manage its modmail</CardDescription>
          </CardHeader>
          <CardContent>
            <Select value={selectedSubreddit} onValueChange={setSelectedSubreddit}>
              <SelectTrigger className="w-full">
                <SelectValue placeholder="Select a subreddit to manage modmail" />
              </SelectTrigger>
              <SelectContent>
                {moderatedSubreddits && Array.isArray(moderatedSubreddits) ? 
                  moderatedSubreddits.map((sub: any) => (
                    <SelectItem key={sub.name} value={sub.name}>
                      r/{sub.name}
                    </SelectItem>
                  )) : null
                }
              </SelectContent>
            </Select>
            
            {isLoadingSubreddits && (
              <div className="mt-2 text-sm text-muted-foreground">Loading your moderated subreddits...</div>
            )}
          </CardContent>
        </Card>

        {selectedSubreddit && (
          <>
            {/* Show reconnection notice if modmail scope is missing */}
            {modmailError && (modmailError as any)?.message?.includes('reconnect') && (
              <Card className="border-yellow-200 bg-yellow-50">
                <CardContent className="p-4">
                  <div className="flex items-center gap-3">
                    <AlertTriangle className="h-5 w-5 text-yellow-600" />
                    <div>
                      <p className="font-medium text-yellow-800">Additional Permissions Required</p>
                      <p className="text-sm text-yellow-700 mt-1">
                        To access modmail functionality, you need to reconnect your Reddit account with modmail permissions.
                      </p>
                      <Button 
                        className="mt-2" 
                        size="sm" 
                        onClick={() => window.open('/api/reddit/auth', '_blank')}
                      >
                        Reconnect Reddit Account
                      </Button>
                    </div>
                  </div>
                </CardContent>
              </Card>
            )}

            {/* Search and Filters */}
            <Card>
              <CardHeader>
                <CardTitle className="text-lg">Filter Conversations</CardTitle>
              </CardHeader>
              <CardContent className="space-y-4">
                <div className="grid grid-cols-1 md:grid-cols-3 gap-4">
                  <div>
                    <label className="text-sm font-medium mb-2 block">Search</label>
                    <div className="relative">
                      <Search className="absolute left-3 top-1/2 transform -translate-y-1/2 h-4 w-4 text-gray-400" />
                      <Input
                        placeholder="Search subjects, users..."
                        value={searchQuery}
                        onChange={(e) => setSearchQuery(e.target.value)}
                        className="pl-10"
                      />
                    </div>
                  </div>
                  <div>
                    <label className="text-sm font-medium mb-2 block">State</label>
                    <Select value={filterState} onValueChange={setFilterState}>
                      <SelectTrigger>
                        <SelectValue />
                      </SelectTrigger>
                      <SelectContent>
                        <SelectItem value="all">All States</SelectItem>
                        <SelectItem value="new">New</SelectItem>
                        <SelectItem value="inprogress">In Progress</SelectItem>
                        <SelectItem value="highlighted">Highlighted</SelectItem>
                        <SelectItem value="archived">Archived</SelectItem>
                        <SelectItem value="mod">Mod Discussions</SelectItem>
                      </SelectContent>
                    </Select>
                  </div>
                  <div className="flex items-end">
                    <Button variant="outline" onClick={() => {
                      setSearchQuery('');
                      setFilterState('all');
                    }}>
                      Clear Filters
                    </Button>
                  </div>
                </div>
              </CardContent>
            </Card>

            {/* Modmail Conversations */}
            <div className="space-y-4">
              <div className="flex items-center justify-between">
                <h2 className="text-xl font-semibold">
                  Modmail for r/{selectedSubreddit}
                </h2>
                <div className="text-sm text-muted-foreground">
                  {filteredConversations.length} of {conversations.length} conversations
                </div>
              </div>

              {isLoadingModmail ? (
                <Card>
                  <CardContent className="p-6">
                    <div className="flex items-center justify-center">
                      <RefreshCw className="h-6 w-6 animate-spin mr-2" />
                      Loading modmail conversations...
                    </div>
                  </CardContent>
                </Card>
              ) : filteredConversations.length === 0 ? (
                <Card>
                  <CardContent className="p-6">
                    <div className="text-center text-muted-foreground">
                      {conversations.length === 0 ? 
                        'No modmail conversations found for this subreddit.' :
                        'No conversations match your current filters.'
                      }
                    </div>
                  </CardContent>
                </Card>
              ) : (
                <div className="grid gap-4">
                  {filteredConversations.map((conversation) => (
                    <Card key={conversation.id} className="border-l-4 border-l-orange-500">
                      <CardContent className="p-6">
                        <div className="space-y-3">
                          <div className="flex items-start justify-between">
                            <div className="space-y-2 flex-1">
                              <div className="flex items-center gap-2 flex-wrap">
                                <Badge className={getStateColor(conversation.state)}>
                                  {getStateIcon(conversation.state)}
                                  <span className="ml-1">{conversation.state.toUpperCase()}</span>
                                </Badge>
                                {conversation.isAuto && (
                                  <Badge variant="outline">Auto</Badge>
                                )}
                                <span className="text-sm text-muted-foreground">
                                  {conversation.numMessages} message(s)
                                </span>
                              </div>
                              <h3 className="font-semibold text-lg">{conversation.subject}</h3>
                              <div className="text-sm text-muted-foreground">
                                <span>with u/{conversation.participant}</span> • 
                                <span className="ml-1">
                                  {new Date(conversation.lastModUpdate).toLocaleString()}
                                </span>
                              </div>
                            </div>
                            <div className="flex gap-2 ml-4">
                              <Button
                                size="sm"
                                variant="outline"
                                onClick={() => setSelectedConversation(conversation)}
                              >
                                <Reply className="h-3 w-3 mr-1" />
                                View & Reply
                              </Button>
                              {conversation.state !== 'archived' && (
                                <Button
                                  size="sm"
                                  variant="outline"
                                  onClick={() => handleArchiveConversation()}
                                  disabled={archiveConversationMutation.isPending}
                                >
                                  <Archive className="h-3 w-3 mr-1" />
                                  Archive
                                </Button>
                              )}
                            </div>
                          </div>
                        </div>
                      </CardContent>
                    </Card>
                  ))}
                </div>
              )}
            </div>
          </>
        )}

        {/* Conversation Detail Dialog */}
        <Dialog open={!!selectedConversation} onOpenChange={() => setSelectedConversation(null)}>
          <DialogContent className="max-w-4xl max-h-[80vh] overflow-hidden flex flex-col">
            <DialogHeader>
              <DialogTitle className="flex items-center gap-2">
                <MessageSquare className="h-5 w-5" />
                {selectedConversation?.subject}
              </DialogTitle>
              <DialogDescription>
                Conversation with u/{selectedConversation?.participant}
              </DialogDescription>
            </DialogHeader>
            
            {selectedConversation && (
              <div className="flex-1 flex flex-col space-y-4 overflow-hidden">
                {/* Messages */}
                <div className="flex-1 overflow-y-auto space-y-3 max-h-96 border rounded-lg p-4">
                  {selectedConversation.messages.map((message) => (
                    <div key={message.id} className={`p-3 rounded-lg ${
                      message.isFromModerator 
                        ? 'bg-blue-50 dark:bg-blue-950 ml-8' 
                        : 'bg-gray-50 dark:bg-gray-900 mr-8'
                    }`}>
                      <div className="flex items-center justify-between mb-2">
                        <div className="flex items-center gap-2">
                          <User className="h-4 w-4" />
                          <span className="font-medium">u/{message.author}</span>
                          {message.isFromModerator && (
                            <Badge variant="outline" className="text-xs">
                              <Shield className="h-3 w-3 mr-1" />
                              Moderator
                            </Badge>
                          )}
                          {message.distinguished && (
                            <Badge variant="default" className="text-xs">
                              {message.distinguished}
                            </Badge>
                          )}
                        </div>
                        <span className="text-xs text-muted-foreground">
                          {new Date(message.created).toLocaleString()}
                        </span>
                      </div>
                      <p className="text-sm whitespace-pre-wrap">{message.body}</p>
                    </div>
                  ))}
                </div>

                {/* Reply Form */}
                {selectedConversation.isRepliable && (
                  <div className="space-y-3 border-t pt-4">
                    <div>
                      <label className="text-sm font-medium mb-2 block">Reply</label>
                      <Textarea
                        value={replyText}
                        onChange={(e) => setReplyText(e.target.value)}
                        placeholder="Type your reply here..."
                        rows={4}
                        className="w-full"
                      />
                    </div>
                    <div className="flex gap-2 justify-end">
                      <Button
                        variant="outline"
                        onClick={() => setSelectedConversation(null)}
                      >
                        Cancel
                      </Button>
                      <Button
                        variant="outline"
                        onClick={() => handleSendReply(true)}
                        disabled={!replyText.trim() || sendReplyMutation.isPending}
                      >
                        <Shield className="h-3 w-3 mr-1" />
                        Mod Note
                      </Button>
                      <Button
                        onClick={() => handleSendReply(false)}
                        disabled={!replyText.trim() || sendReplyMutation.isPending}
                      >
                        {sendReplyMutation.isPending ? (
                          <>
                            <RefreshCw className="h-3 w-3 mr-1 animate-spin" />
                            Sending...
                          </>
                        ) : (
                          <>
                            <Send className="h-3 w-3 mr-1" />
                            Send Reply
                          </>
                        )}
                      </Button>
                    </div>
                  </div>
                )}

                {!selectedConversation.isRepliable && (
                  <div className="text-center text-sm text-muted-foreground border-t pt-4">
                    This conversation cannot be replied to.
                  </div>
                )}
              </div>
            )}
          </DialogContent>
        </Dialog>
      </div>
    </DashboardLayout>
  );
}