import { useQuery, useMutation, useQueryClient } from "@tanstack/react-query";
import { Button } from "@/components/ui/button";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Badge } from "@/components/ui/badge";
import { Tabs, TabsContent, TabsList, TabsTrigger } from "@/components/ui/tabs";
import { useState } from "react";
import { apiRequest } from "@/lib/queryClient";
import { DashboardLayout } from "@/components/dashboard-layout";
import { useToast } from "@/hooks/use-toast";
import { RefreshCw, Eye, Users, Clock, DollarSign, TrendingUp, Loader2 } from "lucide-react";

interface VideoOrder {
  id: number;
  videoId: number;
  channelId: number;
  quantity: number;
  orderId: string | null;
  status: string;
  service: string;
  serviceType: string;
  charge?: string;
  startCount?: string;
  remains?: string;
  currency?: string;
  createdAt: string;
  updatedAt: string;
  video?: {
    id: number;
    title: string;
    videoId: string;
    thumbnailUrl?: string;
    duration?: string;
    views?: number;
  } | null;
  channel?: {
    id: number;
    channelName: string;
    channelHandle?: string;
    customUrl?: string;
    channelId: string;
  } | null;
}

interface SubscriberOrder {
  id: number;
  channelId: number;
  quantity: number;
  orderId: string | null;
  status: string;
  service: string;
  serviceType: string;
  charge?: string;
  startCount?: string;
  remains?: string;
  currency?: string;
  channelUrl?: string;
  createdAt: string;
  updatedAt: string;
  channel?: {
    id: number;
    channelName: string;
    channelHandle?: string;
    customUrl?: string;
    channelId: string;
  } | null;
}

interface AllOrdersResponse {
  videoOrders: VideoOrder[];
  subscriberOrders: SubscriberOrder[];
  totalOrders: number;
}

export default function Orders() {
  const { toast } = useToast();
  const queryClient = useQueryClient();
  const [activeTab, setActiveTab] = useState("all");

  // Fetch all orders (admin only endpoint)
  const { data: allOrders, isLoading } = useQuery<AllOrdersResponse>({
    queryKey: ['/api/all-orders'],
    refetchInterval: 30000, // Auto-refresh every 30 seconds
  });

  // Fetch active orders for live tracking
  const { data: activeOrders } = useQuery({
    queryKey: ['/api/active-orders'],
    refetchInterval: 10000, // More frequent updates for active orders
  });

  // Sync order statuses mutation
  const { mutate: syncOrderStatuses, isPending: isSyncing } = useMutation({
    mutationFn: async () => {
      const response = await apiRequest("/api/sync-all-order-statuses", "POST", {});
      return response.json();
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ['/api/all-orders'] });
      queryClient.invalidateQueries({ queryKey: ['/api/active-orders'] });
      toast({
        title: "Orders Synced",
        description: "All order statuses have been updated from TubeBoost API",
      });
    },
    onError: (error: any) => {
      toast({
        title: "Sync Failed",
        description: error.message || "Failed to sync order statuses",
        variant: "destructive"
      });
    }
  });

  // Function to get order status color
  const getOrderStatusColor = (status: string) => {
    switch (status) {
      case 'Pending':
        return 'bg-yellow-100 text-yellow-800';
      case 'In progress':
        return 'bg-blue-100 text-blue-800';
      case 'Processing':
        return 'bg-blue-100 text-blue-800';
      case 'Partial':
        return 'bg-orange-100 text-orange-800';
      case 'Completed':
        return 'bg-green-100 text-green-800';
      case 'Canceled':
        return 'bg-red-100 text-red-800';
      default:
        return 'bg-gray-100 text-gray-800';
    }
  };

  if (isLoading) {
    return (
      <DashboardLayout>
        <div className="min-h-screen bg-gray-50 flex items-center justify-center">
          <div className="text-center">
            <div className="animate-spin rounded-full h-12 w-12 border-b-2 border-purple-600 mx-auto mb-4"></div>
            <p className="text-gray-600">Loading orders...</p>
          </div>
        </div>
      </DashboardLayout>
    );
  }

  const getStatusBadgeVariant = (status: string) => {
    switch (status.toLowerCase()) {
      case 'completed':
        return 'default';
      case 'pending':
      case 'processing':
      case 'in progress':
        return 'secondary';
      case 'partial':
        return 'outline';
      case 'canceled':
        return 'destructive';
      default:
        return 'secondary';
    }
  };

  return (
    <DashboardLayout>
      <div className="p-6 bg-gray-50 dark:bg-gray-900">
        <div className="max-w-7xl mx-auto">
        <div className="flex flex-col sm:flex-row justify-between items-start sm:items-center mb-6 sm:mb-8 space-y-4 sm:space-y-0">
          <div>
            <h1 className="text-2xl sm:text-3xl font-bold text-gray-900">Orders Management</h1>
            <p className="text-gray-600 mt-1 sm:mt-2 text-sm sm:text-base">View and manage all TubeBoost system orders</p>
          </div>
          
          <Button
            onClick={() => syncOrderStatuses()}
            disabled={isSyncing}
            className="flex items-center space-x-2 bg-purple-600 hover:bg-purple-700 w-full sm:w-auto"
          >
            <span className={`material-icons text-sm ${isSyncing ? 'animate-spin' : ''}`}>
              update
            </span>
            <span>{isSyncing ? 'Syncing...' : 'Sync All Orders'}</span>
          </Button>
        </div>

          {/* Summary Cards */}
          <div className="grid grid-cols-1 md:grid-cols-4 gap-6 mb-6">
            <Card>
              <CardContent className="p-6">
                <div className="flex items-center justify-between">
                  <div>
                    <p className="text-sm font-medium text-gray-600 dark:text-gray-400">Video Orders</p>
                    <p className="text-2xl font-bold text-gray-900 dark:text-white">
                      {allOrders?.videoOrders?.length || 0}
                    </p>
                  </div>
                  <Eye className="w-8 h-8 text-blue-600" />
                </div>
              </CardContent>
            </Card>

            <Card>
              <CardContent className="p-6">
                <div className="flex items-center justify-between">
                  <div>
                    <p className="text-sm font-medium text-gray-600 dark:text-gray-400">Subscriber Orders</p>
                    <p className="text-2xl font-bold text-gray-900 dark:text-white">
                      {allOrders?.subscriberOrders?.length || 0}
                    </p>
                  </div>
                  <Users className="w-8 h-8 text-green-600" />
                </div>
              </CardContent>
            </Card>

            <Card>
              <CardContent className="p-6">
                <div className="flex items-center justify-between">
                  <div>
                    <p className="text-sm font-medium text-gray-600 dark:text-gray-400">Active Orders</p>
                    <p className="text-2xl font-bold text-orange-600">
                      {(allOrders?.videoOrders?.filter(o => ['Processing', 'In progress', 'Pending'].includes(o.status)).length || 0) +
                       (allOrders?.subscriberOrders?.filter(o => ['Processing', 'In progress', 'Pending'].includes(o.status)).length || 0)}
                    </p>
                  </div>
                  <Clock className="w-8 h-8 text-orange-600" />
                </div>
              </CardContent>
            </Card>

            <Card>
              <CardContent className="p-6">
                <div className="flex items-center justify-between">
                  <div>
                    <p className="text-sm font-medium text-gray-600 dark:text-gray-400">Total Orders</p>
                    <p className="text-2xl font-bold text-purple-600">
                      {allOrders?.totalOrders || 0}
                    </p>
                  </div>
                  <TrendingUp className="w-8 h-8 text-purple-600" />
                </div>
              </CardContent>
            </Card>
          </div>

          {/* Orders Tabs */}
          <Tabs value={activeTab} onValueChange={setActiveTab} className="space-y-6">
            <TabsList className="grid w-full grid-cols-3">
              <TabsTrigger value="all">All Orders</TabsTrigger>
              <TabsTrigger value="video">Video Orders</TabsTrigger>
              <TabsTrigger value="subscriber">Subscriber Orders</TabsTrigger>
            </TabsList>

            <TabsContent value="all" className="space-y-6">
              <div className="grid gap-6">
                {/* Video Orders */}
                {allOrders?.videoOrders && allOrders.videoOrders.length > 0 && (
                  <Card>
                    <CardHeader>
                      <CardTitle className="flex items-center space-x-2">
                        <Eye className="w-5 h-5" />
                        <span>Video View Orders</span>
                        <Badge variant="secondary">{allOrders.videoOrders.length}</Badge>
                      </CardTitle>
                    </CardHeader>
                    <CardContent>
                      <div className="space-y-3">
                        {allOrders.videoOrders.slice(0, 5).map((order: VideoOrder) => (
                          <div key={order.id} className="flex items-center justify-between p-4 border rounded-lg">
                            <div className="flex items-center space-x-4">
                              <div className={`px-2 py-1 rounded text-xs font-medium ${getOrderStatusColor(order.status)}`}>
                                {order.status}
                              </div>
                              <div className="flex-1">
                                <div className="flex items-center space-x-2">
                                  <p className="font-medium">{order.quantity.toLocaleString()} {order.serviceType || 'views'}</p>
                                  {order.video?.thumbnailUrl && (
                                    <img 
                                      src={order.video.thumbnailUrl} 
                                      alt="Video thumbnail" 
                                      className="w-8 h-6 rounded object-cover"
                                    />
                                  )}
                                </div>
                                <p className="text-sm font-medium text-gray-900 truncate max-w-md">
                                  {order.video?.title || 'Unknown Video'}
                                </p>
                                <p className="text-xs text-gray-500">
                                  Channel: {order.channel?.channelName || 'Unknown'} • Order #{order.orderId?.slice(-6) || order.id}
                                </p>
                              </div>
                            </div>
                            <div className="text-right">
                              <p className="font-medium">${order.charge || 'N/A'} {order.currency || 'USD'}</p>
                              <p className="text-sm text-gray-500">{new Date(order.createdAt).toLocaleDateString()}</p>
                            </div>
                          </div>
                        ))}
                        {allOrders.videoOrders.length > 5 && (
                          <p className="text-center text-sm text-gray-500 py-2">
                            +{allOrders.videoOrders.length - 5} more video orders
                          </p>
                        )}
                      </div>
                    </CardContent>
                  </Card>
                )}

                {/* Subscriber Orders */}
                {allOrders?.subscriberOrders && allOrders.subscriberOrders.length > 0 && (
                  <Card>
                    <CardHeader>
                      <CardTitle className="flex items-center space-x-2">
                        <Users className="w-5 h-5" />
                        <span>Subscriber Orders</span>
                        <Badge variant="secondary">{allOrders.subscriberOrders.length}</Badge>
                      </CardTitle>
                    </CardHeader>
                    <CardContent>
                      <div className="space-y-3">
                        {allOrders.subscriberOrders.slice(0, 5).map((order: SubscriberOrder) => (
                          <div key={order.id} className="flex items-center justify-between p-4 border rounded-lg">
                            <div className="flex items-center space-x-4">
                              <div className={`px-2 py-1 rounded text-xs font-medium ${getOrderStatusColor(order.status)}`}>
                                {order.status}
                              </div>
                              <div className="flex-1">
                                <p className="font-medium">{order.quantity.toLocaleString()} {order.serviceType || 'subscribers'}</p>
                                <p className="text-sm font-medium text-gray-900">
                                  Channel: {order.channel?.channelName || 'Unknown Channel'}
                                </p>
                                <p className="text-xs text-gray-500">
                                  Handle: @{order.channel?.channelHandle || 'unknown'} • Order #{order.orderId?.slice(-6) || order.id}
                                </p>
                              </div>
                            </div>
                            <div className="text-right">
                              <p className="font-medium">${order.charge || 'N/A'} {order.currency || 'USD'}</p>
                              <p className="text-sm text-gray-500">{new Date(order.createdAt).toLocaleDateString()}</p>
                            </div>
                          </div>
                        ))}
                        {allOrders.subscriberOrders.length > 5 && (
                          <p className="text-center text-sm text-gray-500 py-2">
                            +{allOrders.subscriberOrders.length - 5} more subscriber orders
                          </p>
                        )}
                      </div>
                    </CardContent>
                  </Card>
                )}
              </div>
            </TabsContent>

            <TabsContent value="video">
              <Card>
                <CardHeader>
                  <CardTitle className="flex items-center space-x-2">
                    <Eye className="w-5 h-5" />
                    <span>Video View Orders</span>
                    <Badge variant="secondary">{allOrders?.videoOrders?.length || 0}</Badge>
                  </CardTitle>
                </CardHeader>
                <CardContent>
                  {!allOrders?.videoOrders?.length ? (
                    <div className="text-center py-8">
                      <Eye className="w-12 h-12 text-gray-400 mx-auto mb-4" />
                      <p className="text-gray-500">No video orders found.</p>
                    </div>
                  ) : (
                    <div className="space-y-3">
                      {allOrders.videoOrders.map((order: VideoOrder) => (
                        <div key={order.id} className="border rounded-lg p-4">
                          <div className="flex items-center justify-between mb-3">
                            <div className="flex items-center space-x-3">
                              <div className={`px-2 py-1 rounded text-xs font-medium ${getOrderStatusColor(order.status)}`}>
                                {order.status}
                              </div>
                              <span className="font-medium">{order.quantity.toLocaleString()} {order.serviceType || 'views'}</span>
                              <span className="text-sm text-gray-500">#{order.orderId?.slice(-6) || order.id}</span>
                              {order.video?.thumbnailUrl && (
                                <img 
                                  src={order.video.thumbnailUrl} 
                                  alt="Video thumbnail" 
                                  className="w-10 h-6 rounded object-cover"
                                />
                              )}
                            </div>
                            <div className="text-right">
                              <p className="font-medium">${order.charge || 'N/A'} {order.currency || 'USD'}</p>
                              <p className="text-sm text-gray-500">{new Date(order.createdAt).toLocaleDateString()}</p>
                            </div>
                          </div>
                          
                          {/* Video and Channel Information */}
                          <div className="mb-3 space-y-1">
                            <div className="text-sm">
                              <span className="text-gray-500 font-medium">Video: </span>
                              <span className="font-medium text-gray-900">{order.video?.title || 'Unknown Video'}</span>
                            </div>
                            <div className="text-sm">
                              <span className="text-gray-500 font-medium">Channel: </span>
                              <span className="font-medium text-gray-900">{order.channel?.channelName || 'Unknown Channel'}</span>
                              {order.channel?.channelHandle && (
                                <span className="text-gray-500 ml-1">(@{order.channel.channelHandle})</span>
                              )}
                            </div>
                          </div>
                          
                          {(order.startCount || order.remains) && (
                            <div className="grid grid-cols-2 gap-4 text-sm border-t pt-3">
                              {order.startCount && (
                                <div>
                                  <span className="text-gray-500">Start Count: </span>
                                  <span className="font-medium">{parseInt(order.startCount).toLocaleString()}</span>
                                </div>
                              )}
                              {order.remains && (
                                <div>
                                  <span className="text-gray-500">Remaining: </span>
                                  <span className="font-medium">{order.remains}</span>
                                </div>
                              )}
                            </div>
                          )}
                        </div>
                      ))}
                    </div>
                  )}
                </CardContent>
              </Card>
            </TabsContent>

            <TabsContent value="subscriber">
              <Card>
                <CardHeader>
                  <CardTitle className="flex items-center space-x-2">
                    <Users className="w-5 h-5" />
                    <span>Subscriber Orders</span>
                    <Badge variant="secondary">{allOrders?.subscriberOrders?.length || 0}</Badge>
                  </CardTitle>
                </CardHeader>
                <CardContent>
                  {!allOrders?.subscriberOrders?.length ? (
                    <div className="text-center py-8">
                      <Users className="w-12 h-12 text-gray-400 mx-auto mb-4" />
                      <p className="text-gray-500">No subscriber orders found.</p>
                    </div>
                  ) : (
                    <div className="space-y-3">
                      {allOrders.subscriberOrders.map((order: SubscriberOrder) => (
                        <div key={order.id} className="border rounded-lg p-4">
                          <div className="flex items-center justify-between mb-3">
                            <div className="flex items-center space-x-3">
                              <div className={`px-2 py-1 rounded text-xs font-medium ${getOrderStatusColor(order.status)}`}>
                                {order.status}
                              </div>
                              <span className="font-medium">{order.quantity.toLocaleString()} {order.serviceType || 'subscribers'}</span>
                              <span className="text-sm text-gray-500">#{order.orderId?.slice(-6) || order.id}</span>
                            </div>
                            <div className="text-right">
                              <p className="font-medium">${order.charge || 'N/A'} {order.currency || 'USD'}</p>
                              <p className="text-sm text-gray-500">{new Date(order.createdAt).toLocaleDateString()}</p>
                            </div>
                          </div>
                          
                          {/* Channel Information */}
                          <div className="mb-3 space-y-1">
                            <div className="text-sm">
                              <span className="text-gray-500 font-medium">Channel: </span>
                              <span className="font-medium text-gray-900">{order.channel?.channelName || 'Unknown Channel'}</span>
                              {order.channel?.channelHandle && (
                                <span className="text-gray-500 ml-1">(@{order.channel.channelHandle})</span>
                              )}
                            </div>
                            {order.channelUrl && (
                              <div className="text-sm">
                                <span className="text-gray-500 font-medium">URL: </span>
                                <span className="font-medium text-blue-600">{order.channelUrl}</span>
                              </div>
                            )}
                          </div>
                          
                          {(order.startCount || order.remains) && (
                            <div className="grid grid-cols-2 gap-4 text-sm border-t pt-3">
                              {order.startCount && (
                                <div>
                                  <span className="text-gray-500">Start Count: </span>
                                  <span className="font-medium">{parseInt(order.startCount).toLocaleString()}</span>
                                </div>
                              )}
                              {order.remains && (
                                <div>
                                  <span className="text-gray-500">Remaining: </span>
                                  <span className="font-medium">{order.remains}</span>
                                </div>
                              )}
                            </div>
                          )}
                          
                          {(order.startCount || order.remains) && (
                            <div className="grid grid-cols-2 gap-4 text-sm mt-2">
                              {order.startCount && (
                                <div>
                                  <span className="text-gray-500">Start Count: </span>
                                  <span className="font-medium">{parseInt(order.startCount).toLocaleString()}</span>
                                </div>
                              )}
                              {order.remains && (
                                <div>
                                  <span className="text-gray-500">Remaining: </span>
                                  <span className="font-medium">{order.remains}</span>
                                </div>
                              )}
                            </div>
                          )}
                        </div>
                      ))}
                    </div>
                  )}
                </CardContent>
              </Card>
            </TabsContent>
          </Tabs>
        </div>
      </div>
    </DashboardLayout>
  );
}