import { useState, useEffect } from "react";
import { useQuery, useMutation, useQueryClient } from "@tanstack/react-query";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select";
import { Textarea } from "@/components/ui/textarea";
import { Table, TableBody, TableCell, TableHead, TableHeader, TableRow } from "@/components/ui/table";
import { Dialog, DialogContent, DialogDescription, DialogHeader, DialogTitle, DialogTrigger } from "@/components/ui/dialog";
import { Badge } from "@/components/ui/badge";
import { Switch } from "@/components/ui/switch";
import { useToast } from "@/hooks/use-toast";
import { apiRequest } from "@/lib/queryClient";
import { Plus, Edit, Trash2, Package, DollarSign, Users, Clock } from "lucide-react";
import { DashboardLayout } from "@/components/dashboard-layout";

interface SubscriptionPackage {
  id: number;
  name: string;
  displayName: string;
  description: string;
  price: string;
  currency: string;
  billingCycle: string;
  dailyPoints: number;
  weeklyPoints: number;
  monthlyPoints: number;
  pointAllocationMethod: string;
  maxChannels: number;
  maxVideosPerDay: number;
  features: string[];
  isActive: boolean;
  sortOrder: number;
  createdAt: string;
  updatedAt: string;
}

interface PackageFormData {
  name: string;
  displayName: string;
  description: string;
  price: string;
  currency: string;
  billingCycle: string;
  dailyPoints: number;
  weeklyPoints: number;
  monthlyPoints: number;
  pointAllocationMethod: string;
  maxChannels: number;
  maxVideosPerDay: number;
  features: string[];
  isActive: boolean;
  sortOrder: number;
}

export default function PackageManager() {
  const { toast } = useToast();
  const queryClient = useQueryClient();
  const [isCreateDialogOpen, setIsCreateDialogOpen] = useState(false);
  const [isEditDialogOpen, setIsEditDialogOpen] = useState(false);
  const [selectedPackage, setSelectedPackage] = useState<SubscriptionPackage | null>(null);
  const [formData, setFormData] = useState<PackageFormData>({
    name: "",
    displayName: "",
    description: "",
    price: "0.00",
    currency: "USD",
    billingCycle: "monthly",
    dailyPoints: 600,
    weeklyPoints: 4200,
    monthlyPoints: 18000,
    pointAllocationMethod: "daily",
    maxChannels: 1,
    maxVideosPerDay: 10,
    features: [],
    isActive: true,
    sortOrder: 0
  });

  // Fetch packages
  const { data: packages, isLoading } = useQuery({
    queryKey: ["/api/packages"],
    retry: false,
  });

  // Create package mutation
  const createPackageMutation = useMutation({
    mutationFn: async (data: PackageFormData) => {
      return apiRequest("/api/packages", "POST", data);
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/packages"] });
      setIsCreateDialogOpen(false);
      resetForm();
      toast({
        title: "Package Created",
        description: "Subscription package created successfully",
      });
    },
    onError: (error: any) => {
      toast({
        title: "Error",
        description: error.message || "Failed to create package",
        variant: "destructive",
      });
    },
  });

  // Update package mutation
  const updatePackageMutation = useMutation({
    mutationFn: async (data: { id: number; updates: Partial<PackageFormData> }) => {
      return apiRequest(`/api/packages/${data.id}`, "PUT", data.updates);
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/packages"] });
      setIsEditDialogOpen(false);
      resetForm();
      toast({
        title: "Package Updated",
        description: "Subscription package updated successfully",
      });
    },
    onError: (error: any) => {
      toast({
        title: "Error",
        description: error.message || "Failed to update package",
        variant: "destructive",
      });
    },
  });

  // Delete package mutation
  const deletePackageMutation = useMutation({
    mutationFn: async (id: number) => {
      return apiRequest(`/api/packages/${id}`, "DELETE");
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/packages"] });
      toast({
        title: "Package Deleted",
        description: "Subscription package deleted successfully",
      });
    },
    onError: (error: any) => {
      toast({
        title: "Error",
        description: error.message || "Failed to delete package",
        variant: "destructive",
      });
    },
  });

  const resetForm = () => {
    setFormData({
      name: "",
      displayName: "",
      description: "",
      price: "0.00",
      currency: "USD",
      billingCycle: "monthly",
      dailyPoints: 600,
      weeklyPoints: 4200,
      monthlyPoints: 18000,
      pointAllocationMethod: "daily",
      maxChannels: 1,
      maxVideosPerDay: 10,
      features: [],
      isActive: true,
      sortOrder: 0
    });
    setSelectedPackage(null);
  };

  const handleEdit = (pkg: SubscriptionPackage) => {
    setSelectedPackage(pkg);
    setFormData({
      name: pkg.name,
      displayName: pkg.displayName,
      description: pkg.description || "",
      price: pkg.price,
      currency: pkg.currency,
      billingCycle: pkg.billingCycle,
      dailyPoints: pkg.dailyPoints,
      weeklyPoints: pkg.weeklyPoints,
      monthlyPoints: pkg.monthlyPoints,
      pointAllocationMethod: pkg.pointAllocationMethod,
      maxChannels: pkg.maxChannels,
      maxVideosPerDay: pkg.maxVideosPerDay,
      features: pkg.features || [],
      isActive: pkg.isActive,
      sortOrder: pkg.sortOrder
    });
    setIsEditDialogOpen(true);
  };

  const handleDelete = (id: number) => {
    if (confirm("Are you sure you want to delete this package?")) {
      deletePackageMutation.mutate(id);
    }
  };

  const handleSubmit = (e: React.FormEvent) => {
    e.preventDefault();
    
    if (selectedPackage) {
      updatePackageMutation.mutate({
        id: selectedPackage.id,
        updates: formData
      });
    } else {
      createPackageMutation.mutate(formData);
    }
  };

  const updateFormField = (field: keyof PackageFormData, value: any) => {
    setFormData(prev => ({ ...prev, [field]: value }));
  };

  // Auto-calculate points based on allocation method
  useEffect(() => {
    if (formData.pointAllocationMethod === "daily") {
      setFormData(prev => ({
        ...prev,
        weeklyPoints: prev.dailyPoints * 7,
        monthlyPoints: prev.dailyPoints * 30
      }));
    } else if (formData.pointAllocationMethod === "weekly") {
      setFormData(prev => ({
        ...prev,
        dailyPoints: Math.round(prev.weeklyPoints / 7),
        monthlyPoints: Math.round(prev.weeklyPoints * 4.33) // ~4.33 weeks per month
      }));
    } else if (formData.pointAllocationMethod === "monthly") {
      setFormData(prev => ({
        ...prev,
        dailyPoints: Math.round(prev.monthlyPoints / 30),
        weeklyPoints: Math.round(prev.monthlyPoints / 4.33)
      }));
    }
  }, [formData.dailyPoints, formData.weeklyPoints, formData.monthlyPoints, formData.pointAllocationMethod]);

  const PackageForm = () => (
    <form onSubmit={handleSubmit} className="space-y-4">
      <div className="grid grid-cols-2 gap-4">
        <div>
          <Label htmlFor="name">Package Name</Label>
          <Input
            id="name"
            value={formData.name}
            onChange={(e) => updateFormField("name", e.target.value)}
            placeholder="e.g., starter"
            required
          />
        </div>
        <div>
          <Label htmlFor="displayName">Display Name</Label>
          <Input
            id="displayName"
            value={formData.displayName}
            onChange={(e) => updateFormField("displayName", e.target.value)}
            placeholder="e.g., Starter Plan"
            required
          />
        </div>
      </div>

      <div>
        <Label htmlFor="description">Description</Label>
        <Textarea
          id="description"
          value={formData.description}
          onChange={(e) => updateFormField("description", e.target.value)}
          placeholder="Package description"
        />
      </div>

      <div className="grid grid-cols-3 gap-4">
        <div>
          <Label htmlFor="price">Price</Label>
          <Input
            id="price"
            type="number"
            step="0.01"
            value={formData.price}
            onChange={(e) => updateFormField("price", e.target.value)}
            required
          />
        </div>
        <div>
          <Label htmlFor="currency">Currency</Label>
          <Select value={formData.currency} onValueChange={(value) => updateFormField("currency", value)}>
            <SelectTrigger>
              <SelectValue />
            </SelectTrigger>
            <SelectContent>
              <SelectItem value="USD">USD</SelectItem>
              <SelectItem value="EUR">EUR</SelectItem>
              <SelectItem value="GBP">GBP</SelectItem>
            </SelectContent>
          </Select>
        </div>
        <div>
          <Label htmlFor="billingCycle">Billing Cycle</Label>
          <Select value={formData.billingCycle} onValueChange={(value) => updateFormField("billingCycle", value)}>
            <SelectTrigger>
              <SelectValue />
            </SelectTrigger>
            <SelectContent>
              <SelectItem value="monthly">Monthly</SelectItem>
              <SelectItem value="yearly">Yearly</SelectItem>
            </SelectContent>
          </Select>
        </div>
      </div>

      <div className="grid grid-cols-4 gap-4">
        <div>
          <Label htmlFor="pointAllocationMethod">Point Allocation</Label>
          <Select value={formData.pointAllocationMethod} onValueChange={(value) => updateFormField("pointAllocationMethod", value)}>
            <SelectTrigger>
              <SelectValue />
            </SelectTrigger>
            <SelectContent>
              <SelectItem value="daily">Daily</SelectItem>
              <SelectItem value="weekly">Weekly</SelectItem>
              <SelectItem value="monthly">Monthly</SelectItem>
            </SelectContent>
          </Select>
        </div>
        {/* Primary Point Input - Only show the selected allocation method */}
        {formData.pointAllocationMethod === "daily" && (
          <div>
            <Label htmlFor="dailyPoints">Daily Points</Label>
            <Input
              id="dailyPoints"
              type="number"
              value={formData.dailyPoints}
              onChange={(e) => updateFormField("dailyPoints", parseInt(e.target.value))}
              required
            />
            <p className="text-xs text-blue-600 mt-1">Users receive {formData.dailyPoints} points every day</p>
          </div>
        )}
        
        {formData.pointAllocationMethod === "weekly" && (
          <div>
            <Label htmlFor="weeklyPoints">Weekly Points</Label>
            <Input
              id="weeklyPoints"
              type="number"
              value={formData.weeklyPoints}
              onChange={(e) => updateFormField("weeklyPoints", parseInt(e.target.value))}
              required
            />
            <p className="text-xs text-blue-600 mt-1">Users receive {formData.weeklyPoints} points every week</p>
          </div>
        )}
        
        {formData.pointAllocationMethod === "monthly" && (
          <div>
            <Label htmlFor="monthlyPoints">Monthly Points</Label>
            <Input
              id="monthlyPoints"
              type="number"
              value={formData.monthlyPoints}
              onChange={(e) => updateFormField("monthlyPoints", parseInt(e.target.value))}
              required
            />
            <p className="text-xs text-blue-600 mt-1">Users receive {formData.monthlyPoints} points on their renewal date</p>
          </div>
        )}
        
        {/* Calculated Values Summary - Show as read-only info */}
        <div className="bg-gray-50 p-3 rounded-lg">
          <Label className="text-sm font-medium text-gray-700">Calculated Point Summary</Label>
          <div className="grid grid-cols-3 gap-4 mt-2 text-sm">
            <div>
              <span className="text-gray-600">Daily:</span>
              <span className="ml-1 font-medium">{formData.dailyPoints}</span>
            </div>
            <div>
              <span className="text-gray-600">Weekly:</span>
              <span className="ml-1 font-medium">{formData.weeklyPoints}</span>
            </div>
            <div>
              <span className="text-gray-600">Monthly:</span>
              <span className="ml-1 font-medium">{formData.monthlyPoints}</span>
            </div>
          </div>
        </div>
      </div>

      <div className="grid grid-cols-3 gap-4">
        <div>
          <Label htmlFor="maxChannels">Max Channels</Label>
          <Input
            id="maxChannels"
            type="number"
            value={formData.maxChannels}
            onChange={(e) => updateFormField("maxChannels", parseInt(e.target.value))}
            required
          />
        </div>
        <div>
          <Label htmlFor="maxVideosPerDay">Max Videos Per Day</Label>
          <Input
            id="maxVideosPerDay"
            type="number"
            value={formData.maxVideosPerDay}
            onChange={(e) => updateFormField("maxVideosPerDay", parseInt(e.target.value))}
            required
          />
        </div>
        <div>
          <Label htmlFor="sortOrder">Sort Order</Label>
          <Input
            id="sortOrder"
            type="number"
            value={formData.sortOrder}
            onChange={(e) => updateFormField("sortOrder", parseInt(e.target.value))}
          />
        </div>
      </div>

      <div className="flex items-center space-x-2">
        <Switch
          id="isActive"
          checked={formData.isActive}
          onCheckedChange={(checked) => updateFormField("isActive", checked)}
        />
        <Label htmlFor="isActive">Package Active</Label>
      </div>

      <div className="flex justify-end space-x-2">
        <Button type="button" variant="outline" onClick={() => selectedPackage ? setIsEditDialogOpen(false) : setIsCreateDialogOpen(false)}>
          Cancel
        </Button>
        <Button type="submit" disabled={createPackageMutation.isPending || updatePackageMutation.isPending}>
          {selectedPackage ? "Update Package" : "Create Package"}
        </Button>
      </div>
    </form>
  );

  return (
    <DashboardLayout>
      <div className="space-y-6">
        <div className="flex justify-between items-center">
          <div>
            <h1 className="text-3xl font-bold">Package Manager</h1>
            <p className="text-gray-600">Manage subscription packages and point allocations</p>
          </div>
          <Dialog open={isCreateDialogOpen} onOpenChange={setIsCreateDialogOpen}>
            <DialogTrigger asChild>
              <Button className="bg-blue-600 hover:bg-blue-700" onClick={resetForm}>
                <Plus className="w-4 h-4 mr-2" />
                Create Package
              </Button>
            </DialogTrigger>
            <DialogContent className="max-w-4xl max-h-[90vh] overflow-y-auto">
              <DialogHeader>
                <DialogTitle>Create New Package</DialogTitle>
                <DialogDescription>
                  Create a new subscription package with custom point allocations and features.
                </DialogDescription>
              </DialogHeader>
              <PackageForm />
            </DialogContent>
          </Dialog>
        </div>

        {/* Package Statistics */}
        <div className="grid grid-cols-1 md:grid-cols-4 gap-6">
          <Card>
            <CardHeader className="pb-2">
              <CardTitle className="text-sm font-medium flex items-center">
                <Package className="w-4 h-4 mr-2" />
                Total Packages
              </CardTitle>
            </CardHeader>
            <CardContent>
              <div className="text-2xl font-bold">{packages?.length || 0}</div>
            </CardContent>
          </Card>
          <Card>
            <CardHeader className="pb-2">
              <CardTitle className="text-sm font-medium flex items-center">
                <DollarSign className="w-4 h-4 mr-2" />
                Active Packages
              </CardTitle>
            </CardHeader>
            <CardContent>
              <div className="text-2xl font-bold">{packages?.filter((p: SubscriptionPackage) => p.isActive).length || 0}</div>
            </CardContent>
          </Card>
          <Card>
            <CardHeader className="pb-2">
              <CardTitle className="text-sm font-medium flex items-center">
                <Users className="w-4 h-4 mr-2" />
                Price Range
              </CardTitle>
            </CardHeader>
            <CardContent>
              <div className="text-2xl font-bold">
                {packages?.length > 0 ? 
                  `$${Math.min(...packages.map((p: SubscriptionPackage) => parseFloat(p.price)))} - $${Math.max(...packages.map((p: SubscriptionPackage) => parseFloat(p.price)))}` 
                  : "$0"
                }
              </div>
            </CardContent>
          </Card>
          <Card>
            <CardHeader className="pb-2">
              <CardTitle className="text-sm font-medium flex items-center">
                <Clock className="w-4 h-4 mr-2" />
                Point Methods
              </CardTitle>
            </CardHeader>
            <CardContent>
              <div className="text-2xl font-bold">
                {packages?.length > 0 ? 
                  [...new Set(packages.map((p: SubscriptionPackage) => p.pointAllocationMethod))].length 
                  : 0
                }
              </div>
            </CardContent>
          </Card>
        </div>

        {/* Packages Table */}
        <Card>
          <CardHeader>
            <CardTitle>Subscription Packages</CardTitle>
          </CardHeader>
          <CardContent>
            {isLoading ? (
              <div className="text-center py-8">Loading packages...</div>
            ) : packages?.length === 0 ? (
              <div className="text-center py-8">
                <Package className="w-12 h-12 text-gray-400 mx-auto mb-4" />
                <h3 className="text-lg font-medium text-gray-900 mb-2">No packages found</h3>
                <p className="text-gray-500 mb-6">Get started by creating your first subscription package.</p>
                <Button onClick={() => { resetForm(); setIsCreateDialogOpen(true); }}>
                  <Plus className="w-4 h-4 mr-2" />
                  Create First Package
                </Button>
              </div>
            ) : (
              <Table>
                <TableHeader>
                  <TableRow>
                    <TableHead>Package</TableHead>
                    <TableHead>Price</TableHead>
                    <TableHead>Points</TableHead>
                    <TableHead>Limits</TableHead>
                    <TableHead>Status</TableHead>
                    <TableHead>Actions</TableHead>
                  </TableRow>
                </TableHeader>
                <TableBody>
                  {packages?.map((pkg: SubscriptionPackage) => (
                    <TableRow key={pkg.id}>
                      <TableCell>
                        <div>
                          <div className="font-medium">{pkg.displayName}</div>
                          <div className="text-sm text-gray-500">{pkg.name}</div>
                          {pkg.description && (
                            <div className="text-xs text-gray-400 mt-1">{pkg.description}</div>
                          )}
                        </div>
                      </TableCell>
                      <TableCell>
                        <div className="font-medium">${pkg.price}</div>
                        <div className="text-sm text-gray-500">per {pkg.billingCycle}</div>
                      </TableCell>
                      <TableCell>
                        <div className="space-y-1">
                          <div className="text-sm">
                            <span className="font-medium">{pkg.dailyPoints}</span> daily
                          </div>
                          <div className="text-sm text-gray-500">
                            {pkg.weeklyPoints} weekly • {pkg.monthlyPoints} monthly
                          </div>
                          <Badge variant="outline" className="text-xs">
                            {pkg.pointAllocationMethod}
                          </Badge>
                        </div>
                      </TableCell>
                      <TableCell>
                        <div className="space-y-1">
                          <div className="text-sm">
                            <span className="font-medium">{pkg.maxChannels}</span> channels
                          </div>
                          <div className="text-sm text-gray-500">
                            {pkg.maxVideosPerDay} videos/day
                          </div>
                        </div>
                      </TableCell>
                      <TableCell>
                        <Badge variant={pkg.isActive ? "default" : "secondary"}>
                          {pkg.isActive ? "Active" : "Inactive"}
                        </Badge>
                      </TableCell>
                      <TableCell>
                        <div className="flex space-x-2">
                          <Button
                            variant="outline"
                            size="sm"
                            onClick={() => handleEdit(pkg)}
                          >
                            <Edit className="w-3 h-3" />
                          </Button>
                          <Button
                            variant="outline"
                            size="sm"
                            onClick={() => handleDelete(pkg.id)}
                            disabled={deletePackageMutation.isPending}
                          >
                            <Trash2 className="w-3 h-3" />
                          </Button>
                        </div>
                      </TableCell>
                    </TableRow>
                  ))}
                </TableBody>
              </Table>
            )}
          </CardContent>
        </Card>

        {/* Edit Dialog */}
        <Dialog open={isEditDialogOpen} onOpenChange={setIsEditDialogOpen}>
          <DialogContent className="max-w-4xl max-h-[90vh] overflow-y-auto">
            <DialogHeader>
              <DialogTitle>Edit Package</DialogTitle>
              <DialogDescription>
                Update package details, point allocations, and features.
              </DialogDescription>
            </DialogHeader>
            <PackageForm />
          </DialogContent>
        </Dialog>
      </div>
    </DashboardLayout>
  );
}