import { useState, useEffect } from "react";
import { useQuery, useMutation, useQueryClient } from "@tanstack/react-query";
import { apiRequest } from "@/lib/queryClient";
import { Button } from "@/components/ui/button";
import { Card, CardContent } from "@/components/ui/card";
import { Textarea } from "@/components/ui/textarea";
import { Badge } from "@/components/ui/badge";
import { Avatar, AvatarFallback, AvatarImage } from "@/components/ui/avatar";
import { Dialog, DialogContent, DialogHeader, DialogTitle, DialogTrigger, DialogDescription } from "@/components/ui/dialog";
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select";
import { Separator } from "@/components/ui/separator";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import { useToast } from "@/hooks/use-toast";
import { formatDistanceToNow } from "date-fns";
import { 
  ArrowUp, 
  ArrowDown, 
  MessageSquare, 
  Share2, 
  Bookmark, 
  ExternalLink,
  Eye,
  EyeOff,
  Flag,
  MoreHorizontal,
  Award,
  Pin,
  Lock,
  Archive,
  Zap,
  RefreshCw,
  Filter,
  Reply,
  ChevronUp,
  ChevronDown,
  Shuffle,
  TrendingUp
} from "lucide-react";
import DashboardLayout from "@/components/dashboard-layout";

interface RedditComment {
  id: string;
  fullname: string;
  author: string;
  body: string;
  bodyHtml: string;
  score: number;
  upvoteRatio?: number;
  created: string;
  createdUtc: number;
  permalink: string;
  parentId: string;
  linkId: string;
  subreddit: string;
  depth: number;
  collapsed: boolean;
  distinguished?: string;
  stickied: boolean;
  gilded: number;
  controversiality: number;
  edited: boolean | number;
  archived: boolean;
  scoreHidden: boolean;
  likes?: boolean | null;
  saved: boolean;
  replies: RedditComment[];
  isSubmitter: boolean;
  canGild: boolean;
  canMod: boolean;
}

interface RedditPost {
  id: string;
  fullname: string;
  title: string;
  author: string;
  subreddit: string;
  subredditNamePrefixed: string;
  score: number;
  upvoteRatio: number;
  numComments: number;
  created: string;
  createdUtc: number;
  selftext: string;
  selftextHtml: string;
  url: string;
  permalink: string;
  domain: string;
  isNsfw: boolean;
  isSelf: boolean;
  isVideo: boolean;
  stickied: boolean;
  locked: boolean;
  archived: boolean;
  spoiler: boolean;
  linkFlairText?: string;
  linkFlairBackgroundColor?: string;
  linkFlairTextColor?: string;
  distinguished?: string;
  gilded: number;
  awards: any[];
  media?: {
    type: 'image' | 'video';
    url: string;
    width?: number;
    height?: number;
    thumbnail?: string;
    duration?: number;
  };
  thumbnail?: string;
  likes?: boolean | null;
  saved: boolean;
  hidden: boolean;
  postHint?: string;
}

interface FeedData {
  posts: RedditPost[];
  pagination: {
    after: string | null;
    before: string | null;
    hasMore: boolean;
  };
}

// Component to display orders for a specific post
const PostOrdersDisplay = ({ postId }: { postId: string }) => {
  const { data: ordersData } = useQuery({
    queryKey: ['/api/redupvotes/orders', postId],
    queryFn: async () => {
      const response = await fetch(`/api/redupvotes/orders/${postId}`);
      if (!response.ok) throw new Error('Failed to fetch orders');
      return response.json();
    },
    refetchInterval: 30000, // Refresh every 30 seconds for status updates
  });

  if (!ordersData?.orders || ordersData.orders.length === 0) {
    return null;
  }

  return (
    <div className="mt-3 pt-3 border-t border-gray-200 dark:border-gray-700">
      <h4 className="text-sm font-medium text-gray-900 dark:text-gray-100 mb-2 flex items-center">
        <TrendingUp className="h-4 w-4 mr-1 text-yellow-600" />
        Gold Upvotes Orders
      </h4>
      <div className="space-y-2">
        {ordersData.orders.map((order: any) => (
          <div key={order.id} className="bg-yellow-50 dark:bg-yellow-900/20 rounded-lg p-3 text-sm">
            <div className="flex justify-between items-start">
              <div>
                <p className="font-medium text-yellow-800 dark:text-yellow-200">
                  Order #{order.redupvotesOrderId}
                </p>
                <p className="text-yellow-700 dark:text-yellow-300">
                  {order.quantity} upvotes • Status: {order.status}
                </p>
                {order.charge && (
                  <p className="text-yellow-600 dark:text-yellow-400">
                    Cost: ${order.charge} {order.currency}
                  </p>
                )}
                {order.remains !== null && (
                  <p className="text-yellow-600 dark:text-yellow-400">
                    Remaining: {order.remains}
                  </p>
                )}
              </div>
              <Badge 
                variant={order.status === 'completed' ? 'default' : 
                        order.status === 'in_progress' ? 'secondary' : 'outline'}
                className="text-xs"
              >
                {order.status}
              </Badge>
            </div>
            <p className="text-xs text-yellow-600 dark:text-yellow-500 mt-1">
              Created: {new Date(order.createdAt).toLocaleDateString()}
            </p>
          </div>
        ))}
      </div>
    </div>
  );
};

export default function RedditHomeFeed() {
  const [sortType, setSortType] = useState<'hot' | 'new' | 'rising' | 'top'>('hot');
  const [selectedPost, setSelectedPost] = useState<RedditPost | null>(null);
  const [showCommentsDialog, setShowCommentsDialog] = useState(false);
  const [commentText, setCommentText] = useState('');
  const [replyingTo, setReplyingTo] = useState<string | null>(null);
  const [commentsSort, setCommentsSort] = useState<'top' | 'new' | 'controversial' | 'old' | 'random'>('top');
  const [collapsedComments, setCollapsedComments] = useState<Set<string>>(new Set());
  const [upvoteQuantity, setUpvoteQuantity] = useState<string>('10');
  const [showUpvoteDialog, setShowUpvoteDialog] = useState(false);
  const [orderingPost, setOrderingPost] = useState<RedditPost | null>(null);
  const { toast } = useToast();
  const queryClient = useQueryClient();

  // Fetch Reddit home feed
  const { 
    data: feedData, 
    isLoading, 
    error, 
    refetch 
  } = useQuery<FeedData>({
    queryKey: ['/api/reddit/home-feed', sortType],
    queryFn: async () => {
      const response = await fetch(`/api/reddit/home-feed?sort=${sortType}&limit=25`);
      if (!response.ok) {
        throw new Error(`Failed to fetch home feed: ${response.status}`);
      }
      return response.json();
    },
    refetchInterval: 300000, // Refresh every 5 minutes
    retry: 2,
  });

  // Fetch comments for selected post
  const { 
    data: commentsData, 
    isLoading: commentsLoading,
    refetch: refetchComments 
  } = useQuery({
    queryKey: ['/api/reddit/comments', selectedPost?.id, commentsSort],
    queryFn: async () => {
      if (!selectedPost?.id) return null;
      const response = await fetch(`/api/reddit/comments/${selectedPost.id}?sort=${commentsSort}`);
      if (!response.ok) {
        throw new Error(`Failed to fetch comments: ${response.status}`);
      }
      return response.json();
    },
    enabled: !!selectedPost && showCommentsDialog,
    retry: 2,
  });

  // Vote mutation for posts and comments
  const voteMutation = useMutation({
    mutationFn: async ({ fullname, direction }: { fullname: string; direction: number }) => {
      return apiRequest('/api/reddit/vote', 'POST', { fullname, direction });
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ['/api/reddit/home-feed'] });
      queryClient.invalidateQueries({ queryKey: ['/api/reddit/comments'] });
      toast({
        title: "Vote submitted",
        description: "Your vote has been recorded.",
      });
    },
    onError: (error: any) => {
      toast({
        title: "Vote failed",
        description: error.message || "Failed to submit vote. Please try again.",
        variant: "destructive",
      });
    },
  });

  // Comment mutation
  const commentMutation = useMutation({
    mutationFn: async ({ parentId, text }: { parentId: string; text: string }) => {
      return apiRequest('/api/reddit/comment', 'POST', { parentId, text });
    },
    onSuccess: () => {
      setCommentText('');
      setReplyingTo(null);
      toast({
        title: "Comment posted",
        description: "Your comment has been posted successfully.",
      });
    },
    onError: (error: any) => {
      toast({
        title: "Comment failed",
        description: error.message || "Failed to post comment. Please try again.",
        variant: "destructive",
      });
    },
  });

  // Redupvotes order mutation
  const redupvotesMutation = useMutation({
    mutationFn: async ({ redditPostId, redditPostUrl, quantity }: { 
      redditPostId: string; 
      redditPostUrl: string; 
      quantity: number; 
    }) => {
      return apiRequest('/api/redupvotes/order', 'POST', { 
        redditPostId, 
        redditPostUrl, 
        quantity 
      });
    },
    onSuccess: (data) => {
      setShowUpvoteDialog(false);
      setOrderingPost(null);
      setUpvoteQuantity('10');
      queryClient.invalidateQueries({ queryKey: ['/api/redupvotes/orders'] });
      queryClient.invalidateQueries({ queryKey: ['/api/points'] }); // Refresh points balance
      toast({
        title: "Order placed successfully",
        description: `Order ${data.redupvotesOrderId} placed for ${upvoteQuantity} upvotes. 2 points deducted. Remaining: ${data.remainingPoints} points.`,
      });
    },
    onError: (error: any) => {
      const errorMessage = error.message || "Failed to place upvote order";
      const isInsufficientPoints = errorMessage.includes("Insufficient points") || errorMessage.includes("need 2 points");
      
      toast({
        title: isInsufficientPoints ? "Insufficient Points" : "Order failed",
        description: isInsufficientPoints 
          ? "You need at least 2 points to place a Gold Upvotes order. Please add more points to your account."
          : errorMessage,
        variant: "destructive",
      });
    },
  });

  // Fetch existing orders for a post
  const usePostOrders = (postId: string) => {
    return useQuery({
      queryKey: ['/api/redupvotes/orders', postId],
      queryFn: async () => {
        const response = await fetch(`/api/redupvotes/orders/${postId}`);
        if (!response.ok) throw new Error('Failed to fetch orders');
        return response.json();
      },
      enabled: !!postId,
    });
  };

  const handleVote = (fullname: string, direction: number) => {
    voteMutation.mutate({ fullname, direction });
  };

  const handleCommentVote = (comment: RedditComment, direction: number) => {
    voteMutation.mutate({ fullname: comment.fullname, direction });
  };

  const handleComment = (parentId: string) => {
    if (!commentText.trim()) {
      toast({
        title: "Empty comment",
        description: "Please enter a comment before submitting.",
        variant: "destructive",
      });
      return;
    }
    commentMutation.mutate({ parentId, text: commentText });
  };

  const toggleCommentCollapse = (commentId: string) => {
    setCollapsedComments(prev => {
      const newSet = new Set(prev);
      if (newSet.has(commentId)) {
        newSet.delete(commentId);
      } else {
        newSet.add(commentId);
      }
      return newSet;
    });
  };

  const handleReply = (commentId: string) => {
    setReplyingTo(commentId);
  };

  const renderComment = (comment: RedditComment, depth: number = 0): React.ReactNode => {
    const isCollapsed = collapsedComments.has(comment.id);
    const indentClass = depth > 0 ? `ml-${Math.min(depth * 4, 16)}` : '';

    return (
      <div key={comment.id} className={`${indentClass} border-l-2 border-gray-200 dark:border-gray-700 pl-3 mb-3`}>
        {/* Comment header */}
        <div className="flex items-center space-x-2 mb-2">
          <Avatar className="h-5 w-5">
            <AvatarFallback className="text-xs">
              {comment.author.slice(0, 2).toUpperCase()}
            </AvatarFallback>
          </Avatar>
          <span className="text-sm font-medium text-blue-600 dark:text-blue-400">
            u/{comment.author}
          </span>
          {comment.isSubmitter && (
            <Badge variant="secondary" className="text-xs">OP</Badge>
          )}
          {comment.distinguished && (
            <Badge variant="outline" className="text-xs bg-green-100 text-green-800">
              {comment.distinguished}
            </Badge>
          )}
          <span className="text-xs text-gray-500">•</span>
          <span className="text-xs text-gray-500">
            {formatTimeAgo(comment.createdUtc)}
          </span>
          {comment.edited && (
            <>
              <span className="text-xs text-gray-500">•</span>
              <span className="text-xs text-gray-500">edited</span>
            </>
          )}
        </div>

        {/* Comment body */}
        <div className="mb-2">
          {!isCollapsed && (
            <div className="text-sm text-gray-700 dark:text-gray-300 mb-2">
              {comment.body}
            </div>
          )}
          
          {/* Comment actions */}
          <div className="flex items-center space-x-3 text-xs">
            <Button
              variant="ghost"
              size="sm"
              onClick={() => toggleCommentCollapse(comment.id)}
              className="h-6 px-1 text-gray-500 hover:text-gray-700"
            >
              {isCollapsed ? <ChevronDown className="h-3 w-3" /> : <ChevronUp className="h-3 w-3" />}
              {isCollapsed ? 'Expand' : 'Collapse'}
            </Button>
            
            {!isCollapsed && (
              <>
                <div className="flex items-center space-x-1">
                  <Button 
                    variant="ghost" 
                    size="sm" 
                    onClick={() => handleCommentVote(comment, comment.likes === true ? 0 : 1)}
                    className={`h-6 px-1 ${comment.likes === true ? 'text-orange-500' : 'text-gray-500 hover:text-orange-500'}`}
                    disabled={voteMutation.isPending}
                  >
                    <ArrowUp className="h-3 w-3" />
                  </Button>
                  <span className="text-gray-600 dark:text-gray-400">
                    {comment.scoreHidden ? '•' : formatScore(comment.score)}
                  </span>
                  <Button 
                    variant="ghost" 
                    size="sm" 
                    onClick={() => handleCommentVote(comment, comment.likes === false ? 0 : -1)}
                    className={`h-6 px-1 ${comment.likes === false ? 'text-blue-500' : 'text-gray-500 hover:text-blue-500'}`}
                    disabled={voteMutation.isPending}
                  >
                    <ArrowDown className="h-3 w-3" />
                  </Button>
                </div>
                
                <Button
                  variant="ghost"
                  size="sm"
                  onClick={() => handleReply(comment.fullname)}
                  className="h-6 px-1 text-gray-500 hover:text-gray-700"
                >
                  <Reply className="h-3 w-3 mr-1" />
                  Reply
                </Button>

                <Button variant="ghost" size="sm" className="h-6 px-1 text-gray-500 hover:text-gray-700">
                  <Share2 className="h-3 w-3 mr-1" />
                  Share
                </Button>

                {comment.gilded > 0 && (
                  <div className="flex items-center">
                    <Award className="h-3 w-3 text-yellow-500" />
                    <span className="ml-1">{comment.gilded}</span>
                  </div>
                )}
              </>
            )}
          </div>

          {/* Reply input */}
          {replyingTo === comment.fullname && (
            <div className="mt-3 space-y-2">
              <Textarea
                value={commentText}
                onChange={(e) => setCommentText(e.target.value)}
                placeholder={`Reply to u/${comment.author}...`}
                className="min-h-[80px] text-sm"
              />
              <div className="flex space-x-2">
                <Button
                  size="sm"
                  onClick={() => handleComment(comment.fullname)}
                  disabled={commentMutation.isPending || !commentText.trim()}
                >
                  {commentMutation.isPending ? 'Posting...' : 'Reply'}
                </Button>
                <Button
                  size="sm"
                  variant="outline"
                  onClick={() => setReplyingTo(null)}
                >
                  Cancel
                </Button>
              </div>
            </div>
          )}
        </div>

        {/* Nested replies */}
        {!isCollapsed && comment.replies && comment.replies.length > 0 && (
          <div className="mt-2">
            {comment.replies.map(reply => renderComment(reply, depth + 1))}
          </div>
        )}
      </div>
    );
  };

  const formatTimeAgo = (timestamp: number) => {
    if (!timestamp || isNaN(timestamp)) {
      return 'Unknown time';
    }
    
    const date = new Date(timestamp * 1000);
    if (isNaN(date.getTime())) {
      return 'Unknown time';
    }
    
    return formatDistanceToNow(date, { addSuffix: true });
  };

  const formatScore = (score: number) => {
    if (score >= 1000) {
      return `${(score / 1000).toFixed(1)}k`;
    }
    return score.toString();
  };

  const getVoteColor = (post: RedditPost, type: 'up' | 'down') => {
    if (post.likes === true && type === 'up') return 'text-orange-500';
    if (post.likes === false && type === 'down') return 'text-blue-500';
    return 'text-gray-500 hover:text-gray-700';
  };

  if (isLoading) {
    return (
      <DashboardLayout>
        <div className="min-h-screen bg-gray-50 dark:bg-gray-900">
          <div className="max-w-4xl mx-auto px-4 py-6">
            <div className="flex items-center justify-center h-64">
              <div className="flex items-center space-x-2">
                <RefreshCw className="h-5 w-5 animate-spin" />
                <span className="text-lg">Loading your Reddit home feed...</span>
              </div>
            </div>
          </div>
        </div>
      </DashboardLayout>
    );
  }

  if (error) {
    return (
      <DashboardLayout>
        <div className="min-h-screen bg-gray-50 dark:bg-gray-900">
          <div className="max-w-4xl mx-auto px-4 py-6">
            <Card className="border-red-200 bg-red-50 dark:bg-red-900/10">
              <CardContent className="pt-6">
                <div className="text-center">
                  <Flag className="h-12 w-12 mx-auto text-red-500 mb-4" />
                  <h3 className="text-lg font-semibold text-red-800 dark:text-red-200 mb-2">
                    Reddit Connection Required
                  </h3>
                  <p className="text-red-600 dark:text-red-300 mb-4">
                    {error.message || "Failed to load Reddit home feed"}
                  </p>
                  <Button onClick={() => refetch()} className="mr-2">
                    <RefreshCw className="h-4 w-4 mr-2" />
                    Retry
                  </Button>
                  <Button variant="outline" onClick={() => window.location.href = '/api/reddit/auth'}>
                    Connect Reddit Account
                  </Button>
                </div>
              </CardContent>
            </Card>
          </div>
        </div>
      </DashboardLayout>
    );
  }



  return (
    <DashboardLayout>
      <div className="min-h-screen bg-gray-50 dark:bg-gray-900">
        <div className="max-w-4xl mx-auto px-4 py-6">
          {/* Header with sort options */}
          <div className="mb-6">
            <div className="flex items-center justify-between mb-4">
              <h1 className="text-2xl font-bold flex items-center">
                <Zap className="h-6 w-6 mr-2 text-orange-500" />
                Reddit Home Feed
              </h1>
              <Button onClick={() => refetch()} variant="outline" size="sm">
                <RefreshCw className="h-4 w-4 mr-2" />
                Refresh
              </Button>
            </div>
            
            <div className="flex items-center space-x-2">
              <Filter className="h-4 w-4 text-gray-500" />
              <Select value={sortType} onValueChange={(value: any) => setSortType(value)}>
                <SelectTrigger className="w-32">
                  <SelectValue />
                </SelectTrigger>
                <SelectContent>
                  <SelectItem value="hot">🔥 Hot</SelectItem>
                  <SelectItem value="new">🆕 New</SelectItem>
                  <SelectItem value="rising">📈 Rising</SelectItem>
                  <SelectItem value="top">⭐ Top</SelectItem>
                </SelectContent>
              </Select>
            </div>
          </div>

          {/* Feed */}
          <div className="space-y-4">
            {feedData?.posts.map((post) => (
              <Card key={post.id} className="hover:shadow-lg transition-shadow">
                <CardContent className="p-0">
                  <div className="flex">
                    {/* Vote buttons */}
                    <div className="flex flex-col items-center p-3 bg-gray-50 dark:bg-gray-800">
                      <Button
                        variant="ghost"
                        size="sm"
                        onClick={() => handleVote(post.fullname, post.likes === true ? 0 : 1)}
                        className={`p-1 ${getVoteColor(post, 'up')}`}
                        disabled={voteMutation.isPending}
                      >
                        <ArrowUp className="h-4 w-4" />
                      </Button>
                      <span className="text-sm font-medium py-1">
                        {formatScore(post.score)}
                      </span>
                      <Button
                        variant="ghost"
                        size="sm"
                        onClick={() => handleVote(post.fullname, post.likes === false ? 0 : -1)}
                        className={`p-1 ${getVoteColor(post, 'down')}`}
                        disabled={voteMutation.isPending}
                      >
                        <ArrowDown className="h-4 w-4" />
                      </Button>
                    </div>

                    {/* Post content */}
                    <div className="flex-1 p-4">
                      {/* Header */}
                      <div className="flex items-center space-x-2 mb-2">
                        <Avatar className="h-6 w-6">
                          <AvatarImage src={`https://www.reddit.com/user/${post.author}/about.json`} />
                          <AvatarFallback className="text-xs">
                            {post.author.slice(0, 2).toUpperCase()}
                          </AvatarFallback>
                        </Avatar>
                        <span className="text-sm font-medium text-gray-900 dark:text-gray-100">
                          {post.subredditNamePrefixed}
                        </span>
                        <span className="text-sm text-gray-500">•</span>
                        <span className="text-sm text-gray-500">
                          Posted by u/{post.author}
                        </span>
                        <span className="text-sm text-gray-500">•</span>
                        <span className="text-sm text-gray-500">
                          {formatTimeAgo(post.createdUtc)}
                        </span>
                        
                        {/* Post badges */}
                        {post.stickied && <Pin className="h-4 w-4 text-green-500" />}
                        {post.locked && <Lock className="h-4 w-4 text-yellow-500" />}
                        {post.archived && <Archive className="h-4 w-4 text-gray-500" />}
                        {post.isNsfw && <Badge variant="destructive" className="text-xs">NSFW</Badge>}
                        {post.spoiler && <Badge variant="outline" className="text-xs">Spoiler</Badge>}
                        {post.gilded > 0 && (
                          <div className="flex items-center">
                            <Award className="h-4 w-4 text-yellow-500" />
                            <span className="text-xs ml-1">{post.gilded}</span>
                          </div>
                        )}
                      </div>

                      {/* Title */}
                      <h3 className="text-lg font-semibold mb-2 text-gray-900 dark:text-gray-100">
                        {post.title}
                        {post.linkFlairText && (
                          <Badge 
                            variant="outline" 
                            className="ml-2 text-xs"
                            style={{
                              backgroundColor: post.linkFlairBackgroundColor || undefined,
                              color: post.linkFlairTextColor || undefined
                            }}
                          >
                            {post.linkFlairText}
                          </Badge>
                        )}
                      </h3>

                      {/* Self text */}
                      {post.selftext && (
                        <div className="mb-3 text-gray-700 dark:text-gray-300">
                          <p className="line-clamp-3">
                            {post.selftext.length > 300 
                              ? `${post.selftext.slice(0, 300)}...` 
                              : post.selftext
                            }
                          </p>
                        </div>
                      )}

                      {/* Media */}
                      {post.media && (
                        <div className="mb-3">
                          {post.media.type === 'image' && (
                            <img
                              src={post.media.url}
                              alt={post.title}
                              className="max-w-full h-auto rounded-lg shadow-md cursor-pointer"
                              onClick={() => window.open(post.media?.url, '_blank')}
                              onError={(e) => {
                                e.currentTarget.style.display = 'none';
                              }}
                            />
                          )}
                          {post.media.type === 'video' && (
                            <video
                              src={post.media.url}
                              controls
                              className="max-w-full h-auto rounded-lg shadow-md"
                              poster={post.media.thumbnail}
                            />
                          )}
                        </div>
                      )}

                      {/* External link preview */}
                      {!post.isSelf && post.url && !post.media && post.domain !== 'reddit.com' && (
                        <div className="mb-3 p-3 border rounded-lg bg-gray-50 dark:bg-gray-800">
                          <div className="flex items-center space-x-2">
                            <ExternalLink className="h-4 w-4 text-blue-500" />
                            <a
                              href={post.url}
                              target="_blank"
                              rel="noopener noreferrer"
                              className="text-blue-600 hover:underline text-sm truncate"
                            >
                              {post.url.replace(/^https?:\/\//, '').substring(0, 50)}...
                            </a>
                          </div>
                          <p className="text-xs text-gray-600 dark:text-gray-400 mt-1">
                            {post.domain}
                          </p>
                        </div>
                      )}

                      {/* Action buttons */}
                      <div className="flex items-center space-x-4 pt-2 border-t">
                        <Dialog>
                          <DialogTrigger asChild>
                            <Button 
                              variant="ghost" 
                              size="sm" 
                              className="text-gray-500 hover:text-gray-700"
                              onClick={() => {
                                setSelectedPost(post);
                                setShowCommentsDialog(true);
                                setReplyingTo(null);
                                setCommentText('');
                              }}
                            >
                              <MessageSquare className="h-4 w-4 mr-1" />
                              {post.numComments} Comments
                            </Button>
                          </DialogTrigger>
                          <DialogContent className="max-w-4xl max-h-[90vh] overflow-hidden flex flex-col">
                            <DialogHeader>
                              <DialogTitle>Comments - {post.title}</DialogTitle>
                              <DialogDescription>
                                {post.subredditNamePrefixed} • u/{post.author} • {formatTimeAgo(post.createdUtc)}
                              </DialogDescription>
                            </DialogHeader>
                            
                            <div className="flex-1 overflow-hidden flex flex-col">
                              {/* Sort options */}
                              <div className="flex items-center justify-between mb-4 pb-3 border-b">
                                <div className="flex items-center space-x-2">
                                  <span className="text-sm font-medium">Sort by:</span>
                                  <Select value={commentsSort} onValueChange={(value: any) => setCommentsSort(value)}>
                                    <SelectTrigger className="w-40">
                                      <SelectValue />
                                    </SelectTrigger>
                                    <SelectContent>
                                      <SelectItem value="top">🔥 Hot (Top)</SelectItem>
                                      <SelectItem value="new">🆕 Newest First</SelectItem>
                                      <SelectItem value="random">🎲 Random</SelectItem>
                                      <SelectItem value="controversial">⚡ Controversial</SelectItem>
                                      <SelectItem value="old">⏰ Oldest</SelectItem>
                                    </SelectContent>
                                  </Select>
                                </div>
                                <Button 
                                  variant="outline" 
                                  size="sm"
                                  onClick={() => refetchComments()}
                                  disabled={commentsLoading}
                                >
                                  <RefreshCw className={`h-4 w-4 mr-1 ${commentsLoading ? 'animate-spin' : ''}`} />
                                  Refresh
                                </Button>
                              </div>

                              {/* Add new comment */}
                              <div className="mb-4 p-3 bg-gray-50 dark:bg-gray-800 rounded-lg">
                                <Textarea
                                  value={commentText}
                                  onChange={(e) => setCommentText(e.target.value)}
                                  placeholder="What are your thoughts?"
                                  className="min-h-[80px] mb-2"
                                />
                                <div className="flex justify-end">
                                  <Button
                                    onClick={() => handleComment(post.fullname)}
                                    disabled={commentMutation.isPending || !commentText.trim()}
                                    size="sm"
                                  >
                                    {commentMutation.isPending ? 'Posting...' : 'Add Comment'}
                                  </Button>
                                </div>
                              </div>

                              {/* Comments list */}
                              <div className="flex-1 overflow-y-auto">
                                {commentsLoading ? (
                                  <div className="flex items-center justify-center h-32">
                                    <RefreshCw className="h-5 w-5 animate-spin mr-2" />
                                    <span>Loading comments...</span>
                                  </div>
                                ) : commentsData?.comments?.length > 0 ? (
                                  <div className="space-y-3">
                                    {commentsData.comments.map((comment: RedditComment) => renderComment(comment))}
                                  </div>
                                ) : (
                                  <div className="text-center py-8 text-gray-500">
                                    <MessageSquare className="h-8 w-8 mx-auto mb-2 opacity-50" />
                                    <p>No comments yet. Be the first to comment!</p>
                                  </div>
                                )}
                              </div>
                            </div>
                          </DialogContent>
                        </Dialog>

                        <Button variant="ghost" size="sm" className="text-gray-500 hover:text-gray-700">
                          <Share2 className="h-4 w-4 mr-1" />
                          Share
                        </Button>

                        <Button variant="ghost" size="sm" className="text-gray-500 hover:text-gray-700">
                          <Bookmark className={`h-4 w-4 mr-1 ${post.saved ? 'fill-current text-yellow-500' : ''}`} />
                          Save
                        </Button>

                        {/* Gold Upvotes Button */}
                        <Dialog open={showUpvoteDialog} onOpenChange={setShowUpvoteDialog}>
                          <DialogTrigger asChild>
                            <Button 
                              variant="ghost" 
                              size="sm" 
                              className="text-yellow-600 hover:text-yellow-700 font-medium"
                              onClick={() => {
                                setOrderingPost(post);
                                setShowUpvoteDialog(true);
                              }}
                            >
                              <TrendingUp className="h-4 w-4 mr-1" />
                              Gold Upvotes
                            </Button>
                          </DialogTrigger>
                          <DialogContent className="max-w-md">
                            <DialogHeader>
                              <DialogTitle className="flex items-center">
                                <TrendingUp className="h-5 w-5 mr-2 text-yellow-600" />
                                Purchase Reddit Upvotes
                              </DialogTitle>
                              <DialogDescription>
                                Boost your post's visibility with high-quality Reddit upvotes
                              </DialogDescription>
                            </DialogHeader>
                            
                            {orderingPost && (
                              <div className="space-y-4">
                                <div className="p-3 bg-gray-50 dark:bg-gray-800 rounded-lg">
                                  <h4 className="font-medium text-sm line-clamp-2">
                                    {orderingPost.title}
                                  </h4>
                                  <p className="text-xs text-gray-500 mt-1">
                                    {orderingPost.subredditNamePrefixed} • {orderingPost.score} current upvotes
                                  </p>
                                </div>

                                <div className="space-y-2">
                                  <Label htmlFor="quantity">Number of Upvotes</Label>
                                  <Input
                                    id="quantity"
                                    type="number"
                                    min="1"
                                    max="1000"
                                    value={upvoteQuantity}
                                    onChange={(e) => setUpvoteQuantity(e.target.value)}
                                    placeholder="Enter quantity (1-1000)"
                                  />
                                  <div className="bg-yellow-50 dark:bg-yellow-900/20 border border-yellow-200 dark:border-yellow-800 rounded-lg p-3">
                                    <p className="text-xs text-yellow-800 dark:text-yellow-200 font-medium">
                                      💰 Cost: 2 points per order (regardless of quantity)
                                    </p>
                                    <p className="text-xs text-yellow-600 dark:text-yellow-400 mt-1">
                                      Service will deliver upvotes gradually to appear natural
                                    </p>
                                  </div>
                                </div>

                                <div className="flex justify-end space-x-2">
                                  <Button 
                                    variant="outline" 
                                    onClick={() => {
                                      setShowUpvoteDialog(false);
                                      setOrderingPost(null);
                                    }}
                                  >
                                    Cancel
                                  </Button>
                                  <Button
                                    onClick={() => {
                                      if (orderingPost && upvoteQuantity) {
                                        const postUrl = `https://www.reddit.com${orderingPost.permalink}`;
                                        redupvotesMutation.mutate({
                                          redditPostId: orderingPost.id,
                                          redditPostUrl: postUrl,
                                          quantity: parseInt(upvoteQuantity)
                                        });
                                      }
                                    }}
                                    disabled={redupvotesMutation.isPending || !upvoteQuantity || parseInt(upvoteQuantity) < 1}
                                    className="bg-yellow-600 hover:bg-yellow-700"
                                  >
                                    {redupvotesMutation.isPending ? 'Placing Order...' : 'Place Order'}
                                  </Button>
                                </div>
                              </div>
                            )}
                          </DialogContent>
                        </Dialog>

                        <Button variant="ghost" size="sm" className="text-gray-500 hover:text-gray-700">
                          <MoreHorizontal className="h-4 w-4" />
                        </Button>
                      </div>

                      {/* Display existing orders for this post */}
                      <PostOrdersDisplay postId={post.id} />
                    </div>
                  </div>
                </CardContent>
              </Card>
            ))}
          </div>

          {/* Load more */}
          {feedData?.pagination.hasMore && (
            <div className="text-center mt-6">
              <Button variant="outline">
                Load More Posts
              </Button>
            </div>
          )}

          {/* Empty state */}
          {feedData?.posts.length === 0 && (
            <Card>
              <CardContent className="pt-6">
                <div className="text-center">
                  <MessageSquare className="h-12 w-12 mx-auto text-gray-400 mb-4" />
                  <h3 className="text-lg font-semibold text-gray-900 dark:text-gray-100 mb-2">
                    No posts found
                  </h3>
                  <p className="text-gray-500 mb-4">
                    Your Reddit home feed is empty. Try subscribing to some subreddits!
                  </p>
                  <Button onClick={() => refetch()}>
                    <RefreshCw className="h-4 w-4 mr-2" />
                    Refresh Feed
                  </Button>
                </div>
              </CardContent>
            </Card>
          )}
        </div>
      </div>
    </DashboardLayout>
  );
}