import { useState, useEffect } from 'react';
import { useQuery, useMutation, useQueryClient } from '@tanstack/react-query';
import { Button } from '@/components/ui/button';
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from '@/components/ui/card';
import { Badge } from '@/components/ui/badge';
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from '@/components/ui/select';
import { Tabs, TabsContent, TabsList, TabsTrigger } from '@/components/ui/tabs';
import { Alert, AlertDescription } from '@/components/ui/alert';
import { Textarea } from '@/components/ui/textarea';
import { Label } from '@/components/ui/label';
import { Dialog, DialogContent, DialogDescription, DialogHeader, DialogTitle, DialogTrigger } from '@/components/ui/dialog';
import { DropdownMenu, DropdownMenuContent, DropdownMenuItem, DropdownMenuTrigger } from '@/components/ui/dropdown-menu';
import { useToast } from '@/hooks/use-toast';
import { apiRequest } from '@/lib/queryClient';
import { DashboardLayout } from '@/components/dashboard-layout';
import { 
  Shield, 
  Bot, 
  AlertTriangle, 
  CheckCircle, 
  XCircle, 
  Eye, 
  Clock, 
  FileText, 
  MessageSquare, 
  ExternalLink,
  User,
  RefreshCw,
  Zap,
  ChevronDown,
  ChevronUp,
  Image as ImageIcon,
  History
} from 'lucide-react';

interface ModqueueItem {
  id: string;
  fullname: string;
  type: 'submission' | 'comment';
  author: string;
  title?: string;
  content: string;
  url?: string;
  permalink: string;
  score: number;
  numReports: number;
  userReports: any[];
  modReports: any[];
  createdUtc: number;
  subreddit: string;
}

interface AIAnalysis {
  violationsFound: Array<{
    ruleIndex: number;
    ruleName: string;
    violation: string;
    severity: 'low' | 'medium' | 'high';
    confidence: number;
  }>;
  suggestedAction: 'approve' | 'remove' | 'spam' | 'manual_review';
  reasoning: string;
  confidenceScore: number;
  severity: 'low' | 'medium' | 'high';
  removalReason?: string;
  requiresHumanReview: boolean;
  additionalActions: string[];
}

interface SubredditRules {
  rules: Array<{
    kind: string;
    shortName: string;
    violationReason: string;
    description: string;
    priority: number;
  }>;
  description?: string;
  submissionType?: string;
  subredditType?: string;
}

interface ModlogEntry {
  id: string;
  action: string;
  moderator: string;
  description: string;
  details: string;
  createdUtc: number;
  created: string;
  targetFullname: string;
  targetPermalink: string;
  targetTitle: string;
  targetAuthor: string;
  srName: string;
}

export default function RedditModeration() {
  const [selectedSubreddit, setSelectedSubreddit] = useState<string>('');
  const [selectedItem, setSelectedItem] = useState<ModqueueItem | null>(null);
  const [analysis, setAnalysis] = useState<AIAnalysis | null>(null);
  const [analysisResults, setAnalysisResults] = useState<{[key: string]: AIAnalysis}>({});
  const [actionReason, setActionReason] = useState('');
  const [isAnalyzing, setIsAnalyzing] = useState(false);
  const [isRefreshing, setIsRefreshing] = useState(false);
  const [showActedUponItems, setShowActedUponItems] = useState(false);
  const [errorLog, setErrorLog] = useState<string[]>([]);
  const [expandedItems, setExpandedItems] = useState<Set<string>>(new Set());
  const [actedUponItems, setActedUponItems] = useState<Set<string>>(new Set());
  const [removalDialogItem, setRemovalDialogItem] = useState<ModqueueItem | null>(null);
  const [modlogData, setModlogData] = useState<{[key: string]: ModlogEntry[]}>({});
  
  const { toast } = useToast();
  const queryClient = useQueryClient();

  // Fetch moderated subreddits
  const { data: moderatedSubreddits = [], isLoading: isLoadingSubreddits } = useQuery({
    queryKey: ['/api/subreddits/moderated'],
    enabled: true,
  });

  // Auto-select the first available subreddit when moderatedSubreddits loads
  useEffect(() => {
    if (moderatedSubreddits && moderatedSubreddits.length > 0 && !selectedSubreddit) {
      // Find r/mildlyinteresting if it exists, otherwise use first subreddit
      const preferredSubreddit = moderatedSubreddits.find((sub: any) => 
        sub.name === 'mildlyinteresting' || sub.name.includes('mildlyinteresting')
      );
      const defaultSubreddit = preferredSubreddit || moderatedSubreddits[0];
      setSelectedSubreddit(defaultSubreddit.name);
    }
  }, [moderatedSubreddits, selectedSubreddit]);

  // Fetch modqueue items
  const { data: modqueueItems = [], isLoading: isLoadingQueue, refetch: refetchModqueue } = useQuery({
    queryKey: ['/api/reddit/modqueue', selectedSubreddit],
    queryFn: async () => {
      const response = await apiRequest(`/api/reddit/modqueue?subreddit=${selectedSubreddit}`, 'GET');
      console.log('[MODQUEUE DEBUG] API Response:', response);
      console.log('[MODQUEUE DEBUG] Response type:', typeof response);
      console.log('[MODQUEUE DEBUG] Is Array:', Array.isArray(response));
      return Array.isArray(response) ? response : [];
    },
    enabled: !!selectedSubreddit,
  });

  // Auto-expand all modqueue items when they load
  useEffect(() => {
    if (modqueueItems && Array.isArray(modqueueItems) && modqueueItems.length > 0) {
      const allItemIds = new Set(modqueueItems.map((item: ModqueueItem) => item.id));
      setExpandedItems(allItemIds);
      
      // Only reset acted-upon items if this is a new subreddit selection, not a refresh
      // We'll handle refresh differently to maintain better UX
      const currentItemIds = new Set(modqueueItems.map((item: ModqueueItem) => item.id));
      setActedUponItems(prev => {
        // Remove acted-upon items that are no longer in the modqueue (they were actually processed by Reddit)
        const stillActedUpon = new Set();
        prev.forEach(itemId => {
          if (currentItemIds.has(itemId)) {
            stillActedUpon.add(itemId);
          }
        });
        return stillActedUpon;
      });
    }
  }, [modqueueItems]);

  // Filter out acted-upon items from display unless showActedUponItems is true
  const filteredModqueueItems = modqueueItems ? 
    (showActedUponItems ? modqueueItems : modqueueItems.filter((item: ModqueueItem) => !actedUponItems.has(item.id))) : [];

  // Fetch subreddit rules
  const { data: subredditRules } = useQuery({
    queryKey: ['/api/reddit/rules', selectedSubreddit],
    queryFn: async () => {
      const response = await apiRequest(`/api/reddit/rules?subreddit=${selectedSubreddit}`, 'GET');
      console.log('[RULES DEBUG] API Response:', response);
      console.log('[RULES DEBUG] Rules array:', response?.rules);
      return response;
    },
    enabled: !!selectedSubreddit,
  });

  // Fetch moderation history
  const { data: moderationHistory = [] } = useQuery({
    queryKey: ['/api/reddit/moderation-history', selectedSubreddit],
    queryFn: async () => {
      const response = await apiRequest(`/api/reddit/moderation-history?subreddit=${selectedSubreddit}`, 'GET');
      return Array.isArray(response) ? response : [];
    },
    enabled: !!selectedSubreddit,
  });

  // AI Analysis mutation
  const analyzeItemMutation = useMutation({
    mutationFn: async ({ subreddit, itemId, itemData }: { subreddit: string; itemId: string; itemData: ModqueueItem }) => {
      console.log('[AI Analysis] Starting analysis for item:', itemId);
      const result = await apiRequest(`/api/reddit/analyze-modqueue-item`, 'POST', {
        subreddit,
        itemId,
        itemData
      });
      
      console.log('[AI Analysis] Result:', result);
      return { analysis: result.analysis, itemId, itemData };
    },
    onSuccess: ({ analysis: analysisResult, itemId, itemData }) => {
      console.log('[AI Analysis] Success:', analysisResult);
      
      // Store analysis result in the map using the item ID
      setAnalysisResults(prev => ({
        ...prev,
        [itemData.id]: analysisResult
      }));
      
      // For single item analysis, also set the main analysis state
      if (selectedItem?.id === itemData.id) {
        setAnalysis(analysisResult);
      }
      
      setIsAnalyzing(false);
      
      toast({
        title: "AI Analysis Complete",
        description: `Analysis complete with ${Math.round(analysisResult.confidenceScore * 100)}% confidence`,
      });
    },
    onError: (error) => {
      console.error('[AI Analysis] Error:', error);
      setIsAnalyzing(false);
      setErrorLog(prev => [...prev, `AI Analysis Error: ${error.message}`]);
      
      // Check if error is about insufficient points
      if (error.message?.includes('Insufficient points')) {
        toast({
          title: "Insufficient Points",
          description: "You need 5 points for each AI recommendation. Please top up your points to continue.",
          variant: "destructive",
        });
      } else {
        toast({
          title: "Analysis Failed",
          description: `Failed to analyze item: ${error.message}`,
          variant: "destructive",
        });
      }
    }
  });

  // Execute action mutation
  const executeActionMutation = useMutation({
    mutationFn: async ({ subreddit, itemId, action, reason, aiAnalysisId, itemData }: { 
      subreddit: string; 
      itemId: string; 
      action: string; 
      reason: string;
      aiAnalysisId?: number;
      itemData?: ModqueueItem;
    }) => {
      const response = await apiRequest(`/api/reddit/execute-moderation-action`, 'POST', {
        subreddit,
        itemId,
        action,
        reason,
        aiAnalysisId
      });
      
      return { response, itemData };
    },
    onSuccess: ({ response, itemData }) => {
      toast({
        title: "Action Executed",
        description: "Moderation action completed successfully",
      });
      
      // Mark item as acted upon to remove from display
      if (itemData) {
        setActedUponItems(prev => new Set([...prev, itemData.id]));
        
        // Clear analysis results for this item
        setAnalysisResults(prev => {
          const newResults = { ...prev };
          delete newResults[itemData.id];
          return newResults;
        });
      }
      
      // Reset state
      setSelectedItem(null);
      setAnalysis(null);
      setActionReason('');
      
      // Refresh data after a short delay to get updated modqueue
      setTimeout(() => {
        refetchModqueue();
      }, 2000);
      
      queryClient.invalidateQueries({ queryKey: ['/api/reddit/moderation-history', selectedSubreddit] });
    },
    onError: (error) => {
      toast({
        title: "Action Failed",
        description: `Failed to execute action: ${error.message}`,
        variant: "destructive",
      });
    }
  });

  const handleAnalyzeItem = async (item: ModqueueItem) => {
    if (!selectedSubreddit) return;
    
    setIsAnalyzing(true);
    setSelectedItem(item);
    setAnalysis(null);
    
    await analyzeItemMutation.mutateAsync({
      subreddit: selectedSubreddit,
      itemId: item.fullname,
      itemData: item
    });
  };

  const handleExecuteAction = async (action: string, aiAnalysisId?: number, aiReason?: string, targetItem?: ModqueueItem) => {
    // Use targetItem if provided, otherwise fall back to selectedItem
    const item = targetItem || selectedItem;
    if (!item || !selectedSubreddit) return;
    
    // Use AI reasoning if provided, otherwise fall back to manual reason
    const effectiveReason = aiReason || actionReason || `${action} action executed by moderator`;
    
    await executeActionMutation.mutateAsync({
      subreddit: selectedSubreddit,
      itemId: item.fullname,
      action,
      reason: effectiveReason,
      aiAnalysisId,
      itemData: item
    });
  };

  const handleDirectAction = async (item: ModqueueItem, action: string, customReason?: string) => {
    if (!selectedSubreddit) return;
    
    const reason = customReason || `Direct ${action} action by moderator`;
    
    await executeActionMutation.mutateAsync({
      subreddit: selectedSubreddit,
      itemId: item.fullname,
      action,
      reason,
      aiAnalysisId: undefined,
      itemData: item
    });
  };

  const handleRefresh = async () => {
    if (!selectedSubreddit) return;
    
    setIsRefreshing(true);
    try {
      await refetchModqueue();
      toast({
        title: "Modqueue Refreshed",
        description: "Successfully updated modqueue items from Reddit",
      });
    } catch (error) {
      toast({
        title: "Refresh Failed",
        description: "Failed to refresh modqueue items",
        variant: "destructive",
      });
    } finally {
      setIsRefreshing(false);
    }
  };

  // Function to clean HTML tags and decode entities from rule descriptions
  const cleanRuleDescription = (description: string): string => {
    if (!description) return '';
    
    return description
      // Remove HTML comments
      .replace(/<!--[\s\S]*?-->/g, '')
      // Remove HTML tags
      .replace(/<[^>]*>/g, '')
      // Decode common HTML entities
      .replace(/&amp;/g, '&')
      .replace(/&lt;/g, '<')
      .replace(/&gt;/g, '>')
      .replace(/&quot;/g, '"')
      .replace(/&#39;/g, "'")
      .replace(/&nbsp;/g, ' ')
      // Clean up extra whitespace
      .replace(/\s+/g, ' ')
      .trim();
  };

  // Refresh rules mutation
  const refreshRulesMutation = useMutation({
    mutationFn: async () => {
      if (!selectedSubreddit) throw new Error('No subreddit selected');
      
      const response = await fetch(`/api/reddit/rules?subreddit=${selectedSubreddit}&refresh=true`);
      if (!response.ok) throw new Error('Failed to refresh rules');
      return response.json();
    },
    onSuccess: (data) => {
      queryClient.invalidateQueries({ queryKey: ['/api/reddit/rules', selectedSubreddit] });
      toast({
        title: "Rules Refreshed",
        description: `Updated rules for r/${selectedSubreddit} from Reddit API`,
      });
    },
    onError: (error) => {
      toast({
        title: "Refresh Failed",
        description: "Could not refresh rules from Reddit API",
        variant: "destructive",
      });
    }
  });

  const handleAnalyzeAllItems = async () => {
    if (!selectedSubreddit || !filteredModqueueItems || !Array.isArray(filteredModqueueItems) || filteredModqueueItems.length === 0) return;
    
    setIsAnalyzing(true);
    toast({
      title: "Bulk Analysis Started",
      description: `Analyzing ${filteredModqueueItems.length} modqueue items with OpenWeb 3.2 AI...`,
    });

    try {
      for (let i = 0; i < filteredModqueueItems.length; i++) {
        const item = filteredModqueueItems[i];
        console.log(`[AI Bulk Analysis] Processing item ${i + 1}/${filteredModqueueItems.length}: ${item.id}`);
        
        setSelectedItem(item);
        
        await analyzeItemMutation.mutateAsync({
          subreddit: selectedSubreddit,
          itemId: item.fullname,
          itemData: item
        });
        
        if (i < filteredModqueueItems.length - 1) {
          await new Promise(resolve => setTimeout(resolve, 1000));
        }
      }
      
      toast({
        title: "Bulk Analysis Complete",
        description: `Successfully analyzed all ${filteredModqueueItems.length} modqueue items`,
      });
    } catch (error) {
      console.error('[AI Bulk Analysis] Error:', error);
      toast({
        title: "Bulk Analysis Failed",
        description: "Some items could not be analyzed. Check individual items for details.",
        variant: "destructive",
      });
    } finally {
      setIsAnalyzing(false);
    }
  };

  const toggleExpanded = (itemId: string) => {
    const newExpanded = new Set(expandedItems);
    if (newExpanded.has(itemId)) {
      newExpanded.delete(itemId);
    } else {
      newExpanded.add(itemId);
    }
    setExpandedItems(newExpanded);
  };

  // Function to fetch modlog data for a specific post
  const fetchModlogForPost = async (item: ModqueueItem) => {
    if (modlogData[item.id]) return; // Already fetched

    try {
      const response = await apiRequest(`/api/reddit/modlog/${selectedSubreddit}/${item.id}`, 'GET');
      setModlogData(prev => ({
        ...prev,
        [item.id]: Array.isArray(response) ? response : []
      }));
    } catch (error) {
      console.error('Failed to fetch modlog for post:', item.id, error);
      setModlogData(prev => ({
        ...prev,
        [item.id]: []
      }));
    }
  };

  const isExpanded = (itemId: string) => expandedItems.has(itemId);

  // Automatically fetch modlog data when items are expanded
  useEffect(() => {
    if (modqueueItems && Array.isArray(modqueueItems) && selectedSubreddit) {
      modqueueItems.forEach((item: ModqueueItem) => {
        if (isExpanded(item.id) && !modlogData[item.id]) {
          fetchModlogForPost(item);
        }
      });
    }
  }, [expandedItems, modqueueItems, selectedSubreddit]);

  const getSeverityColor = (severity: string) => {
    switch (severity) {
      case 'high': return 'destructive';
      case 'medium': return 'default';
      case 'low': return 'secondary';
      default: return 'outline';
    }
  };

  const getConfidenceColor = (confidence: number) => {
    if (confidence >= 0.8) return 'text-green-600 dark:text-green-400';
    if (confidence >= 0.6) return 'text-yellow-600 dark:text-yellow-400';
    return 'text-red-600 dark:text-red-400';
  };

  return (
    <DashboardLayout>
      <div className="space-y-4 sm:space-y-6 p-3 sm:p-4 lg:p-6">
        <div className="flex flex-col sm:flex-row sm:items-center sm:justify-between gap-4">
          <div>
            <h1 className="text-2xl sm:text-3xl font-bold flex items-center gap-2">
              <Shield className="h-6 w-6 sm:h-8 sm:w-8 text-blue-600" />
              AI-Powered Reddit Moderation
            </h1>
            <p className="text-muted-foreground mt-2 text-sm sm:text-base">
              Semi-automatic content moderation using Azure Llama 3.2 90B AI analysis
            </p>
          </div>
          
          <div className="text-xs sm:text-sm bg-muted p-2 sm:p-3 rounded border">
            <div>Selected Subreddit: {selectedSubreddit || 'None'}</div>
            <div>Component State: Loaded</div>
            <div>Error Count: {errorLog.length}</div>
          </div>
        </div>

        <Card>
          <CardHeader>
            <CardTitle className="text-lg sm:text-xl">Select Subreddit</CardTitle>
            <CardDescription className="text-sm">Choose a subreddit you moderate to begin analyzing the modqueue</CardDescription>
          </CardHeader>
          <CardContent>
            <Select value={selectedSubreddit} onValueChange={setSelectedSubreddit}>
              <SelectTrigger className="w-full">
                <SelectValue placeholder="Select a subreddit to moderate" />
              </SelectTrigger>
              <SelectContent>
                {moderatedSubreddits && Array.isArray(moderatedSubreddits) ? 
                  moderatedSubreddits.map((sub: any) => (
                    <SelectItem key={sub.name} value={sub.name}>
                      r/{sub.name}
                    </SelectItem>
                  )) : null
                }
              </SelectContent>
            </Select>
          
          {isLoadingSubreddits && (
            <div className="mt-2 text-sm text-muted-foreground">Loading your moderated subreddits...</div>
          )}
          </CardContent>
        </Card>

        {selectedSubreddit && (
        <Tabs defaultValue="modqueue" className="space-y-4">
          <TabsList className="grid w-full grid-cols-3">
            <TabsTrigger value="modqueue" className="flex items-center gap-2">
              <AlertTriangle className="h-4 w-4" />
              Modqueue ({modqueueItems && Array.isArray(modqueueItems) ? modqueueItems.length : 0})
            </TabsTrigger>
            <TabsTrigger value="rules" className="flex items-center gap-2">
              <FileText className="h-4 w-4" />
              Rules
            </TabsTrigger>
            <TabsTrigger value="history" className="flex items-center gap-2">
              <Clock className="h-4 w-4" />
              History
            </TabsTrigger>
          </TabsList>

          <TabsContent value="modqueue" className="space-y-4">
            {isLoadingQueue ? (
              <Card>
                <CardContent className="flex items-center justify-center p-8">
                  <div className="text-center">
                    <div className="animate-spin rounded-full h-8 w-8 border-b-2 border-blue-600 mx-auto mb-4"></div>
                    <p>Loading modqueue items...</p>
                  </div>
                </CardContent>
              </Card>
            ) : !modqueueItems || !Array.isArray(modqueueItems) || modqueueItems.length === 0 ? (
              <Card>
                <CardContent className="flex items-center justify-center p-8">
                  <div className="text-center">
                    <CheckCircle className="h-12 w-12 text-green-600 mx-auto mb-4" />
                    <h3 className="text-lg font-semibold mb-2">Modqueue is Clean!</h3>
                    <p className="text-muted-foreground">No items require moderation in r/{selectedSubreddit}</p>
                  </div>
                </CardContent>
              </Card>
            ) : (
              <>
                <div className="flex flex-col sm:flex-row sm:items-center sm:justify-between gap-4">
                  <div className="flex flex-wrap items-center gap-2">
                    <h3 className="text-lg font-semibold">Modqueue Items</h3>
                    <Badge variant="outline" className="text-xs">{filteredModqueueItems && Array.isArray(filteredModqueueItems) ? filteredModqueueItems.length : 0} pending</Badge>
                    {actedUponItems.size > 0 && (
                      <Badge variant="secondary" className="bg-green-100 text-green-700 text-xs">
                        {actedUponItems.size} acted upon
                      </Badge>
                    )}
                  </div>
                  <div className="flex flex-wrap gap-2">
                    <Button
                      onClick={handleRefresh}
                      variant="outline"
                      size="sm"
                      disabled={isRefreshing}
                      className="text-xs whitespace-nowrap"
                    >
                      <RefreshCw className={`h-4 w-4 mr-1 ${isRefreshing ? 'animate-spin' : ''}`} />
                      {isRefreshing ? 'Refreshing...' : 'Refresh'}
                    </Button>
                    {actedUponItems.size > 0 && (
                      <Button
                        onClick={() => setShowActedUponItems(!showActedUponItems)}
                        variant="outline"
                        size="sm"
                        className={`text-xs whitespace-nowrap ${showActedUponItems ? "bg-green-50 border-green-200 text-green-700" : ""}`}
                      >
                        <Eye className="h-4 w-4 mr-1" />
                        <span className="hidden sm:inline">{showActedUponItems ? 'Hide Acted Upon' : 'Show All Items'}</span>
                        <span className="sm:hidden">{showActedUponItems ? 'Hide' : 'Show'}</span>
                      </Button>
                    )}
                    <Button
                      onClick={handleAnalyzeAllItems}
                      disabled={isAnalyzing || !filteredModqueueItems || !Array.isArray(filteredModqueueItems) || filteredModqueueItems.length === 0}
                      size="sm"
                      className="bg-blue-600 hover:bg-blue-700 text-xs whitespace-nowrap"
                    >
                      <Zap className="h-4 w-4 mr-1" />
                      <span className="hidden sm:inline">{isAnalyzing ? "Analyzing All..." : `Analyze All (${filteredModqueueItems?.length || 0})`}</span>
                      <span className="sm:hidden">{isAnalyzing ? "Analyzing..." : `All (${filteredModqueueItems?.length || 0})`}</span>
                    </Button>
                  </div>
                </div>

                <div className="grid gap-4">
                  {filteredModqueueItems && Array.isArray(filteredModqueueItems) && filteredModqueueItems.map((item: ModqueueItem) => (
                    <Card key={item.id} className={`border-l-4 ${actedUponItems.has(item.id) ? 'border-l-green-500 bg-green-50 dark:bg-green-950 opacity-75' : 'border-l-orange-500'}`}>
                      <CardContent className="p-6">
                        <div className="flex items-start justify-between">
                          <div className="space-y-3 flex-1">
                            <div className="flex items-center gap-2">
                              {item.type === 'submission' ? (
                                <FileText className="h-4 w-4 text-blue-600" />
                              ) : (
                                <MessageSquare className="h-4 w-4 text-green-600" />
                              )}
                              <Badge variant="outline">
                                {item.type === 'submission' ? 'Post' : 'Comment'}
                              </Badge>
                              {actedUponItems.has(item.id) && (
                                <Badge variant="secondary" className="bg-green-100 text-green-700">
                                  ✓ Acted Upon
                                </Badge>
                              )}
                              <span className="text-sm text-muted-foreground">by u/{item.author}</span>
                              <span className="text-sm text-muted-foreground">
                                {new Date(item.createdUtc * 1000).toLocaleString()}
                              </span>
                            </div>
                            
                            {item.title && (
                              <h4 className="font-semibold text-lg">{item.title}</h4>
                            )}
                            
                            {/* Collapsible Full Content Display */}
                            {isExpanded(item.id) && (
                              <div className="space-y-3 mt-4 p-4 bg-gray-50 dark:bg-gray-900 rounded-lg border">
                                <div className="flex items-center gap-2 mb-3">
                                  <FileText className="h-4 w-4 text-blue-600" />
                                  <h5 className="font-semibold text-sm">Full Post Details</h5>
                                </div>

                                {item.content && (
                                  <div className="bg-white dark:bg-gray-800 p-4 rounded-lg border">
                                    <h6 className="font-medium text-sm text-muted-foreground mb-2 flex items-center gap-1">
                                      <MessageSquare className="h-3 w-3" />
                                      Post Content:
                                    </h6>
                                    <div className="prose prose-sm max-w-none dark:prose-invert">
                                      <p className="whitespace-pre-wrap text-sm">{item.content}</p>
                                    </div>
                                  </div>
                                )}
                                
                                {/* Enhanced Image Display */}
                                {item.url && (item.url.includes('.jpg') || item.url.includes('.jpeg') || item.url.includes('.png') || item.url.includes('.gif') || item.url.includes('.webp') || item.url.includes('i.redd.it')) && (
                                  <div className="bg-white dark:bg-gray-800 p-4 rounded-lg border">
                                    <h6 className="font-medium text-sm text-muted-foreground mb-3 flex items-center gap-1">
                                      <ImageIcon className="h-3 w-3" />
                                      Attached Media:
                                    </h6>
                                    <div className="relative bg-gray-100 dark:bg-gray-700 rounded-lg p-2">
                                      <img 
                                        src={item.url} 
                                        alt={item.title || "Post image"}
                                        className="w-full h-auto rounded-lg shadow-lg max-h-[500px] object-contain mx-auto"
                                        onLoad={(e) => {
                                          const target = e.target as HTMLImageElement;
                                          target.style.opacity = '1';
                                        }}
                                        onError={(e) => {
                                          const target = e.target as HTMLImageElement;
                                          target.style.display = 'none';
                                          const errorDiv = document.createElement('div');
                                          errorDiv.className = 'p-4 bg-red-50 dark:bg-red-950 rounded border border-red-200 text-red-600 text-sm text-center';
                                          errorDiv.innerHTML = `<div class="flex items-center justify-center gap-2"><svg class="h-4 w-4" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 9v2m0 4h.01m-6.938 4h13.856c1.54 0 2.502-1.667 1.732-2.5L13.732 4c-.77-.833-1.964-.833-2.732 0L4.082 16.5c-.77.833.192 2.5 1.732 2.5z"></path></svg>Image failed to load</div><div class="text-xs mt-1 opacity-75">${item.url}</div>`;
                                          target.parentNode?.appendChild(errorDiv);
                                        }}
                                        style={{ opacity: '0', transition: 'opacity 0.3s ease' }}
                                      />
                                      <div className="absolute top-2 right-2">
                                        <a 
                                          href={item.url} 
                                          target="_blank" 
                                          rel="noopener noreferrer"
                                          className="bg-black bg-opacity-50 text-white text-xs px-2 py-1 rounded flex items-center gap-1 hover:bg-opacity-70 transition-all"
                                        >
                                          <ExternalLink className="h-3 w-3" />
                                          Open Original
                                        </a>
                                      </div>
                                    </div>
                                  </div>
                                )}
                                
                                {/* External Links */}
                                {item.url && !(item.url.includes('.jpg') || item.url.includes('.jpeg') || item.url.includes('.png') || item.url.includes('.gif') || item.url.includes('.webp') || item.url.includes('i.redd.it')) && (
                                  <div className="bg-white dark:bg-gray-800 p-4 rounded-lg border">
                                    <h6 className="font-medium text-sm text-muted-foreground mb-2 flex items-center gap-1">
                                      <ExternalLink className="h-3 w-3" />
                                      External Link:
                                    </h6>
                                    <a 
                                      href={item.url} 
                                      target="_blank" 
                                      rel="noopener noreferrer"
                                      className="flex items-center gap-2 text-blue-600 hover:underline text-sm break-all p-2 bg-blue-50 dark:bg-blue-950 rounded border"
                                    >
                                      <ExternalLink className="h-3 w-3 flex-shrink-0" />
                                      {item.url}
                                    </a>
                                  </div>
                                )}
                                
                                {/* Enhanced Reports Section */}
                                {(item.userReports?.length > 0 || item.modReports?.length > 0) && (
                                  <div className="bg-white dark:bg-gray-800 p-4 rounded-lg border">
                                    <h6 className="font-medium text-sm text-muted-foreground mb-3 flex items-center gap-1">
                                      <AlertTriangle className="h-3 w-3" />
                                      Detailed Reports:
                                    </h6>
                                    
                                    {item.userReports && item.userReports.length > 0 && (
                                      <div className="mb-3">
                                        <div className="bg-yellow-50 dark:bg-yellow-950 p-3 rounded border border-yellow-200">
                                          <h6 className="font-medium text-sm text-yellow-700 dark:text-yellow-300 mb-2">
                                            User Reports ({item.userReports.length})
                                          </h6>
                                          <div className="space-y-2">
                                            {item.userReports.map((report: any, idx: number) => (
                                              <div key={idx} className="bg-yellow-100 dark:bg-yellow-900 p-2 rounded text-xs">
                                                {Array.isArray(report) ? (
                                                  <div>
                                                    <span className="font-medium">"{report[0]}"</span>
                                                    <span className="text-yellow-600 dark:text-yellow-400 ml-2">
                                                      ({report[1]} report{report[1] > 1 ? 's' : ''})
                                                    </span>
                                                  </div>
                                                ) : (
                                                  <span>{report}</span>
                                                )}
                                              </div>
                                            ))}
                                          </div>
                                        </div>
                                      </div>
                                    )}

                                    {item.modReports && item.modReports.length > 0 && (
                                      <div className="bg-red-50 dark:bg-red-950 p-3 rounded border border-red-200">
                                        <h6 className="font-medium text-sm text-red-700 dark:text-red-300 mb-2">
                                          Moderator Reports ({item.modReports.length})
                                        </h6>
                                        <div className="space-y-1">
                                          {item.modReports.map((report: any, idx: number) => (
                                            <div key={idx} className="bg-red-100 dark:bg-red-900 p-2 rounded text-xs text-red-600 dark:text-red-400">
                                              {Array.isArray(report) ? report[0] : report}
                                            </div>
                                          ))}
                                        </div>
                                      </div>
                                    )}
                                  </div>
                                )}
                              </div>
                            )}

                            {/* Reddit Moderation History */}
                            {modlogData[item.id] && (
                              <div className="bg-purple-50 dark:bg-purple-950 p-4 rounded-lg border border-purple-200">
                                <h6 className="font-medium text-sm text-purple-700 dark:text-purple-300 mb-3 flex items-center gap-1">
                                  <History className="h-3 w-3" />
                                  Previous Moderator Actions ({modlogData[item.id].length})
                                </h6>
                                {modlogData[item.id].length > 0 ? (
                                  <div className="space-y-2 max-h-40 overflow-y-auto">
                                    {modlogData[item.id].map((entry: ModlogEntry, idx: number) => (
                                      <div key={idx} className="bg-purple-100 dark:bg-purple-900 p-3 rounded border text-xs">
                                        <div className="flex items-center justify-between mb-1">
                                          <div className="flex items-center gap-2">
                                            <Badge variant="outline" className="text-xs px-2 py-0.5">
                                              {entry.action}
                                            </Badge>
                                            <span className="font-medium text-purple-700 dark:text-purple-300">
                                              u/{entry.moderator}
                                            </span>
                                          </div>
                                          <span className="text-purple-600 dark:text-purple-400">
                                            {new Date(entry.created).toLocaleDateString()} {new Date(entry.created).toLocaleTimeString()}
                                          </span>
                                        </div>
                                        {entry.description && (
                                          <p className="text-purple-600 dark:text-purple-400 mt-1">
                                            {entry.description}
                                          </p>
                                        )}
                                        {entry.details && (
                                          <p className="text-purple-500 dark:text-purple-500 text-xs mt-1">
                                            Details: {entry.details}
                                          </p>
                                        )}
                                        {entry.targetTitle && (
                                          <p className="text-purple-500 dark:text-purple-500 text-xs mt-1 truncate">
                                            Target: {entry.targetTitle}
                                          </p>
                                        )}
                                      </div>
                                    ))}
                                  </div>
                                ) : (
                                  <p className="text-purple-600 dark:text-purple-400 text-xs">
                                    No previous moderation actions found for this post
                                  </p>
                                )}
                              </div>
                            )}

                            {/* Comprehensive Post Metadata */}
                            <div className="grid grid-cols-2 md:grid-cols-4 gap-3 text-sm">
                              <div className="bg-gray-50 dark:bg-gray-800 p-2 rounded">
                                <span className="text-muted-foreground">Score:</span>
                                <span className="ml-1 font-medium">{item.score}</span>
                              </div>
                              <div className="bg-gray-50 dark:bg-gray-800 p-2 rounded">
                                <span className="text-muted-foreground">Reports:</span>
                                <span className="ml-1 font-medium">{item.numReports}</span>
                              </div>
                              {(item as any).upvotes && (
                                <div className="bg-gray-50 dark:bg-gray-800 p-2 rounded">
                                  <span className="text-muted-foreground">Upvotes:</span>
                                  <span className="ml-1 font-medium text-green-600">{(item as any).upvotes}</span>
                                </div>
                              )}
                              {(item as any).upvoteRatio && (
                                <div className="bg-gray-50 dark:bg-gray-800 p-2 rounded">
                                  <span className="text-muted-foreground">Ratio:</span>
                                  <span className="ml-1 font-medium">{Math.round((item as any).upvoteRatio * 100)}%</span>
                                </div>
                              )}
                              {(item as any).numComments && (
                                <div className="bg-gray-50 dark:bg-gray-800 p-2 rounded">
                                  <span className="text-muted-foreground">Comments:</span>
                                  <span className="ml-1 font-medium">{(item as any).numComments}</span>
                                </div>
                              )}
                              {(item as any).gilded > 0 && (
                                <div className="bg-gray-50 dark:bg-gray-800 p-2 rounded">
                                  <span className="text-muted-foreground">Gilded:</span>
                                  <span className="ml-1 font-medium text-yellow-600">{(item as any).gilded}</span>
                                </div>
                              )}
                              {(item as any).totalAwardsReceived > 0 && (
                                <div className="bg-gray-50 dark:bg-gray-800 p-2 rounded">
                                  <span className="text-muted-foreground">Awards:</span>
                                  <span className="ml-1 font-medium text-purple-600">{(item as any).totalAwardsReceived}</span>
                                </div>
                              )}
                              <div className="bg-gray-50 dark:bg-gray-800 p-2 rounded">
                                <a 
                                  href={`https://reddit.com${item.permalink}`} 
                                  target="_blank" 
                                  rel="noopener noreferrer"
                                  className="flex items-center gap-1 text-blue-600 hover:underline"
                                >
                                  <ExternalLink className="h-3 w-3" />
                                  View on Reddit
                                </a>
                              </div>
                            </div>

                            {/* Author and Flair Information */}
                            {((item as any).authorFlairText || (item as any).linkFlairText) && (
                              <div className="bg-blue-50 dark:bg-blue-950 p-3 rounded-lg border border-blue-200">
                                <h6 className="font-medium text-sm text-blue-700 dark:text-blue-300 mb-2 flex items-center gap-1">
                                  <User className="h-3 w-3" />
                                  Author & Post Information
                                </h6>
                                <div className="grid grid-cols-1 md:grid-cols-2 gap-3 text-sm">
                                  {(item as any).authorFlairText && (
                                    <div>
                                      <span className="text-muted-foreground">Author Flair:</span>
                                      <span className="ml-2 px-2 py-1 bg-blue-100 dark:bg-blue-900 rounded text-blue-700 dark:text-blue-300 text-xs">
                                        {(item as any).authorFlairText}
                                      </span>
                                    </div>
                                  )}
                                  {(item as any).linkFlairText && (
                                    <div>
                                      <span className="text-muted-foreground">Post Flair:</span>
                                      <span className="ml-2 px-2 py-1 bg-green-100 dark:bg-green-900 rounded text-green-700 dark:text-green-300 text-xs">
                                        {(item as any).linkFlairText}
                                      </span>
                                    </div>
                                  )}
                                  {(item as any).domain && (
                                    <div>
                                      <span className="text-muted-foreground">Domain:</span>
                                      <span className="ml-2 font-mono text-xs">{(item as any).domain}</span>
                                    </div>
                                  )}
                                  {(item as any).postHint && (
                                    <div>
                                      <span className="text-muted-foreground">Content Type:</span>
                                      <span className="ml-2 capitalize">{(item as any).postHint}</span>
                                    </div>
                                  )}
                                </div>
                              </div>
                            )}

                            {/* Moderation Status Indicators */}
                            {((item as any).stickied || (item as any).locked || (item as any).archived || (item as any).spoiler || (item as any).nsfw) && (
                              <div className="flex flex-wrap gap-2">
                                {(item as any).stickied && (
                                  <Badge variant="secondary" className="text-xs bg-green-100 text-green-700">
                                    📌 Stickied
                                  </Badge>
                                )}
                                {(item as any).locked && (
                                  <Badge variant="secondary" className="text-xs bg-red-100 text-red-700">
                                    🔒 Locked
                                  </Badge>
                                )}
                                {(item as any).archived && (
                                  <Badge variant="secondary" className="text-xs bg-gray-100 text-gray-700">
                                    📦 Archived
                                  </Badge>
                                )}
                                {(item as any).spoiler && (
                                  <Badge variant="secondary" className="text-xs bg-yellow-100 text-yellow-700">
                                    ⚠️ Spoiler
                                  </Badge>
                                )}
                                {(item as any).nsfw && (
                                  <Badge variant="secondary" className="text-xs bg-red-100 text-red-700">
                                    🔞 NSFW
                                  </Badge>
                                )}
                                {(item as any).isOriginalContent && (
                                  <Badge variant="secondary" className="text-xs bg-blue-100 text-blue-700">
                                    ✨ OC
                                  </Badge>
                                )}
                              </div>
                            )}

                            {/* User Reports Display */}
                            {item.userReports && item.userReports.length > 0 && (
                              <div className="bg-yellow-50 dark:bg-yellow-950 p-3 rounded-lg border border-yellow-200">
                                <h5 className="font-medium text-sm text-yellow-700 dark:text-yellow-300 mb-2 flex items-center gap-1">
                                  <AlertTriangle className="h-3 w-3" />
                                  User Reports ({item.userReports.length})
                                </h5>
                                <div className="space-y-1">
                                  {item.userReports.map((report: any, idx: number) => (
                                    <div key={idx} className="text-xs text-yellow-600 dark:text-yellow-400">
                                      {Array.isArray(report) ? (
                                        <span>"{report[0]}" - {report[1]} report{report[1] > 1 ? 's' : ''}</span>
                                      ) : (
                                        <span>{report}</span>
                                      )}
                                    </div>
                                  ))}
                                </div>
                              </div>
                            )}

                            {/* Mod Reports Display */}
                            {item.modReports && item.modReports.length > 0 && (
                              <div className="bg-red-50 dark:bg-red-950 p-3 rounded-lg border border-red-200">
                                <h5 className="font-medium text-sm text-red-700 dark:text-red-300 mb-2 flex items-center gap-1">
                                  <Shield className="h-3 w-3" />
                                  Moderator Reports ({item.modReports.length})
                                </h5>
                                <div className="space-y-1">
                                  {item.modReports.map((report: any, idx: number) => (
                                    <div key={idx} className="text-xs text-red-600 dark:text-red-400">
                                      {Array.isArray(report) ? report[0] : report}
                                    </div>
                                  ))}
                                </div>
                              </div>
                            )}
                          </div>

                          <div className="flex flex-col gap-2 ml-4">
                            <Button
                              size="sm"
                              variant="outline"
                              onClick={() => toggleExpanded(item.id)}
                              className="min-w-[100px]"
                            >
                              {isExpanded(item.id) ? (
                                <>
                                  <ChevronUp className="h-3 w-3 mr-1" />
                                  Collapse Details
                                </>
                              ) : (
                                <>
                                  <ChevronDown className="h-3 w-3 mr-1" />
                                  Show Details
                                </>
                              )}
                            </Button>
                            <Button
                              size="sm"
                              variant="outline"
                              onClick={() => handleAnalyzeItem(item)}
                              disabled={isAnalyzing}
                              className="min-w-[100px]"
                            >
                              {isAnalyzing && selectedItem?.id === item.id ? (
                                <>
                                  <div className="animate-spin h-3 w-3 border border-gray-300 border-t-blue-600 rounded-full mr-2"></div>
                                  Analyzing...
                                </>
                              ) : (
                                <>
                                  <Bot className="h-3 w-3 mr-1" />
                                  AI Analysis
                                </>
                              )}
                            </Button>
                            <Button
                              size="sm"
                              variant="outline"
                              onClick={() => setSelectedItem(item)}
                            >
                              <Eye className="h-4 w-4 mr-1" />
                              Review
                            </Button>
                          </div>
                        </div>

                        {/* Direct Moderation Actions */}
                        <div className="mt-4 pt-4 border-t border-gray-200 dark:border-gray-700">
                          <div className="flex gap-2 justify-end">
                            <Button
                              size="sm"
                              variant="outline"
                              onClick={() => handleDirectAction(item, 'approve')}
                              disabled={executeActionMutation.isPending}
                              className="text-green-600 border-green-200 hover:bg-green-50 hover:border-green-300"
                            >
                              <CheckCircle className="h-3 w-3 mr-1" />
                              Approve
                            </Button>
                            <DropdownMenu>
                              <DropdownMenuTrigger asChild>
                                <Button
                                  size="sm"
                                  variant="outline"
                                  disabled={executeActionMutation.isPending}
                                  className="text-red-600 border-red-200 hover:bg-red-50 hover:border-red-300"
                                >
                                  <XCircle className="h-3 w-3 mr-1" />
                                  Remove
                                  <ChevronDown className="h-3 w-3 ml-1" />
                                </Button>
                              </DropdownMenuTrigger>
                              <DropdownMenuContent align="end" className="w-80">
                                {subredditRules && (subredditRules as any)?.rules && Array.isArray((subredditRules as any).rules) ? (
                                  (subredditRules as any).rules.map((rule: any, index: number) => (
                                    <DropdownMenuItem 
                                      key={index}
                                      onClick={() => handleDirectAction(item, 'remove', `Rule ${index + 1}: ${rule.shortName} - ${cleanRuleDescription(rule.description).substring(0, 100)}...`)}
                                      className="flex-col items-start p-3 cursor-pointer hover:bg-red-50"
                                    >
                                      <div className="font-medium text-red-700">Rule {index + 1}: {rule.shortName}</div>
                                      <div className="text-xs text-gray-600 mt-1 line-clamp-2">
                                        {cleanRuleDescription(rule.description).substring(0, 120)}...
                                      </div>
                                    </DropdownMenuItem>
                                  ))
                                ) : (
                                  <DropdownMenuItem onClick={() => handleDirectAction(item, 'remove', 'Manual removal')}>
                                    <div className="text-center w-full">
                                      <div className="font-medium">Manual Removal</div>
                                      <div className="text-xs text-gray-600">No specific rule selected</div>
                                    </div>
                                  </DropdownMenuItem>
                                )}
                              </DropdownMenuContent>
                            </DropdownMenu>
                          </div>
                        </div>

                        {/* AI Analysis Results Display */}
                        {analysisResults[item.id] && (
                          <div className="mt-4 p-4 bg-blue-50 dark:bg-blue-950 rounded-lg border">
                            <div className="flex items-center justify-between mb-3">
                              <div className="flex items-center gap-2">
                                <Bot className="h-4 w-4 text-blue-600" />
                                <span className="font-semibold text-blue-700 dark:text-blue-300">AI Analysis Results</span>
                                <Badge variant={getSeverityColor(analysisResults[item.id].severity)}>
                                  {analysisResults[item.id].severity.toUpperCase()} RISK
                                </Badge>
                                <span className={`text-sm font-medium ${getConfidenceColor(analysisResults[item.id].confidenceScore)}`}>
                                  {Math.round(analysisResults[item.id].confidenceScore * 100)}% confidence
                                </span>
                              </div>
                              <Button 
                                size="sm" 
                                variant="ghost" 
                                onClick={() => {
                                  setAnalysisResults(prev => {
                                    const newResults = { ...prev };
                                    delete newResults[item.id];
                                    return newResults;
                                  });
                                  if (selectedItem?.id === item.id) {
                                    setAnalysis(null);
                                    setSelectedItem(null);
                                  }
                                }}
                              >
                                ✕
                              </Button>
                            </div>

                            <div className="space-y-3">
                              <div className="bg-white dark:bg-gray-800 p-3 rounded border">
                                <div className="flex items-center gap-2 mb-2">
                                  <span className="font-medium text-gray-700 dark:text-gray-300">Recommended Action:</span>
                                  <Badge variant={analysisResults[item.id].suggestedAction === 'approve' ? 'default' : 'destructive'}>
                                    {analysisResults[item.id].suggestedAction.toUpperCase()}
                                  </Badge>
                                </div>
                                <p className="text-sm text-gray-600 dark:text-gray-400">{analysisResults[item.id].reasoning}</p>
                              </div>

                              {analysisResults[item.id].violationsFound && analysisResults[item.id].violationsFound.length > 0 && (
                                <div className="bg-red-50 dark:bg-red-950 p-3 rounded border border-red-200">
                                  <h4 className="font-medium text-red-700 dark:text-red-300 mb-2">Rule Violations Detected:</h4>
                                  <div className="space-y-2">
                                    {analysisResults[item.id].violationsFound.map((violation, idx) => (
                                      <div key={idx} className="text-sm">
                                        <div className="flex items-center gap-2">
                                          <Badge variant="destructive" className="text-xs">
                                            Rule {violation.ruleIndex + 1}
                                          </Badge>
                                          <span className="font-medium">{violation.ruleName}</span>
                                          <Badge variant={getSeverityColor(violation.severity)} className="text-xs">
                                            {violation.severity}
                                          </Badge>
                                        </div>
                                        <p className="text-red-600 dark:text-red-400 mt-1">{violation.violation}</p>
                                      </div>
                                    ))}
                                  </div>
                                </div>
                              )}

                              <div className="flex gap-2 pt-2">
                                <Button
                                  size="sm"
                                  variant="default"
                                  onClick={() => handleExecuteAction(analysisResults[item.id].suggestedAction, analysisResults[item.id].id, analysisResults[item.id].reasoning, item)}
                                  className="bg-green-600 hover:bg-green-700"
                                >
                                  <CheckCircle className="h-3 w-3 mr-1" />
                                  Accept AI Suggestion
                                </Button>
                                <Button
                                  size="sm"
                                  variant="outline"
                                  onClick={() => {
                                    setAnalysisResults(prev => {
                                      const newResults = { ...prev };
                                      delete newResults[item.id];
                                      return newResults;
                                    });
                                    if (selectedItem?.id === item.id) {
                                      setAnalysis(null);
                                      setSelectedItem(null);
                                    }
                                  }}
                                >
                                  <XCircle className="h-3 w-3 mr-1" />
                                  Ignore Suggestion
                                </Button>
                                <Button
                                  size="sm"
                                  variant="outline"
                                  onClick={() => setSelectedItem(item)}
                                >
                                  <Eye className="h-3 w-3 mr-1" />
                                  Manual Review
                                </Button>
                              </div>
                            </div>
                          </div>
                        )}
                      </CardContent>
                    </Card>
                  ))}
                </div>
              </>
            )}
          </TabsContent>

          <TabsContent value="rules" className="space-y-4">
            {subredditRules ? (
              <Card>
                <CardHeader>
                  <div className="flex items-center justify-between">
                    <div>
                      <CardTitle>r/{selectedSubreddit} Rules</CardTitle>
                      <CardDescription>
                        Rules used by the AI for content analysis
                      </CardDescription>
                    </div>
                    <Button
                      variant="outline"
                      size="sm"
                      onClick={() => refreshRulesMutation.mutate()}
                      disabled={refreshRulesMutation.isPending}
                    >
                      {refreshRulesMutation.isPending ? (
                        <>
                          <div className="animate-spin rounded-full h-3 w-3 border-b-2 border-blue-600 mr-2" />
                          Refreshing...
                        </>
                      ) : (
                        <>
                          <RefreshCw className="h-3 w-3 mr-2" />
                          Refresh Rules
                        </>
                      )}
                    </Button>
                  </div>
                </CardHeader>
                <CardContent>
                  <div className="space-y-4">
                    {subredditRules && (subredditRules as any)?.rules && Array.isArray((subredditRules as any).rules) ? (subredditRules as any).rules.map((rule: any, index: number) => (
                      <div key={index} className="p-4 border rounded-lg">
                        <div className="flex items-start gap-3">
                          <Badge variant="outline" className="mt-1">
                            Rule {index + 1}
                          </Badge>
                          <div className="flex-1">
                            <h4 className="font-semibold">{rule.shortName}</h4>
                            <p className="text-sm text-muted-foreground mt-1">
                              {cleanRuleDescription(rule.description)}
                            </p>
                            {rule.violationReason && (
                              <p className="text-xs text-red-600 mt-2 bg-red-50 dark:bg-red-900/20 p-2 rounded">
                                Violation: {rule.violationReason}
                              </p>
                            )}
                          </div>
                        </div>
                      </div>
                    )) : (
                      <div className="p-4 text-center text-muted-foreground">
                        No rules available for this subreddit
                      </div>
                    )}
                  </div>
                </CardContent>
              </Card>
            ) : (
              <Card>
                <CardContent className="flex items-center justify-center p-8">
                  <div className="text-center">
                    <div className="animate-spin rounded-full h-8 w-8 border-b-2 border-blue-600 mx-auto mb-4"></div>
                    <p>Loading subreddit rules...</p>
                  </div>
                </CardContent>
              </Card>
            )}
          </TabsContent>

          <TabsContent value="history" className="space-y-4">
            {moderationHistory && Array.isArray(moderationHistory) && moderationHistory.length > 0 ? (
              <Card>
                <CardHeader>
                  <CardTitle>Recent Moderation Actions</CardTitle>
                  <CardDescription>
                    History of moderation decisions for r/{selectedSubreddit}
                  </CardDescription>
                </CardHeader>
                <CardContent>
                  <div className="space-y-4">
                    {moderationHistory && Array.isArray(moderationHistory) && moderationHistory.map((action: any) => (
                      <div key={action.id} className="flex items-center justify-between p-4 border rounded-lg">
                        <div className="space-y-1">
                          <div className="flex items-center gap-2">
                            <Badge variant={action.wasAiSuggested ? "default" : "secondary"}>
                              {action.wasAiSuggested ? <Bot className="h-3 w-3 mr-1" /> : <User className="h-3 w-3 mr-1" />}
                              {action.wasAiSuggested ? 'AI Suggested' : 'Manual'}
                            </Badge>
                            <span className="font-medium">{action.action}</span>
                          </div>
                          <p className="text-sm text-muted-foreground">
                            {action.reason || 'No reason provided'}
                          </p>
                        </div>
                        <div className="text-sm text-muted-foreground">
                          {new Date(action.createdAt).toLocaleString()}
                        </div>
                      </div>
                    ))}
                  </div>
                </CardContent>
              </Card>
            ) : (
              <Card>
                <CardContent className="flex items-center justify-center p-8">
                  <div className="text-center">
                    <Clock className="h-12 w-12 text-gray-400 mx-auto mb-4" />
                    <h3 className="text-lg font-semibold mb-2">No History Yet</h3>
                    <p className="text-muted-foreground">
                      Moderation actions will appear here once you start moderating
                    </p>
                  </div>
                </CardContent>
              </Card>
            )}
          </TabsContent>
        </Tabs>
      )}

      {errorLog.length > 0 && (
        <Alert variant="destructive">
          <AlertTriangle className="h-4 w-4" />
          <AlertDescription>
            <details>
              <summary>Debug Information ({errorLog.length} errors)</summary>
              <div className="mt-2 space-y-1 text-xs">
                {errorLog.map((error, idx) => (
                  <div key={idx} className="bg-red-100 p-2 rounded">
                    {error}
                  </div>
                ))}
              </div>
            </details>
          </AlertDescription>
        </Alert>
      )}
      </div>
    </DashboardLayout>
  );
}