import { useState } from "react";
import { useQuery, useMutation, useQueryClient } from "@tanstack/react-query";
import { apiRequest } from "@/lib/queryClient";
import { DashboardLayout } from "@/components/dashboard-layout";
import { Button } from "@/components/ui/button";
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card";
import { Badge } from "@/components/ui/badge";
import { Dialog, DialogContent, DialogDescription, DialogFooter, DialogHeader, DialogTitle, DialogTrigger } from "@/components/ui/dialog";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select";
import { Tabs, TabsContent, TabsList, TabsTrigger } from "@/components/ui/tabs";
import { Separator } from "@/components/ui/separator";
import { useToast } from "@/hooks/use-toast";
import { Settings as SettingsIcon, Brain, Server, Zap, User, Bell, Shield, Palette } from "lucide-react";

interface AiConfiguration {
  id: number;
  name: string;
  provider: string;
  model: string;
  apiKey?: string;
  systemPrompt: string;
  apiEndpoint?: string;
  maxTokens: number;
  temperature: number;
  topP: number;
  isActive: boolean;
  isDefault: boolean;
  createdBy: number;
  createdAt: string;
  updatedAt: string;
}

interface AiErrorLog {
  id: number;
  provider: string;
  model: string;
  errorMessage: string;
  httpStatus?: number;
  createdAt: string;
}

interface AiUsageStats {
  id: number;
  provider: string;
  model: string;
  requestCount: number;
  successCount: number;
  errorCount: number;
  totalTokensUsed: number;
  averageResponseTime: number;
  dateRecord: string;
}

export default function Settings() {
  const { toast } = useToast();
  const queryClient = useQueryClient();
  const [editingConfig, setEditingConfig] = useState<AiConfiguration | null>(null);
  const [isCreateDialogOpen, setIsCreateDialogOpen] = useState(false);
  const [activeTab, setActiveTab] = useState("ai-settings");

  // AI Configuration Queries
  const { data: aiConfigurations, isLoading: configsLoading } = useQuery({
    queryKey: ["/api/ai-configurations"],
    queryFn: () => apiRequest("/api/ai-configurations"),
  });

  const { data: aiErrors } = useQuery({
    queryKey: ["/api/ai-error-logs"],
    queryFn: () => apiRequest("/api/ai-error-logs"),
  });

  const { data: aiUsage } = useQuery({
    queryKey: ["/api/ai-usage-stats"],
    queryFn: () => apiRequest("/api/ai-usage-stats"),
  });

  // AI Configuration Mutations
  const activateConfigMutation = useMutation({
    mutationFn: (configId: number) => apiRequest(`/api/ai-configurations/${configId}/activate`, "POST"),
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/ai-configurations"] });
      toast({ title: "AI configuration activated successfully" });
    },
    onError: (error: any) => {
      toast({ 
        title: "Failed to activate configuration", 
        description: error.message,
        variant: "destructive"
      });
    },
  });

  const testConfigMutation = useMutation({
    mutationFn: (configId: number) => apiRequest(`/api/ai-configurations/${configId}/test`, "POST"),
    onSuccess: (data) => {
      toast({ 
        title: data.success ? "Configuration test passed" : "Configuration test failed",
        description: data.message,
        variant: data.success ? "default" : "destructive"
      });
    },
    onError: (error: any) => {
      toast({ 
        title: "Test failed", 
        description: error.message,
        variant: "destructive"
      });
    },
  });

  const createConfigMutation = useMutation({
    mutationFn: (config: Partial<AiConfiguration>) => apiRequest("/api/ai-configurations", "POST", config),
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/ai-configurations"] });
      setIsCreateDialogOpen(false);
      toast({ title: "AI configuration created successfully" });
    },
    onError: (error: any) => {
      toast({ 
        title: "Failed to create configuration", 
        description: error.message,
        variant: "destructive"
      });
    },
  });

  const updateConfigMutation = useMutation({
    mutationFn: ({ id, ...config }: AiConfiguration) => 
      apiRequest(`/api/ai-configurations/${id}`, "PUT", config),
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/ai-configurations"] });
      setEditingConfig(null);
      toast({ title: "AI configuration updated successfully" });
    },
    onError: (error: any) => {
      toast({ 
        title: "Failed to update configuration", 
        description: error.message,
        variant: "destructive"
      });
    },
  });

  const deleteConfigMutation = useMutation({
    mutationFn: (configId: number) => apiRequest(`/api/ai-configurations/${configId}`, "DELETE"),
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/ai-configurations"] });
      toast({ title: "AI configuration deleted successfully" });
    },
    onError: (error: any) => {
      toast({ 
        title: "Failed to delete configuration", 
        description: error.message,
        variant: "destructive"
      });
    },
  });

  const getProviderIcon = (provider: string) => {
    switch (provider) {
      case 'anthropic': return '🤖';
      case 'openai': return '🧠';
      case 'azure': return '☁️';
      default: return '⚙️';
    }
  };

  const getProviderColor = (provider: string) => {
    switch (provider) {
      case 'anthropic': return 'bg-orange-100 text-orange-800 dark:bg-orange-900 dark:text-orange-200';
      case 'openai': return 'bg-green-100 text-green-800 dark:bg-green-900 dark:text-green-200';
      case 'azure': return 'bg-blue-100 text-blue-800 dark:bg-blue-900 dark:text-blue-200';
      default: return 'bg-gray-100 text-gray-800 dark:bg-gray-900 dark:text-gray-200';
    }
  };

  return (
    <DashboardLayout>
      <div className="container mx-auto p-6 space-y-6">
        <div className="flex items-center gap-3 mb-6">
          <SettingsIcon className="h-8 w-8" />
          <div>
            <h1 className="text-3xl font-bold">Settings</h1>
            <p className="text-muted-foreground">Manage your account and application preferences</p>
          </div>
        </div>

        <Tabs value={activeTab} onValueChange={setActiveTab} className="space-y-6">
          <TabsList className="grid w-full grid-cols-4">
            <TabsTrigger value="ai-settings" className="flex items-center gap-2">
              <Brain className="h-4 w-4" />
              OpenWeb 3.2 AI
            </TabsTrigger>
            <TabsTrigger value="account" className="flex items-center gap-2">
              <User className="h-4 w-4" />
              Account
            </TabsTrigger>
            <TabsTrigger value="notifications" className="flex items-center gap-2">
              <Bell className="h-4 w-4" />
              Notifications
            </TabsTrigger>
            <TabsTrigger value="security" className="flex items-center gap-2">
              <Shield className="h-4 w-4" />
              Security
            </TabsTrigger>
          </TabsList>

          <TabsContent value="ai-settings" className="space-y-6">
            <Card>
              <CardHeader>
                <CardTitle className="flex items-center gap-2">
                  <Brain className="h-5 w-5" />
                  OpenWeb 3.2 AI Configuration
                </CardTitle>
                <CardDescription>
                  Manage AI providers and configurations for content analysis and moderation.
                  Currently displayed as "OpenWeb 3.2 AI" throughout the platform.
                </CardDescription>
              </CardHeader>
              <CardContent>
                <div className="flex justify-between items-center mb-6">
                  <div className="text-sm text-muted-foreground">
                    {aiConfigurations?.length || 0} configuration(s) available
                  </div>
                  <Dialog open={isCreateDialogOpen} onOpenChange={setIsCreateDialogOpen}>
                    <DialogTrigger asChild>
                      <Button>Add Configuration</Button>
                    </DialogTrigger>
                    <DialogContent>
                      <DialogHeader>
                        <DialogTitle>Create AI Configuration</DialogTitle>
                        <DialogDescription>
                          Add a new AI provider configuration for content analysis.
                        </DialogDescription>
                      </DialogHeader>
                      <CreateConfigForm 
                        onSubmit={(config) => createConfigMutation.mutate(config)}
                        isLoading={createConfigMutation.isPending}
                      />
                    </DialogContent>
                  </Dialog>
                </div>

                {configsLoading ? (
                  <div className="text-center py-8">Loading configurations...</div>
                ) : (
                  <div className="grid gap-4">
                    {aiConfigurations?.map((config: AiConfiguration) => (
                      <Card key={config.id} className={`relative ${config.isActive ? 'ring-2 ring-primary' : ''}`}>
                        <CardHeader className="pb-3">
                          <div className="flex items-center justify-between">
                            <div className="flex items-center gap-3">
                              <span className="text-2xl">{getProviderIcon(config.provider)}</span>
                              <div>
                                <CardTitle className="text-lg">{config.name}</CardTitle>
                                <div className="flex items-center gap-2 mt-1">
                                  <Badge className={getProviderColor(config.provider)}>
                                    {config.provider.toUpperCase()}
                                  </Badge>
                                  <Badge variant="outline">{config.model}</Badge>
                                  {config.isActive && <Badge variant="default">Active</Badge>}
                                  {config.isDefault && <Badge variant="secondary">Default</Badge>}
                                </div>
                              </div>
                            </div>
                            <div className="flex items-center gap-2">
                              <Button
                                variant="outline"
                                size="sm"
                                onClick={() => testConfigMutation.mutate(config.id)}
                                disabled={testConfigMutation.isPending}
                              >
                                <Zap className="h-4 w-4 mr-2" />
                                Test
                              </Button>
                              {!config.isActive && (
                                <Button
                                  size="sm"
                                  onClick={() => activateConfigMutation.mutate(config.id)}
                                  disabled={activateConfigMutation.isPending}
                                >
                                  Activate
                                </Button>
                              )}
                              <Button
                                variant="outline"
                                size="sm"
                                onClick={() => setEditingConfig(config)}
                              >
                                Edit
                              </Button>
                              <Button
                                variant="destructive"
                                size="sm"
                                onClick={() => deleteConfigMutation.mutate(config.id)}
                                disabled={config.isActive || deleteConfigMutation.isPending}
                              >
                                Delete
                              </Button>
                            </div>
                          </div>
                        </CardHeader>
                        <CardContent className="pt-0">
                          <div className="grid grid-cols-2 md:grid-cols-4 gap-4 text-sm">
                            <div>
                              <Label className="font-medium">Max Tokens</Label>
                              <p className="text-muted-foreground">{config.maxTokens}</p>
                            </div>
                            <div>
                              <Label className="font-medium">Temperature</Label>
                              <p className="text-muted-foreground">{config.temperature}</p>
                            </div>
                            <div>
                              <Label className="font-medium">Top P</Label>
                              <p className="text-muted-foreground">{config.topP}</p>
                            </div>
                            <div>
                              <Label className="font-medium">Created</Label>
                              <p className="text-muted-foreground">
                                {new Date(config.createdAt).toLocaleDateString()}
                              </p>
                            </div>
                          </div>
                        </CardContent>
                      </Card>
                    ))}
                  </div>
                )}

                {/* Usage Statistics */}
                {aiUsage && aiUsage.length > 0 && (
                  <div className="mt-8">
                    <h3 className="text-lg font-semibold mb-4">Usage Statistics</h3>
                    <div className="grid gap-4 md:grid-cols-2 lg:grid-cols-3">
                      {aiUsage.slice(0, 6).map((stat: AiUsageStats) => (
                        <Card key={stat.id}>
                          <CardHeader className="pb-2">
                            <CardTitle className="text-sm flex items-center gap-2">
                              <span>{getProviderIcon(stat.provider)}</span>
                              {stat.provider} - {stat.model}
                            </CardTitle>
                          </CardHeader>
                          <CardContent>
                            <div className="space-y-2 text-sm">
                              <div className="flex justify-between">
                                <span>Requests:</span>
                                <span className="font-medium">{stat.requestCount}</span>
                              </div>
                              <div className="flex justify-between">
                                <span>Success Rate:</span>
                                <span className="font-medium">
                                  {((stat.successCount / stat.requestCount) * 100).toFixed(1)}%
                                </span>
                              </div>
                              <div className="flex justify-between">
                                <span>Tokens Used:</span>
                                <span className="font-medium">{stat.totalTokensUsed.toLocaleString()}</span>
                              </div>
                              <div className="flex justify-between">
                                <span>Avg Response:</span>
                                <span className="font-medium">{stat.averageResponseTime.toFixed(0)}ms</span>
                              </div>
                            </div>
                          </CardContent>
                        </Card>
                      ))}
                    </div>
                  </div>
                )}

                {/* Recent Errors */}
                {aiErrors && aiErrors.length > 0 && (
                  <div className="mt-8">
                    <h3 className="text-lg font-semibold mb-4">Recent Errors</h3>
                    <Card>
                      <CardContent className="p-0">
                        <div className="divide-y">
                          {aiErrors.slice(0, 5).map((error: AiErrorLog) => (
                            <div key={error.id} className="p-4">
                              <div className="flex items-center justify-between">
                                <div className="flex items-center gap-2">
                                  <span>{getProviderIcon(error.provider)}</span>
                                  <Badge className={getProviderColor(error.provider)}>
                                    {error.provider}
                                  </Badge>
                                  <span className="text-sm font-medium">{error.model}</span>
                                  {error.httpStatus && (
                                    <Badge variant="destructive">HTTP {error.httpStatus}</Badge>
                                  )}
                                </div>
                                <span className="text-sm text-muted-foreground">
                                  {new Date(error.createdAt).toLocaleString()}
                                </span>
                              </div>
                              <p className="text-sm text-muted-foreground mt-2">{error.errorMessage}</p>
                            </div>
                          ))}
                        </div>
                      </CardContent>
                    </Card>
                  </div>
                )}
              </CardContent>
            </Card>
          </TabsContent>

          <TabsContent value="account" className="space-y-6">
            <Card>
              <CardHeader>
                <CardTitle>Account Information</CardTitle>
                <CardDescription>Manage your account details and preferences</CardDescription>
              </CardHeader>
              <CardContent>
                <div className="space-y-4">
                  <div>
                    <Label>Email Address</Label>
                    <Input disabled value="ceo@openweb.co.za" />
                  </div>
                  <div>
                    <Label>Account Type</Label>
                    <Input disabled value="Administrator" />
                  </div>
                  <div>
                    <Label>Plan</Label>
                    <Input disabled value="Free Plan (Beta)" />
                  </div>
                </div>
              </CardContent>
            </Card>
          </TabsContent>

          <TabsContent value="notifications" className="space-y-6">
            <Card>
              <CardHeader>
                <CardTitle>Notification Preferences</CardTitle>
                <CardDescription>Configure how you receive notifications</CardDescription>
              </CardHeader>
              <CardContent>
                <p className="text-muted-foreground">Notification settings will be available in a future update.</p>
              </CardContent>
            </Card>
          </TabsContent>

          <TabsContent value="security" className="space-y-6">
            <Card>
              <CardHeader>
                <CardTitle>Security Settings</CardTitle>
                <CardDescription>Manage your account security and privacy</CardDescription>
              </CardHeader>
              <CardContent>
                <p className="text-muted-foreground">Security settings will be available in a future update.</p>
              </CardContent>
            </Card>
          </TabsContent>
        </Tabs>

        {/* Edit Configuration Dialog */}
        {editingConfig && (
          <Dialog open={!!editingConfig} onOpenChange={() => setEditingConfig(null)}>
            <DialogContent>
              <DialogHeader>
                <DialogTitle>Edit AI Configuration</DialogTitle>
                <DialogDescription>
                  Update the AI configuration settings.
                </DialogDescription>
              </DialogHeader>
              <EditConfigForm 
                config={editingConfig}
                onSubmit={(config) => updateConfigMutation.mutate(config)}
                isLoading={updateConfigMutation.isPending}
              />
            </DialogContent>
          </Dialog>
        )}
      </div>
    </DashboardLayout>
  );
}

function CreateConfigForm({ onSubmit, isLoading }: { 
  onSubmit: (config: Partial<AiConfiguration>) => void;
  isLoading: boolean;
}) {
  const [formData, setFormData] = useState({
    name: "",
    provider: "",
    model: "",
    apiKey: "",
    systemPrompt: "You are an AI Moderator for Reddit and are moderating Subreddits to ensure people follow the rules.",
    maxTokens: 1000,
    temperature: 0.3,
    topP: 0.1,
  });

  const getDefaultModel = (provider: string) => {
    switch (provider) {
      case 'anthropic': return 'claude-sonnet-4';
      case 'openai': return 'gpt-4o';
      case 'azure': return 'gpt-4.1';
      default: return '';
    }
  };

  const handleProviderChange = (provider: string) => {
    setFormData({ 
      ...formData, 
      provider, 
      model: getDefaultModel(provider),
      name: `${provider.charAt(0).toUpperCase() + provider.slice(1)} Configuration`
    });
  };

  const handleSubmit = (e: React.FormEvent) => {
    e.preventDefault();
    onSubmit(formData);
  };

  return (
    <form onSubmit={handleSubmit} className="space-y-4">
      <div>
        <Label htmlFor="name">Configuration Name</Label>
        <Input
          id="name"
          value={formData.name}
          onChange={(e) => setFormData({ ...formData, name: e.target.value })}
          placeholder="My AI Configuration"
          required
        />
      </div>
      <div>
        <Label htmlFor="provider">Provider</Label>
        <Select value={formData.provider} onValueChange={handleProviderChange}>
          <SelectTrigger>
            <SelectValue placeholder="Select AI provider" />
          </SelectTrigger>
          <SelectContent>
            <SelectItem value="anthropic">Anthropic Claude</SelectItem>
            <SelectItem value="openai">OpenAI GPT</SelectItem>
            <SelectItem value="azure">Azure AI</SelectItem>
          </SelectContent>
        </Select>
      </div>
      {formData.provider && (
        <>
          <div>
            <Label htmlFor="model">Model</Label>
            <Input
              id="model"
              value={formData.model}
              onChange={(e) => setFormData({ ...formData, model: e.target.value })}
              placeholder="AI model version"
              required
            />
            <p className="text-sm text-muted-foreground mt-1">
              {formData.provider === 'anthropic' && 'Recommended: claude-sonnet-4'}
              {formData.provider === 'openai' && 'Recommended: gpt-4o'}
              {formData.provider === 'azure' && 'Recommended: gpt-4.1'}
            </p>
          </div>
          <div>
            <Label htmlFor="apiKey">API Key</Label>
            <Input
              id="apiKey"
              type="password"
              value={formData.apiKey}
              onChange={(e) => setFormData({ ...formData, apiKey: e.target.value })}
              placeholder="Enter your API key"
              required
            />
            <p className="text-sm text-muted-foreground mt-1">
              Your API key will be stored securely and used for moderation requests.
            </p>
          </div>
          <div>
            <Label htmlFor="systemPrompt">System Prompt for Moderation</Label>
            <textarea
              id="systemPrompt"
              className="flex min-h-[80px] w-full rounded-md border border-input bg-background px-3 py-2 text-sm ring-offset-background placeholder:text-muted-foreground focus-visible:outline-none focus-visible:ring-2 focus-visible:ring-ring focus-visible:ring-offset-2 disabled:cursor-not-allowed disabled:opacity-50"
              value={formData.systemPrompt}
              onChange={(e) => setFormData({ ...formData, systemPrompt: e.target.value })}
              placeholder="Instructions for the AI moderator..."
              rows={3}
              required
            />
            <p className="text-sm text-muted-foreground mt-1">
              This prompt will be sent to the AI with every moderation request.
            </p>
          </div>
        </>
      )}
      <DialogFooter>
        <Button type="submit" disabled={isLoading || !formData.provider}>
          {isLoading ? "Creating..." : "Create Configuration"}
        </Button>
      </DialogFooter>
    </form>
  );
}

function EditConfigForm({ config, onSubmit, isLoading }: { 
  config: AiConfiguration;
  onSubmit: (config: AiConfiguration) => void;
  isLoading: boolean;
}) {
  const [formData, setFormData] = useState({
    ...config,
    apiKey: config.apiKey || "",
    systemPrompt: config.systemPrompt || "You are an AI Moderator for Reddit and are moderating Subreddits to ensure people follow the rules.",
  });

  const handleSubmit = (e: React.FormEvent) => {
    e.preventDefault();
    onSubmit(formData);
  };

  return (
    <form onSubmit={handleSubmit} className="space-y-4">
      <div>
        <Label htmlFor="name">Configuration Name</Label>
        <Input
          id="name"
          value={formData.name}
          onChange={(e) => setFormData({ ...formData, name: e.target.value })}
          required
        />
      </div>
      <div>
        <Label htmlFor="provider">Provider</Label>
        <Input
          id="provider"
          value={formData.provider}
          disabled
          className="bg-muted"
        />
        <p className="text-sm text-muted-foreground mt-1">
          Provider cannot be changed after creation.
        </p>
      </div>
      <div>
        <Label htmlFor="model">Model</Label>
        <Input
          id="model"
          value={formData.model}
          onChange={(e) => setFormData({ ...formData, model: e.target.value })}
          required
        />
        <p className="text-sm text-muted-foreground mt-1">
          {formData.provider === 'anthropic' && 'Recommended: claude-sonnet-4'}
          {formData.provider === 'openai' && 'Recommended: gpt-4o'}
          {formData.provider === 'azure' && 'Recommended: gpt-4.1'}
        </p>
      </div>
      <div>
        <Label htmlFor="apiKey">API Key</Label>
        <Input
          id="apiKey"
          type="password"
          value={formData.apiKey}
          onChange={(e) => setFormData({ ...formData, apiKey: e.target.value })}
          placeholder="Enter your API key"
        />
        <p className="text-sm text-muted-foreground mt-1">
          Leave blank to keep existing key. Your API key is stored securely.
        </p>
      </div>
      <div>
        <Label htmlFor="systemPrompt">System Prompt for Moderation</Label>
        <textarea
          id="systemPrompt"
          className="flex min-h-[80px] w-full rounded-md border border-input bg-background px-3 py-2 text-sm ring-offset-background placeholder:text-muted-foreground focus-visible:outline-none focus-visible:ring-2 focus-visible:ring-ring focus-visible:ring-offset-2 disabled:cursor-not-allowed disabled:opacity-50"
          value={formData.systemPrompt}
          onChange={(e) => setFormData({ ...formData, systemPrompt: e.target.value })}
          placeholder="Instructions for the AI moderator..."
          rows={3}
          required
        />
        <p className="text-sm text-muted-foreground mt-1">
          This prompt will be sent to the AI with every moderation request.
        </p>
      </div>
      <DialogFooter>
        <Button type="submit" disabled={isLoading}>
          {isLoading ? "Updating..." : "Update Configuration"}
        </Button>
      </DialogFooter>
    </form>
  );
}